//
//  ISD51  In-System Debugger for 8051 based microcontrollers
//  Copyright KEIL ELEKTRONIK GmbH and Keil Software, Inc. 2000 - 2003
//  Version 2.04
//------------------------------------------------------------------------------

//#include "ch552_keil.h" 

/** ISD 51 configuration file

    This is a special version of the config file designed for use with WCH 
    micro controllers CH5xx. It defines an ISD_PORT to represent the debug
    port. This port can be assigned to uart0 or uart1. Baudrate can be adjusted
    from 9600 to 115200 assuming the device is running at 24 MHz.  
    Critical config values are validated to ensure no false configuration.
    There are a few extra macros for initializing the baudrate of the isd port.
    For more infos on ISD51 check the helpfile and the original files from Keil
    
    For testing the trafic of the isd port the LEDs can be used. Make sure to 
    switch them of by some timer if you use this feature.
     
    WARNING : when ISD_CBLK_SZ == 2 the debugger will be able to use 
              flash break points. If use this it can wear out the flosh very 
              quickly and should be used only when its neccesarry to debug at 
              fullspeed. In all other cases use ISD_CBLK_SZ == 0 to disable 
              flash break points or configure Vision not to us flash breaks.
    
    Open questions which the docs dont answer:
    - What should be done with AUXR this one is not available on CH55x devices
    - Is it save to use XBUS_AUX instead? 
    - For what purpose does Keil use AUXR? AUXR just enables ALE and XRAM on some
      devices 
    - Why does the original code include absacc.h? It should include intrins.h
    - Is it guaranteed that flashing will always start at even addresses this 
      would simplify the flashroutine    
*/

#include "ch552_keil.h"


#define RAMSIZE      0x100      // all ch55x devices have 256 bytes IDATA RAM
#define CMP_START    0x0000     // compare starts at 0   
#define CMP_END      0x37FF     // and ends at 0x37FF 
#define WCH_CHIP_ID  0x52       // set the correct chip
#define ISD_PORT     0x01       // 0 | 1 if the mcu has more than one uart 
#define ISD_BAUDRATE 0x00       // 0 .. 4 0 means 9600 Baud
#define ISD_CBLK_SZ  0          // just 0 or 2 is allowed
#define USE_LEDS     0          // set this to 0 to disable the leds

//#define ISD_INCLUDE  1        // use that to include ISD51 into the app 

// do some checks on the config values
#if WCH_CHIP_ID < 0x51 
  #error "This chip is not supportet!" 
#endif

#if WCH_CHIP_ID > 0x54 
  #error "This chip is not supportet!" 
#endif

#if ISD_BAUDRATE > 4
  #error "Bautrate is wrong!" 
#endif

#if WCH_CHIP_ID & 0x01 && ISD_PORT > 0 
	#error "This chip just supports uart 0!"
#endif	

#if ISD_CBLK_SZ != 0 && ISD_CBLK_SZ  != 2
  #error "Just blocksize 0 or 2 is allowed"
#endif	

#if ISD_PORT > 1
  #error "Error this uart is not supported"
#endif

#if ISD_PORT == 0 
  #define ISD_SCON SCON
  #define ISD_SBUF SBUF
  #define ISD_TI   TI 
  #define ISD_RI   RI 
  #define ISD_ES   ES
  #define SINTRVEC 0x0023
#endif  
#if ISD_PORT == 1
  #define ISD_SCON SCON1
  #define ISD_SBUF SBUF1
  #define ISD_TI   U1TI 
  #define ISD_RI   U1RI 
  #define ISD_ES   IE_UART1
  #define SINTRVEC 0x0053
#endif

#if ISD_BAUDRATE==0
  #define ISD_RELOAD  0-FREQ_SYS/16/9600
#elif ISD_BAUDRATE==1
  #define ISD_RELOAD  0-FREQ_SYS/16/19200
#elif ISD_BAUDRATE==2
  #define ISD_RELOAD  0-FREQ_SYS/16/38400
#elif ISD_BAUDRATE==3
  #define ISD_RELOAD  0-FREQ_SYS/16/57600
#elif ISD_BAUDRATE==4
  #define ISD_RELOAD  0-FREQ_SYS/16/115200
#endif
#if USE_LEDS == 1
  #define LED1         TXD        
  #define LED3         RXD            
#endif	

#ifndef __C51__
// some assembler macros controlling the ISD_UART
   
   CLR_TI   MACRO               ; Clear Transmit Interrupt Flag 
            CLR     ISD_TI      ; No CPU register may be changed here
            ENDM
   
   SET_TI   MACRO               ; Set   Transmit Interrupt Flag
            SETB    ISD_TI      ; No CPU register may be changed here
            ENDM
   
   JNB_TI   MACRO   label       ; Jump if Transmit Interrupt Flag not set
            JNB     ISD_TI,label; PSW may be modified without saving it
            ENDM
   
   WR_SBUF  MACRO               ; Write ACC to SBUF
            MOV     ISD_SBUF,A  ; ACC and PSW may be modified without saving it
#if USE_LEDS == 1
            SETB    LED3        ; place txled here            
#endif
            ENDM
   
   CLR_RI   MACRO               ; Clear Receiver Interrupt Flag 
            CLR     ISD_RI      ; No CPU register may be changed here
            ENDM
   
   JB_RI    MACRO   label       ; Jump if Receiver Interrupt Flag set
            JB      ISD_RI,label; ACC and PSW may be modified without saving it
            ENDM
   
   JNB_RI   MACRO   label       ; Jump if Receiver Interrupt Flag not set
            JNB     ISD_RI,label; ACC and PSW may be modified without saving it
            ENDM
   
   RD_SBUF  MACRO               ; Return SBUF in ACC
            MOV    A,ISD_SBUF   ; ACC and PSW may be modified without saving it
#if USE_LEDS == 1
            SETB   LED1         ; place rxled here            
#endif
            ENDM
   
   CLR_ES   MACRO               ; Disable Serial Interrupt
            CLR    ISD_ES       ; No CPU register may be changed here
            ENDM
   
   SET_ES   MACRO               ; Enable Serial Interrupt
            SETB   ISD_ES       ; No CPU register may be changed here
            ENDM
   
   JNB_ES   MACRO   label       ; Jump if Receiver Interrupt Flag not set
            JNB     ISD_ES,label; ACC and PSW may be modified without saving it
            ENDM
   
   SAVE_ES  MACRO               ; Save Serial Interrupt enable bit to Carry
            MOV    C,ISD_ES     ; ACC and PSW may be modified without saving it
            ENDM
   
   RESTO_ES MACRO               ; Restore Serial Interrupt enable bit from Carry
            MOV    ISD_ES,C     ; ACC and PSW may be modified without saving it
            ENDM
#endif

//------------------------------------------------------------------------------
//
//  ISD51 CODE MEMORY ACCESS FUNCTIONS FOR HARDWARE BREAKPOINTS 
//  ===========================================================
//
//  For devices that support IAP (In-Application Programming) ISD51 may modify
//  the code memory to insert CALL instructions for flash breakpoints.  The IAP
//  functions are configured below:
//
//  The macro CWRITE defines a flash erase and programming function that writes
//  new content to a Flash CODE ROM block.  The new content data are store in
//  on-chip IDATA RAM at address CBLK.  The Flash CODE ROM block address is
//  passed in the register A:R0 (A is MSB, R0 is LSB).

// ch55x dont need to erase flash and always write 16 bits to the flash
// therefore we set CBLK_SZ = 2


#ifndef __C51__

   // execute flashing
   CH55X_FLASH MACRO
        clr   E_DIS
        mov   ROM_CTRL, # ROM_CMD_WRITE
        mov   A, ROM_CTRL           ;
        xrl   A, # bROM_ADDR_OK     ;
        setb  E_DIS
   ENDM
      
// CH55x always write 16 bit flashdata on even addresses
// therefore blksize is set to 2
//
// - there is no info if DPTR can be used savely
// - its supposed the debuger may try to flash at odd addresses
// - on entry flashaddr is set in ACC/R0 (MSB in ACC LSB in R0)
// - on exit ACC contains 0 or error code
  
   CWRITE  MACRO           
        ;push  DPL
        ;push  DPH                 ;save dptr
        mov   DPH,A
        mov   DPL,R0               ;and load the flshaddr

        mov   SAFE_MOD, #0x55      ;
        mov   SAFE_MOD, #0xAA      ;
        orl   GLOBAL_CFG, #bCODE_WE;       
        
        mov   A,R0        
        jb    ACC.0, ODD_ADDRESS    ;check even address

        mov   ROM_ADDR_H,DPH
        mov   ROM_ADDR_L,DPL
        
        mov   R0,#CBLK
        mov   ROM_DATA_H, @R0       ;Hi byte comes from CBLK buffer
        inc   R0 
        mov   ROM_DATA_L, @R0       ;lo byte comes from CBLK buffer
        
        mov   A, ROM_CTRL           ;chk addr ok
        jnb   ACC.6, Exit           ;

        CH55X_FLASH                 ;execute flash macro

        sjmp  Exit

ODD_ADDRESS:
	      dec   DPL                   ;make dptr even
        clr   A                     ;read a byte from cmem 
        movc  A,@A+DPTR             ;  -> will become our msb data
        mov   ROM_ADDR_H, DPH       ;setup the flash address
        mov   ROM_ADDR_L, DPL       ;
        
        mov   ROM_DATA_H,A          ;hi byte comes from cmem
        mov   R0,#CBLK
        mov   ROM_DATA_L, @R0       ;lo byte comes from CBLK buffer
        
        mov   A, ROM_CTRL           ;chk addr ok
        jnb   ACC.6, Exit           ;

        CH55X_FLASH                 ;execute the flash macro

        inc   DPTR                  ;next even addr
        inc   DPTR
        mov   ROM_ADDR_H, DPH       ;setup the flash address
        mov   ROM_ADDR_L, DPL       ;
        inc   R0                    ;the second byte from buffer
        mov   ROM_DATA_H,@R0        ; -> will become our msb
        
        inc   DPTR
        clr   A                     ;read a byte from cmem 
        movc  A,@A+DPTR             ;  -> will become our lsb data
        mov   ROM_DATA_L,A          ; lo byte comes from cmmem

        mov   A, ROM_CTRL           ;chk addr ok
        jnb   ACC.6, Exit           ;

        CH55X_FLASH                 ;execute the flash macro

Exit:   
	      mov   SAFE_MOD, #0x55      ;
        mov   SAFE_MOD, #0xAA      ;
        anl   GLOBAL_CFG, #not bCODE_WE;       

        ;pop   dph
        ;pop   dpl 
        ENDM

#endif


//------------------------------------------------------------------------------
//
//  ISD51 specific Serial User I/O
//  ==============================
//
//  ISD51 offers serial user input/output functions that work via the serial
//  interface that is used for debugging.  The serial user I/O functions are
//  configured below.

//  enable/disable ISD51 specific putchar function for user output.
#define  ISD_PUTCHAR  1    //  set to 0 to disable the putchar function.
                           //  set to 1 to enable the putchar function.

//  enable/disable ISD51 specific _getkey & _iskey function for user input.
#define  ISD_GETKEY   1    //  set to 0 to disable the _getkey/_iskey function.
                           //  set to 1 to enable the _getkey/_iskey function.

//------------------------------------------------------------------------------
//
//  ISD51 Functions
//  ===============
//
//  The following C Macro define the IDS51 functions
#ifdef __C51__

//#include <absacc.h>  //why should i include that?
#include <intrins.h>
extern void __isd_init (void);   // ISD51 core init function

#define ISDinit()                                                      \
  if (ISD_ES == 0)                                                     \
  {                      /* Is ISD51 UART already initialized?     */  \
    __isd_init ();       /* Init ISD51 core & enable ISD interrupt */  \
  }

#define ISDwait()                                                      \
  while (1)                                                            \
  {                                                                    \
    if (ISD_RI)                                                        \  
    {                    /* wait until Debugger sends 0xA5 */          \
      if (ISD_SBUF == 0xA5)  break;                                    \    
      ISD_RI = 0;                                                      \
    }                                                                  \
  }                                                                    \
  __isd_init ();         /* Init ISD51 core & enable ISD interrupt */

#define ISDcheck()                                                     \
  if (ISD_ES == 0)                                                     \
  {                      /* Is ISD51 UART already initialized?     */  \
    if (ISD_RI)                                                        \
    {                    /* wait until Debugger sends 0xA5 */          \
      if (ISD_SBUF != 0xA5) ISD_RI = 0;                                \
      else              __isd_init (); /* Init core & ISD interrupt */ \
    }                                                                  \
  }

#if ISD_PORT ==0 /*init for UART0*/
#define ISDInitUart()                                                  \
   TMOD  = (TMOD &~ MASK_T1_MOD) | bT1_M1; /*auto reload 8 bit timer*/ \
   T2MOD = T2MOD | bTMR_CLK | bT1_CLK;     /*runing at fsys */         \ 
   TH1   = ISD_RELOAD;           /* load the calculated reload value*/ \  
   TR1   = 1;                                                          \
   ISD_SCON  = 0x50;                        /* 8bit var baudrate     */\
   ISD_ES= 1;                               /* enable the irq        */\ 
   PS    = 1;                               /* high priority         */
#endif

#if ISD_PORT ==1  /*init for UART1*/
#define ISDInitUart()                                                  \
   ISD_SCON = 0x30;                         /* 8bit var baudrate     */\ 
   SBAUD1   = ISD_RELOAD;        /* load the calculated reload value */\
   IE_UART1 = 1;                                                       \
   IP_EX   |= bIP_UART1;                    /* high priority         */
#endif
     
#define ISDdisable()     /* Disable ISD Interrupt                  */  \
    ISD_ES = 0;

#define ISDenable()      /* Enable  ISD Interrupt                  */  \
    ISD_ES = 1;

#define ISDbreak()       /* hard-code program stop (breakpoint) */     \
    ISD_TI = 1;              /* Enter ISD Interrupt function */        \
    _nop_ ();

#if ISD_GETKEY            /* user input function                */          
extern bit _iskey (void); /* check if input character available */
#endif

#endif

// #define ISDInitUart() /*this is a new one*/
//-----------------------------------------------------------------------------
//---------------- !!! End of User Configuration Part    !!! ------------------
//-----------------------------------------------------------------------------

