/* Taktgenerator SI5351A - Berechnungen */

/*
   Verwendet wird hier ein 27 MHz VCTCXO.
   Siehe auch: Silabs AN619
*/

#include "StdTypes.h"
#include "si5351a_ll.h"
#include "radio.h"
#include "si5351a.h"
#include <stdlib.h>
#include <math.h>
#include "systick.h"


/* Blockdefinition fr die Multisynths */
struct TPll_Block
{ byte P3_hi;          // Bits 15..8
  byte P3_lo;          // Bits 7..0
  byte P1_top;         // idxOutDiv, ena Div:4, P1_17..16
  byte P1_hi;          // Bits 15..8
  byte P1_lo;          // Bits 7..0
  byte P32_top;        // P3 und P2 Bits 19..16
  byte P2_hi;          // Bits 15..8
  byte P2_lo;          // Bits 7..0
};


/* interne Variablen */
byte Register3;        // Output-Enables
long Frequenz_VCOA;    // 600..900 MHz erlaubt
long Frequenz_VCOB;    // 600..900 MHz erlaubt
long TeilerClock0;     // Merker fr VCOA/CLOCK0
long TeilerClock1;     // Merker fr VCOB/CLOCK1


void Pll_Reset (void)
{ byte nix;

  nix = (1<<7) | (1<<5);
  Write_SI5351 (177, &nix, 1);     // PLL Reset
}



/* die Vorbelegungen der generellen Register des Chips */
const byte RegMap_ab_R15 [11] =
{  0x00, // Register 15 Clockteiler und Sourcen fr PLLA+B

         // Register 16 fr CLK0
  (0<<7) | // PowerDown: CLK0 power-up
  (1<<6) | // Multisynth_0 in Integermode
  (0<<5) | // PLLA fr Multisynth_0
  (0<<4) | // CLK0 nicht invertiert
  (3<<2) | // Multisynth_0 treibt CLK0
  (1<<0),  // CLK0 mit 4 mA Strke

         // Register 17 fr CLK1
  (0<<7) | // PowerDown: CLK1 power-up
  (1<<6) | // Multisynth_1 in Integermode
  (1<<5) | // PLLB fr Multisynth_1
  (0<<4) | // CLK1 nicht invertiert
  (3<<2) | // Multisynth_1 treibt CLK1
  (1<<0),  // CLK1 mit 8 mA Strke

         // Register 18 fr CLK2
  (0<<7) | // PowerDown: CLK2 power-up
  (1<<6) | // Multisynth_2 in Integermode
  (1<<5) | // PLLB fr Multisynth_2
  (0<<4) | // CLK2 nicht invertiert
  (3<<2) | // Multisynth_2 treibt CLK2
  (1<<0),  // CLK2 mit 8 mA Strke

  0x80,  // Register 19 fr CLK3
  0x80,  // Register 20 fr CLK4
  0x80,  // Register 21 fr CLK5
  0x80,  // Register 22 fr CLK6
  0x80,  // Register 23 fr CLK7

  0x00,  // Register 24 CLK0..3 Pegel wenn disabled
  0x00   // Register 25 CLK4..7 Pegel wenn disabled
};


bool SetzeRegister3 (void)
{ return Write_SI5351(3, &Register3, 1); }

/* die Clock-Ausgnge ein- und ausschalten */
bool Clock0_aus (void)
{ Register3 |= 1;  return SetzeRegister3(); }

bool Clock1_aus (void)
{ Register3 |= 2;  return SetzeRegister3(); }

bool Clock2_aus (void)
{ Register3 |= 4;  return SetzeRegister3(); }

bool Clock0_ein (void)
{ Register3 &= ~1;  return SetzeRegister3(); }

bool Clock1_ein (void)
{ Register3 &= ~2;  return SetzeRegister3(); }

bool Clock2_ein (void)
{ Register3 &= ~4;  return SetzeRegister3(); }




/*
merke zu den MultiSynth-Teilern:
erlaubte Teilerverhltnisse sind 4, 6, 8, 9 .. 900 .. <901
(zwischen 8.0 und 900.9 sind fraktionale Teilerverh. erlaubt)

Die VCO's knnen zwischen 600 MHz und 900 MHz schwingen.

P2 = 0, P3 = 1 forces an integer value for the divider
*/

#define F_max      150000000     // maximal 150 MHz (da Teiler mindestens 6)
#define F_min         666667     // minimal geht 600 MHz / 900 --> 0.666666 MHz
#define F_outmin        5208     // 0.666 MHz / 128 --> 5.208 kHz


bool MacheMultisynthRecord (struct TPll_Block* aSynth, long Fvco, long Fsynt)
{ bool erg;
  long A, B, C;            // merke: Fvco = Fsynt * (A + B/C)  // AN619 Ch.3.2
  ldiv_t T;
  float f;
  long P1, P2, P3, X;


  A = Fvco;
  T = ldiv(Fvco, Fsynt);    // T enthlt Quotient und Rest
  A = T.quot;               // A ist der ganzzahlige Teil

  if (T.rem==0)             // es ist ganzzahlig
  { P1 = 128 * A - 512;
    P2 = 0;
    P3 = 1;
  }
  else
  { f = T.rem;                // der integer Rest, ist kleiner als Fsynt
    f = f / Fsynt;            // jetzt ist f = B/C und damit < 1.0

     //  Fr C whlen wir etwas, das sich gut teilen lt:
     //  2*2*3*5*7*11*13*17 = 1021020 = 0xF945C;
    C = 1021020;
    B = f * C;                // B ist damit immer kleiner als C

     // P1..3 berechnen
    X = floor(128 * f);  // f war B/C
    P1 = 128 * A + X - 512;
    P2 = 128 * B - C * X;
    P3 = C;
  }

  // Struct zusammensetzen
  aSynth->P3_hi   = (P3 >> 8) & 0xFF;            // Bits 15..8
  aSynth->P3_lo   =  P3 & 0xFF;                  // Bits 7..0
  aSynth->P1_top  = (P1 >>16) & 3;               // (idxOutDiv, ena Div:4,) P1_top
  aSynth->P1_hi   = (P1 >> 8) & 0xFF;            // Bits 15..8
  aSynth->P1_lo   = P1 & 0xFF;                   // Bits 7..0
  aSynth->P32_top = (((P3 >> 16) & 0x0F) << 4) | // P3 und P2 Bits 19..16
                    ((P2 >> 16) & 0x0F);
  aSynth->P2_hi   = (P2 >> 8) & 0xFF;            // Bits 15..8
  aSynth->P2_lo   =  P2 & 0xFF;                  // Bits 7..0

  erg = true;
  return erg;
}

bool TesteVCO (long aFrequenz)
{ if (aFrequenz < 600000000) return false;
  if (aFrequenz > 800000000) return false;
  return true;
}


// LO abstimmen
bool SetzeClock0 (long Frequenz)
{ long VCO;
  long Teiler;
  bool erg, ok;
  byte nix;
  struct TPll_Block Synth;

  VCO = TeilerClock0 * Frequenz;
  ok  = TesteVCO(VCO);

  if (!ok)                // neues Teilerverhltnis aufsetzen
  { Teiler = (700000000 / Frequenz) & 0xFFFFFFFE;
    VCO = Teiler * Frequenz;
    erg = TesteVCO(VCO);
    if (!erg) return false;

    TeilerClock0  = Teiler;
    Frequenz_VCOA = VCO;

    // Ausgangsteiler aufsetzen
    erg = MacheMultisynthRecord (&Synth, VCO, Frequenz);
    if (!erg) return erg;
    erg = Write_SI5351 (42, &Synth, 8);  // CLK0 Synth aufsetzen
    if (!erg) return erg;
  }

  // PLLA aufsetzen
  erg = MacheMultisynthRecord (&Synth, VCO, JustageWerte.Mutterfrequenz);
  if (!erg) return erg;

  erg = Write_SI5351 (26, &Synth, 8);     // PLL1: VCOA einrichten
  if (!erg) return erg;

  if (!ok)
  { nix = (1<<5);
    erg = Write_SI5351 (177, &nix, 1);     // PLLA Reset
    if (!erg) return erg;
  }

  erg = Clock0_ein();
  return erg;
}

bool SetzeClock1 (long Frequenz)
{ long VCO;
  long Teiler;
  bool erg, ok;
  byte nix;
  struct TPll_Block Synth;

  VCO = TeilerClock1 * Frequenz;
  ok  = false; //TesteVCO(VCO);

  if (!ok)                // neues Teilerverhltnis aufsetzen
  { Teiler = (700000000 / Frequenz) & 0xFFFFFFFE;
    VCO = Teiler * Frequenz;
    erg = TesteVCO(VCO);
    if (!erg) return false;

    TeilerClock1  = Teiler;
    Frequenz_VCOB = VCO;

    // Ausgangsteiler aufsetzen
    erg = MacheMultisynthRecord (&Synth, VCO, Frequenz);
    if (!erg) return erg;
    erg = Write_SI5351 (50, &Synth, 8);  // CLK1 Synth aufsetzen
    if (!erg) return erg;
  }

  // PLLB aufsetzen
  erg = MacheMultisynthRecord (&Synth, VCO, JustageWerte.Mutterfrequenz);
  if (!erg) return erg;

  erg = Write_SI5351 (34, &Synth, 8);     // PLL2: VCOB einrichten
  if (!erg) return erg;

  if (!ok)
  { nix = (1<<7);
    erg = Write_SI5351 (177, &nix, 1);     // PLLA Reset
    if (!erg) return erg;
  }

  erg = Clock1_ein();
  return erg;
}


/* Clock2 darf den VCOB nicht verndern
da der Teiler max. 900 sein darf und
der VCOB bei etwa 700 MHz liegt,
mu das Ausgangssignal :2 geteilt werden
700MHz : 900 --> 777 kHz
777 kHz : 2 --> 388 kHz,
das reicht fr den BFO
im Bereich 450..470 kHz
*/

bool SetzeClock2 (long Frequenz)
{ long F;
  bool erg;
  struct TPll_Block Synth;

  F = Frequenz * 2;
  // Ausgangsteiler aufsetzen

  erg = MacheMultisynthRecord (&Synth, Frequenz_VCOB, F);
  if (!erg) return erg;
  Synth.P1_top |= (1<<4);              // Ausgangsteiler :2 setzen
  erg = Write_SI5351 (58, &Synth, 8);  // CLK2 Synth aufsetzen
  if (!erg) return erg;

  erg = Clock2_ein();
  return erg;
}


bool SI5351_Init (void)
{ bool erg;
  byte nix;
  long L;
  struct TPll_Block Synth;

  /******* alles Clocks erstmal aus **********/
  Register3 = 0;
  erg = SetzeRegister3();
  if (!erg) return erg;

  /******* alle generellen Register setzen ***/
  erg = Write_SI5351(15, (void*) &RegMap_ab_R15, sizeof(RegMap_ab_R15));
  if (!erg) return erg;

  /******* Spread-Spectrum ausdrcklich ausstellen ***/
  nix = 0;
  erg = Write_SI5351(149, &nix, 1);
  if (!erg) return erg;

  /******* Kondensator setzen *****/
  nix = (3<<6) + 0x12;              // = default
  erg = Write_SI5351(183, &nix, 1);
  if (!erg) return erg;

  /******* die 3 Clocks erstmal auf Defaultwerte setzen *****/
  /*
  Dazu setzen wir zunchst eine uns passend erscheinende
  VCO-Frequenz aus, denn im Gegensatz zu SiLabs heit es,
  da die VCO-Frequenzen bei gleichbleibendem Ausgangsteiler
  ohne PLL-Reset und damit ohne Knacken gendert werden knnen
  und: da die VCO's eher von weniger als 400 MHz bis etwa 750 MHz
  gehen.

  Also sollte VCOA und VCOB erst einmal bei etwa 700 MHz liegen.
  */


  Frequenz_VCOA =
  Frequenz_VCOB = 700000000;

  erg = MacheMultisynthRecord (&Synth, Frequenz_VCOA, JustageWerte.Mutterfrequenz);
  if (!erg) return erg;

  erg = Write_SI5351 (26, &Synth, 8);     // PLL1: VCOA einrichten
  if (!erg) return erg;
  Warte_ms(10);

  erg = Write_SI5351 (34, &Synth, 8);     // PLL2: VCOB einrichten
  if (!erg) return erg;

  // so, beide PLL's sollten jetzt auf 700 MHz stehen
  // jetzt knnen wir die gewhnlichen Einstellungen
  // durchfhren. Die Frequenzen sind Default-Frequenzen

  L = (Frequenz_VCOA / 58112500 + 10000000) & 0xFFFFFFFE;
  TeilerClock0  = L;
  Frequenz_VCOA = L * 58112500 + 10000000;
  erg = SetzeClock0 (58112500 + 10000000);       // CLK0 auf 10 MHz Radio-Input
  if (!erg) return erg;

  L = (Frequenz_VCOB / 58112500 - 455000) & 0xFFFFFFFE;
  TeilerClock1 = L;
  Frequenz_VCOB = L * 58112500 - 455000;
  erg = SetzeClock1 (58112500 - 455000);          // CLK1 auf  58 MHz
  if (!erg) return erg;

  erg = SetzeClock2 (455000);            // CLK2 auf 455 kHz
  if (!erg) return erg;
  Warte_ms(10);

  // einmalig PLL-Reset fr beide PLL's
  nix = (1<<7) | (1<<5);
  erg = Write_SI5351 (177, &nix, 1);     // PLL Reset
  if (!erg) return erg;

  // Last-C und 'geheime' PLL-Bandbreite aufsetzen
  nix = (2<<6) | (1<<4) | (1<<1);        // (CL=8pF) (PLLB) (PLLA)
  erg = Write_SI5351 (183, &nix, 1);
  Warte_ms(10);

  return erg;
}



  /***/
