#include "CombieAdc.h"
#include <avr/power.h>

namespace Combie
{
    
    
    AdcCallBack Adc::callback = nullptr;
    
    #if defined(ADC_MODELL_MEGA_BIG)   ||  defined(ADC_MODELL_MEGA_U4)    
      byte  Adc::lastMux = 0;
    #endif    
    
    
    void Adc::isr()
    {
      int value = readValue();
      if(callback) callback(value);
    }
    
    
    Adc & Adc::setCallBack(AdcCallBack callback)
    {
      Adc::callback = callback;
      return *this;
    }
    
    Adc & Adc::setReference(AnalogReference reference, bool wait)
    {
    
      byte mux = ADMUX;
      if((mux & refMask) != reference)   // Referenz unveraendert
      {
        ADMUX = (mux & ~refMask) | reference; // neue Referenz setzen
        if(wait) delay(10);                     // settle Time
      }
      return *this;
    }
    
    
    
    #if defined(ADC_MODELL_MEGA_BIG) || defined(ADC_MODELL_MEGA_U4)
      Adc &  Adc::setSource(AnalogSource source)
      {
        lastMux = source;
        ADMUX =(ADMUX & ~sourceMask) | (source & sourceMask);
        ADCSRB = (source & _BV(5))?ADCSRB | _BV(MUX5):ADCSRB & ~_BV(MUX5);
        return *this;
      }
    #endif
    
    #if defined(ADC_MODELL_MEGA_SMALL) ||  defined(ADC_MODELL_TINY_BIG)
      Adc &  Adc::setSource(AnalogSource source)
      {
        ADMUX = (ADMUX & ~sourceMask) | source;
        return *this;
      }
    #endif
    
    
    
    Adc & Adc::setTrigger(TriggerSource source)
    {
      ADCSRB = (ADCSRB & ~trigMask) | source;
      return *this;
    }
    
    Adc & Adc::setClockDivisor(AnalogClockDivisor divisor)
    {
       ADCSRA = (ADCSRA & ~divMask) | divisor;
       return *this;
    }
    
    Adc & Adc::setClockDivisor()
    {
       setClockDivisor(ADC_DEFAULT_CLOCK_PRESCALER);
       return *this;
    }
    
    Adc & Adc::setResolution(AnalogResolution resolution)
    {
       ADMUX = (ADMUX & ~resMask) | resolution;
       return *this;
    }
    
    Adc & Adc::enable()
    {
       power_adc_enable();
       ADCSRA |= _BV(ADEN);  
       return *this;
    }
    
    Adc & Adc::reset() // Kaltstart Zustand wieder herstellen
    {
       power_adc_enable();
       ADCSRA = 0;
       ADCSRB = 0;
       ADMUX  = 0;
       return *this;
    }
    
    Adc & Adc::disable()
    {
       ADCSRA &= ~_BV(ADEN);
       power_adc_disable();
       return *this;
    }
    
    #if defined(ADC_MODELL_MEGA_BIG)
      int Adc::readValue()
      {
    //    byte muxbits = ADMUX & 0b00011111; // sourceMask
        if((lastMux > MUX_ADC7) && (lastMux < MUX_REF)) // differenzalinput
        {
          int value = ADCW;
          return (value & _BV(9))?value | 0b1111111000000000:value; // Vorzeichen erweitern
        }
        return (ADMUX&_BV(ADLAR))?ADCH:ADCW;
      }
    #endif
    
    #if defined(ADC_MODELL_MEGA_U4)
      int Adc::readValue()
      {
    //    byte muxbits = ADMUX & 0b00011111; // sourceMask
        if(((lastMux > MUX_ADC7) && (lastMux < MUX_REF)) || (lastMux == MUX_ADC1_ADC0_G40) || (lastMux > MUX_THERMO)) // differenzalinput
        {
          int value = ADCW;
          return (value & _BV(9))?value | 0b1111111000000000:value; // Vorzeichen erweitern
        }
        return (ADMUX&_BV(ADLAR))?ADCH:ADCW;
      }
    #endif
    
    
    
    #if defined(ADC_MODELL_MEGA_SMALL)
      int Adc::readValue()
      {
         return (ADMUX&_BV(ADLAR))?ADCH:ADCW;
      }
    #endif
    
    
    #if defined(ADC_MODELL_TINY_BIG)
      int Adc::readValue()
      {
        if(ADCSRB & _BV(7)) // bipolar differenzal input
        {
          int value = ADCW;
          return (value & _BV(9))?value | 0b1111111000000000:value; // Vorzeichen erweitern
        }
       // int reverse = (ADCSRB & _BV(IPR))?-1:1; // Input Polarity Reversal
        return ((ADMUX&_BV(ADLAR))?ADCH:ADCW);// * reverse;
      }
    #endif
    
    
    
    
    int Adc::getValue()
    {
       startConversion();
       while(ADCSRA & _BV(ADSC)); // warte auf Wandlung fertig
       return readValue();
    }
    
    int Adc::operator()()
    {
      return getValue();
    }
    
    Adc::operator int()
    {
      return getValue();
    }
    
    int Adc::operator()(AnalogSource source)
    {
      setSource(source);
      return getValue();
    }
    
    int Adc::operator[](AnalogSource source)
    {
      setSource(source);
      return getValue();
    }
    
    bool Adc::getIrqFlag()
    {
      return ADCSRA & _BV(ADIF);
    }
    
    Adc & Adc::enableIrq()
    {
      ADCSRA |= _BV(ADIE);  
      return *this;
    
    }
    
    Adc & Adc::disableIrq()
    {
      ADCSRA &= ~_BV(ADIE);
      return *this;
    }
    
    Adc & Adc::enableAutoTrigger()
    {
      ADCSRA |= _BV(ADATE);  
      return *this;
    }
    
    Adc & Adc::disableAutoTrigger()
    {
      ADCSRA &= ~_BV(ADATE);
      return *this;
    }
    
    Adc & Adc::startConversion()
    {
      ADCSRA |= _BV(ADSC);       // Wandlung starten
      return *this;
    }
    
    #if defined(ADC_MODELL_MEGA_U4)
      Adc & Adc::enableHighSpeed()
      {
         ADCSRB |= _BV(ADHSM); 
         return *this;
      } 
      
      Adc & Adc::disableHighSpeed()
      {
         ADCSRB &= ~_BV(ADHSM); 
         return *this;
      }
    #endif 
    
    
    
    
    #if defined(ADC_MODELL_TINY_BIG)
      Adc & Adc::enableBipolar()
      {
         ADCSRB |= _BV(7); // BIN
         return *this;
      } 
      Adc & Adc::disableBipolar()
      {
         ADCSRB &= ~_BV(7); // BIN
         return *this;
      }
      Adc & Adc::enablePolRev()
      {
         ADCSRB |= _BV(IPR); 
         return *this;
      } 
      Adc & Adc::disablePolRev()
      {
         ADCSRB &= ~_BV(IPR);
         return *this;
      }   
   #endif 

}



