#pragma once
#include <Arduino.h>   

// ATMega328P Datasheet
// http://www.atmel.com/Images/Atmel-42735-8-bit-AVR-Microcontroller-ATmega328-328P_Datasheet.pdf

// ATMega2560 Datasheet
// http://www.atmel.com/Images/Atmel-2549-8-bit-AVR-Microcontroller-ATmega640-1280-1281-2560-2561_datasheet.pdf

// ATMega32U4  Datasheet
// http://www.atmel.com/Images/Atmel-7766-8-bit-AVR-ATmega16U4-32U4_Datasheet.pdf

// ATTiny85 Datasheet
// http://www.atmel.com/images/atmel-2586-avr-8-bit-microcontroller-attiny25-attiny45-attiny85_datasheet.pdf

/**
 * 
 * Prozessor Check
 * http://www.atmel.com/webdoc/avrlibcreferencemanual/using_tools_1using_avr_gcc_mach_opt.html
*/

#if       defined(__AVR_ATmega640__)  || defined(__AVR_ATmega1280__) || defined(__AVR_ATmega2560__)

  #define ADC_MODELL_MEGA_BIG
  
#elif     defined(__AVR_ATmega88__)   || defined(__AVR_ATmega168__)

  #define ADC_MODELL_MEGA_SMALL
  
#elif     defined(__AVR_ATmega16U4__)   || defined(__AVR_ATmega32U4__)

  #define ADC_MODELL_MEGA_U4
  #define ADC_MODELL_WITH_THERMO
  
#elif     defined(__AVR_ATmega328P__) || defined(__AVR_ATmega168P__)

  #define ADC_MODELL_MEGA_SMALL
  #define ADC_MODELL_WITH_THERMO
  
#elif     defined(__AVR_ATtiny85__)   || defined(__AVR_ATtiny45__)   || defined(__AVR_ATtiny25__)

  #define ADC_MODELL_TINY_BIG
  #define ADC_MODELL_WITH_THERMO
  
#else 
  #error Dieser ADC wird leider (noch) nicht unterstuetzt 
#endif  




#ifndef ADC_DEFAULT_CLOCK_PRESCALER
  #if   F_CPU > 6400000
    #define ADC_DEFAULT_CLOCK_PRESCALER  Adc::DIV_128   
  #elif F_CPU > 3200000 
    #define ADC_DEFAULT_CLOCK_PRESCALER  Adc::DIV_64   
  #elif F_CPU > 1600000 
    #define ADC_DEFAULT_CLOCK_PRESCALER  Adc::DIV_32 
  #elif F_CPU >  800000 
    #define ADC_DEFAULT_CLOCK_PRESCALER  Adc::DIV_16 
  #elif F_CPU >  400000 
    #define ADC_DEFAULT_CLOCK_PRESCALER  Adc::DIV_8   
  #elif F_CPU >  200000 
    #define ADC_DEFAULT_CLOCK_PRESCALER  Adc::DIV_4   
  #else 
    #define ADC_DEFAULT_CLOCK_PRESCALER  Adc::DIV_2   
  #endif
#endif




#define ADC_ISR_STR(x) #x
#define ADC_ISR_HLP_STR(x) ADC_ISR_STR(x)


namespace Combie
{
    
    typedef void (*AdcCallBack)(int value);
    
    class Adc  
    {
      
      private:
      static AdcCallBack callback;
      static void isr() __asm__(ADC_ISR_HLP_STR(ADC_vect)) __attribute__((__signal__, __used__, __externally_visible__));
    
      protected:
        static int readValue();
        
    #if defined(ADC_MODELL_MEGA_BIG) || defined(ADC_MODELL_MEGA_U4)    
        static byte lastMux;
    #endif  
     
      
        const byte divMask    = 0b00000111;
        
    #if defined(ADC_MODELL_MEGA_BIG) || defined(ADC_MODELL_MEGA_SMALL) || defined(ADC_MODELL_TINY_BIG)
        const byte trigMask   = 0b00000111;
    #endif
        
    #if  defined(ADC_MODELL_MEGA_U4)
        const byte trigMask   = 0b00001111;
    #endif
    
        const byte resMask    = 0b00100000;
    
    #if defined(ADC_MODELL_MEGA_BIG) || defined(ADC_MODELL_MEGA_SMALL) || defined(ADC_MODELL_MEGA_U4)
        const byte refMask    = 0b11000000;
    #endif
    
    #if defined(ADC_MODELL_TINY_BIG)
        const byte refMask    = 0b11010000;
    #endif
        
    #if defined(ADC_MODELL_MEGA_BIG) || defined(ADC_MODELL_MEGA_U4)
        const byte sourceMask = 0b00011111;
    #endif
    
    #if defined(ADC_MODELL_MEGA_SMALL) || defined(ADC_MODELL_TINY_BIG) 
        const byte sourceMask = 0b00001111;
    #endif
    
        
        
    
      public:
        enum AnalogResolution : byte
        {
          RES_10BIT = 0b00000000,  
          RES_8BIT  = 0b00100000,  
        };
        
        
        enum AnalogClockDivisor : byte
        {
          DIV_2   = 0b00000001,
          DIV_4   = 0b00000010,
          DIV_8   = 0b00000011,
          DIV_16  = 0b00000100,
          DIV_32  = 0b00000101,
          DIV_64  = 0b00000110,
          DIV_128 = 0b00000111,
        };
    
     
    
    
    #if defined(ADC_MODELL_MEGA_BIG)
        enum AnalogReference : byte 
        {
          REF_EXT = 0b00000000,
          REF_VCC = 0b01000000,
          REF_11  = 0b10000000,
          REF_256 = 0b11000000,
        };
    #endif
        
    #if defined(ADC_MODELL_MEGA_SMALL)
        enum AnalogReference : byte 
        {
          REF_EXT = 0b00000000,
          REF_VCC = 0b01000000,
          REF_11  = 0b11000000,
        };
    #endif   
     
    #if defined(ADC_MODELL_MEGA_U4)
        enum AnalogReference : byte 
        {
          REF_EXT  = 0b00000000,
          REF_VCC  = 0b01000000,
          REF_256  = 0b11000000,
        };
    #endif    
        
    #if defined(ADC_MODELL_TINY_BIG)
        enum AnalogReference : byte 
        {
          REF_VCC   = 0b00000000,
          REF_EXT   = 0b01000000,
          REF_11    = 0b11000000,
          REF_256   = 0b10010000,
          REF_256_C = 0b11010000,
        };
    #endif    
    
    
    #if defined(ADC_MODELL_MEGA_BIG)
        enum AnalogSource : byte
        {
          MUX_ADC0    = 0b00000000,
          MUX_ADC1    = 0b00000001,
          MUX_ADC2    = 0b00000010,
          MUX_ADC3    = 0b00000011,
          MUX_ADC4    = 0b00000100,
          MUX_ADC5    = 0b00000101,
          MUX_ADC6    = 0b00000110,
          MUX_ADC7    = 0b00000111, 
    
          MUX_ADC0_ADC0_G10    = 0b00001000, 
          MUX_ADC1_ADC0_G10    = 0b00001001, 
          MUX_ADC0_ADC0_G200   = 0b00001010, 
          MUX_ADC1_ADC0_G200   = 0b00001011, 
          MUX_ADC2_ADC2_G10    = 0b00001100, 
          MUX_ADC3_ADC2_G10    = 0b00001101, 
          MUX_ADC2_ADC2_G200   = 0b00001110, 
          MUX_ADC3_ADC2_G200   = 0b00001111,
          MUX_ADC0_ADC1_G1     = 0b00010000, 
          MUX_ADC1_ADC1_G1     = 0b00010001, 
          MUX_ADC2_ADC1_G1     = 0b00010010,
          MUX_ADC3_ADC1_G1     = 0b00010011, 
          MUX_ADC4_ADC1_G1     = 0b00010100, 
          MUX_ADC5_ADC1_G1     = 0b00010101, 
          MUX_ADC6_ADC1_G1     = 0b00010110,
          MUX_ADC7_ADC1_G1     = 0b00010111,
          MUX_ADC0_ADC2_G1     = 0b00011000, 
          MUX_ADC1_ADC2_G1     = 0b00011001, 
          MUX_ADC2_ADC2_G1     = 0b00011010, 
          MUX_ADC3_ADC2_G1     = 0b00011011, 
          MUX_ADC4_ADC2_G1     = 0b00011100,  
          MUX_ADC5_ADC2_G1     = 0b00011101, 
          
          MUX_REF     = 0b00011110,
          MUX_GND     = 0b00011111,
    
          MUX_ADC8    = 0b00100000,
          MUX_ADC9    = 0b00100001,
          MUX_ADC10   = 0b00100010,
          MUX_ADC11   = 0b00100011,
          MUX_ADC12   = 0b00100100,
          MUX_ADC13   = 0b00100101,
          MUX_ADC14   = 0b00100110,
          MUX_ADC15   = 0b00100111,
           
          MUX_ADC8_ADC8_G10     = 0b00101000, 
          MUX_ADC9_ADC8_G10     = 0b00101001, 
          MUX_ADC8_ADC8_G200    = 0b00101010,
          MUX_ADC9_ADC8_G200    = 0b00101011, 
          MUX_ADC10_ADC10_G10   = 0b00101100, 
          MUX_ADC11_ADC10_G10   = 0b00101101, 
          MUX_ADC10_ADC10_G200  = 0b00101110, 
          MUX_ADC11_ADC10_G200  = 0b00101111, 
          MUX_ADC8_ADC9_G1      = 0b00110000, 
          MUX_ADC9_ADC9_G1      = 0b00110001, 
          MUX_ADC10_ADC9_G1     = 0b00110010,
          MUX_ADC11_ADC9_G1     = 0b00110011,
          MUX_ADC12_ADC9_G1     = 0b00110100, 
          MUX_ADC13_ADC9_G1     = 0b00110101, 
          MUX_ADC14_ADC9_G1     = 0b00110110, 
          MUX_ADC15_ADC9_G1     = 0b00110111, 
          MUX_ADC8_ADC10_G1     = 0b00111000, 
          MUX_ADC9_ADC10_G1     = 0b00111001, 
          MUX_ADC10_ADC10_G1    = 0b00111010, 
          MUX_ADC11_ADC10_G1    = 0b00111011, 
          MUX_ADC12_ADC10_G1    = 0b00111100, 
          MUX_ADC13_ADC10_G1    = 0b00111101,
        };
    #endif
    
    #if defined(ADC_MODELL_MEGA_U4)
        enum AnalogSource : byte
        {
           MUX_ADC0    = 0b00000000,
           MUX_ADC1    = 0b00000001,
           MUX_ADC4    = 0b00000100,
           MUX_ADC5    = 0b00000101,
           MUX_ADC6    = 0b00000110,
           MUX_ADC7    = 0b00000111, 
          
           MUX_ADC1_ADC0_G10     = 0b00001001,
           MUX_ADC1_ADC0_G200    = 0b00001011,
           MUX_ADC0_ADC1_G1      = 0b00010000,
           MUX_ADC4_ADC1_G1      = 0b00010100,
           MUX_ADC5_ADC1_G1      = 0b00010101,
           MUX_ADC6_ADC1_G1      = 0b00010110,
           MUX_ADC7_ADC1_G1      = 0b00010111,
         
           MUX_REF      = 0b00011110,
           MUX_GND      = 0b00011111,
         
           MUX_ADC8    = 0b00100000,
           MUX_ADC9    = 0b00100001,
           MUX_ADC10   = 0b00100010,
           MUX_ADC11   = 0b00100011,
           MUX_ADC12   = 0b00100100,
           MUX_ADC13   = 0b00100101,
          
           MUX_ADC1_ADC0_G40     = 0b00100110,
    
           MUX_THERMO   = 0b00100111,
    
         MUX_ADC4_ADC0_G10    = 0b00101000,
         MUX_ADC5_ADC0_G10    = 0b00101001,
         MUX_ADC6_ADC0_G10    = 0b00101010,
         MUX_ADC7_ADC0_G10    = 0b00101011,
         MUX_ADC4_ADC1_G10    = 0b00101100,
         MUX_ADC5_ADC1_G10    = 0b00101101,
         MUX_ADC6_ADC1_G10    = 0b00101110,
         MUX_ADC7_ADC1_G10    = 0b00101111,
         MUX_ADC4_ADC0_G40    = 0b00110000,
         MUX_ADC5_ADC0_G40    = 0b00110001,
         MUX_ADC6_ADC0_G40    = 0b00110010,
         MUX_ADC7_ADC0_G40    = 0b00110011,
         MUX_ADC4_ADC1_G40    = 0b00110100,
         MUX_ADC5_ADC1_G40    = 0b00110101,
         MUX_ADC6_ADC1_G40    = 0b00110110,
         MUX_ADC7_ADC1_G40    = 0b00110111,
         MUX_ADC4_ADC0_G200   = 0b00111000,
         MUX_ADC5_ADC0_G200   = 0b00111001,
         MUX_ADC6_ADC0_G200   = 0b00111010,
         MUX_ADC7_ADC0_G200   = 0b00111011,
         MUX_ADC4_ADC1_G200   = 0b00111100,
         MUX_ADC5_ADC1_G200   = 0b00111101,
         MUX_ADC6_ADC1_G200   = 0b00111110,
         MUX_ADC7_ADC1_G200   = 0b00111111,
       };  
    
    #endif
    
    #if defined(ADC_MODELL_MEGA_SMALL)
        enum AnalogSource : byte
        {
          MUX_ADC0    = 0b00000000,
          MUX_ADC1    = 0b00000001,
          MUX_ADC2    = 0b00000010,
          MUX_ADC3    = 0b00000011,
          MUX_ADC4    = 0b00000100,
          MUX_ADC5    = 0b00000101,
          MUX_ADC6    = 0b00000110,
          MUX_ADC7    = 0b00000111, 
          #if defined(ADC_MODELL_WITH_THERMO) 
            MUX_THERMO  = 0b00001000,
          #endif
          MUX_REF     = 0b00001110,
          MUX_GND     = 0b00001111,
        };
    #endif  
    
    #ifdef ADC_MODELL_TINY_BIG
        enum AnalogSource : byte
        {
          MUX_ADC0          = 0b00000000,
          MUX_ADC1          = 0b00000001,
          MUX_ADC2          = 0b00000010,
          MUX_ADC3          = 0b00000011,
          MUX_ADC2_ADC2_G1  = 0b00000100,
          MUX_ADC2_ADC2_G20 = 0b00000101,
          MUX_ADC2_ADC3_G1  = 0b00000110,
          MUX_ADC2_ADC3_G20 = 0b00000111, 
          MUX_ADC0_ADC0_G1  = 0b00001000, 
          MUX_ADC0_ADC0_G20 = 0b00001001, 
          MUX_ADC0_ADC1_G1  = 0b00001010, 
          MUX_ADC0_ADC1_G20 = 0b00001011, 
          MUX_REF           = 0b00001100,
          MUX_GND           = 0b00001101,
          MUX_THERMO        = 0b00001111,
        };
    #endif     
    
    #if defined(ADC_MODELL_MEGA_BIG) | defined(ADC_MODELL_MEGA_SMALL)
        enum TriggerSource : byte
        {
          TRIG_FREERUNNING  = 0b00000000,
          TRIG_AC           = 0b00000001,
          TRIG_IRQ0         = 0b00000010,
          TRIG_TIMER0_COMPA = 0b00000011,
          TRIG_TIMER0_OVF   = 0b00000100,
          TRIG_TIMER1_COMPB = 0b00000101,
          TRIG_TIMER1_OVF   = 0b00000110,
          TRIG_TIMER1_CAPT  = 0b00000111, 
        };
    #endif
    
    #if defined(ADC_MODELL_MEGA_U4)
        enum TriggerSource : byte
        {
          TRIG_FREERUNNING  = 0b00000000,
          TRIG_AC           = 0b00000001,
          TRIG_IRQ0         = 0b00000010,
          TRIG_TIMER0_COMPA = 0b00000011,
          TRIG_TIMER0_OVF   = 0b00000100,
          TRIG_TIMER1_COMPB = 0b00000101,
          TRIG_TIMER1_OVF   = 0b00000110,
          TRIG_TIMER1_CAPT  = 0b00000111, 
          TRIG_TIMER4_OVF   = 0b00001000,
          TRIG_TIMER4_COMPA = 0b00001001,
          TRIG_TIMER4_COMPB = 0b00001010,
          TRIG_TIMER4_COMPD = 0b00001011,
    
        };
    #endif
    
    #if defined(ADC_MODELL_TINY_BIG)
        enum TriggerSource : byte
        {
          TRIG_FREERUNNING  = 0b00000000,
          TRIG_AC           = 0b00000001,
          TRIG_IRQ0         = 0b00000010,
          TRIG_TIMER0_COMPA = 0b00000011,
          TRIG_TIMER0_OVF   = 0b00000100,
          TRIG_TIMER0_COMPB = 0b00000101,
          TRIG_PCINT        = 0b00000110,
        };
    #endif   
     
    #if defined(ADC_MODELL_MEGA_U4)
      Adc & enableHighSpeed();
      Adc & disableHighSpeed();
    #endif 
    
    #if defined(ADC_MODELL_TINY_BIG)
      Adc & enableBipolar();
      Adc & disableBipolar();
      Adc & enablePolRev();
      Adc & disablePolRev();
    #endif 
        
      Adc & setCallBack(AdcCallBack callback);
      Adc & setReference(AnalogReference reference, bool wait = true);
      Adc & setSource(AnalogSource source);
      Adc & setTrigger(TriggerSource source);
      Adc & setClockDivisor(AnalogClockDivisor divisor);
      Adc & setClockDivisor();
      Adc & setResolution(AnalogResolution resolution);
      Adc & enable();
      Adc & reset();
      Adc & disable();
      Adc & enableIrq();
      Adc & disableIrq();
      Adc & enableAutoTrigger();
      Adc & disableAutoTrigger();
      Adc & startConversion(); 
      bool  getIrqFlag();
      int   getValue();
      int   operator()();
      operator int();
      int   operator()(AnalogSource source);
      int   operator[](AnalogSource source);
    
    };
 }