/** 
 * Situation:
 * Der Arduino bezieht seine Betriebsspannung aus einer 
 * LiPo Zelle und soll diese überwachen.
 * 
 * Bei einer Spannung von unter 3,1V soll die LED an Pin 13 blinken.
 * 
 * Ablauf:
 * Kalibrierwert aus EEPROM lesen
 * Referenzspannung auf Vcc setzen
 * Die interne Bandgab Referenz messen
 * Vcc mit Hilfe eines gespeicherten Kallibrierwertes berechnen
 * Gleitenden Mittelwert bilden 
 * Spannung seriell ausgeben
 * Unter 3,1V mit der LED blinken
 * 
 * Kalibrierung:
 * Arduino mit dem Programm laufen lassen
 * (zeigt evtl falsche Werte an)
 * Mit einem Multimeter die Akku- bzw Versorgungsspannung möglichst genau messen
 * Angenommen: 3,42V gemesssen, dann in der Seriellen Konsole 1,3.42; eingeben
 * und absenden.
 * 1 ist das Kommando
 * , trennt Kommando von Parameter
 * 3.42 die gemessene Spannung
 * ; schliesst das Kommando ab
 * 
 * Ab dann sollten die Werte erheblich genauer sein.
 * 
 * Der Kallibrierwert wird im EEPROM gespeichert.
 * 
 * 
 * 
 * 
 */

#include <TaskMacro.h>
// Siehe: https://forum.arduino.cc/index.php?topic=415229.0

#include <CombieAdc.h>
// Siehe: https://forum.arduino.cc/index.php?topic=496305.0

#include <CmdMessenger.h>
// siehe: http://playground.arduino.cc/Code/CmdMessenger

#include <EEPROM.h>

using Combie::Adc;
Adc adc;

CmdMessenger cmdMessenger(Serial);
//CmdMessenger cmdMessenger = CmdMessenger(Serial);



struct EepromData
{
  float konkreteReferenz;
};

EepromData EEMEM eepdata = {1.1}; // eep Datei wird erzeugt

float ReferenzSpannung;

float gmw; // gleitender Mittelwert
const float GmwFaktor = 0.1; // Faktor, mit welchem neue Messungen eingehen


void tueGmw(float messung)
{
  gmw = gmw - (gmw*GmwFaktor) + (messung*GmwFaktor);
}

float adc2Spannung(int adcValue)
{
  return ReferenzSpannung * 1024.0 / float(adcValue);  
}

void blink() // Task
{
  taskBegin();
  while(1)
  {
    taskWaitFor(gmw < 3.1); // warte, bis Grenze erreicht
    digitalWrite(LED_BUILTIN, 1);
    taskPause(500);
    digitalWrite(LED_BUILTIN, 0);
    taskPause(500);
  }
  taskEnd();
}

void messe() // Task
{
  taskBegin();
  while(1)
  {
    tueGmw(adc2Spannung(adc));
    taskPause(100);
  }
  taskEnd();
}

void show() // Task
{
  taskBegin();
  while(1)
  {
    taskPause(1000);
    Serial.print(F("Versorgungsspannung des Prozessors: "));
    Serial.println(gmw,3);
  }
  taskEnd();
}


void setup() 
{                
   Serial.begin(9600); 
   
   pinMode(LED_BUILTIN,OUTPUT);
   
   EEPROM.get(eepdata.konkreteReferenz,ReferenzSpannung);
   if(isnan(ReferenzSpannung)) ReferenzSpannung = 1.1; // EEPROM noch leer?
   
   Serial.print(F("Referenz: "));  Serial.print(ReferenzSpannung);  Serial.println(F("V"));
   
   cmdMessenger.attach(1, [](){
                                 float kalValue   = cmdMessenger.readFloatArg();
                                 gmw = kalValue;  
                                 ReferenzSpannung = kalValue/1024.0*adc;
                                 EEPROM.put(eepdata.konkreteReferenz,ReferenzSpannung);
                               });
    
   adc  .enable()
        .setClockDivisor() 
        .setReference(Adc::REF_VCC)
        .setSource(Adc::MUX_REF); 
        
   delay(20); // dem ADC etwas Zeit geben zum setzen
   gmw =  adc2Spannung(adc); // gleitenden Mittelwert vorbesetzen
}



void loop() 
{
    messe();
    blink();
    show();
    cmdMessenger.feedinSerialData();
 }
