#pragma once


// Makro zur definetion einer Variaben in de .noinit Section 
// https://www.nongnu.org/avr-libc/user-manual/mem_sections.html
#define NOINIT __attribute__ ((section (".noinit"))) 



namespace Combie
{
  namespace Tools
  {

    class TriggerableBool // Interface 
    {
       protected:
       bool state = false;

       virtual bool doTrigger(const bool trigger) = 0;
       
       bool operator() (const bool trigger)
       {
         return doTrigger(trigger);
       }
      
       bool operator= (const bool trigger)
       {
         return doTrigger(trigger);
       }
       
       operator bool() const
       {
         return state;
       }
       
       //public:
      // TriggerableBool():state(false){}
    };
    
    class FlankenErkennung : protected TriggerableBool
    {
     
      public:
     // using TriggerableBool::TriggerableBool;
      
      virtual bool doTrigger(const bool trigger) override
      {
        bool result = trigger && !state;
        state = trigger;
        return result;
      }
      using TriggerableBool::operator();
      using TriggerableBool::operator=;
    };

    
    template<typename DatenType>   
    class Counter
    {
      private:
      using CounterCallback = void (*) (Counter &counter);
      DatenType count= 0;
      CounterCallback onCountCallback  = 0;
      CounterCallback onChangeCallback = 0;
      
      public:
      Counter():count(0),onCountCallback(0),onChangeCallback(0){}
      
      void onCount(CounterCallback callback)
      {
        onCountCallback = callback;
      }
      
      void onChange(CounterCallback callback)
      {
        onChangeCallback = callback;
      }
      
      void reStart()
      {
        count = 0;
        if(onChangeCallback) onChangeCallback(*this);
      }
      
      DatenType doTrigger(const bool trigger)
      {
        count += trigger;
        if(trigger && onCountCallback) onCountCallback(*this);
        if(trigger && onChangeCallback) onChangeCallback(*this);
        return count;
      }
      
      DatenType operator () (const bool trigger)
      {
        return doTrigger(trigger);
      }
      
      DatenType operator = (const bool trigger)
      {
        return doTrigger(trigger);
      }
      
      DatenType operator()() const
      {
        return count;
      }
      
      operator DatenType()  const
      {
        return count;
      }

      Counter & operator++() // preincrement
      {
        doTrigger(true);
        return *this;
      }
      
      DatenType operator++(int) // postincrement
      { 
        DatenType temp = count; 
        doTrigger(true);
        return temp;
      }
    };
    
    template<typename DatenType> 
    class GleitenderMittelwert   // eher Tiefpassfilter
    {
			private:
				const double factor;
				double mittelwert;
				
			public:
				 GleitenderMittelwert(const double factor):factor(factor),mittelwert(0){}
				 
				 void setInitial(double value)
				 {
				 		mittelwert = value;
				 }
				 
				 DatenType doValue(DatenType value) // neuen Wert verarbeiten
				 {
				 		mittelwert *= 1.0 - factor;
				 		mittelwert += factor * value;
				 		return  mittelwert;
				 }
				 
				 DatenType operator= (DatenType value)
				 {
				   	return doValue(value);			 
				 }
				 
				 DatenType operator() (DatenType value)
				 {
				   	return doValue(value);			 
				 }
				 
				 DatenType operator() () const
				 {
				   	return mittelwert;			 
				 }
		
			   operator DatenType() const
				 {
				 	 return mittelwert;
				 }	
		};
		
		
		
		template<typename DatenType>
		class SchmittTrigger
		{
		  private:
		    bool state;     // Merker, Ist Zustandes
		    DatenType high;
		    DatenType low;
		    
		  
			public:
			 SchmittTrigger(const DatenType high,const DatenType low):state(0),high(high),low(low){}
 
			 void setHighLevel(const DatenType value)
			 {
			 		high = value;
			 }

			 void setLowLevel(const DatenType value)
			 {
			 		low = value;
			 }
			 
			 void setLevel(const DatenType highValue,const DatenType lowValue)
			 {
			 		high = highValue;
			 		low  = lowValue;
			 }
			 
			 bool doValue(DatenType value) // neuen Wert verarbeiten
			 {
			 		if(value >= high) state = true;			
			 		if(value <= low)  state = false;			
			 		return  state;
			 }
			 
      bool operator () (DatenType value)
      {
        return  doValue(value);
      }
			 
			 bool operator= (DatenType value)
			 {
			   	return doValue(value);			 
			 }
			 
	  	 bool operator() ()  const
			 {
			   	return state;			 
			 }
			 
			 operator bool()  const
			 {
			   	return state;			 
			 }
		};

// -------------------------



// integer Type mit eingeschaenktem Werteberech
template<int startWrapValue,int highWrapValue>
class LimitedInteger 
{
  private:
   int value;

   void wrapAround()
   {
     while(value >= highWrapValue)  value -= highWrapValue-startWrapValue;
     while(value <  startWrapValue) value += highWrapValue-startWrapValue;
   }

  public:
   LimitedInteger():value(startWrapValue){}
   LimitedInteger(int _value)
   {
    value = _value;
    wrapAround();
   }

   operator int() const
   {
     return value;
   }
   
   int operator ++() // pre
   {
     value++;
     wrapAround();
     return value;
   }
      
   int operator ++(int) // post     
   { 
     int temp = value; 
     value++;
     wrapAround();
     return temp;
   }
   
   int operator --() // pre
   {
     value--;
     wrapAround();
     return value;
   }
      
   int operator --(int) // post     
   { 
     int temp = value; 
     value--;
     wrapAround();
     return temp;
   }
   
   int operator =(int newValue)
   {
     value = newValue;
     wrapAround();
     return value;
   }
   
   int operator +=(LimitedInteger & addValue)
   {
     value += addValue;
     wrapAround();
     return value;
   }
   
   int operator -=(LimitedInteger & subValue)
   {
     value -= subValue;
     wrapAround();
     return value;
   }
   
   int operator +=(int addValue)
   {
     value += addValue;
     wrapAround();
     return value;
   }
   
   int operator -=(int subValue)
   {
     value -= subValue;
     wrapAround();
     return value;
   }
};



// -------------------------
// helper functions unsorted
    
    template<typename AnyType>void swap(AnyType &x, AnyType &y) 
    {
       AnyType temp = x;
       x = y;   
       y = temp;
    }


template<typename ArrayType, size_t count> constexpr size_t arrayCount(const ArrayType (&)[count])
{
  return count;
}

    
 /*   
    constexpr size_t arrayCount(const auto &array)
    {
      return sizeof(array)/sizeof(array[0]);
    }
*/
    
  }
}

