/* ----------------------------------------------------------
                    toene_melodie.c

     Einfacher Rechteckgenerator fuer Noten der C-DUR
     Tonleiter mit Notenstring-Parser


     MCU     : ATmega8, ATmegaxx8, ATtinyx4, ATtinyx5
               ATtiny13, ATtiny2313
     F_CPU   : 4,8MHz, 8MHz, 9,6MHz, 16MHz

     28.11.2019  R. Seelig
   ---------------------------------------------------------- */

#include "toene_melodie.h"

// -------------- Lautsprecheranschluss ---------------------
#define speaker_port     B
#define speaker_portnum  4


// ----------------- globale Variable -----------------------
volatile uint8_t  sound;                 // kann generell Frequenzerzeugung ein- oder ausschalten
                                         // 0 = Ton aus; 1 = Ton an

         uint16_t playtempo = 5;             // Grundgeschwindigkeit, mit der ein Notenstring
                                         // abgespielt wird


uint8_t togglespk= 0;
uint8_t t0_ovf_rel= 185;


/* ---------------------------------------------------------------------------
     Reloadwerte fuer TCNT0 im Overflow-Betrieb.

     Fuer F_CPU Werte von 16MHz und 9,6MHz muss der Prescaler /256, fuer
     8MHZ und 4,8MHz auf /64 eingestellt werden
   --------------------------------------------------------------------------- */

#if ( F_CPU == 16000000ul )
  uint8_t freqreload [24] = { 137, 143, 150, 156, 161, 167, 172, 176, 181, 185, 189, 193,
                              196, 200, 203, 206, 209, 211, 214, 216, 218, 220, 222, 224 };
#elif ( F_CPU == 9600000ul )
    uint8_t freqreload [24] = { 184, 188, 192, 196, 199, 202, 205, 208, 211, 213, 216, 218,
                                220, 222, 224, 226, 228, 229, 231, 232, 233, 235, 236, 237 };
#elif ( F_CPU == 8000000ul )
    uint8_t freqreload [24] = {  17,  31,  43,  55,  66,  77,  87,  97, 106, 114, 122, 129,
                                137, 143, 150, 156, 161, 167, 172, 176, 181, 185, 189, 193 };
#elif ( F_CPU == 4800000ul )
    uint8_t freqreload [24] = { 113, 121, 128, 135, 142, 149, 155, 160, 166, 171, 176, 180,
                                184, 188, 192, 196, 199, 202, 205, 208, 211, 213, 216, 218  };
#endif

#if (( F_CPU != 16000000ul ) && ( F_CPU != 9600000ul ) && ( F_CPU != 8000000ul ) && ( F_CPU != 4800000ul ))
  #error Taktfrequenz muss 16, 9.6, 8, oder 4.8MHZ betragen
#endif



static const unsigned char schneew  [] PROGMEM = { "c4d4e2g4e2g4e1d4e4f2g4f2g4f1g4a4h2+f4-h2+f4-h1a4h4+c2e4c2-a4g1e4g4+c1-h1+d1c1-a2+c4-a2+c4-a1" };


/* ---------------------------------------------------------------------------
     Die Namen des Timer 0 Interruptvectors sind in libc (leider) mit
     unterschiedlichen Namen gegeben
  --------------------------------------------------------------------------- */
#if defined __AVR_ATtiny24__ || __AVR_ATtiny44__ || __AVR_ATtiny84__ || __AVR_ATtiny13__   \
         || __AVR_ATtiny45__ || __AVR_ATtiny25__ || __AVR_ATtiny85__

  ISR (TIM0_OVF_vect)

#else
  // fuer ATmega und ATtiny2313

  ISR (TIMER0_OVF_vect)

#endif
{
  TCNT0= t0_ovf_rel;
  if (sound)
  {
    if (togglespk & 1)
    {
      setspk();
      togglespk= 0;
    }
    else
    {
      clrspk();
      togglespk= 1;
    }
  }
  else
  {
    togglespk= 1;
    clrspk();
  }
}


/* ---------------------------------------------------------------------------
                                timer0_init

       initialisiert Timer0 fuer Overflow und Interrupt - Betrieb

       fuer CPU Frequenzen > 8 MHz wird ein Taktvorteiler von /256 benoetigt,
       fuer CPU Frequenzen <= 8 MHz ein Teiler von /64
   --------------------------------------------------------------------------- */
void timer0_init(void)
/*
  TCCR0B - Timer/Counter0 Register B

  CS02 | CS01 | CS00 |  Bedeutung
  ---------------------------------------------------
    0  |   0  |   0  |  keine Taktquelle = Timer halt
    0  |   0  |   1  |  F_CPU / 1
    0  |   1  |   0  |  F_CPU / 8
    0  |   1  |   1  |  F_CPU / 64
    1  |   0  |   0  |  F_CPU / 256
    1  |   0  |   1  |  F_CPU / 1024
    1  |   1  |   0  |  ext. Takt an T0-Pin, neg. Taktflanke
    1  |   1  |   1  |  dto. pos. Taktflanke
*/

{
  #if ( F_CPU > 8000000ul )
    TCCR0B = (1 << CS02);                   // Taktteiler / 256
  #else
    TCCR0B = (1 << CS01) | ( 1 << CS00);   // Taktteiler / 64
  #endif

  TIMSK0 = (1 << TOIE0);                    // Interrupt bei Ueberlauf

  TCNT0= t0_ovf_rel;                        // Reloadwert, bis Ueberlauf sind es noch 71 Timer-Takte
  sei();                                    // Interrupt grundsaetzlich zulassen

}

/* ---------------------------------------------------
                 Ende Timer0 Interrupt
   --------------------------------------------------- */

void spk_delay(int wert)
{
  int i;

  for (i= 0; i< wert; i++)
  {
    _delay_us(1);
  }
}


void tonlen(int w)
{
  int cx;

  w = w*100;
  for (cx= 0; cx< w; cx++) { _delay_us(100); }
  sound= 0;
  _delay_ms(30);
}

void playnote(char note)
{
  t0_ovf_rel= freqreload[note];
  sound= 1;
}

void playstring(const unsigned char* const s)
{
  char ch;
  char aokt;
  int dind;

  aokt= 0; dind= 0;
  ch= pgm_read_byte(&(s[dind]));
  while (ch)
  {
    ch= pgm_read_byte(&(s[dind]));
    switch(ch)
    {
      case '-': { aokt= aokt-12; break; }
      case '+': { aokt= aokt+12; break; }
      case 'c': { playnote(aokt); break; }
      case 'C': { playnote(aokt+1); break; }
      case 'd': { playnote(aokt+2); break; }
      case 'D': { playnote(aokt+3); break; }
      case 'e': { playnote(aokt+4); break; }
      case 'f': { playnote(aokt+5); break; }
      case 'F': { playnote(aokt+6); break; }
      case 'g': { playnote(aokt+7); break; }
      case 'G': { playnote(aokt+8); break; }
      case 'a': { playnote(aokt+9); break; }
      case 'A': { playnote(aokt+10); break; }
      case 'h': { playnote(aokt+11); break; }
      case 'P': { sound= 1; break; }
      case 'p': { sound= 0; break; }
      case '1': { tonlen(16*playtempo); break; }
      case '2': { tonlen(8*playtempo); break; }
      case '3': { tonlen(6*playtempo); break; }
      case '4': { tonlen(4*playtempo); break; }
      case '5': { tonlen(3*playtempo); break; }
      case '8': { tonlen(2*playtempo); break; }
      case '9': { tonlen(2*playtempo); break; }
    }
    dind++;
  }
}

