;################################################################################
;#										#
;# avr-chipbasic2 - single chip basic computer with ATmega644			#
;# expression parser								#
;# copyright (c) 2006-2008 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 2		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################
; ereg			= error code
;		 0	= OK
;		 1	= break
;		 2	= overflow (add,sub,mul)
;		 3	= division by zero
;		 4	= square root from negative value
;		 5	= constant overflow (basic expression parser)
;		 6	= garbage in expression
;		 7	= syntax error (basic expression parser)
;
;		expar_ros	= address of stack root
;##############################################################################

;##############################################################################
; the basic parser
; Z= pointer in ram
;##############################################################################
expar:		ldi	YL,LOW(expar_ros)	;set stack pointer
		ldi	YH,HIGH(expar_ros)	;
		st	-Y,const_0		;write low value
		st	-Y,const_0		;write high value
		ldi	ereg,0x00		;no error
		push	const_0			;stack stopper
		clr	XL			;OP L
		clr	XH			;OP H
		clr	tempreg1		;clear status (last=op)

;------------------------------------------------------------------------------
; get character from string
;------------------------------------------------------------------------------
expar_bp_01:	ld	tempreg4,Z+		;get char
		cpi	tempreg4,0x20		;space
		breq	expar_bp_01
expar_bp_02:

;------------------------------------------------------------------------------
; get precompressed decimal 8 bit
;------------------------------------------------------------------------------
expar_pcr_01:	cpi	tempreg4,0x1c		;token
		brne	expar_pcr_02
		ld	XL,Z+
		clr	XH
		ld	tempreg4,Z+		;get next char
		rjmp	expar_bp_n3

;------------------------------------------------------------------------------
; get precompressed hexadecimal 8 bit
;------------------------------------------------------------------------------
expar_pcr_02:	cpi	tempreg4,0x1e		;token
		brne	expar_pcr_03
		ld	XL,Z+
		clr	XH
		ld	tempreg4,Z+		;get next char
		rjmp	expar_bp_n3

;------------------------------------------------------------------------------
; get precompressed decimal 16 bit
;------------------------------------------------------------------------------
expar_pcr_03:	cpi	tempreg4,0x1d		;token
		brne	expar_pcr_04
		ld	XH,Z+
		ld	XL,Z+
		ld	tempreg4,Z+		;get next char
		rjmp	expar_bp_n3

;------------------------------------------------------------------------------
; get precompressed hexadecimal 16 bit
;------------------------------------------------------------------------------
expar_pcr_04:	cpi	tempreg4,0x1f		;token
		brne	expar_pcr_05
		ld	XH,Z+
		ld	XL,Z+
		ld	tempreg4,Z+		;get next char
		rjmp	expar_bp_n3

;------------------------------------------------------------------------------
; get precompressed function
;------------------------------------------------------------------------------
expar_pcr_05:	cpi	tempreg4,0x1c		;upper limit for OPs
		brcc	expar_hex1
		mov	tempreg2,tempreg4	;copy OP
		mov	tempreg5,YL		;save registers
		mov	tempreg6,YH
		mov	YL,ZL			;copy RAM pointer
		mov	YH,ZH
		rjmp	expar_mo_pcr
		
;------------------------------------------------------------------------------
; check for a hexadecimal number
;------------------------------------------------------------------------------
expar_hex1:	cpi	tempreg4,'$'		;hex numbers start with $
		brne	expar_bp_n1		;no
expar_hex2:	ld	tempreg4,Z+		;get next char
		cpi	tempreg4,48		;"0"
		brcs	expar_hex3		;less than "0"
		cpi	tempreg4,58		;"9"+1
		brcc	expar_hex3		;greater then "9"
		subi	tempreg4,48		;number base
		rjmp	expar_hex4
expar_hex3:	cpi	tempreg4,65		;"A"
		brcs	expar_hex5		;less than "A"
		cpi	tempreg4,71		;"F"+1
		brcc	expar_hex5		;greater then "F"
		subi	tempreg4,55		;hex base	
expar_hex4:	cpi	XH,0x10
		brcc	expar_hex6		;error
		swap	XL			;*16
		swap	XH
		mov	tempreg2,XL
		andi	tempreg2,0x0f
		andi	XL,0xf0
		or	XH,tempreg2
		add	XL,tempreg4
		rjmp	expar_hex2
expar_hex5:	rjmp	expar_bp_n3
expar_hex6:	rjmp	expar_bp_n5		
		
;------------------------------------------------------------------------------
; check for a decimal number
;------------------------------------------------------------------------------
expar_bp_n1:	cpi	tempreg4,48		;"0"
		brcs	expar_bp_m1		;next check
		cpi	tempreg4,58		;"9"+1
		brcc	expar_bp_m1		;next check
		cpi	tempreg1,0x80		;last was op?		
		brcs	expar_bp_n2
		ldi	ereg,6			;missing operator
		rjmp	expar_bp_err		;error handling

;mult result with 10 and add actual value		
expar_bp_n2:	lsl	XL			;*2
		rol	XH			;
		brcs	expar_bp_n5		;overflow
		mov	tempreg5,XL		;result *2
		mov	tempreg6,XH		
		lsl	XL			;*2
		rol	XH			;
		brcs	expar_bp_n5		;overflow
		lsl	XL			;*2
		rol	XH			;
		brcs	expar_bp_n5		;overflow
		add	XL,tempreg5		;-> x*10
		adc	XH,tempreg6
		brcs	expar_bp_n5		;overflow
		subi	tempreg4,48		;add value
		add	XL,tempreg4
		adc	XH,const_0
		brcs	expar_bp_n5		;overflow
		sbrc	XH,7			;negative
		rjmp	expar_bp_n5
		ld	tempreg4,Z+		;get next char
		cpi	tempreg4,48		;"0"
		brcs	expar_bp_n3		;not a digit
		cpi	tempreg4,58		;"9"+1
		brcc	expar_bp_n3		;not a digit
		rjmp	expar_bp_n2		;add this digit			
		
expar_bp_n3:	cpi	tempreg1,0x7f		;inverter?
		brne	expar_bp_n4
		rcall	expar_inva		;invert value
expar_bp_n4:	st	-Y,XL			;write low value
		st	-Y,XH			;write high value
		ldi	tempreg1,0x80		;last element is a value
		rjmp	expar_bp_02		;next			
		
expar_bp_n5:	ldi	ereg,5			;constant overflow
		rjmp	expar_bp_err		;error handling

;------------------------------------------------------------------------------
; multichar ops AR(, ABS(...
;------------------------------------------------------------------------------
expar_mo_nx:	rjmp	expar_bp_i1		;jump extender

expar_bp_m1:	cpi	tempreg4,65		;>="A"
		brcs	expar_mo_nx		;no 
		cpi	tempreg4,91		;<"Z"+1
		brcc	expar_mo_nx		;no
		
		ld	tempreg3,Z
		cpi	tempreg3,65		;>="A"
		brcs	expar_bp_v1		;no second alpha -> variable
		cpi	tempreg3,91		;<"Z"+1
		brcc	expar_bp_v1		;no second alpha -> variable

		mov	tempreg5,YL		;save registers
		mov	tempreg6,YH
		mov	YL,ZL			;copy RAM pointer
		mov	YH,ZH
		sbiw	YL,1			;YL points to first char
		mov	r16,YL
		mov	r17,YH
		ldi	ZL,LOW(expar_ttab*2)	;token table
		ldi	ZH,HIGH(expar_ttab*2)	;token table
		ldi	tempreg2,0x02		;first token
expar_mo_01:	lpm	tempreg4,Z+		;get char from table		
		cpi	tempreg4,0		;end of table
		brne	expar_mo_02
		ldi	ereg,8			;unknown
		rjmp	expar_bp_err		;error handling
expar_mo_02:	cpi	tempreg4,' '		;found
		brne	expar_mo_03
		ld	tempreg3,Y+
		cpi	tempreg3,'('
		brne	expar_mo_05
expar_mo_pcr:	push	tempreg2		;push OP
		mov	tempreg1,tempreg2	;set status
		clr	XL			;clear operand
		clr	XH
		mov	ZL,YL
		mov	ZH,YH
		mov	YL,tempreg5
		mov	YH,tempreg6
		rjmp	expar_bp_01		;next char
expar_mo_03:	ld	tempreg3,Y+		;ram char
		cp	tempreg3,tempreg4
		breq	expar_mo_01		;next char				
expar_mo_04:	lpm	tempreg4,Z+
		cpi	tempreg4,0x20		;space
		brne	expar_mo_04
		inc	tempreg2		;next token		
		mov	YL,r16			;points to first char in RAM
		mov	YH,r17
		rjmp	expar_mo_01
expar_mo_05:	ldi	ereg,7			;syntax error
		rjmp	expar_bp_err		;error handling
		
;------------------------------------------------------------------------------
; a variable "A...Z"
;------------------------------------------------------------------------------
expar_bp_v1:	cpi	tempreg1,0x80		;last was op?		
		brcs	expar_bp_v2
		ldi	ereg,6			;missing operator
		rjmp	expar_bp_err		;error handling

expar_bp_v2:	subi	tempreg4,65		;-"A"
		lsl	tempreg4		;*2
		push	ZL			;save Z-reg
		push	ZH
		ldi	ZL,LOW(varspace)	;pointer to vars
		ldi	ZH,HIGH(varspace)
		add	ZL,tempreg4		;set actual pointer
		adc	ZH,const_0
		ld	XL,Z+
		ld	XH,Z+
		pop	ZH			;restore Z-reg			
		pop	ZL
		ld	tempreg4,Z+		;get next char
		rjmp	expar_bp_n3

;------------------------------------------------------------------------------
; an inverter
;------------------------------------------------------------------------------
expar_bp_i1:	cpi	tempreg4,45		;"-"
		brne	expar_bp_p1		;no
		cpi	tempreg1,0x7f		;is already inverted
		brne	expar_bp_i2		;no
		clr	tempreg1		;
		rjmp	expar_bp_01		;thats all
		
expar_bp_i2:	brcc	expar_bp_i3		;last was not an operator
		cpi	tempreg1,0x20
		brcs	expar_bp_i3
		ldi	tempreg1,0x7f		;set inverter
		rjmp	expar_bp_01		;next			
expar_bp_i3:	rjmp	expar_bp_c1		

;------------------------------------------------------------------------------
; check for parantheses
;------------------------------------------------------------------------------
expar_bp_p1:	cpi	tempreg4,40		;"("
		brne	expar_bp_p2
		ldi	tempreg4,0x01		;code for left paranthesis
		push	tempreg4		;push to stack	
		clr	XL			;clear operand
		clr	XH
		mov	tempreg1,tempreg4
		rjmp	expar_bp_01		;next character

expar_bp_p2:	cpi	tempreg4,41		;")"
		breq	expar_bp_p3
		rjmp	expar_bp_l1

expar_bp_p3:	pop	tempreg2		;get operator
		cpi	tempreg2,0x01		;left paranthesis
		breq	expar_cp_ab1		;yes

expar_cp_ab:	cpi	tempreg2,0x02		;ABS(
		brne	expar_cp_sg		;not
		rcall	expar_abs
expar_cp_ab1:	rjmp	expar_bp_p9		;goto end

expar_cp_sg:	cpi	tempreg2,0x03		;SGN(
		brne	expar_cp_sq		;not
		rcall	expar_sgn
		rjmp	expar_bp_p9		;goto end

expar_cp_sq:	cpi	tempreg2,0x04		;SQR(
		brne	expar_cp_no		;not
		rcall	expar_sqr
		cpi	ereg,0x00
		breq	expar_bp_pax		;goto end
		rjmp	expar_bp_err
expar_bp_pax:	rjmp	expar_bp_pa		;jump extender		

expar_cp_no:	cpi	tempreg2,0x05		;NOT(
		brne	expar_cp_rn		;not
		ld	XH,Y+			;get high value of TOS/SEC
		ld	XL,Y+			;get low value of TOS/SEC
		com	XH			;invert
		com	XL			
		rjmp	expar_bp_pa		;goto end

expar_cp_rn:	cpi	tempreg2,0x06		;RND(
		brne	expar_cp_ad		;not
		rjmp	expar_rnd
		
expar_cp_ad:	cpi	tempreg2,0x07		;ADC(
		brne	expar_cp_di		;not
		rjmp	expar_adc
		
expar_cp_di:	cpi	tempreg2,0x08		;DIN(
		brne	expar_cp_pe		;not
		rjmp	expar_din

expar_cp_pe:	cpi	tempreg2,0x09		;XPEEK(
		brne	expar_cp_te		;not
		rjmp	expar_xpeek

expar_cp_te:	cpi	tempreg2,0x0a		;TEMP(
		brne	expar_cp_lo		;not
		rjmp	expar_temp

expar_cp_lo:	cpi	tempreg2,0x0b		;LO(
		brne	expar_cp_hi		;not
		rjmp	expar_lo

expar_cp_hi:	cpi	tempreg2,0x0c		;HI(
		brne	expar_cp_ke		;not
		rjmp	expar_hi

expar_cp_ke:	cpi	tempreg2,0x0d		;KEY(
		brne	expar_cp_epk		;not
		rjmp	expar_key

expar_cp_epk:	cpi	tempreg2,0x0e		;EPEEK(
		brne	expar_cp_ar		;not
		rjmp	expar_epeek

expar_cp_ar:	cpi	tempreg2,0x10		;AR(
		brne	expar_cp_sin		;not
		rjmp	expar_ar

expar_cp_sin:	cpi	tempreg2,0x11		;SIN(
		brne	expar_cp_cos		;not
		rjmp	expar_sin

expar_cp_cos:	cpi	tempreg2,0x12		;COS(
		brne	expar_cp_fil		;not
		rjmp	expar_cos

expar_cp_fil:	cpi	tempreg2,0x13		;FTYPE(
		brne	expar_cp_fsiz		;not
		rjmp	expar_ftype

expar_cp_fsiz:	cpi	tempreg2,0x14		;FSIZE(
		brne	expar_cp_erh		;not
		rjmp	expar_fsize

expar_cp_erh:	cpi	tempreg2,0x0f		;ERR(
		brne	expar_cp_pst		;not
		rjmp	expar_geterr

expar_cp_pst:	cpi	tempreg2,0x15		;PSTAT(
		brne	expar_cp_spi		;not
		rjmp	expar_pstat

expar_cp_spi:	cpi	tempreg2,0x16		;SPI(
		brne	expar_cp_men		;not
		rjmp	expar_spi

expar_cp_men:	cpi	tempreg2,0x17		;MENU(
		brne	expar_bp_p7		;not
		rjmp	expar_menu

expar_bp_p7:	cpi	tempreg2,0x00		;end
		breq	expar_bp_p8		;no open 
		rcall	expar_bp_x
		cpi	ereg,0x00
		breq	expar_bp_p3x
		rjmp	expar_bp_err
expar_bp_p3x:	rjmp	expar_bp_p3		;jump extender		

expar_bp_p8:	ldi	ereg,6			;error code
		ret				;thats all

expar_bp_pa:	st	-Y,XL			;write result to stack
		st	-Y,XH			;

expar_bp_p9:	clr	XL			;clear operand
		clr	XH
		ldi	tempreg1,0x02		;code for right paranthesis
		rjmp	expar_bp_01		;next character
	
;------------------------------------------------------------------------------
; check if last was an operand
;------------------------------------------------------------------------------
expar_bp_l1:	cpi	tempreg1,0x7f		;upper border for operators
		brcc	expar_bp_c1		;last was no operator
		cpi	tempreg1,0x0f		;lower border for operators
		brcs	expar_bp_c1
		ldi	ereg,6			;missing operand
		rjmp	expar_bp_err

;------------------------------------------------------------------------------
; a comparsion
;------------------------------------------------------------------------------
expar_bp_c1:	cpi	tempreg4,60		;"<"
		brne	expar_bp_c4
		ldi	tempreg4,0x21		;code for less
		ld	tempreg3,Z		;get next
		cpi	tempreg3,62		;">"
		brne	expar_bp_c2		;no
		ldi	tempreg4,0x22		;code for not equal
		ld	tempreg3,Z+		;correct Z-register
		rjmp	expar_bp_d0		;do operation
expar_bp_c2:	cpi	tempreg3,61		;"="
		brne	expar_bp_c3		;no
		ldi	tempreg4,0x23		;code for less or equal
		ld	tempreg3,Z+		;correct Z-register		
expar_bp_c3:	rjmp	expar_bp_d0		;do operation

expar_bp_c4:	cpi	tempreg4,62		;">"
		brne	expar_bp_c6
		ldi	tempreg4,0x24		;code for greater
		ld	tempreg3,Z		;get next
		cpi	tempreg3,61		;"="
		brne	expar_bp_c5		;no
		ldi	tempreg4,0x25		;code for greater or equal
		ld	tempreg3,Z+		;correct Z-register		
expar_bp_c5:	rjmp	expar_bp_d0		;do operation
				
expar_bp_c6:	cpi	tempreg4,61		;"="
		brne	expar_bp_c7
		ldi	tempreg4,0x26		;code for equal
		rjmp	expar_bp_d0		;do operation
		
expar_bp_c7:		
;------------------------------------------------------------------------------
; an arithmetic operator 
;------------------------------------------------------------------------------
expar_bp_o1:	cpi	tempreg4,43		;"+"
		brne	expar_bp_o2
		ldi	tempreg4,0x30		;code for +
		rjmp	expar_bp_d0
			
expar_bp_o2:	cpi	tempreg4,45		;"-"
		brne	expar_bp_o3
		ldi	tempreg4,0x31		;code for sub
		rjmp	expar_bp_d0

expar_bp_o3:	cpi	tempreg4,35		;"#"
		brne	expar_bp_o4
		ldi	tempreg4,0x32		;code for or
		rjmp	expar_bp_d0
		
;next level		
					
expar_bp_o4:	cpi	tempreg4,42		;"*"
		brne	expar_bp_o5
		ldi	tempreg4,0x40		;code for mult
		rjmp	expar_bp_d0
			
expar_bp_o5:	cpi	tempreg4,47		;"/"
		brne	expar_bp_o6
		ldi	tempreg4,0x41		;code for div
		rjmp	expar_bp_d0	

expar_bp_o6:	cpi	tempreg4,37		;"%"
		brne	expar_bp_o7
		ldi	tempreg4,0x42		;code for mod
		rjmp	expar_bp_d0	

expar_bp_o7:	cpi	tempreg4,38		;"&"
		brne	expar_bp_o8
		ldi	tempreg4,0x43		;code for and
		rjmp	expar_bp_d0	

expar_bp_o8:
;------------------------------------------------------------------------------
; we are at the end of expression
;------------------------------------------------------------------------------
expar_bp_e2:	pop	tempreg2		;get last operand
		cpi	tempreg2,0x00		;end?
		breq	expar_bp_e3		;->end
		cpi	tempreg2,0x1f
		brcs	expar_bp_e5
		rcall	expar_bp_x
		rjmp	expar_bp_e2
expar_bp_e3:	ld	XH,Y+			;get high result
		ld	XL,Y+			;get low result
expar_bp_e4:	sbiw	ZL,1			;corect Z-pointer to next char
		ret
expar_bp_e5:	ldi	ereg,6
expar_bp_e6:	pop	tempreg2		;get last operand
		cpi	tempreg2,0x00		;end?
		breq	expar_bp_e3		;->end
		rjmp	expar_bp_e6

;------------------------------------------------------------------------------
; clear stack on error 
;------------------------------------------------------------------------------
expar_bp_err:	pop	tempreg1		;get operator
		cpi	tempreg1,0		;stack ist empty?
		brne	expar_bp_err		;loop
expar_bp_er1:	ret				;thats all		
		
;------------------------------------------------------------------------------
; do operator function 
;------------------------------------------------------------------------------
expar_bp_d0:	pop	tempreg2		;get last operator from stack
	
	    	mov	XL,tempreg4		;recent operator
		mov	XH,tempreg2		;last operator		
		andi	XL,0xf0			;we need only priority
		andi	XH,0xf0			;		
		cp	XH,XL			;compare last with recent
		brcs	expar_bp_d2		;branch if last is not less
		
expar_bp_d1:	rcall	expar_bp_x		;do last operation
		cpi	ereg,0x00		;error?
		brne	expar_bp_err		;yes
		rjmp	expar_bp_d0
		
;write last and recent operator on stack
		
expar_bp_d2:	push	tempreg2		;put last on stack
		push	tempreg4		;put recent on stack
		clr	XL			;clear operand
		clr	XH
		mov	tempreg1,tempreg4	;set status
		rjmp	expar_bp_01		;get next char

;------------------------------------------------------------------------------
; execute function with 2 parameters 
;------------------------------------------------------------------------------
expar_bp_x:	push	r21
		push	r20
		ld	r17,Y+			;get high value of TOS
		ld	r16,Y+			;get low value of TOS
		ld	XH,Y+			;get high value of SEC
		ld	XL,Y+			;get low value of SEC

		cpi	tempreg2,0x21		;<
		brne	expar_bp_x1
		rcall	expar_gt
		rjmp	expar_bp_xy		
		
expar_bp_x1:	cpi	tempreg2,0x22		;<>
		brne	expar_bp_x2
		rcall	expar_eq
		subi	XL,1
		andi	XL,1
		rjmp	expar_bp_xy		
		
expar_bp_x2:	cpi	tempreg2,0x23		;<=
		brne	expar_bp_x3
		rcall	expar_lt
		subi	XL,1
		andi	XL,1
		rjmp	expar_bp_xy		
		
expar_bp_x3:	cpi	tempreg2,0x24		;>
		brne	expar_bp_x4
		rcall	expar_lt
		rjmp	expar_bp_xy		
		
		
expar_bp_x4:	cpi	tempreg2,0x25		;>=
		brne	expar_bp_x5
		rcall	expar_gt
		subi	XL,1
		andi	XL,1
		rjmp	expar_bp_xy		
		
expar_bp_x5:	cpi	tempreg2,0x26		;=
		brne	expar_bp_x6
		rcall	expar_eq
		rjmp	expar_bp_xy
		
expar_bp_x6:	cpi	tempreg2,0x30		;+
		brne	expar_bp_x7
		rcall	expar_add
		rjmp	expar_bp_xy
		
expar_bp_x7:	cpi	tempreg2,0x31		;-
		brne	expar_bp_x8
		rcall	expar_sub
		rjmp	expar_bp_xy
		
expar_bp_x8:	cpi	tempreg2,0x32		;or
		brne	expar_bp_x9
		or	XL,r16
		or	XH,r17
		rjmp	expar_bp_xy
		
expar_bp_x9:	cpi	tempreg2,0x40		;*
		brne	expar_bp_xa
		rcall	expar_mul
		rjmp	expar_bp_xy		;put TOS

expar_bp_xa:	cpi	tempreg2,0x41		;/
		brne	expar_bp_xb
		rcall	expar_div		;call division
		rjmp	expar_bp_xy		;put TOS
		
expar_bp_xb:	cpi	tempreg2,0x42		;%
		brne	expar_bp_xc
		rcall	expar_div		;call division
		mov	XL,r18			;get remainder
		mov	XH,r19
		rjmp	expar_bp_xy		;put TOS
		
expar_bp_xc:	cpi	tempreg2,0x43		;and
		brne	expar_bp_xy
		and	XL,r16
		and	XH,r17
		rjmp	expar_bp_xy
		
expar_bp_xy:	st	-Y,XL			;write result to stack
		st	-Y,XH			
		pop	r20
		pop	r21
		ret
		
;-------------------------------------------------------------------------------
; absolute value
; TOS <= abs(TOS)
;-------------------------------------------------------------------------------
expar_abs:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		rcall	expar_absa		;calc absolute value
expar_abs1:	st	-Y,XL			;write low value
		st	-Y,XH			;write high value
		ret

;-------------------------------------------------------------------------------
; absolute value of op1
;-------------------------------------------------------------------------------
expar_absa:	sbrs	XH,7			;skip if negative
		ret				;return if op was positive
expar_inva:	com	XL			;switch sign
		com	XH			;
		add	XL,const_1		;
		adc	XH,const_0		;
		ret				;thats all

;-------------------------------------------------------------------------------
; absolute value of op2
;-------------------------------------------------------------------------------
expar_absb:	sbrs	r17,7			;skip if negative
		ret				;return if op was positive
expar_invb:	com	r16			;switch sign
		com	r17			;
		add	r16,const_1		;
		adc	r17,const_0		;
		ret				;thats all

;-------------------------------------------------------------------------------
; signum
; TOS <= sign(TOS) (-1,0,1)
;-------------------------------------------------------------------------------
expar_sgn:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		or	XL,XH			;or to check if zero
		breq	expar_sgn_1		;branch if not zero
		bst	XH,7			;write sign to T
		ldi	XL,0x01			;set to 1
		clr	XH		
		brtc	expar_sgn_1
		sbiw	XL,2
expar_sgn_1:	rjmp	expar_abs1		;put result to stack
		

;-------------------------------------------------------------------------------
; 16 Bit subtraction
; (TOS) <= (SEC)-(TOS)
;-------------------------------------------------------------------------------
expar_sub:	rcall	expar_invb		;-(TOS)

;-------------------------------------------------------------------------------
; 16 Bit addition
; (TOS) <= (TOS)+(SEC)
;-------------------------------------------------------------------------------
expar_add:	mov	r18,XH			;High 1
		eor	r18,r17			;exclusive or with high 2
		add	XL,r16			;add low bytes
		adc	XH,r17			;add high bytes
		sbrc	r18,7			;skip if signs equal
		ret				;result is ok
		mov	r18,XH			;High result
		eor	r18,r17			;High OP2
		sbrc	r18,7			;skip if signs equal
		ldi	ereg,2			;set error
		ret				;thats all

;-------------------------------------------------------------------------------
; 16 Bit multiplication
; (TOS) <= (TOS) * (SEC)
;-------------------------------------------------------------------------------
expar_mul:	rcall	mul16
expar_mul_1:	mov	XL,r18			;result low
		mov	XH,r19			;result high
		andi	XH,0x7f			;set temp result positive
		brtc	expar_mul_2		;result is positive
		rcall	expar_inva		;invert result
expar_mul_2:	sbrc	r19,7			;skip if no overflow
		rjmp	expar_mul_e		;overflow in bit 15 -> error
		or	r20,r21
		brne	expar_mul_e		;overflow
		ret
expar_mul_e:	ldi	ereg,2			;set error
		ret				;thats all


;-------------------------------------------------------------------------------
; 16*16 Bit signed multiplication
;-------------------------------------------------------------------------------
mul16:		rcall	expar_2abs		
mul16a:		rcall	expar_absa		;absolute val 2
mul16b:		mul	XH,r17			;high bytes
		mov	r20,r0
		mov	r21,r1
		mul	XL,r16			;low bytes
		mov	r18,r0
		mov	r19,r1
		mul	XL,r17			;first cross multiplication
		add	r19,r0		
		adc	r20,r1
		adc	r21,const_0
		mul	XH,r16			;second cross multiplication
		add	r19,r0		
		adc	r20,r1
		adc	r21,const_0
		ret

;-------------------------------------------------------------------------------
; shift value XH,XL to 0...359 
;-------------------------------------------------------------------------------
expar_shan:	ldi	r16,LOW(360)		;interval
		ldi	r17,HIGH(360)
expar_shan1:	cpi	XH,0x80			;sign
		brcs	expar_shan2		;branch if positive
		add	XL,r16			;+360
		adc	XH,r17
		rjmp	expar_shan1	
expar_shan2:	cp	XL,r16			;compare with 360
		cpc	XH,r17
		brcs	expar_shan4		;OK
expar_shan3:	sub	XL,r16
		sbc	XH,r17
		rjmp	expar_shan2
expar_shan4:	ret

;-------------------------------------------------------------------------------
; calc sinus 
;-------------------------------------------------------------------------------
expar_sin:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		ldi	r16,90			;shift
		sub	XL,r16
		sbc	XH,const_0
		rjmp	expar_cos_0

;-------------------------------------------------------------------------------
; calc cosinus 
;-------------------------------------------------------------------------------
expar_cos:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
expar_cos_0:	rcall	expar_shan		;shift to 0..359
		push	ZH
		push	ZL
		ldi	ZL,LOW(bas_sintab*2)
		ldi	ZH,HIGH(bas_sintab*2)
		ldi	r16,LOW(90)
		ldi	r17,HIGH(90)
		cp	XL,r16
		cpc	XH,r17
		brcc	expar_cos_2		;not Q1
		add	ZL,r16
		adc	ZH,r17
		sub	ZL,XL
		sbc	ZH,XH
expar_cos_1:	lpm	XL,Z
		pop	ZL
		pop	ZH
		clr	XH
		rjmp	expar_bp_pa		;goto end

expar_cos_2:	ldi	r16,LOW(180)
		ldi	r17,HIGH(180)
		cp	XL,r16
		cpc	XH,r17
		brcc	expar_cos_4		;not Q2
		ldi	r16,LOW(90)
		ldi	r17,HIGH(90)
		sub	ZL,r16
		sbc	ZH,r17
		add	ZL,XL
		adc	ZH,XH
expar_cos_3:	lpm	XL,Z
		pop	ZL
		pop	ZH
		clr	XH
		com	XL			;switch sign
		com	XH			;
		add	XL,const_1		;
		adc	XH,const_0		;
		rjmp	expar_bp_pa		;goto end

expar_cos_4:	ldi	r16,LOW(270)
		ldi	r17,HIGH(270)
		cp	XL,r16
		cpc	XH,r17
		brcc	expar_cos_5		;->Q4
		add	ZL,r16
		adc	ZH,r17
		sub	ZL,XL
		sbc	ZH,XH
		rjmp	expar_cos_3
		
expar_cos_5:	ldi	r16,LOW(270)
		ldi	r17,HIGH(270)
		sub	ZL,r16
		sbc	ZH,r17
		add	ZL,XL
		adc	ZH,XH
		rjmp	expar_cos_1
							
;-------------------------------------------------------------------------------
; 16 Bit signed division
; r16,r17 / XL,XH => XH,XL + remainder r18,r19
;-------------------------------------------------------------------------------
expar_div:	mov	r18,r17			;HIGH divisor
		or	r18,r16			;or LOW divisor
		brne	expar_div_1		;jump if no zero
		ldi	ereg,3			;set error
		ret				;return (error)

expar_div_1:	rcall	expar_2abs		
		clr	r18			;clear remainder L
		clr	r19			;clear remainder H
		clr	r20			;clear carry source
	    	ldi	r21,17			;loop counter
		
expar_div_2:	rol	r20
		rol	XL			;lshift dividend
		rol	XH			;
		dec	r21			;loop counter
		breq	expar_div_3		;end of loop
		rol	r18			;shift into remainder
		rol	r19			;
		ldi	r20,0xff		;set carry source
		sub	r18,r16			;sub divisor
		sbc	r19,r17			;
		brcc	expar_div_2		;branch if no borrow
		add	r18,r16			;add divisor to restore
		adc	r19,r17			;
		ldi	r20,0x00		;clear carry source
		rjmp	expar_div_2		;goto loop
expar_div_3:	brtc	expar_div_4		;jump if result is positive
		rjmp	expar_inva		;invert result
expar_div_4:	ret				;thats all

;-------------------------------------------------------------------------------
; get sign and absolute value 
;-------------------------------------------------------------------------------
expar_2abs:	mov	r18,XH			;High dividend
		eor	r18,r17			;exclusive or with high divisor
		bst	r18,7			;store sign of result in T
		rcall	expar_absa		;absolute val 1
		rjmp	expar_absb		;absolute val 2
	
;-------------------------------------------------------------------------------
; 16 Bit equal (return 1 if TOS == SEC)
;-------------------------------------------------------------------------------
expar_eq:	eor	r16,XL			;compare low
		eor	r17,XH			;compare high
		clr	XL			;set result to 0
		clr	XH
		or	r16,r17			;diff bits
		breq	expar_true		
		ret

;-------------------------------------------------------------------------------
; 16 Bit greater (return 1 if TOS > SEC)
;-------------------------------------------------------------------------------
expar_gt:  	subi	XH,0x80			;invert sign-bit of OP1
		subi	r17,0x80		;invert sign-bit of OP2
		sub	XL,r16			;SEC-TOS
		sbc	XH,r17
		clr	XL			;set to zero
		clr	XH
		brcs	expar_true		;no carry		
		ret

;-------------------------------------------------------------------------------
; 16 Bit less (return 1 if TOS < SEC)
;-------------------------------------------------------------------------------
expar_lt:	subi	XH,0x80			;invert sign-bit of OP1
		subi	r17,0x80		;invert sign-bit of OP2
		sub	r16,XL			;TOS-SEC
		sbc	r17,XH
		clr	XL			;set to zero
		clr	XH
		brcs	expar_true				
		ret

;-------------------------------------------------------------------------------
; result is true
;-------------------------------------------------------------------------------
expar_true:	ldi	XL,0x01
		ldi	XH,0x00
		ret

;-------------------------------------------------------------------------------
; 16 Bit square root
; TOS <= sqrt(TOS)
; t is set, if operand is negative
;-------------------------------------------------------------------------------
expar_sqr:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		ldi	ereg,4			;set error
		sbrc	XH,7			;skip if positive
		ret
		push	r16
		push	r17
		push	r18
		push	r19
		clr	ereg			;clear error
		mov	r16,XL			;copy value
		mov	r17,XH		
		ldi	XH,2			;offset for tempreg
		clr	XL			;clear result
		mov	r18,const_1		;set tempreg to 1
		clr	r19
expar_sqr_1:	sub	r16,r18			;sub tempreg
		sbc	r17,r19
		brcs	expar_sqr_2		;overflow
		inc	XL			;result+1
		add	r18,XH			;tempreg=tempreg+2
		adc	r19,const_0		;
		rjmp	expar_sqr_1		;loop
expar_sqr_2:	clr	XH			;
		st	-Y,XL			;write low value
		st	-Y,XH			;write high value
		pop	r19
		pop	r18
		pop	r17
		pop	r16
		ret
		
;-------------------------------------------------------------------------------
; 16 Bit RND
;-------------------------------------------------------------------------------
expar_rnd:	push	r20
		push	r21
		libmio_random			;calc a new value
		ld	r17,Y+			;get high value of TOS
		ld	r16,Y+			;get low value of TOS
		rcall	mul16b			;mult
		mov	XL,r20
		mov	XH,r21
		pop	r21
		pop	r20
		rjmp	expar_bp_pa		;goto end


;-------------------------------------------------------------------------------
; player status
;-------------------------------------------------------------------------------
expar_pstat:	ld	r17,Y+			;get high value of TOS
		ld	r16,Y+			;get low value of TOS
		cpi	r16,0x02		;stopp sofort
		brne	expar_pstat_1		;no
		sts	libmio_seqspeed,const_0	;stopp sequencer
		ldi	XL,0x02
		sts	libmio_seqstat,XL
		clr	XH
		rjmp	expar_bp_pa		;goto end

expar_pstat_1:	cpi	r16,0x01		;stopp am Ende
		brne	expar_pstat_2		;no
		lds	XL,libmio_seqstat
		cpi	XL,0x00			;still running?
		brne	expar_pstat_2		;no
		sts	libmio_seqstat,const_1	;stopp sequencer
expar_pstat_2:	lds	XL,libmio_seqstat
		clr	XH
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; SPI IO
;-------------------------------------------------------------------------------
expar_spi:	ld	r17,Y+			;get high value of TOS
		ld	r16,Y+			;get low value of TOS
		out	SPDR0,r16		;put out LOW value
		call	spi_wait
		in	XL,SPDR0
		clr	XH
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; menu
;-------------------------------------------------------------------------------
expar_menu:	lds	r17,libmio_vidmode	;get video mode
		cpi	r17,0x00
		breq	expar_menu_1
		ldi	ereg,28			;not in graphics mode
		ret
		
expar_menu_1:	ld	r17,Y+			;get high value of TOS
		ld	r16,Y+			;get low value of TOS
		ldi	XL,LOW(708)
		ldi	XH,HIGH(708)
		cp	r16,XL
		cpc	r17,XH
		brcs	expar_menu_2
		ldi	ereg,18			;out of array
		ret

expar_menu_2:	push	ZH			;save Z register1
		push	ZL
		ldi	ZH,HIGH(bas_array)
		ldi	ZL,LOW(bas_array)
		add	ZL,r16
		adc	ZH,r17
		set				;read menu data from RAM
		libmio_menu			;show menu
		pop	ZL
		pop	ZH
		
		mov	XL,tempreg1		;set result to 0
		clr	XH 		
		rjmp	expar_bp_pa		;goto end
		
;-------------------------------------------------------------------------------
; arrayelement value
;-------------------------------------------------------------------------------
expar_ar:	ld	XH,Y+			;get high value of TOS
		sts	bas_ram+11,XH		;store area
		ld	XL,Y+			;get low value of TOS
		sts	bas_ram+10,XL		;store offset
		push	YL
		push	YH
		call	bas_rarr
		pop	YH
		pop	YL
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; filetype
;-------------------------------------------------------------------------------
expar_ftype:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		cpi	XH,0xff
		breq	expar_ftype_2		;-1
		sts	bas_partab,XL		;file number
		call	fsys_gettype		;get file type
		cpi	ereg,0x00
		breq	expar_ftype_1
expar_ftype_0:	ldi	XL,0xff			;-1
		ldi	XH,0xff
		clr	ereg		
		rjmp	expar_bp_pa		;goto end
		
expar_ftype_1:	mov	XL,tempreg1
		clr	XH
		jmp	expar_bp_pa		;goto end
		
expar_ftype_2:	call	fsys_size
		clr	ereg
		mov	XH,tempreg1
		clr	XL
		rjmp	expar_bp_pa		;goto end
	
;-------------------------------------------------------------------------------
; flash/file-size
; -1 free files
; -2 free pages
; other 
;-------------------------------------------------------------------------------
expar_fsize:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		cpi	XH,0xff
		brne	expar_fsize_4		;
		sbrc	XL,0
		rjmp	expar_fsize_2
;free data pages		
		push	YH
		push	YL
		push	ZH
		push	ZL
		call	fsys_free		;calc pages
		cpi	ereg,0x00
		breq	expar_fsize_1
		clr	ereg
		clr	YL
		clr	YH
expar_fsize_1:	movw	XL,YL			;copy value
		pop	ZL
		pop	ZH
		pop	YL
		pop	YH
		rjmp	expar_bp_pa		;goto end

expar_fsize_2:	push	YH
		push	YL
		push	ZH
		push	ZL
		call	fsys_free		;calc pages
		cpi	ereg,0x00
		breq	expar_fsize_3
		clr	ereg
		clr	ZL
		clr	ZH
expar_fsize_3:	movw	XL,ZL			;copy value
		pop	ZL
		pop	ZH
		pop	YL
		pop	YH
		rjmp	expar_bp_pa		;goto end
						

expar_fsize_4:	sts	bas_partab,XL		;file number
		call	fsys_gettype		;get type byte (dummy) and set page
		cpi	ereg,0x00
		breq	expar_fsize_6
expar_fsize_5:	clr	ereg
		clr	XL			;result is 0
		clr	XH		
		rjmp	expar_bp_pa		;goto end

expar_fsize_6:	push	YH
		push	YL
		push	tempreg4
		clr	tempreg4
expar_fsize_7:	call	fsys_gword		;get data word
		cpi	YH,0xff
		brne	expar_fsize_9
		cpi	YL,0xff
		brne	expar_fsize_9
expar_fsize_8:	mov	XL,tempreg4
		clr	XH
		pop	tempreg4		;restore registers
		pop	YL
		pop	YH
		rjmp	expar_bp_pa		;goto end
	
expar_fsize_9:	inc	tempreg4
		cpi	tempreg4,0x80
		brne	expar_fsize_7
		rjmp	expar_fsize_8		

;-------------------------------------------------------------------------------
; analog input
;-------------------------------------------------------------------------------
expar_adc:	ld	XL,Y+			;get high value of TOS (dummy)
		ld	XL,Y+			;get low value of TOS
		andi	XL,0x07
		ori	XL,0xc0
		sts	ADMUX,XL
		lds	XL,ADCSRA
		ori	XL,0x40			;set ADSC
		sts	ADCSRA,XL
expar_adc1:	lds	XL,ADCSRA
		sbrc	XL,6
		rjmp	expar_adc1
		lds	XL,ADCL
		lds	XH,ADCH		
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; digital input
;-------------------------------------------------------------------------------
expar_din:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		cpi	XH,0x00			;mask
		brne	expar_inm		;masked
		cpi	XL,255
		breq	expar_inb
		andi	XL,0x07
		inc	XL
		in	XH,PINA
expar_din_1:	lsr	XH
		dec	XL
		brne	expar_din_1	
		clr	XL
		rol	XL
		clr 	XH
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; masked input byte
;-------------------------------------------------------------------------------
expar_inm:	mov	XH,XL			;copy mask
		in	XL,PINA
		and	XL,XH			;mask bits
		clr	XH
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; input byte
;-------------------------------------------------------------------------------
expar_inb:	in	XL,PINA
		clr	XH
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; xpeek
;-------------------------------------------------------------------------------
expar_xpeek:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		call	mem_getconf1		;chip address
		libi2c_read			;get byte
		cpi	ereg,0
		brne	expar_i2err
		clr	XH
		mov	XL,tempreg2
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; epeek
;-------------------------------------------------------------------------------
expar_epeek:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		push	YL
		push	YH
		mov	YL,XL
		mov	YH,XH
		libeep_read			;get byte
		pop	YH
		pop	YL	
		cpi	ereg,0
		brne	expar_i2err
		clr	XH
		mov	XL,tempreg1
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; temp
;-------------------------------------------------------------------------------
expar_temp:	ld	XL,Y+			;get high value of TOS (dummy)
		ld	tempreg4,Y+
		libi2c_rlm75
		cpi	ereg,0
		brne	expar_i2err
		rjmp	expar_bp_pa		;goto end
expar_i2err:	clr	XL
		clr	XH
		ldi	ereg,15
		rjmp	expar_bp_e6

;-------------------------------------------------------------------------------
; lo
;-------------------------------------------------------------------------------
expar_lo:	ld	XH,Y+			;get high value of TOS (dummy)
		ld	XL,Y+			;get low value of TOS
		clr	XH
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; hi
;-------------------------------------------------------------------------------
expar_hi:	ld	XH,Y+			;get high value of TOS (dummy)
		ld	XL,Y+			;get low value of TOS
		mov	XL,XH
		clr	XH
		rjmp	expar_bp_pa		;goto end

;-------------------------------------------------------------------------------
; key
;-------------------------------------------------------------------------------
expar_key:	push	r16
		ld	XH,Y+			;get high value of TOS (dummy)
		ld	r16,Y+			;get low value of TOS
		clr	XL			;neutral value
		clr	XH
		cpi	r16,0			;getkey
		brne	expar_key_01		;no
		lds	XL,libmio_keycode	
		rjmp	expar_key_end
		
expar_key_01:	cpi	r16,1			;waitkey
		brne	expar_key_02			
expar_key_01a:	lds	XL,libmio_keycode	;get keycode
		cpi	XL,0x00			;no key
		breq	expar_key_01a
		rjmp	expar_key_end
		
expar_key_02:	cpi	r16,2			;waitnokey
		brne	expar_key_03
expar_key_02a:	lds	XL,libmio_keycode	;get keycode
		cpi	XL,0x00			;no key
		brne	expar_key_02a
		lds	XL,libmio_lastkey
		rjmp	expar_key_end
						
expar_key_03:	cpi	r16,3
		brne	expar_key_04
		lds	XL,libmio_kbdstate	;keyboard status
		rjmp	expar_key_end
				
		
expar_key_04:	cpi	r16,4			;left s/c
		brne	expar_key_05		;jump if no lshift
		lds	r16,libmio_kbdstate	
		sbrc	r16,0			;left shift
		adiw	XL,1
		sbrc	r16,2			;left ctrl
		sbiw	XL,1
		rjmp	expar_key_end
		
expar_key_05:	cpi	r16,5							
		brne	expar_key_06		;jump if no rshift
		lds	r16,libmio_kbdstate	
		sbrc	r16,1			;right shift
		adiw	XL,1
		sbrc	r16,3			;right ctrl
		sbiw	XL,1
		rjmp	expar_key_end
		
expar_key_06:	cpi	r16,6							
		brne	expar_key_07		;jump if no xcursor
		lds	r16,libmio_keycode
		cpi	r16,0xe2		;left
		brne	expar_key_06a
		sbiw	XL,1
expar_key_06a:	cpi	r16,0xe3		;right
		brne	expar_key_end
		adiw	XL,1
		rjmp	expar_key_end
						
expar_key_07:	cpi	r16,7
		brne	expar_key_08
		lds	r16,libmio_keycode
		cpi	r16,0xe4		;up
		brne	expar_key_07a
		adiw	XL,1
expar_key_07a:	cpi	r16,0xe5		;down
		brne	expar_key_end
		sbiw	XL,1
		rjmp	expar_key_end
		
expar_key_08:	push	r17
expar_key_08a:	lds	r17,libmio_keycode
		cp	r16,r17
		brne	expar_key_08a
		pop	r17
		mov	XL,r16
		clr	XH		
expar_key_end:	pop	r16			;restore reg
		rjmp	expar_bp_pa		;goto end



;-------------------------------------------------------------------------------
; err(
;-------------------------------------------------------------------------------
expar_geterr:	ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		clr	XH
		cpi	XL,0x01			;line
		brne	expar_geterr1		;no
		lds	XL,bas_ram+5
		inc	XL			;+1
		rjmp	expar_bp_pa		;goto end

expar_geterr1:	cpi	XL,0x02			;statement
		brne	expar_geterr2		;no
		lds	XL,bas_ram+6
		inc	XL			;+1
		rjmp	expar_bp_pa		;goto end

expar_geterr2:	lds	XL,bas_ram+7		;errno
		rjmp	expar_bp_pa		;goto end

		
expar_ttab:	.db "ABS SGN SQR NOT RND ADC IN XPEEK TEMP LO HI KEY EPEEK ERR AR SIN COS FTYPE FSIZE PSTAT SPI ",0