;################################################################################
;#										#
;# avr-chipbasic2 - single chip basic computer with ATmega644			#
;# runtime engine								#
;# copyright (c) 2006-2008 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 2		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################

tbrun:
;-----------------------------------------------------------------------
; clear all variables
;-----------------------------------------------------------------------
		ldi	ZH,HIGH(varspace)	
		ldi	ZL,LOW(varspace)
		ldi	XL,0			;zero
		ldi	XH,180			;variables
tbrun_i1:	st	Z+,XL
		dec	XH
		brne	tbrun_i1
		ldi	ZH,HIGH(bas_ram)	
		ldi	ZL,LOW(bas_ram)	
		ldi	XH,16			;basic system
tbrun_i2:	st	Z+,XL
		dec	XH
		brne	tbrun_i2
		
		ldi	ZH,HIGH(bas_array)	
		ldi	ZL,LOW(bas_array)	
		ldi	XH,0			;array
tbrun_i3:	st	Z+,XL
		st	Z+,XL
		st	Z+,XL
		dec	XH
		brne	tbrun_i3


		clr	r13			;statement 1
		clr	r12			;line 1
		sts	libmio_border,const_0
		sts	libmio_font,const_0
		ldi	XL,0x0e			;white on black
		sts	libmio_color,XL		;set color
		ldi	XL,libmio_v0cols	;set clipping to mode 0
		sts	libmio_clipx2,XL
		ldi	XL,libmio_v0rows
		sts	libmio_clipy2,XL
		ldi	XL,0xff			;no error handling
		sts	bas_ram+15,XL
		libmio_clrscr			;clear screen
		sts	libmio_seqspeed,const_0	;stop sequencer
		ldi	XL,0x02
		sts	libmio_seqstat,XL	;status=stopped
		ldi	XL,0xc6
		sts	ADCSRA,XL		;enable ADC
		lds	tempreg1,libmio_lastmenu;program number
		sts	libmio_prog,tempreg1	;temp program number (run)
		sts	libmio_tprog,tempreg1
		call	mem_setprog		;set program number
		
;-----------------------------------------------------------------------
; main loop for all lines
;-----------------------------------------------------------------------
tbrun_02:	mov	ctrl,r12		;set line number
		call	mem_readline		;read line from EEPROM
		rcall	basrun			;interpret line
		cpi	ereg,0x00		
		brne	tbrun_03
		mov	XL,r12			;linenumber
		cpi	XL,bas_lines
		brcs	tbrun_02
		ldi	ctrl,0x0c		;standard format
		sts	libmio_seqspeed,const_0	;stop sequencer
		ret				;line numer exceeds
;error		
tbrun_03:	sts	bas_ram+5,r12		;line
		sts	bas_ram+6,r13		;statement
		sts	bas_ram+7,ereg		;error
		cpi	ereg,1
		breq	tbrun_04
		sts	libmio_seqspeed,const_0	;stop sequencer
		lds	tempreg1,bas_ram+15	;on error
		cpi	tempreg1,bas_lines	;deactivated
		brcc	tbrun_04		
		clr	r13
		mov	r12,tempreg1
		clr	ereg			;clear error
		rjmp	tbrun_02		;error handling
		
tbrun_04:	ldi	ctrl,0x0c		;standard format
		sts	libmio_seqspeed,const_0	;stop sequencer
		ret				;line numer exceeds

	
tbrun_loop:	movw	ZL,r14			;get pointer
tbrun_mtest:	rjmp	basrun_05		;end
	
;-----------------------------------------------------------------------
; interpret line from Z
;-----------------------------------------------------------------------
basrun:		ldi	ZH,HIGH(bas_linebuf)	;start of buffer
		ldi	ZL,LOW(bas_linebuf)	;start of buffer
		sts	libmio_channel,const_0	;set output to screen
		clr	ereg			;clear Error
		mov	XL,r13			;statement to go
		clr	XH
basrun_01:	cp	XL,XH			;found?
		breq	basrun_05		;yes

basrun_02:	ld	tempreg4,Z+		;get char
		cpi	tempreg4,0xff		;end of line
		breq	basrun_07		;next line

		cpi	tempreg4,0x1c		;numbers
		breq	basrun_cnum8
		cpi	tempreg4,0x1e		;numbers
		breq	basrun_cnum8
		cpi	tempreg4,0x1d		;numbers
		breq	basrun_cnum16
		cpi	tempreg4,0x1f		;numbers
		breq	basrun_cnum16

		cpi	tempreg4,0x27		;comment
		breq	basrun_07		;next line
		cpi	tempreg4,'"'		;string
		brne	basrun_04
basrun_03:	ld	tempreg4,Z+		;get char
		cpi	tempreg4,0xff		;end of line
		breq	basrun_07		;yes
		cpi	tempreg4,0x27		;comment
		breq	basrun_07	
		cpi	tempreg4,'"'		;string
		brne	basrun_03
		rjmp	basrun_02
basrun_04:	cpi	tempreg4,':'		;separator	
		brne	basrun_02
		inc	XH			;statement+1
		cp	XL,XH			;found?
		brne	basrun_02
		rjmp	basrun_05
					
basrun_cnum16:	ld	tempreg4,Z+		;ignore compressed value
basrun_cnum8:	ld	tempreg4,Z+		;ignore compressed value
		rjmp	basrun_02

basrun_05:	lds	tempreg4,libmio_kflags	;break
		sbrs	tempreg4,0
		rjmp	basrun_mtest
		
basrun_brk:	ldi	ereg,1			;set ereg to 1
		andi	tempreg4,0xfe		;clear break flag
		sts	libmio_kflags,tempreg4			
		ret
		
basrun_mtest:	sbrs	tempreg4,2
		rjmp	basrun_06		;no monitor
		andi	XL,0xfb			;clear monitor flag
		sts	libmio_kflags,tempreg4	;store back
		lds	XL,libmio_vidmode
		cpi	XL,0x05
		breq	basrun_mtest0		
		cpi	XL,0x00
		brne	basrun_mtest1
basrun_mtest0:	call	mon_main		;call monitor
basrun_mtest1:	sbrs	XL,0			;esc flag
		rjmp	basrun_06
		pop	r0			;kill stack
		pop	r0
		ldi	ereg,1
		ret	
						;end

basrun_06:	libmio_screenshot
		ld	tempreg4,Z+		;get char
		cpi	tempreg4,0xff		;EOL?
		brne	basrun_08
basrun_07:	clr	r13			;first statement
		inc	r12			;next line
		ret				;end of line

basrun_08:	sbrc	tempreg4,7
		rjmp	basrun_kw		;token
		cpi	tempreg4,32		;space
		breq	basrun_05
		cpi	tempreg4,':'
		brne	basrun_10
		inc	r13
		rjmp	basrun_05

basrun_10:	cpi	tempreg4,0x27		;comment
		breq	basrun_07		;next line

		cpi	tempreg4,'A'
		brcs	basrun_se
		cpi	tempreg4,'Z'+1
		brcs	basrun_vset

		subi	tempreg4,0x20		;set to caps

		cpi	tempreg4,'A'
		brcs	basrun_se
		cpi	tempreg4,'Z'+1
		brcc	basrun_se
		
		
;-----------------------------------------------------------------------
; check for variable setting
;-----------------------------------------------------------------------
basrun_vset:	mov	tempreg7,tempreg4	;store variable number
		call	bas_ispace    	;space
	    	cpi	tempreg4,'='
		brne	basrun_se
		rcall	expar			;call parser
		cpi	ereg,0
		brne	basrun_end
		mov	tempreg4,tempreg7
		subi	tempreg4,'A'
		lsl	tempreg4			
		ldi	YH,high(varspace)	
		ldi	YL,low(varspace)
		add	YL,tempreg4
		adc	YH,const_0					
		st	Y+,XL
		st	Y+,XH
		rjmp	tbrun_mtest		;monitor test

basrun_arse:	pop	ZL			;restore pointer
		pop	ZH
basrun_se:	ldi	ereg,7			;syntax error
basrun_end:	ret		


		cpi	tempreg4,0x80
		brcc	basrun_kw
		ldi	ereg,7			;syntax error		
		rjmp	tbrun_loop
		
;-----------------------------------------------------------------------
; check for keywords
;-----------------------------------------------------------------------
basrun_kw:	movw	r14,ZL			;store pointer
		ldi	ZL,LOW(basrun_kw1)
		ldi	ZH,HIGH(basrun_kw1)
		andi	tempreg4,0x3f		;64 token
		add	ZL,tempreg4
		adc	ZH,const_0
		ijmp
    		
basrun_kw1:	rjmp	bas_print		;80
		rjmp	bas_cls			;81
		rjmp	bas_goto		;82
		rjmp	bas_box			;83
		rjmp	bas_acopy		;84
		rjmp	bas_sync		;85
		rjmp	bas_end			;86
		rjmp	bas_bcopy		;87
		rjmp	bas_then		;88
		rjmp	bas_posxy		;89
		rjmp	bas_note		;8a
		rjmp	basrun_kw2		;8b
		rjmp	bas_rread		;8c
		rjmp	bas_rwrite		;8d
		rjmp	bas_gosub		;8e
		rjmp	bas_return		;8f
		
		rjmp	bas_vmode		;90
		rjmp	bas_next		;91
		rjmp	bas_input		;92
		rjmp	bas_plot		;93
		rjmp	bas_wait		;94
		rjmp	bas_tset		;95
		rjmp	bas_tget		;96
		rjmp	bas_fsel		;97
		rjmp	bas_onerr		;98 
		rjmp	bas_sput		;99
		rjmp	bas_sget		;9a
		rjmp	bas_epoke		;9b
		rjmp	bas_gpix		;9c
		rjmp	bas_circle		;9d
		rjmp	bas_color		;9e
		rjmp	bas_border		;9f
		
		rjmp	bas_font		;a0
		rjmp	bas_chpump		;a1
		rjmp	bas_xpoke		;a2
		rjmp	basrun_kw2		;a3 not used
		rjmp	bas_limit		;a4
		rjmp	bas_gchar		;a5
		rjmp	bas_drawto		;a6
		rjmp	bas_play		;a7
		rjmp	bas_sprite		;a8
		rjmp	bas_scroll		;a9
		rjmp	bas_breakpoint		;aa
		rjmp	bas_fast		;ab
		rjmp	bas_slow		;ac
		rjmp	bas_draw		;ad
		rjmp	bas_icomm		;ae
		rjmp	bas_data		;af
		
		rjmp	bas_ctext		;b0
		rjmp	bas_fread		;b1 read page
		rjmp	bas_fwrite		;b2 write page
		rjmp	bas_cbox		;b3
		rjmp	bas_gattr		;b4
		rjmp	bas_fdelete		;b5 
		rjmp	bas_fbox		;b6
		rjmp	bas_setpal		;b7
		rjmp	bas_for			;b8
		rjmp	bas_if			;b9
		rjmp	bas_dir			;ba
		rjmp	bas_out			;bb
		rjmp	bas_fcircle		;bc
		rjmp	bas_alert		;bd 
		rjmp	bas_fcreate		;be
		rjmp	bas_ask			;bf
				
		
basrun_kw2:	ldi	ereg,8			;unknown
		ret
basrun_noval:	ldi	ereg,20			;no value expected
		ret		

tbrun_wrongpar:	ldi	ereg,19			;incomplete parameter		
		ret

						
;-----------------------------------------------------------------------
; keyword is PRINT
;-----------------------------------------------------------------------
bas_print:	call	bas_rlp			;restore Z-register
bas_print_01:	ldi	ctrl,0x0c		;standard format
		mov	tempreg8,const_1		;enable newline
		sts	libmio_channel,const_0	;print to screen
bas_print_02:	ld	XH,Z+
		cpi	XH,32			;leading space
		breq	bas_print_02
		cpi	XH,':'			;end of statement
		breq	bas_print_end
		cpi	XH,0xff			;end of line
		brne	bas_print_sep
bas_print_end:	sbiw	ZL,1			;points to this char		
		sbrc	tempreg8,0		;newline?
		libmio_newline
		lds	tempreg1,libmio_channel
		cpi	tempreg1,4		;I2C
		brcs	bas_print_end1
		libi2c_stop	
bas_print_end1:	rjmp	tbrun_mtest		;return to main loop
	
;-------------- separator ----------------------------------------------		
bas_print_sep:	cpi	XH,0x3b			;";"
		brne	bas_print_tab
		clr	tempreg8		;no newline
		rjmp	bas_print_02		;print loop

;-------------- tabulator ----------------------------------------------		
bas_print_tab:	cpi	XH,','			;tab
		brne	bas_print_05

		lds	tempreg1,libmio_channel
		cpi	tempreg1,0		;screen
		brne	bas_print_tab2
bas_print_tab1:	libmio_outspace
		mov	XH,libmio_cur_x
		andi	XH,0x07
		brne	bas_print_tab1
		rjmp	bas_print_tab3
		
bas_print_tab2:	libmio_outspace
		lds	XH,libmio_lpos
		andi	XH,0x07
		brne	bas_print_tab2
		
bas_print_tab3:	clr	tempreg8		;no newline
	    	rjmp	bas_print_02		;print loop1
		
;-------------- enable newline if no separator --------------------------		
bas_print_05:	mov	tempreg8,const_1        	;enable newline

;------------- set channel --------------------------------------------------------
bas_print_ch:	cpi	XH,'#'			;channel set?
		brne	bas_print_at		;no
		sts	bas_partab,const_0	;set array index to 0
		sts	bas_partab+1,const_0
		rcall	expar			;get value
		mov	tempreg1,XL		;copy value
		sts	libmio_channel,tempreg1	;store channel number
		cpi	tempreg1,4		;above 3 -> I2C
		brcc	bas_print_ch2
		cpi	tempreg1,3
		brne	bas_print_ch1
bas_print_ch1:	rjmp	bas_print_02		;return to print loop
						
bas_print_ch2:	libi2c_start			;start
		cpi	tempreg3,0x08
		brne	bas_print_i2e
		lds	tempreg2,libmio_channel
		andi	tempreg2,0xfe		;mask bit 0 (write)
		libi2c_wbyte			;write address to bus
		cpi	tempreg3,0x18
		brne	bas_print_i2e
		rjmp	bas_print_02		;return to print loop		

;------------- set position --------------------------------------------------------
bas_print_at:	cpi	XH,'@'			;position set?
		brne	bas_print_for		;no
		rcall	bas_g2b			;get two byte values
		lds	tempreg1,libmio_channel ;get channel number
		cpi	tempreg1,3
		brcc	bas_print_at1
		libmio_gotoxy			;set position
		rjmp	bas_print_02		;print loop
				
bas_print_at1:	brne	bas_print_at2
		sts	bas_partab,XL		;set array index
		sts	bas_partab+1,XH
		sts	libmio_lpos,XL
		rjmp	bas_print_02		;print loop
				
bas_print_at2:	ldi	tempreg2,0xff		;cordic statement
		libi2c_wbyte			;write to I2C
		cpi	tempreg3,0x28
		brne	bas_print_i2e
		mov	tempreg2,XL		;X-coord
		libi2c_wbyte			;write to I2C
		cpi	tempreg3,0x28
		brne	bas_print_i2e
		mov	tempreg2,XH		;Y-coord
		libi2c_wbyte			;write to I2C
		cpi	tempreg3,0x28
		brne	bas_print_i2e
bas_print_at3:	rjmp	bas_print_02		;print loop		

bas_print_i2e:	ldi	ereg,15			;I2C error
		ret

;------------- set format --------------------------------------------------------
bas_print_for:	cpi	XH,'!'			;format?
		brne	bas_print_str		;no
		rcall	expar
		mov	ctrl,XL
		rjmp	bas_print_02				

bas_print_ex:	rjmp	bas_print_end		
bas_print_02x:	rjmp	bas_print_02	

;-------------- a string ------------------------------------------------		
bas_print_str:	cpi	XH,'"'			;is it a string?
		brne	bas_print_atx		;no
bas_print_str1:	ld	tempreg1,Z+		;get string char
		cpi	tempreg1,0xff
		breq	bas_print_ex		;yes
		cpi	tempreg1,'"'		;end of string
		breq	bas_print_02x		;yes
		libmio_outchar			;output char
		rjmp	bas_print_str1		;string loop

;------------- array text --------------------------------------------------------
bas_print_atx:	cpi	XH,'&'			;format?
		brne	bas_print_char		;no
		rcall	expar
		cpi	XH,0x03
		brcs	bas_print_atx1
		ldi	ereg,18			;out of array
		ret
bas_print_atx1:	ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		add	YL,XL
		adc	YH,XH
bas_print_atx2:	ld	tempreg1,Y+
		cpi	tempreg1,0x00
		breq	bas_print_atx4
		libmio_outchar
		adiw	XL,1
		cpi	XH,0x03
		brcs	bas_print_atx2			
	
bas_print_atx4:	rjmp	bas_print_02				


;------------- a single char --------------------------------------------
bas_print_char:	cpi	XH,'%'			;char
		brne	bas_print_equ
		rcall	expar
		mov	tempreg1,XL
		libmio_outchar
		rjmp	bas_print_02				

;------------- an equation --------------------------------------------
bas_print_equ:	sbiw	ZL,1			;set pointer to char before
		rcall	expar			;mathematics
		cpi	ereg,0
		breq	bas_print_14
		ret
		
bas_print_14:	cpi	ctrl,0x80		;Hex/dez
		brcc	bas_print_15
		libmio_outdez		
		rjmp	bas_print_02				

bas_print_15:	libmio_outhex
		rjmp	bas_print_02				

;-----------------------------------------------------------------------
; keyword is CLS
;-----------------------------------------------------------------------
bas_cls:	ldi	XL,0			;max 0 parameters
		rcall	bas_getpar		;get parameters
		libmio_clrscr
		rjmp	tbrun_loop		;goto main interpreter loop

		
;-----------------------------------------------------------------------
; keyword is color
;-----------------------------------------------------------------------
bas_color:	ldi	XL,2			;max 0 parameters
		rcall	bas_getpar		;get parameters
		cpi	XL,0			;no parameter
		breq	bas_color_e
		cpi	XL,1			;only foreground
		breq	bas_color_1
		lds	XL,bas_partab+2		;background
		andi	XL,0x07
		lsl	XL			;color*2 in vm0	
bas_color_0:	lds	XH,libmio_color
		andi	XH,0x07
		swap	XL
		or	XH,XL		
		sts	libmio_color,XH
bas_color_1:	lds	XL,bas_partab		;foreground
		andi	XL,0x07
		lsl	XL			;color*2 in vm0	
bas_color_2:	lds	XH,libmio_color
		andi	XH,0xf0
		or	XH,XL
		sts	libmio_color,XH
bas_color_3:	rjmp	tbrun_loop		;goto main interpreter loop
bas_color_e:	rjmp	tbrun_wrongpar


;-----------------------------------------------------------------------
; keyword is acopy
;-----------------------------------------------------------------------
bas_acopy:	ldi	XL,0x03			;max 3 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x03			;min parameters
		brcs	bas_acopy_e		;<3 is not OK
		lds	XL,bas_partab+0		;source
		lds	XH,bas_partab+1
		sts	bas_ram+10,XL		;copy
		sts	bas_ram+11,XH
		lds	XL,bas_partab+2		;dest
		lds	XH,bas_partab+3
		sts	bas_partab+0,XL		;dest
		sts	bas_partab+1,XH				
		lds	XL,bas_partab+4		;par3 is number
		lds	XH,bas_partab+5
bas_acopy_1:	cpi	XH,0x00
		brne	bas_acopy_2
		cpi	XL,0x00
		breq	bas_acopy_9

bas_acopy_2:	movw	r0,XL
		call	bas_rarr		;read array value
		call	bas_warr		;write array value
		movw	XL,r0
		cpi	ereg,0x00		;OK?
		brne	bas_acopy_o
		sbiw	XL,1
		rjmp	bas_acopy_1				
	    
bas_acopy_9:	rjmp	tbrun_loop		;goto main interpreter loop
bas_acopy_e:	rjmp	tbrun_wrongpar		;error
bas_acopy_o:	ret


;-----------------------------------------------------------------------
; keyword is box
;-----------------------------------------------------------------------
bas_box:	ldi	XL,0x05			;max 5 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x04			;min parameters
		brcs	bas_box_e		;<4 is not OK
		rcall	bas_ccheck
		push	tempreg4
		rcall	bas_g4c
		sts	bas_ram+3,ZL		;last X
		sts	bas_ram+4,ZH		;last Y		
		libmio_box			;draw box
		pop	tempreg4
		sts	libmio_color,tempreg4	;restore color
		rjmp	tbrun_loop		;goto main interpreter loop
bas_box_e:	rjmp	tbrun_wrongpar		;error

;-----------------------------------------------------------------------
; keyword is font
;-----------------------------------------------------------------------
bas_font:	ldi	XL,0x01			;max 5 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x01			;min parameters
		brcs	bas_box_e		;<4 is not OK
		lds	tempreg1,bas_partab
		andi	tempreg1,0x01
		sts	libmio_font,tempreg1
		rjmp	tbrun_loop		;goto main interpreter loop

;-----------------------------------------------------------------------
; keyword is fbox
;-----------------------------------------------------------------------
bas_fbox:	ldi	XL,0x05			;max 5 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x04			;min parameters
		brcs	bas_fbox_e		;<4 is not OK
		rcall	bas_ccheck
		push	tempreg4
		rcall	bas_g4c
		sts	bas_ram+3,ZL		;last X
		sts	bas_ram+4,ZH		;last Y		
		libmio_fbox			;draw box
		pop	tempreg4
		sts	libmio_color,tempreg4	;restore color
		rjmp	tbrun_loop		;goto main interpreter loop
bas_fbox_e:	rjmp	tbrun_wrongpar		;error


;-----------------------------------------------------------------------
; keyword is circle
;-----------------------------------------------------------------------
bas_circle:	ldi	XL,0x05			;max 5 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x04			;min parameters
		brcs	bas_circle_e		;<4 is not OK
		rcall	bas_ccheck
		push	tempreg4
		rcall	bas_g4c
		sts	bas_ram+3,YL		;last X
		sts	bas_ram+4,YH		;last Y		
		libmio_circle			;draw circle
		pop	tempreg4
		sts	libmio_color,tempreg4	;restore color
		rjmp	tbrun_loop		;goto main interpreter loop
bas_circle_e:	rjmp	tbrun_wrongpar		;error


;-----------------------------------------------------------------------
; keyword is fcircle
;-----------------------------------------------------------------------
bas_fcircle:	ldi	XL,0x05			;max 5 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x04			;min parameters
		brcs	bas_fcircle_e		;<4 is not OK
		rcall	bas_ccheck
		push	tempreg4
		rcall	bas_g4c
		sts	bas_ram+3,YL		;last X
		sts	bas_ram+4,YH		;last Y		
		libmio_fcircle			;draw circle
		pop	tempreg4
		sts	libmio_color,tempreg4	;restore color
		rjmp	tbrun_loop		;goto main interpreter loop
bas_fcircle_e:	rjmp	tbrun_wrongpar		;error


;-----------------------------------------------------------------------
; keyword is plot
;-----------------------------------------------------------------------
bas_plot:	ldi	XL,0x03
		rcall	bas_getpar		;get parameters
		cpi	XL,0x02			;with color
		brcs	bas_plot_e		;<2 is not OK
		rcall	bas_ccheck
		push	tempreg4
		lds	XH,bas_partab		;X1
		lds	XL,bas_partab+2		;Y1
		sts	bas_ram+3,XL		;last X
		sts	bas_ram+4,XH		;last Y		
		libmio_plot
		pop	tempreg4
		sts	libmio_color,tempreg4	;restore color	
		rjmp	tbrun_loop		;goto main interpreter loop
bas_plot_e:	rjmp	tbrun_wrongpar		;error

				
;-----------------------------------------------------------------------
; keyword is draw
;-----------------------------------------------------------------------
bas_draw:	ldi	XL,0x05			;max 5 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x04			;min parameters
		brcs	bas_draw_e
		rcall	bas_ccheck		;color value?
		push	tempreg4
		rcall	bas_g4c
		sts	bas_ram+3,ZL		;set new old
		sts	bas_ram+4,ZH
		libmio_draw
		pop	tempreg4
		sts	libmio_color,tempreg4	;restore color	
		rjmp	tbrun_loop		;goto main interpreter loop
bas_draw_e:	rjmp	tbrun_wrongpar		;error


;-----------------------------------------------------------------------
; keyword is alert
;-----------------------------------------------------------------------
bas_alert:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_alert_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_alert_tm:	ldi	XL,0x01			;get 1 parameter
		rcall	bas_getpar
		cpi	XL,0x01
		brcs	bas_draw_e		;error
		lds	XL,bas_partab		;lo ptr
		lds	XH,bas_partab+1		;hi ptr
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		add	YL,XL
		adc	YH,XH
		lds	tempreg1,libmio_color
		push	tempreg1		;save status
		push	libmio_cur_x
		push	libmio_cur_y
		libmio_alert			;alert
		pop	libmio_cur_y		;restore status
		pop	libmio_cur_x
		pop	tempreg1
		sts	libmio_color,tempreg1
		rjmp	tbrun_loop		;goto main interpreter loop

;-----------------------------------------------------------------------
; keyword is ask
;-----------------------------------------------------------------------
bas_ask:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_ask_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_ask_tm:	rcall	bas_getvar		;variable
		sts	bas_ram+10,YL		;save var address
		sts	bas_ram+11,YH
		rcall	bas_ispace
		cpi	tempreg4,','
		brne	bas_ask_err
		ldi	XL,0x01			;get 1 parameter
		rcall	bas_getpar_0
		cpi	XL,0x01
		brcs	bas_draw_e		;error
		lds	XL,bas_partab		;lo ptr
		lds	XH,bas_partab+1		;hi ptr
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		add	YL,XL
		adc	YH,XH
		lds	tempreg1,libmio_color
		push	tempreg1		;save status
		push	libmio_cur_x
		push	libmio_cur_y
		libmio_ask			;ask box
		pop	libmio_cur_y		;restore status
		pop	libmio_cur_x
		lds	YL,bas_ram+10
		lds	YH,bas_ram+11
		st	Y+,tempreg1		;result is 0/1
		st	Y+,const_0
		pop	tempreg1
		sts	libmio_color,tempreg1
		rjmp	tbrun_loop		;goto main interpreter loop
		
bas_ask_err:	ldi	ereg,0x07		;syntax error
		rjmp	tbrun_loop		;error


;-----------------------------------------------------------------------
; keyword is fsel
;-----------------------------------------------------------------------
bas_fsel:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_fsel_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_fsel_tm:	rcall	bas_getvar		;variable
		sts	bas_ram+10,YL		;save var address
		sts	bas_ram+11,YH
		rcall	bas_ispace
		cpi	tempreg4,','
		brne	bas_ask_err
		ldi	XL,0x01			;get 1 parameter
		rcall	bas_getpar_0
		cpi	XL,0x01
		breq	bas_fsel_01		;no error
		rjmp	tbrun_wrongpar

bas_fsel_01:			

bas_fsel_02:	push	ZH
		push	ZL
		push	libmio_cur_x
		push	libmio_cur_y
		lds	tempreg1,libmio_color
		push	tempreg1		;save status
		push	tempreg4
		andi	tempreg1,0x0f		;set bg to black
		sts	libmio_color,tempreg1
		call	mon_backup
		ldi	YL,1
		ldi	YH,2
		ldi	ZL,28
		ldi	ZH,21		
		libmio_whbox
		lds	YL,libmio_color
		swap	YL
		sts	libmio_color,YL
		libmio_thistext
		.db	3,2,0,0			;set pos
		lds	XL,bas_partab		;lo ptr
		lds	XH,bas_partab+1		;hi ptr
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		add	YL,XL
		adc	YH,XH
		ldi	XL,22			;max chars		
bas_fsel_03:	ld	tempreg1,Y+		;get char
		cpi	tempreg1,32		;space
		brcs	bas_fsel_04		;less
		cpi	tempreg1,0x80
		brcc	bas_fsel_04
		libmio_outchar
		dec	XL
		brne	bas_fsel_03
		
bas_fsel_04:	clr	tempreg4		;file 1
		ldi	tempreg3,0x01		;no valid selection
		rcall	bas_fsel_tcall		;select file
		call	mon_restore	
		cpi	tempreg3,0x02		;valid
		brne	bas_fsel_05		
		lds	YL,bas_ram+10
		lds	YH,bas_ram+11
		st	Y+,tempreg4		;result
		st	Y+,const_0
		pop	tempreg4
		pop	tempreg1
		sts	libmio_color,tempreg1
		pop	libmio_cur_y		;restore status
		pop	libmio_cur_x
		pop	ZL
		pop	ZH
		rjmp	tbrun_loop		;goto main interpreter loop

bas_fsel_05:	pop	tempreg4
		pop	tempreg1
		sts	libmio_color,tempreg1
		pop	libmio_cur_y		;restore status
		pop	libmio_cur_x
		pop	ZL
		pop	ZH
		lds	YL,bas_ram+10
		lds	YH,bas_ram+11
		ldi	tempreg1,0xff
		st	Y+,tempreg1		;result is -1
		st	Y+,tempreg1		
		rjmp	tbrun_loop		;goto main interpreter loop
		
bas_fsel_err:	ldi	ereg,0x07		;syntax error
		rjmp	tbrun_loop		;error

;use this!
bas_fsel_tcall:	call	bas_fcheck		;check for flash
		clr	ereg
		cpi	tempreg3,0x00
		brne	bas_fsel_tc1
		ret
		
bas_fsel_tc1:	call	fsys_select		;call selector
		clr	ereg
		ldi	tempreg3,0x02		;set to valid
		ret

;-----------------------------------------------------------------------
; keyword is drawto
;-----------------------------------------------------------------------
bas_drawto:	ldi	XL,0x03			;max 5 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x02			;only end parameters
		brcs	bas_drawto_e
		rcall	bas_ccheck		;color value?
		push	tempreg4
		lds	YH,bas_ram+4		;Y1
		lds	YL,bas_ram+3		;X1
		lds	ZH,bas_partab+0		;Y2
		lds	ZL,bas_partab+2		;X2
		sts	bas_ram+3,ZL		;set new old
		sts	bas_ram+4,ZH
		libmio_draw
		pop	tempreg4
		sts	libmio_color,tempreg4	;restore color	
		rjmp	tbrun_loop		;goto main interpreter loop
bas_drawto_e:	rjmp	tbrun_wrongpar		;error
		

;-----------------------------------------------------------------------
; keyword is cbox
;-----------------------------------------------------------------------
bas_cbox:	ldi	XL,0x04			;max 4 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x04			;min parameters
		brne	bas_cbox_e		;<4 is not OK
		rcall	bas_g4c
		libmio_cbox			;draw clear box
		rjmp	tbrun_loop		;goto main interpreter loop
bas_cbox_e:	rjmp	tbrun_wrongpar		;error

;-----------------------------------------------------------------------
; keyword is BORDER
;-----------------------------------------------------------------------
bas_border:	ldi	XL,0x01			;max 1 par
		rcall	bas_getpar		;get parameters
		cpi	XL,0x01			;min parameters
		brcs	bas_goto_e		;<1 is not OK
		lds	XL,bas_partab		;get parameter
		swap	XL
		lsl	XL
		andi	XL,0xef
		sts	libmio_border,XL
		rjmp	tbrun_loop		;goto main interpreter loop				
						    				    
;-----------------------------------------------------------------------
; keyword is GOTO
;-----------------------------------------------------------------------
bas_goto:	ldi	XL,0x01			;max 1 par
		rcall	bas_getpar		;get parameters
		cpi	XL,0x01			;min parameters
		brcs	bas_goto_e		;<1 is not OK
		lds	XL,bas_partab		;get parameter
		subi	XL,1
		mov	r12,XL		
		clr	r13
		ret	
			
bas_goto_e:	rjmp	tbrun_wrongpar		;error


;-----------------------------------------------------------------------
; keyword is onerr
;-----------------------------------------------------------------------
bas_onerr:	ldi	XL,0x01			;max 1 par
		rcall	bas_getpar		;get parameters
		cpi	XL,0x01			;min parameters
		brcs	bas_goto_e		;<1 is not OK
		lds	XL,bas_partab		;get parameter
		subi	XL,1
		sts	bas_ram+15,XL		;store number		
		rjmp	tbrun_loop
		
bas_onerr_e:	rjmp	tbrun_wrongpar		;error


;-----------------------------------------------------------------------
; keyword is palette
;-----------------------------------------------------------------------
bas_setpal:	ldi	XL,0x05			;max 4 palette entries
		rcall	bas_getpar		;get parameters
		cpi	XL,0x02			;min 2
		brcs	bas_setpal_e
		ldi	ZH,0x01			;palette start
		lds	ZL,bas_partab		;set
		andi	ZL,0x07			;max 8 palette entries
bas_setpal_1:	lds	XH,bas_partab+2		;val 1
		andi	XH,0x07
		lsl	XH
		swap 	XH
		st	Z+,XH
		cpi	XL,0x02
		breq	bas_setpal_2
		andi	ZL,0x07			;max 16 palette entries
		
		lds	XH,bas_partab+4		;val 2
		andi	XH,0x07
		lsl	XH
		swap 	XH
		st	Z+,XH
		cpi	XL,0x03
		breq	bas_setpal_2
		andi	ZL,0x07			;max 16 palette entries

		lds	XH,bas_partab+6		;val 3
		andi	XH,0x07
		lsl	XH
		swap 	XH		
		st	Z+,XH
		cpi	XL,0x04
		breq	bas_setpal_2
		andi	ZL,0x07			;max 16 palette entries

		lds	XH,bas_partab+8		;val 4
		andi	XH,0x07
		lsl	XH
		swap 	XH
		st	Z+,XH

bas_setpal_2:	rjmp	tbrun_loop		;goto main interpreter loop
						
bas_setpal_e:	rjmp	tbrun_wrongpar		;error		

		
;-----------------------------------------------------------------------
; keyword is vmode
;-----------------------------------------------------------------------
bas_vmode_e:	rjmp	tbrun_wrongpar		;error
bas_vmode:	ldi	XL,0x01			;max 1 par
		rcall	bas_getpar		;get parameters
		cpi	XL,0x01			;min parameters
		brcs	bas_vmode_e		;<1 is not OK
		lds	XL,bas_partab		;get parameter
		cpi	XL,5			;handheld mode?
		breq	bas_vmode_1		;yes
		andi	XL,0x03			;mask bits		
bas_vmode_1:	libmio_setvmode
		libmio_clrscr
		rjmp	tbrun_loop		

		
;-----------------------------------------------------------------------
; keyword is sync
;-----------------------------------------------------------------------
bas_sync:	ldi	XL,0x01			;max 1 par
		rcall	bas_getpar		;get parameters
		cpi	XL,0x01			;min parameters
		brcs	bas_sync_e		;<1 is not OK
		lds	XL,bas_partab		;get parameter
		lds	XH,bas_partab+1
bas_sync_1:	cpi	XH,0x00
		brne	bas_sync_2
		cpi	XL,0x00
		brne	bas_sync_2
		rjmp	tbrun_loop
		
bas_sync_2:	libmio_sync
		sbiw	XL,1
		rjmp	bas_sync_1
					
bas_sync_e:	rjmp	tbrun_wrongpar		;error

;-----------------------------------------------------------------------
; keyword is play
;-----------------------------------------------------------------------
bas_play:	sts	libmio_seqspeed,const_0	;stopp sequencer
		ldi	XL,0x02
		sts	libmio_seqstat,XL	;status=stopped
		ldi	XL,0x03			;max 1 par
		rcall	bas_getpar		;get parameters
		cpi	XL,0x03			;min parameters
		brcs	bas_play_s		;<1 is not OK -> we hold stop state
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)				
		lds	XL,bas_partab		;get parameter 1 (start)
		lds	XH,bas_partab+1
		cpi	XH,0x03
		brcc	bas_play_e
		add	XL,YL			;add array base
		adc	XH,YH
		sts	libmio_seql,XL
		sts	libmio_seqal,XL
		sts	libmio_seqh,XH
		sts	libmio_seqah,XH
		lds	XL,bas_partab+2		;get parameter 2 (end)
		lds	XH,bas_partab+3
		cpi	XH,0x03
		brcc	bas_play_e
		add	XL,YL			;add array base
		adc	XH,YH
		sts	libmio_seqbl,XL
		sts	libmio_seqbh,XH
		lds	XL,bas_partab+4		;get parameter 3 (speed)
		sts	libmio_seqspeed,XL
		sts	libmio_seqstat,const_0	;status=running
bas_play_s:	rjmp	tbrun_loop
		
bas_play_e:	ldi	ereg,18			;out of array
		rjmp	tbrun_loop

		
;-----------------------------------------------------------------------
; keyword is end
;-----------------------------------------------------------------------
bas_end:	sts	libmio_seqspeed,const_0	;stop sequencer
		pop	XH
		pop	XH
		ret
		
;-----------------------------------------------------------------------
; keyword is if
;-----------------------------------------------------------------------
bas_if:		rcall	bas_expar
		or	XL,XH
		brne	bas_if1
		rjmp	basrun_07		;next line
bas_if1:	rjmp	tbrun_mtest		;continue


;-----------------------------------------------------------------------
; keyword is then
;-----------------------------------------------------------------------
bas_then:	rcall	bas_rlp
		rjmp	tbrun_mtest		;continue

		
;-----------------------------------------------------------------------
; keyword is pos
;-----------------------------------------------------------------------
bas_posxy:	movw	ZL,r14			;restore pointer
		rcall	bas_g2b			;get 2 byte parameters
		libmio_gotoxy
bas_posxy_ext:	rjmp	tbrun_mtest		


;-----------------------------------------------------------------------
; keyword is note
;-----------------------------------------------------------------------
bas_note:	rcall	bas_expar
		libmio_note
		rjmp	tbrun_mtest

;-----------------------------------------------------------------------
; keyword is emit
;-----------------------------------------------------------------------
bas_emit:	mov	tempreg3,tempreg4
		push	tempreg3
		rcall	bas_expar
		pop	tempreg3
bas_emit_01:	mov	tempreg1,XL
		libmio_outchar
		sbrc	tempreg3,5		;skip if (X)Emit
		libmio_vcur			;move cursor for YEmit
		rcall	bas_ispc
		cpi	tempreg4,','
		brne	bas_posxy_ext
		rcall	bas_ispace
		call	expar
		rjmp	bas_emit_01	

;-----------------------------------------------------------------------
; keyword is GOSUB
;-----------------------------------------------------------------------
bas_gosub:	ldi	XL,0x02			;max 2 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x01			;min  1
		breq	bas_gosub_00
		cpi	XL,0x02			;min  1
		breq	bas_xcall
		rjmp	tbrun_wrongpar		;error

bas_gosub_00:	lds	XL,bas_partab		;get line number
		dec	XL			;-1
		cpi	XL,bas_lines
		brcs	bas_gosub_02
bas_gosub_01:	ldi	ereg,10			;no valid linenumber
		rjmp	tbrun_loop		;goto main interpreter loop
			
bas_gosub_02:	lds	XH,bas_stackp		;stackpointer
		cpi	XH,bas_stackdepth-1	;maximum-1
		brcs	bas_gosub_03
		ldi	ereg,13			;too many gosub
		rjmp	tbrun_loop		;goto main interpreter loop
			
bas_gosub_03:	push	XL			;this is the new linenumber			
		ldi	YL,LOW(bas_stacks)	;stack root
		ldi	YH,HIGH(bas_stacks)
		ldi	XL,5			;bytes/record
		mul	XL,XH
		add	YL,r0
		adc	YH,r1
		inc	XH			;inc pointer
		sts	bas_stackp,XH
		mov	XH,r13			;statement
		inc	XH			;must return to next statement
		ori	XH,0x80			;mark as gosub
		st	Y+,XH			;return statement number
		mov	XH,r12			;our line number
		st	Y+,XH
		lds	XH,libmio_lastmenu	;prog number
		st	Y+,XH			;store
		clr	r13			;statement=0
		pop	r12			;new linenumber
		ret				;end of working on this line

;-----------------------------------------------------------------------
; keyword is GOSUB (XCALL)
;-----------------------------------------------------------------------
bas_xcall:	lds	XL,bas_partab+2		;get line number
		dec	XL
		cpi	XL,bas_lines
		brcc	bas_gosub_01		;invalid line number
;		rjmp	bas_gosub_02		;debug
		lds	XH,bas_partab		;prog number
		dec	XH
		andi	XH,0x07
		sts	libmio_tprog,XH		;set temp prognumber

bas_xcall_02:	lds	XH,bas_stackp		;stackpointer
		cpi	XH,bas_stackdepth-1	;maximum-1
		brcs	bas_xcall_03
		ldi	ereg,13			;too many gosub
		rjmp	tbrun_loop		;goto main interpreter loop	
		
bas_xcall_03:	push	XL					
		ldi	YL,LOW(bas_stacks)	;stack root
		ldi	YH,HIGH(bas_stacks)
		ldi	XL,5			;bytes/record
		mul	XL,XH
		add	YL,r0
		adc	YH,r1
		inc	XH			;inc pointer
		sts	bas_stackp,XH
		mov	XH,r13			;statement
		inc	XH			;must return to next statement
		ori	XH,0x80			;mark as gosub
		st	Y+,XH			;return statement number
		mov	XH,r12			;our line number
		st	Y+,XH
		lds	XH,libmio_prog		;caller prog number
		st	Y+,XH			;store
		clr	r13			;statement=0
		pop	r12
		lds	tempreg1,libmio_tprog	;get temp program
		sts	libmio_prog,tempreg1	;set as new
		call	mem_setprog		;set program
		ret

;-----------------------------------------------------------------------
; keyword is RETURN
;-----------------------------------------------------------------------
bas_return:	rcall	bas_rlp			;restore Z-register
		lds	XH,bas_stackp		;stackpointer
		cpi	XH,0
		brne	bas_return_02
bas_return_01:	ldi	ereg,12			;return without call
		ret
bas_return_02:	dec	XH
		ldi	YL,LOW(bas_stacks)	;stack root
		ldi	YH,HIGH(bas_stacks)
		ldi	XL,5			;bytes/record
		mul	XL,XH
		add	YL,r0
		adc	YH,r1
		sts	bas_stackp,XH
		ld	XH,Y+	
		sbrs	XH,7			;check if TOS is call
		rjmp	bas_return_01		;no
		andi	XH,0x7f
		mov	r13,XH			;statement
		ld	r12,Y+			;linenumber
		ld	tempreg1,Y+		;program
		sts	libmio_prog,tempreg1	;set as "this"
		call	mem_setprog
		ret

;-----------------------------------------------------------------------
; keyword is FOR
;-----------------------------------------------------------------------
bas_for_err:	ldi	ereg,7
		ret
bas_for:	rcall	bas_rlp			;restore Z-register
		rcall	bas_ispace		;space
		cpi	tempreg4,'A'
		brcs	bas_for_err
		cpi	tempreg4,'Z'+1
		brcc	bas_for_err
		subi	tempreg4,'A'
		lsl	tempreg4
		mov	tempreg7,tempreg4	;store variable number
		rcall	bas_ispace		;space
	    	cpi	tempreg4,'='
		brne	bas_for_err		;syntax error
		call	expar			;call parser
		cpi	ereg,0
		breq	bas_for_1
		ret
		
bas_for_1:	ldi	YH,high(varspace)	
		ldi	YL,low(varspace)
		add	YL,tempreg7
		adc	YH,const_0					
		st	Y+,XL			;set variable to initial value
		st	Y+,XH
		rcall	bas_ispace
		cpi	tempreg4,'T'
		brne	bas_for_err
		ld	tempreg4,Z+
		cpi	tempreg4,'O'
		brne	bas_for_err
		call	expar			;target value
		mov	tempreg5,XL		;save target value
		mov	tempreg6,XH
		cpi	ereg,0x00
		breq	bas_for_2
		ret
		
bas_for_2:	lds	XH,bas_stackp		;stackpointer
		cpi	XH,bas_stackdepth-1	;maximum-1
		brcs	bas_for_3
		ldi	ereg,14			;too many for
		ret
		
bas_for_3:	ldi	YL,LOW(bas_stacks)	;stack root
		ldi	YH,HIGH(bas_stacks)
		ldi	XL,5			;bytes/record
		mul	XL,XH
		add	YL,r0
		adc	YH,r1
		inc	XH			;inc pointer
		sts	bas_stackp,XH
		mov	XH,r13
		inc	XH			;next must skip this
		st	Y+,XH			;return statement number
		mov	XH,r12			;our line number
		st	Y+,XH
		st	Y+,tempreg7		;variable offset
		st	Y+,tempreg5		;target value low
		st	Y+,tempreg6		;target value high
		rjmp	tbrun_mtest	

		
;-----------------------------------------------------------------------
; keyword is NEXT
;-----------------------------------------------------------------------
bas_next:	rcall	bas_rlp			;restore Z-register
		lds	XH,bas_stackp		;stackpointer
		cpi	XH,0
		brne	bas_next_02
bas_next_01:	ldi	ereg,11			;next without for
		ret
bas_next_02:	dec	XH
		ldi	YL,LOW(bas_stacks)	;stack root
		ldi	YH,HIGH(bas_stacks)
		ldi	XL,5			;bytes/record
		mul	XL,XH
		add	YL,r0
		adc	YH,r1
		ld	XH,Y+			;statement
		sbrc	XH,7			;check if TOS is for
		rjmp	bas_next_01		;no
		andi	XH,0x7f
		sts	bas_ram+2,XH		;save statement temporary
		ld	tempreg6,Y+		;get line number to jump	
		ld	tempreg4,Y+		;variable
		ldi	XH,high(varspace)	
		ldi	XL,low(varspace)
		add	XL,tempreg4
		adc	XH,const_0					
		ld	r16,X+			;variable low
		ld	r17,X+			;variable high
		add	r16,const_1
		adc	r17,const_0
		st	-X,r17
		st	-X,r16
		ld	XL,Y+			;target low
		ld	XH,Y+			;target high
		adiw	XL,1			;+1
		cp	r16,XL
		brne	bas_next_03
		cp	r17,XH
		brne	bas_next_03
		lds	XH,bas_stackp		;stackpointer
		dec	XH
		sts	bas_stackp,XH		
		rjmp	tbrun_mtest		;next statement
bas_next_03:	mov	r12,tempreg6
		lds	r13,bas_ram+2
		ret
			
;-----------------------------------------------------------------------
; keyword is INPUT
;-----------------------------------------------------------------------
bas_input:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_input_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_input_tm:	rcall	bas_rlp			;restore Z-register
		ldi	XH,0xff
		sts	bas_inbuf,XH
bas_input_00:	ld	XH,Z+
		cpi	XH,32			;space
		breq	bas_input_00
		mov	tempreg1,XH
		cpi	XH,0xff
		brne	bas_input_02
bas_input_01:	libmio_newline
		sbiw	ZL,1
		rjmp	tbrun_mtest

bas_input_02:	cpi	XH,':'			;end of statement
		breq	bas_input_01
		cpi	XH,0x3b			;";"	ignore this
		breq	bas_input_00
		cpi	XH,','			;tab
		brne	bas_input_04
bas_input_03:	libmio_outspace
		mov	XH,libmio_cur_x
		andi	XH,0x07
		brne	bas_input_03
		rjmp	bas_input_00

bas_input_04:	cpi	XH,'"'			;is it a string?
		brne	bas_input_06		;no
bas_input_05:	ld	tempreg1,Z+		;get string char
		cpi	tempreg1,0xff
		breq	bas_input_01
		cpi	tempreg1,'"'		;end of string
		breq	bas_input_00		;yes
		libmio_outchar			;output char
		rjmp	bas_input_05		;string loop
		
bas_input_06:	cpi	XH,'A'
		brcs	bas_input_err
		cpi	XH,'Z'+1
		brcc	bas_input_err
		subi	XH,'A'
		lsl	XH
		sts	bas_ram+2,XH		;variable number
		ld	XH,Z
		cpi	XH,'A'
		brcs	bas_input_07
		cpi	XH,'Z'+1
		brcc	bas_input_07
		rjmp	bas_input_err
bas_input_07:	rcall	bas_inval		;get value		
		ldi	YH,high(varspace)	
		ldi	YL,low(varspace)
		lds	r17,bas_ram+2
		add	YL,r17
		adc	YH,const_0					
		st	Y+,XL			;set variable to initial value
		st	Y+,XH
		rjmp	bas_input_00

bas_input_err:	ldi	ereg,7			;syntax error
		ret


;-----------------------------------------------------------------------
; keyword is wait
;-----------------------------------------------------------------------
bas_wait:	rcall	bas_expar
		cpi	ereg,0
		breq	bas_wait_01
bas_wait_end:	rjmp	tbrun_mtest	
		
bas_wait_01:	mov	tempreg1,XL
		or	tempreg1,XH
		breq	bas_wait_end
bas_wait_02:	lds	tempreg1,libmio_time0
		cpi	tempreg1,0x00
		breq	bas_wait_02
bas_wait_03:	lds	tempreg1,libmio_time0
		cpi	tempreg1,0x00
		brne	bas_wait_03
		sbiw	XL,1
		rjmp	bas_wait_01			    	

;-----------------------------------------------------------------------
; keyword is tset
;-----------------------------------------------------------------------
bas_tset:	rcall	bas_expar			;get value
		sts	libmio_time_l,XL
		sts	libmio_time_h,XH
		rjmp	bas_wait_01			    	

;-----------------------------------------------------------------------
; keyword is tget
;-----------------------------------------------------------------------
bas_tget:	rcall	bas_getvar		;get variable
		lds	XL,libmio_time_l
		st	Y+,XL			;low timer
		lds	XL,libmio_time_h
		st	Y+,XL			;high timer
		rjmp	tbrun_mtest


;-----------------------------------------------------------------------
; keyword is sput
;-----------------------------------------------------------------------
bas_sput:	rcall	bas_expar
bas_sput_01:	mov	tempreg1,XL
		libmio_pser
		rcall	bas_ispc
		cpi	tempreg4,','
		brne	bas_sput_ext
		rcall	bas_ispace
		call	expar
		rjmp	bas_sput_01
bas_sput_ext:	rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is sget
;-----------------------------------------------------------------------
bas_sget:	libmio_gser
		mov	XL,tempreg1
		clr	XH
	    	rcall	bas_getvar		;get variable
		st	Y+,XL			;low value
		st	Y+,XH			;high value
		rjmp	tbrun_mtest

;-----------------------------------------------------------------------
; keyword is epoke
;-----------------------------------------------------------------------
bas_epoke:	movw	ZL,r14			;restore pointer
		rcall	bas_g2w			;get parameters
		mov	tempreg1,XL
		cpi	YH,HIGH(2000)		;max EEPROM adr
		brcc	bas_epoke1
		cpi	YL,LOW(2000)		;max EEPROM adr
		brcc	bas_epoke1
		libeep_write
bas_epoke1:	rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is xpoke
;-----------------------------------------------------------------------
bas_xpoke:	movw	ZL,r14			;restore pointer
		rcall	bas_g2w
		mov	tempreg2,XL
		mov	XL,YL			;set address
		mov	XH,YH
		call	mem_getconf1		;data EEPROM address
		libi2c_write
		libmio_sync			;wait min 32ms
		libmio_sync
		libmio_sync
	    	rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is dir
;-----------------------------------------------------------------------
bas_dir:	rcall	bas_expar
		out	DDRA,XL
		rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is out (Y=pin XL=val)
;-----------------------------------------------------------------------
bas_out:	movw	ZL,r14			;restore pointer
		rcall	bas_g2w
		cpi	YH,0x00			;single bit?
		brne	bas_out_2		;no
		mov	XH,YL
		andi	XH,0x07			;8 Bits
		inc	XH
		clr	tempreg4		;or mask for reset
		ldi	tempreg3,0x7f		;and mask for reset
		sbrc	XL,0			;value bit
		ldi	tempreg4,0x80		;or mask for set
		sbrc	XL,0			;value bit
		ldi	tempreg3,0xff		;and mask for set
bas_out_1:	bst	tempreg4,7		;rotate or mask
		lsl	tempreg4
		bld	tempreg4,0
		bst	tempreg3,7		;rotate and mask
		lsl	tempreg3
		bld	tempreg3,0
		dec	XH
		brne	bas_out_1
		in	r16,PORTA
		or	r16,tempreg4
		and	r16,tempreg3
		out	PORTA,r16
	    	rjmp	tbrun_mtest
		
bas_out_2:	cpi	YH,0x01			;use mask?
		brne	bas_out_3		;no
		in	XH,PORTA		;get stat
		com	YL			;invert mask
		and	XH,YL			;clear bits
		com	YL			;invert mask again
		and	YL,XL			;and value
		or	XH,YL
		out	PORTA,XH		;out byte
bas_out_e:    	rjmp	tbrun_mtest
				
bas_out_3:	cpi	YH,0x02			;spiset?
		brne	bas_out_e		;no
		call	spi_set
		rjmp	tbrun_mtest
		
;-----------------------------------------------------------------------
; keyword is pump
;-----------------------------------------------------------------------
bas_chpump:	rcall	bas_expar
		libmio_chpump
		rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is limit
;-----------------------------------------------------------------------
bas_limit:	rcall	bas_getvar		;variable
		sts	bas_ram+10,YL		;save var address
		sts	bas_ram+11,YH
		ld	r16,Y+			;set var value
		ld	r17,Y+
		subi	r17,0x80		;change sign bit
		rcall	bas_ispace
		cpi	tempreg4,','
		brne	bas_limit_err
		rcall	bas_g2w			;get min & max value		
		subi	XH,0x80			;change sign bit
		subi	YH,0x80
		cp	YL,r16			;lower limit
		cpc	YH,r17
		brcs	bas_limit1		;is greater
		mov	r16,YL
		mov	r17,YH
bas_limit1:	cp	XL,r16		
		cpc	XH,r17
		brcc	bas_limit2
		mov	r16,XL
		mov	r17,XH
bas_limit2:	lds	YH,bas_ram+11
		lds	YL,bas_ram+10
		subi	r17,0x80
		st	Y+,r16
		st	Y+,r17
	    	rjmp	tbrun_mtest		
bas_limit_err:	ldi	ereg,6			;syntax error
		ret

;-----------------------------------------------------------------------
; keyword is gchar
;-----------------------------------------------------------------------
bas_gchar:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_gchar_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_gchar_tm:	rcall	bas_getvar		;variable
		sts	bas_ram+10,YL		;save var address
		sts	bas_ram+11,YH
		rcall	bas_ispace
		cpi	tempreg4,','
		brne	bas_limit_err
		rcall	bas_g2b			;get X & Y value		
		ldi	YL,LOW(libmio_vram)
		ldi	YH,HIGH(libmio_vram)
		ldi	tempreg4,libmio_cols
		mul	tempreg4,XH
		add	YL,r0
		adc	YH,r1
		add	YL,XL
		adc	YH,const_0
		clr	XH
		ld	XL,Y
		lds	YH,bas_ram+11
		lds	YL,bas_ram+10
		st	Y+,XL
		st	Y+,XH
	    	rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is gattr
;-----------------------------------------------------------------------
bas_gattr:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_gattr_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_gattr_tm:	rcall	bas_getvar		;variable
		sts	bas_ram+10,YL		;save var address
		sts	bas_ram+11,YH
		rcall	bas_ispace
		cpi	tempreg4,','
		brne	bas_limit_err
		rcall	bas_g2b			;get x & y value		
		ldi	YL,LOW(libmio_vram)
		ldi	YH,HIGH(libmio_vram)
		ldi	tempreg4,libmio_cols
		mul	tempreg4,XH
		add	YL,r0
		adc	YH,r1
		add	YL,XL
		adc	YH,const_0
		ldi	XL,LOW(libmio_cols*libmio_rows)
		add	YL,XL
		ldi	XL,HIGH(libmio_cols*libmio_rows)
		add	YH,XL
		clr	XH
		ld	XL,Y
		lsr	XL
		mov	XH,XL
		andi	XL,0x77
		swap 	XH
		andi	XH,0x80
		or	XL,XH
		clr	XH
		lds	YH,bas_ram+11
		lds	YL,bas_ram+10
		st	Y+,XL
		st	Y+,XH
	    	rjmp	tbrun_mtest		


;-----------------------------------------------------------------------
; keyword is GPIX
;-----------------------------------------------------------------------
bas_gpix_err:	ldi	ereg,6			;syntax error
		ret 
bas_gpix:	rcall	bas_getvar		;variable
		sts	bas_ram+10,YL		;save var address
		sts	bas_ram+11,YH
		rcall	bas_ispace
		cpi	tempreg4,','
		brne	bas_gpix_err
		rcall	bas_g2b			;get X & Y value
		ldi	YL,LOW(libmio_vram)
		ldi	YH,HIGH(libmio_vram)
		lds	tempreg4,libmio_vidmode
		andi	tempreg4,0x03		;mask bits
    		cpi	tempreg4,0x00		;mode 0
		brne	bas_gpix_10

bas_gpix_00:	push	XL			;save x coord
		push	XH			;save y			
		lsr	XL
		lsr	XH
		ldi	tempreg4,libmio_cols
		mul	tempreg4,XH
		add	YL,r0
		adc	YH,r1
		add	YL,XL
		adc	YH,const_0		;address of char
		ld	tempreg4,Y		;get char
		pop	XH			;restore coords
		pop	XL
		sbrc	XL,0
		lsr	tempreg4
		sbrc	XH,0
		lsr	tempreg4
		sbrc	XH,0
		lsr	tempreg4
		ldi	XL,LOW(libmio_cols*libmio_rows)
		ldi	XH,HIGH(libmio_cols*libmio_rows)
		add	YL,XL			;address of attribute
		adc	YH,XH
		ld	XL,X			;get color
		sbrc	tempreg4,0
		swap	XL
bas_gpix_01:	lsr	XL
		andi	XL,0x07
		clr	XH
		lds	YH,bas_ram+11
		lds	YL,bas_ram+10
		st	Y+,XL
		st	Y+,XH
	    	rjmp	tbrun_mtest		

bas_gpix_10:	cpi	tempreg4,0x01		;mode 1
		brne	bas_gpix_20
		push	XL			;save x coord
		lsr	XL			;/8
		lsr	XL
		lsr	XL
		ldi	tempreg4,libmio_v1bpl	;bytes per line
		mul	tempreg4,XH
		add	YL,r0
		adc	YH,r1
		add	YL,XL
		adc	YH,const_0		;address of byte
		ld	tempreg4,Y		;get byte
		pop	XL			;restore coords
		sbrc	XL,2
		swap	tempreg4
		sbrc	XL,1
		lsr	tempreg4
		sbrc	XL,1
		lsr	tempreg4
		sbrc	XL,0
		lsr	tempreg4
		andi	tempreg4,0x01
		ldi	YH,0x01
		mov	YL,tempreg4
		ld	XL,Y
		swap	XL
		rjmp	bas_gpix_01

bas_gpix_20:	cpi	tempreg4,0x01		;mode 1
		brne	bas_gpix_20
		push	XL			;save x coord
		lsr	XL			;/4
		lsr	XL
		ldi	tempreg4,libmio_v2bpl	;bytes per line
		mul	tempreg4,XH
		add	YL,r0
		adc	YH,r1
		add	YL,XL
		adc	YH,const_0		;address of byte
		ld	tempreg4,Y		;get byte
		pop	XL			;restore coords
		sbrc	XL,1
		swap	tempreg4
		sbrc	XL,0
		lsr	tempreg4
		sbrc	XL,0
		lsr	tempreg4
		andi	tempreg4,0x03
		ldi	YH,0x01
		mov	YL,tempreg4
		ld	XL,Y
		swap	XL
		rjmp	bas_gpix_01

bas_gpix_30:	push	XL			;save x coord
		lsr	XL
		ldi	tempreg4,libmio_v3bpl	;bytes per line
		mul	tempreg4,XH
		add	YL,r0
		adc	YH,r1
		add	YL,XL
		adc	YH,const_0		;address of byte
		ld	tempreg4,Y		;get byte
		pop	XL			;restore coords
		sbrc	XL,0
		swap	tempreg4
		mov	XL,tempreg4
		rjmp	bas_gpix_01


;-----------------------------------------------------------------------
; keyword is sprite
;-----------------------------------------------------------------------
bas_sprite:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_sprite_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_sprite_tm:	ldi	XL,0x03			;max 3 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x03		
		brcs	bas_sprite_e
		lds	XL,bas_partab		;adr lo
		lds	XH,bas_partab+1		;adr hi
		cpi	XH,0x03
		brcs	bas_sprite_01
		ldi	ereg,18			;out of array
		ret
bas_sprite_01:	cpi	XH,0x02
		brne	bas_sprite_02
		cpi	XL,0xf7			;max
		brcs	bas_sprite_02
bas_sprite_oa:	ldi	ereg,18			;out of array
		ret

bas_sprite_wd:	ldi	ereg,33			;wrong sprite
		ret

bas_sprite_e:	rjmp	tbrun_wrongpar		;error


bas_sprite_02:	ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		add	YL,XL
		adc	YH,XH
		sts	libmio_bcsrcx,YL	;address
		sts	libmio_bcsrcy,YH
		ldd	XL,Y+4			;dx
		cpi	XL,0x00			;=0
		breq	bas_sprite_wd		;error
		cpi	XL,libmio_cols		;>8
		brcc	bas_sprite_wd		;error
		ldd	XH,Y+3			;dy
		cpi	XH,0x00			;=0
		breq	bas_sprite_wd		;error		
		cpi	XH,libmio_rows		;>8
		brcc	bas_sprite_WD		;error
		lds	YL,bas_partab		;adr lo
		lds	YH,bas_partab+1		;adr hi		
		adiw	YL,5			;header
		mul	XH,XL
		lsl	r0
		rol	r1
		lsl	r0
		rol	r1
		add	YL,r0
		adc	YL,r1
		cpi	YH,3
		brcc	bas_sprite_wd		;error
		
		lds	YH,bas_partab+2		;Y
		lds	YL,bas_partab+4		;X
		sts	libmio_bcdestx,YL	;coords
		sts	libmio_bcdesty,YH
		cpi	YL,0xff			;hide
		breq	bas_sprite_03
		cpi	YH,0xff			;hide
		breq	bas_sprite_03		
		add	YL,XL			;x+dx
		dec	YL			;-1
		cpi	YL,libmio_cols
		brcc	bas_sprite_os
		add	YH,XH			;y+dy
		dec	YH			;-1
		cpi	YH,libmio_rows
		brcc	bas_sprite_os
bas_sprite_03:	libmio_sprite			;now we can draw
		rjmp	tbrun_loop
				
bas_sprite_os:	ldi	ereg,22			;out of screen
		ret

;-----------------------------------------------------------------------
; keyword is break
;-----------------------------------------------------------------------
bas_breakpoint:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_break_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_break_tm:	rcall	bas_rlp
		lds	XL,libmio_kflags
		ori	XL,4			;set monitor flag
		sts	libmio_kflags,XL
	  	rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is fast
;-----------------------------------------------------------------------
bas_fast:	lds	XL,libmio_config
		ori	XL,0x20
		sts	libmio_config,XL
	  	rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is slow
;-----------------------------------------------------------------------
bas_slow:	lds	XL,libmio_config
		andi	XL,0xdf
		sts	libmio_config,XL
	  	rjmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is fread (read from dataflash)
; par1=file
; par2=page
; par3=position in array (0/1/2)
;-----------------------------------------------------------------------
bas_fread:	ldi	XL,0x03			;max 3 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x03			;min 3
		breq	bas_fread_1
		rjmp	tbrun_wrongpar		;error

bas_fread_1:	call	fsys_gettype
		cpi	tempreg1,0x30		;user file
		brcs	bas_fread_3
		ldi	ereg,30			;no user file
bas_fread_2:  	ret		
			
bas_fread_3:	call	fsys_read		;read page
		cpi	ereg,0x00
		brne	bas_fread_2		
	  	rjmp	tbrun_loop		

;-----------------------------------------------------------------------
; keyword is fwrite (write to dataflash)
; par1=file
; par2=page
; par3=position in array (0/1/2)
;-----------------------------------------------------------------------
bas_fwrite:	ldi	XL,0x03			;max 3 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x03			;min 3
		breq	bas_fwrite_1
		rjmp	tbrun_wrongpar		;error

bas_fwrite_1:	call	fsys_gettype
		cpi	tempreg1,0x30		;user file
		brcs	bas_fwrite_3
		ldi	ereg,30			;no user file
bas_fwrite_2:  	ret		
			
bas_fwrite_3:	call	fsys_write		;write page
		cpi	ereg,0x00
		brne	bas_fwrite_2				
	  	rjmp	tbrun_loop		

;-----------------------------------------------------------------------
; keyword is fcreate (create/delete file)
; par1=file
; par2=pages
;-----------------------------------------------------------------------
bas_fcreate:	ldi	XL,0x03			;max 3 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x02			;min 2
		breq	bas_fcreate_1
		cpi	XL,0x03			;max 3
		breq	bas_fcreate_1a
		
		rjmp	tbrun_wrongpar		;error

bas_fcreate_1:	ldi	XL,0x14			;file type
		sts	bas_partab+4,XL
bas_fcreate_1a:	call	fsys_create		;create new file
		cpi	ereg,0x00
		brne	bas_fcreate_2				
	  	rjmp	tbrun_loop
bas_fcreate_2:	ret				;error				

bas_fdelete_3:	call	fsys_delete		;delete file
		cpi	ereg,0x00
		brne	bas_fdelete_2		
	  	rjmp	tbrun_loop		

;-----------------------------------------------------------------------
; keyword is dfile (delete file)
;-----------------------------------------------------------------------
bas_fdelete:	ldi	XL,0x01			;max 2 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x01			;min 2
		breq	bas_fdelete_1
		rjmp	tbrun_wrongpar		;error

bas_fdelete_1:	call	fsys_gettype
		cpi	tempreg1,0x14		;user file
		breq	bas_fdelete_3
		ldi	ereg,30			;no user file
bas_fdelete_2:  ret		


;-----------------------------------------------------------------------
; keyword is rread (raw read from dataflash)
; par1=page
; par2=position in array (0/1/2)
;-----------------------------------------------------------------------
bas_rread:	ldi	XL,0x02			;max 3 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x02			;min 3
		breq	bas_rread_1
		rjmp	tbrun_wrongpar		;error

bas_rread_1:	lds	XL,bas_partab
		lds	XH,bas_partab+1
		lds	tempreg1,bas_partab+2
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		cpi	tempreg1,0x00
		breq	bas_rread_2
		inc	YH
		cpi	tempreg1,0x01
		breq	bas_rread_2
		inc	YH
		cpi	tempreg1,0x02
		breq	bas_rread_2
		ldi	ereg,18			;out of array
		ret			

bas_rread_2:	libdfl_readraw
		cpi	ereg,0
		breq	bas_rread_3
		ret
		
bas_rread_3:	rjmp	tbrun_loop		

;-----------------------------------------------------------------------
; keyword is rwrite (raw write to dataflash)
; par1=page
; par2=position in array (0/1/2)
;-----------------------------------------------------------------------
bas_rwrite:	ldi	XL,0x02			;max 3 pars
		rcall	bas_getpar		;get parameters
		cpi	XL,0x02			;min 3
		breq	bas_rwrite_1
		rjmp	tbrun_wrongpar		;error

bas_rwrite_1:	lds	XL,bas_partab
		lds	XH,bas_partab+1
		lds	tempreg1,bas_partab+2
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		cpi	tempreg1,0x00
		breq	bas_rwrite_2
		inc	YH
		cpi	tempreg1,0x01
		breq	bas_rwrite_2
		inc	YH
		cpi	tempreg1,0x02
		breq	bas_rwrite_2
		ldi	ereg,18			;out of array
		ret			

bas_rwrite_2:	libdfl_writeraw
		cpi	ereg,0
		breq	bas_rread_3
		ret

;-----------------------------------------------------------------------
; keyword is icomm
;-----------------------------------------------------------------------
bas_icomm_ex:	rjmp	bas_icomm_e		;jump extender 
bas_icomm:	movw	ZL,r14			;restore pointer
		rcall	bas_g3			;get adr, start and num
		sts	bas_ram+12,XL		;number of bytes
		libi2c_start			;startbit
		cpi	tempreg3,0x08
		brne	bas_icomm_ex		;error
		sbrs	ctrl,0			;skip if read
		rjmp	bas_icomm_w		;jump to write
		mov	tempreg2,ctrl		;address
		libi2c_wbyte			;output
		cpi	tempreg3,0x40
		brne	bas_icomm_ex		;error
		cpi	XL,0x00			;read number?
		breq	bas_icomm_r4		;yes
bas_icomm_r1:	ldi	YH,high(bas_array)	;array
		ldi	YL,low(bas_array)
		lds	tempreg1,bas_ram+12	;number of bytes
		add	tempreg1,XH		;add offset
		sub	tempreg1,XL		;sub number of bytes to do
		add	YL,tempreg1		;array element address
		adc	YH,const_0
		cpi	XL,1			;last?
		breq	bas_icomm_r2
		libi2c_rbyte			;read byte with ack
		cpi	tempreg3,0x50
		brne	bas_icomm_e		;error
		st	Y,tempreg2		;store byte
		dec	XL			;dec number of bytes to read
		rjmp	bas_icomm_r1

bas_icomm_r2:	libi2c_rbyten			;read byte without ack
		cpi	tempreg3,0x58
		brne	bas_icomm_e		;error
		st	Y,tempreg2		;store byte
		libi2c_stop			;
		rjmp	tbrun_mtest		
		
bas_icomm_r4:	libi2c_rbyte			;get number of bytes
		cpi	tempreg2,0x00		;no bytes to read
		breq	bas_icomm_r2		;so it is
		sts	bas_ram+12,tempreg2	;store number of bytes
		mov	XL,tempreg2		;bytes to do
		rjmp	bas_icomm_r1		;read bytes

bas_icomm_w:	mov	tempreg2,ctrl		;address
		libi2c_wbyte			;output
		cpi	tempreg3,0x18
		brne	bas_icomm_e		;error
		mov	r19,XL			;number of bytes
bas_icomm_w1:	ldi	YH,high(bas_array)	;array
		ldi	YL,low(bas_array)
		lds	tempreg1,bas_ram+12	;number of bytes
		add	tempreg1,XH		;add offset
		sub	tempreg1,XL		;sub number of bytes to do
		add	YL,tempreg1		;array element address
		adc	YH,const_0
		ld	tempreg2,Y		;get byte
		cpi	tempreg2,0x00		;zero?
		brne	bas_icomm_w2
		cpi	r19,0x00		;stop at zero?
		breq	bas_icomm_w3		;exit loop
bas_icomm_w2:	libi2c_wbyte			;write byte
		cpi	tempreg3,0x28
		brne	bas_icomm_e		;error
		dec	XL			;dec number of byte to read
		brne	bas_icomm_w1
bas_icomm_w3:	libi2c_stop
		rjmp	tbrun_mtest		
		
bas_icomm_e:	ldi	ereg,15			;i2c error
		libi2c_stop
		ret				;end

;-----------------------------------------------------------------------
; keyword is DATA
;-----------------------------------------------------------------------
bas_data:	rcall	bas_expar	
		sts	bas_partab,XL		;array index
		sts	bas_partab+1,XH
;Z points to char
		
bas_data_01:	rcall	bas_ispc		;wait for no space
		cpi	tempreg4,','		;comma?
		brne	bas_data_02		;no -> end
		adiw	ZL,1
		rcall	bas_ispace		;next nonspace char
		cpi	tempreg4,'"'		;string?
		breq	bas_data_03		;yes
		sbiw	ZL,1
		call	expar
		rcall	bas_warr		;write array value
		rjmp	bas_data_01	

bas_data_02:	rjmp	tbrun_mtest

bas_data_03:	ld	tempreg4,Z+
		cpi	tempreg4,'"'
		breq	bas_data_04
		cpi	tempreg4,0xff
		breq	bas_data_04
		clr	XH
		mov	XL,tempreg4
		rcall	bas_warr		;write array value
		rjmp	bas_data_03		

bas_data_04:	ld	tempreg4,Z
		rjmp	bas_data_01

;-----------------------------------------------------------------------
; keyword is BCOPY
;-----------------------------------------------------------------------
bas_bcopy:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		brne	bas_bcopy_gm
		ldi	ereg,29			;not in text mode
		ret
		
bas_bcopy_gm:	ldi	XL,7			;max 7 parameters
		rcall	bas_getpar		;get parameters		
		lds	XH,bas_partab		;parameter 1
		sts	libmio_bcmode,XH	;mode
    		cpi	XH,0x01			;vmem->vmem
		breq	bas_bcopy_vv
		cpi	XH,0x02			;vmem->mem
		breq	bas_bcopy_vm
bas_copy_1:	cpi	XH,0x03
		brne	bas_bcopy_err
		rjmp	bas_bcopy_mv
		
bas_bcopy_err:	ldi 	ereg,21			;bcopy def err
		ret
	
		
bas_bcopy_vv:	cpi	XL,7			;we need 7 parameters
		brne	bas_bcopy_err		;error
		ldi	YL,LOW(bas_partab)
		ldi	YH,HIGH(bas_partab)
		ldd	XH,Y+2			
		sts	libmio_bcsrcy,XH	;source X
		ldd	XH,Y+4			
		sts	libmio_bcsrcx,XH	;source Y
		ldd	XH,Y+6			
		sts	libmio_bcdy,XH		;blocks v
		ldd	XH,Y+8			
		sts	libmio_bcdx,XH		;blocks h
		ldd	XH,Y+10			
		sts	libmio_bcdesty,XH	;dest Y
		ldd	XH,Y+12			
		sts	libmio_bcdestx,XH	;dest X
		libmio_bcopy			;transfer
		cpi	ereg,0x00
		breq	bas_bcopy_vv1
		ret
bas_bcopy_vv1:	rjmp	tbrun_loop		;goto main loop

bas_bcopy_vm:	cpi	XL,6			;we need 6 parameters
		brne	bas_bcopy_err		;error
		ldi	YL,LOW(bas_partab)
		ldi	YH,HIGH(bas_partab)
		ldd	XH,Y+2			
		sts	libmio_bcsrcy,XH	;source y
		ldd	XH,Y+4			
		sts	libmio_bcsrcx,XH	;source x
		ldd	XH,Y+6			
		sts	libmio_bcdy,XH		;pixel columns
		ldd	XH,Y+8			
		sts	libmio_bcdx,XH		;pixel rows
		ldd	XH,Y+10			;arrayptr low			
		sts	libmio_bcdestx,XH	;low pointer
		ldd	XH,Y+11			;arrayptr high			
		sts	libmio_bcdesty,XH	;high pointer
		libmio_bcopy			;transfer
		cpi	ereg,0x00
		breq	bas_bcopy_vm1
		ret
bas_bcopy_vm1:	jmp	tbrun_loop		;goto main loop
bas_bcopy_errx:	rjmp	bas_bcopy_err		;jump extender

bas_bcopy_mv:	cpi	XL,4			;we need 4 parameters
		brne	bas_bcopy_errx		;error
		ldi	YL,LOW(bas_partab)
		ldi	YH,HIGH(bas_partab)
		ldd	XH,Y+2			;arrayptr low			
		sts	libmio_bcsrcx,XH	;low pointer
		ldd	XH,Y+3			;arrayptr high			
		sts	libmio_bcsrcy,XH	;high pointer
		ldd	XH,Y+4			
		sts	libmio_bcdesty,XH	;dest y
		ldd	XH,Y+6			
		sts	libmio_bcdestx,XH	;dest x
		libmio_bcopy			;transfer
		cpi	ereg,0x00
		breq	bas_bcopy_mv1
		ret
bas_bcopy_mv1:	jmp	tbrun_loop		;goto main loop
		
				
;-----------------------------------------------------------------------
; keyword is CTEXT
;-----------------------------------------------------------------------
bas_ctext:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_ctext_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_ctext_tm:	movw	ZL,r14			;restore pointer
		rcall	bas_g2w
		sts	bas_partab,YL		;offset
		sts	bas_partab+1,YH		;array area
		lds	tempreg2,bas_ram+12	;number of chars
		cp	XL,tempreg2		;limit
		brcc	bas_ctext_00
		mov	tempreg2,XL
	
bas_ctext_00:	ldi	XL,LOW(bas_inbuf)
		ldi	XH,HIGH(bas_inbuf)
			
bas_ctext_01:	cpi	tempreg2,0
		breq	bas_ctext_02
		ld	tempreg1,X+
		sts	bas_ram+13,XL
		sts	bas_ram+14,XH
		mov	XL,tempreg1
		clr	XH
		rcall	bas_warr
		lds	XL,bas_ram+13
		lds	XH,bas_ram+14		
		dec	tempreg2
		rjmp	bas_ctext_01	
bas_ctext_02:	clr	XL
		clr	XH
		rcall	bas_warr
		jmp	tbrun_mtest		

;-----------------------------------------------------------------------
; keyword is scroll
;-----------------------------------------------------------------------
bas_scroll:	lds	XL,libmio_vidmode
		cpi	XL,0x00
		breq	bas_scroll_tm
		ldi	ereg,28			;not in graphics mode
		ret
		
bas_scroll_tm:	ldi	XL,0x01			;get 1 parameter
		rcall	bas_getpar
		cpi	XL,0x01
		brcs	bas_scroll_e		;error
		lds	XL,bas_partab		;lo ptr
		mov	tempreg1,XL
		andi	tempreg1,0x03
		libmio_scroll
		jmp	tbrun_loop		;goto main interpreter loop
bas_scroll_e:	rjmp	tbrun_wrongpar

;-----------------------------------------------------------------------
; get XL parameters to bas_par
;-----------------------------------------------------------------------				
bas_getpar:	movw	ZL,r14			;get line pointer
bas_getpar_0:	cpi	XL,0x00			;parameters
		brne	bas_getpar_1		;yes
		ret
		
bas_getpar_1:	push	YH
		push	YL
		ldi	YL,LOW(bas_partab)	;set ptr
		ldi	YH,HIGH(bas_partab)
		ldi	tempreg1,0x00		;no value
		mov	tempreg2,XL		;max value count

bas_getpar_2:	cpi	tempreg2,0x00		;all done?
		brne	bas_getpar_3
		mov	XL,tempreg1		;numver of values
		movw	r14,ZL			;set new pointer
		pop	YL
		pop	YH
		ret
		
bas_getpar_3:	push	YH			;save registers
		push	YL
		push	tempreg1
		push	tempreg2
		call	expar			;call parser
		pop	tempreg2		;restore registers
		pop	tempreg1
		pop	YL
		pop	YH
		
		cpi	ereg,0x19		;end of list?
		brne	bas_getpar_4		;yes
		clr	ereg			;ignore this error
		movw	r14,ZL			;set new pointer
		mov	XL,tempreg1		;set number  
		ret
							
bas_getpar_4:	st	Y+,XL			;write low
		st	Y+,XH			;write high
		inc	tempreg1		;number of params
		dec	tempreg2
		cpi	tempreg2,0x00		;all done?
		breq	bas_getpar_2		;end now
		
bas_getpar_5:	ld	tempreg4,Z+		;get next
		cpi	tempreg4,32		;space
		breq	bas_getpar_5		;wait for not

		cpi	tempreg4,','		;delimiter		
		breq	bas_getpar_3		;OK, check next value
		sbiw	ZL,1			;set pointer back to this char
		mov	XL,tempreg1		;numver of values
		movw	r14,ZL			;set new pointer
		pop	YL
		pop	YH
		ret
		
;-----------------------------------------------------------------------
; get 3 byte parameters (ctrl,XH,XL)
;-----------------------------------------------------------------------
bas_g3:		ldi	XL,0x03
		rcall	bas_getpar
		cpi	XL,0x03
		brne	bas_g2b_e
		lds	ctrl,bas_partab
		lds	XH,bas_partab+2
		lds	XL,bas_partab+4
		ret

bas_g4c:	lds	YH,bas_partab		;Y1
		lds	YL,bas_partab+2		;X1
		lds	ZH,bas_partab+4		;Y2
		lds	ZL,bas_partab+6		;X2
		sts	libmio_bcsrcx,YL
		sts	libmio_bcsrcy,YH
		sts	libmio_bcdx,ZL
		sts	libmio_bcdy,ZH
		ret		

;-----------------------------------------------------------------------
; get 2 byte parameters (XH,XL)
;-----------------------------------------------------------------------
bas_g2b:	ldi	XL,0x02			;2 parameters
		rcall	bas_getpar_0		;get
		cpi	XL,0x02			;2 found
		brne	bas_g2b_e		;no
		lds	XH,bas_partab
		lds	XL,bas_partab+2
		ret
		
bas_g2b_e:	pop	XL
		pop	XL
		rjmp	tbrun_wrongpar

;-----------------------------------------------------------------------
; get 2 word parameters (Y,X)
;-----------------------------------------------------------------------
bas_g2w:	ldi	XL,0x02
		rcall	bas_getpar_0
		cpi	XL,0x02
		brne	bas_g2b_e
		lds	YL,bas_partab
		lds	YH,bas_partab+1
		lds	XL,bas_partab+2
		lds	XH,bas_partab+3		
		ret
		

;-----------------------------------------------------------------------
; get variable address
;-----------------------------------------------------------------------
bas_getvar:	movw	ZL,r14			;get line pointer
bas_va:		rcall	bas_ispace
bas_va_1:	cpi	tempreg4,'A'
		brcs	bas_va_err
		cpi	tempreg4,'Z'+1
		brcc	bas_va_3
		subi	tempreg4,'A'
bas_va_2:	lsl	tempreg4
		ldi	YH,high(varspace)	
		ldi	YL,low(varspace)
		add	YL,tempreg4
		adc	YH,const_0					
		ret
		
bas_va_3:	cpi	tempreg4,'a'
		brcs	bas_va_err
		cpi	tempreg4,'z'+1
		brcc	bas_va_err
		subi	tempreg4,'a'
		rjmp	bas_va_2
		
bas_va_err:	pop	ereg
		pop	ereg
		ldi	ereg,7
		ret		

;-----------------------------------------------------------------------
; read until char is not space
;-----------------------------------------------------------------------
bas_ispace:	ld	tempreg4,Z+
bas_ispc:	cpi	tempreg4,32			;space
		breq	bas_ispace
bas_ignoresp01:	ret		

;-----------------------------------------------------------------------
; restore line pointer
;-----------------------------------------------------------------------
bas_rlp:	movw	ZL,r14		;get line pointer
		ret

;-----------------------------------------------------------------------
; restore line pointer and get parameter
;-----------------------------------------------------------------------
bas_expar:	movw	ZL,r14		;get line pointer
		jmp	expar

;-----------------------------------------------------------------------
; input value
;-----------------------------------------------------------------------
bas_inval:	push	ZL			;save pointer
		push	ZH
		libmio_calc			;calculate address
		mov	ZL,YL			;set start of pointer
		mov	ZH,YH
		ldi	r16,0			;length
bas_inval_01:	libmio_cursor			;view cursor
		lds	tempreg4,libmio_kflags	;break
		cpi	tempreg4,1
		brne	bas_inval_01b
		mov	ereg,tempreg4
		rjmp	bas_inval_01a	
	
bas_inval_01b:	cpi	tempreg1,0xea		;enter
		brne	bas_inval_04
		sts	bas_ram+12,r16		;number of chars
		cpi	r16,0			;no char
		brne	bas_inval_02	
		clr	XL
		clr	XH
bas_inval_01a:	clr	ereg
		pop	ZH			;restore pointer
		pop	ZL
		ret
		
bas_inval_02:	ldi	YL,LOW(bas_inbuf)
		ldi	YH,HIGH(bas_inbuf)
bas_inval_03:	ld	XL,Z+
		st	Y+,XL
		dec	r16
		brne	bas_inval_03
		ldi	XL,0x20
		st	Y+,XL
		ldi	XL,0xff			;eol
		st	Y+,XL
		ldi	ZL,LOW(bas_inbuf)
		ldi	ZH,HIGH(bas_inbuf)
		call	expar			;calculate value
		rjmp	bas_inval_01a	
bas_inval_04:	cpi	tempreg1,0xec		;backspace
		brne	bas_inval_05
		cpi	r16,0
		breq	bas_inval_01		;nothing
		dec	libmio_cur_x
		libmio_calc
		ldi	XL,0x20
		st	Y,XL
		dec	r16
		rjmp	bas_inval_01
bas_inval_05:	cpi	tempreg1,0x20
		brcs	bas_inval
		cpi	tempreg1,'z'+1
		brcc	bas_inval
		mov	XL,libmio_cur_x		
		cpi	XL,libmio_cols-1
		brcc	bas_inval
		libmio_outchar
		inc	r16
		rjmp	bas_inval_01
		
bas_vline:	ldi	ctrl,2
		clr	XH
		mov	XL,r12		;line
		inc	XL
		libmio_outdez
		ldi	tempreg1,':'
		libmio_outchar
		clr	XH
		mov	XL,r13		;statement
		libmio_outdez
		libmio_outspace
		ret				


;-----------------------------------------------------------------------
; write array element X 
;-----------------------------------------------------------------------
bas_warr:	push	YH
		push	YL
		push	XH
		push	XL
		lds	XL,bas_partab		;LO address
		lds	XH,bas_partab+1		;HI address
		cpi	XH,0x03			;768=limit
		brcc	bas_warr_w		;->word
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		add	YL,XL
		adc	YH,XH
		pop	XL			;restore value
		pop	XH
		st	Y,XL			;store value
		lds	YL,bas_partab		;LO address
		lds	YH,bas_partab+1		;HI address
		adiw	YL,1			;increment pointer
bas_warr_01:	sts	bas_partab,YL		;LO address
		sts	bas_partab+1,YH		;HI address
		pop	YL
		pop	YH					
		ret
		
bas_warr_w:	subi	XH,4			;
		brcs	bas_warr_err		;768<=adr<=1023
		lsl	XL			;*2
		rol	XH			
		cpi	XH,0x03			;limit
		brcc	bas_warr_err		;>1407
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		add	YL,XL
		adc	YH,XH
		pop	XL			;restore value
		pop	XH
		st	Y+,XL
		st	Y+,XH
		lds	YL,bas_partab		;LO address
		lds	YH,bas_partab+1		;HI address
		adiw	YL,2
		rjmp	bas_warr_01

	
bas_warr_err:	ldi	ereg,18			;out of array
		pop	XL
		pop	XH
		pop	YL
		pop	YH	
		ret		

;-----------------------------------------------------------------------
; check for color store and fill flag
;-----------------------------------------------------------------------
bas_ccheck:	lds	tempreg4,libmio_color	;recent color
		sbrs	XL,0
		ret
		cpi	XL,3			;3 paramters
		brne	bas_ccheck_2		;no
		lds	tempreg2,bas_partab+4	;low of p3
bas_ccheck_1:	
		andi	tempreg2,0x0f		;set only fg
		lsl	tempreg2
		andi	tempreg4,0xf0		;no change bg
		or	tempreg2,tempreg4
		lds	tempreg4,libmio_color	;get color
		sts	libmio_color,tempreg2	;temporary color
		ret
		
bas_ccheck_2:	lds	tempreg2,bas_partab+8	;low of p5
		rjmp	bas_ccheck_1
										
				
;------------------------------------------------------------------------------
;check memory for useability
;------------------------------------------------------------------------------
bas_fcheck:	call	fsys_maxpage		;get no of pages
		cpi	tempreg1,0x00
		brne	bas_fcheck_01
		clr	tempreg3
		ret
		
bas_fcheck_01:	sts	bas_partab+8,tempreg1	;store no of pages
		call	fsys_checkf		;test for valid format
		cpi	tempreg1,0x01		;OK
		breq	bas_fcheck_02
		clr	tempreg3
bas_fcheck_02:	ret

					