//----------------------------------------------------------------------
// Titel     	: Basisstation fr Hausalarmanlage
//----------------------------------------------------------------------
// Schaltung 	: Pinbelegung siehe ports.h
// Beschreibung	: berwacht Kontakte und gibt Alarm.
//----------------------------------------------------------------------
// Prozessor 	: ATmega88P
// Takt			: 8 MHz, intern
// Datenrate	: 9600 Baud
// Datum    	: 15.04.2020
// Version   	: s.u.
// Autor     	: P. Fuhsy
// Fuses		: exfuse=0xF9, hfuse=0xD4, lfuse=0xE2, lockfuse=0xFF
/***********************************************************************/
//-----------------------------Includes---------------------------------
#include <stdbool.h>				//fr Bool-Variablen
#include <string.h>					//zur Stringbearbeitung
#include <avr/interrupt.h>			//fr Interrupts
#include "uart.h"					//fr die URT-Schnittstelle
#include "delay_x.h"				//fr Warteschleifen
#include "ports.h"					//zur Portdeklaration
#include "haa_bus.h"				//Bus der Alarmanlage

//----------------------------------------------------------------------
/***********************************************************************/
//-----------------------Konstanten / Variablen-------------------------
#define SWH				5			//Software-Hauptversion
#define SWN				2			//Software-Nebenversion
#define T_SCHARF_S		2			//Totzeit bis Scharf geschaltet wird in sek.		//TODO Zeit ndern	
#define T_ALARM_S		15			//Totzeit bis Voralarm zum Alarm wird in sek.
#define T_SUM_TOT_S		60			//Totzeit bis Summer nach dem Alarm ausgeschaltet wird in sek.	
#define T_CS_ANSTRG_MS	500			//Ansteuerungszeit des CS-Exit-Ausgangs
#define T_RXD_TOT_MS	500			/*Totzeit wielange neue Daten an Empfangene angehangen und nicht neugeschrieben werden.
									Die Gesamtzeit der empfangenen Daten (16 Bytes) ist bei 9600 Baud ca. 18ms.*/

//Zustnde der Anlage
bool _kontakt_alarm		= false;	//Alarm aller Kontakte
bool _kontakt_alarm_oe	= false;	//Alarm aller Kontakte ohne der Eingangstr
bool _voralarm			= false;	//Voralarm wenn Eingangstr geffnet wird
bool _alarm				= false;	//momentaner Alarm der Anlage
bool _anlage_start		= false;	//Anlage startet gerade
bool _anlage_scharf		= false;	//Anlage ist scharf
bool _inputs[7]				   ;	//Kontakte im Array
bool _in_sabo			= false;	//Sabotage der Kontakte
bool _protokoll_senden	= false;	//Bite ob dem Epfnger geantwortet werden soll

//Kommandos Basis
bool _kom_bas_alf		= false;	//Kommando True: Anlage soll starten, False: nur Statusausgabe
bool _kom_error			= false;	//Error beim Empfangen eines Kommandos

//Status des Codeschlosses
bool _cs_sabo			= false;	//Sabotagekontakte des Codeschlosses
bool _cse_best			= false;	//True: Besttigung dass Anlage startet
bool _cse_error			= false;	//Anlage kann nicht von aussen angesteuert werden

//Status der Sirene
bool _sir_sabo			= false;	//Sabotagekontakte der Sirene
bool _sir_fkt			= false;	//Funktionstest der Sirene
bool _sir_stoer			= false;	//Stoerung

//Schnittstelle
volatile bool _rxd_komplett		= false;	//Flag empfangener String abgeschlossen ist
volatile char _rxd_string[_DATA_LEN]	;			//Stringbuffer fr UART
volatile uint8_t _rxd_count		= 0;		//Zhler der empfangenen Bytes

//Zeitmessungen:
//bei ms sollte mind. uint16_t (0...65.535) gewhlt werden
//bei s gentigt uint8_t (0...255), wenn nicht mehr als 255s bentigt werden
volatile uint16_t _t_Led_Rot_ms;
volatile uint16_t _t_Led_Gruen_ms;
volatile uint16_t _t_Summer_ms;
volatile uint16_t _t_Impuls_ms;
volatile uint16_t _t_Sirene_ms;
volatile uint16_t _t_Rxd_tot_ms;
volatile uint16_t _t_Cs_anstrg_ms;
volatile uint8_t _t_Sum_tot_s;
volatile uint8_t _t_Zustand_s;
volatile uint8_t _t_Voralarm_s;

//------------------------------Funktionen------------------------------
void initPorts()			//Ports deklarieren
{
	//Output
	DDRD |= (1<<SUM_OUT);	//Summersausgang	
	DDRC |= (1<<SIR_OUT_1);	//Sirenenausgang 1
	DDRC |= (1<<SIR_OUT_2);	//Sirenenausgang 2
	DDRB |= (1<<CS_LED_RT);	//LED fr allg. Statuszustand
	DDRB |= (1<<CS_LED_GN);	//LED fr "Anlage nicht scharf"
	DDRD |= (1<<CS_EXIT);	//Externer Zugriff auf Codeschloss	

	//Input
	DDRC &=~(1<<IN_0);		//Kontakt 0
	DDRC &=~(1<<IN_1);		//Kontakt 1
	DDRD &=~(1<<IN_2);		//Kontakt 2	
	DDRB &=~(1<<IN_3);		//Kontakt 3
	DDRD &=~(1<<IN_4);		//Kontakt 4
	DDRD &=~(1<<IN_5);		//Kontakt 5
	DDRB &=~(1<<IN_6);		//Kontakt 6
	DDRB &=~(1<<IN_7);		//Kontakt 7	
	DDRC &=~(1<<IN_SABO);	//Sabotagekontakt der Kontake	
	DDRB &=~(1<<CS_REL_1);	//Relais 1 vom Codeschloss	
	DDRB &=~(1<<CS_REL_2);	//Relais 2 vom Codeschloss	
	DDRD &=~(1<<CS_SABO);	//Sabotagekontakt vom Codeschloss	
	DDRC &=~(1<<SIR_SABO);	//Sabotagekontakt der Sirene
	DDRB &=~(1<<SIR_STOER);	//Strungssignal der Sirene
	
	//interne Pull-Up-Widerstnde einschalten
	PORTC |= (1<<IN_0);		//Kontakt 0		
	PORTC |= (1<<IN_1);		//Kontakt 1	
	PORTD |= (1<<IN_2);		//Kontakt 2	
	PORTB |= (1<<IN_3);		//Kontakt 3
	PORTD |= (1<<IN_4);		//Kontakt 4
	PORTD |= (1<<IN_5);		//Kontakt 5
	PORTB |= (1<<IN_6);		//Kontakt 6
	PORTB |= (1<<IN_7);		//Kontakt 7	
	PORTC |= (1<<IN_SABO);	//Sabotagekontakt der Kontake	
	PORTB |= (1<<CS_REL_1);	//Relais 1 vom Codeschloss		
	PORTB |= (1<<CS_REL_2);	//Relais 2 vom Codeschloss		
	PORTD |= (1<<CS_SABO);	//Sabotagekontakt vom Codeschloss		
	PORTC |= (1<<SIR_SABO);	//Sabotagekontakt der Sirene
	PORTB |= (1<<SIR_STOER);//Strungssignal der Sirene		

	//Ports initiaisieren
	Summer_Off();			//Summerausgang
	Sir_Out1_Off();			//Sirenenausgang
	Sir_Out2_Off();			//Sirenenausgang	
	CS_Exit_Off();			//Exit-Ausgang ausschalten
	Led_Rot_Off();			//LED ausschalten
	Led_Gruen_Off();		//LED ausschalten
}
void initTimer() 			//Timer initialisierung
{
	//ISR_Freq = F_CPU : (Prescaler * ORC) in Hz
	//F_CPU		/ Vorteiler	/ Wunschfrequenz	= Vorladezeit
	//8.000.000	/ 256		/ 1.000 Hz			= 31
  
	//Timmer0, 8 Bit
	TCCR0A	= (1<<WGM01);	//CTC Modus
	TCCR0B	= (1<<CS02);	//Vorteiler 256
	OCR0A	= 31;			//Vorladezeit
	TIMSK0	= (1<<OCIE0A);	// Compare Interrupt erlauben
}
void init_Start()			//Power On Prozedur aller Ausgnge
{
	//Ausgnge ansteuern
	Led_Rot_On();
	Led_Gruen_On();
	_delay_ms(1000);
	
	//Ausgnge zurcksetzten
	Led_Rot_Off();
	Led_Gruen_Off();
	
	//Summer kurz testen
	Summer_On();
	_delay_ms(100);
	Summer_Off();
	_delay_ms (1000);
}
void Zustand_Anlage()		//allg. Zustand der Alarmanlage abfragen
{
	/*----------------------Codeschloss-----------------*/
	//Relais 1 offen / aktiviert
	if (CS_Rel1_offen())						
	{
		_anlage_start = true;
		
		//Totzeit ist noch nicht abgelaufen
		//Und prfen ob die Anlage scharf ist damit es nur einmal gesetzt wird
		if (!_anlage_scharf && _t_Zustand_s <= T_SCHARF_S)
		{
			_anlage_scharf = false;
		}
		
		//Totzeit ist abgelaufen
		else
		{
			_anlage_scharf = true;
			_anlage_start = false;
			_t_Zustand_s = 0;
		}
	}
	
	//Relais 1 geschlossen / deaktiviert
	else
	{
		_anlage_start = false;
		_anlage_scharf = false;
		//_alarm_hist = false;
		_t_Zustand_s = 0;
	}
	
	//Sabotagekontakt
	if(CS_Sabo())
	{
		_cs_sabo = true;
	}
		
	else
	{
		_cs_sabo = false;
	}
	/*--------------------------------------------------*/	
	
	/*-------------------------Sirene-------------------*/	
	//Strung
	if(Sir_Stoer())
	{
		_sir_stoer = true;
	}
	
	else
	{
		_sir_stoer = false;
	}	
	
	//Sabotagekontakt
	if(Sir_Sabo())
	{
		_sir_sabo = true;
	}
	
	else
	{
		_sir_sabo = false;
	}	
	/*--------------------------------------------------*/		

	/*------------------------Kontakte------------------*/		
	//Werte der Kontaktewerden ins Array geschrieben
 	_inputs[0] = In_0_offen() ? true : false;	//Auswertung, If-Kurzschreibweise
 	_inputs[1] = In_1_offen() ? true : false;	//Auswertung, If-Kurzschreibweise
 	_inputs[2] = In_2_offen() ? true : false;	//Auswertung, If-Kurzschreibweise
	_inputs[3] = In_3_offen() ? true : false;	//Auswertung, If-Kurzschreibweise
 	_inputs[4] = In_4_offen() ? true : false;	//Auswertung, If-Kurzschreibweise
	_inputs[5] = In_5_offen() ? true : false;	//Auswertung, If-Kurzschreibweise	
	_inputs[6] = In_6_offen() ? true : false;	//Auswertung, If-Kurzschreibweise
	_inputs[7] = In_7_offen() ? true : false;	//Auswertung, If-Kurzschreibweise
	
	//min. einer ist geffnet
	if ((In_0_offen()) || (In_1_offen()) ||											//TODO im Array prfen
		(In_2_offen()) || (In_3_offen()) ||
		(In_4_offen()) || (In_5_offen()) ||
		(In_6_offen()) || (In_7_offen()))
	{
		_kontakt_alarm = true;
	}

	//alle sind geschlossen
	else
	{
		_kontakt_alarm = false;
	}
	
	//min. einer ist geffnet ohne Eingangstr
	if ((In_1_offen()) ||															//TODO im Array prfen
		(In_2_offen()) || (In_3_offen()) ||
		(In_4_offen()) || (In_5_offen()) ||
		(In_6_offen()) || (In_7_offen()))
	{
		_kontakt_alarm_oe = true;
	}

	//alle geschlossen
	else
	{
		_kontakt_alarm_oe = false;
	}
	
	//Sabotagekontakt
	if(In_Sabo())
	{
		_in_sabo = true;
	}
			
	else
	{
		_in_sabo = false;
	}
	/*--------------------------------------------------*/				
}
void Voralarm()				//Abfragen ob ein Voralarm vorhanden ist
{
	//Alarmanlage ist scharf gestellt
	if (_anlage_scharf)
	{	
		//Kontakt 0 ist ausgelst
		if (In_0_offen())
		{
			_voralarm = true;
		}
	}

	//Alarmanlage noch nicht scharf gestellt
	else
	{
		_voralarm = false;
		_t_Voralarm_s = 0;
	}

	//falls es noch kein Voralarm gibt muss der Zhler zurckgesetzt werden
	if (!_voralarm)
	{
		_t_Voralarm_s = 0;
	}

	//Bei Alarm wird der Zeitzhler und Voralarm zurckgesetzt
	if (_alarm)
	{
		_voralarm = false;
		_t_Voralarm_s = 0;
	}
}
void Alarm()				//Abfragen ob ein Alarm vorhanden ist
{
	//Alarmanlage ist scharf geschaltet
	if (_anlage_scharf)
	{	
		//Kontaktalarm ohne Eingangstr oder Sabotagekontakt
		if (_kontakt_alarm_oe || CS_Sabo() || Sir_Sabo() || In_Sabo())		
		{
			_alarm = true;			
		}

	}
	//Alarmanlage nicht scharf gestellt / wieder abgeschaltet
	else
	{
		_alarm = false;		//Alarm zurcksetzten
	}
	
	if (_voralarm)
	{
		//Totzeit fr Voralarm ist abgelaufen
		if (_t_Voralarm_s > T_ALARM_S)
		{
			_alarm = true;
		}
	}	
}
void Summer_Anstrg()		//Summer ansteuern
{
	static bool ausgeloest = false;		//Alarm schon ausgelst
	static int zaehler = 0;				//Um den Summer nur einpaar mal anzusteuern
	uint8_t fall = 0;					//Fallunterscheidung
	uint16_t t_an = 0;
	uint16_t t_aus = 0;

	/*Fallunterscheidungen*/
	//Prio 6: Sirenenstrung
	if (_sir_stoer) {fall = 6;}
		
	//Prio 5: Kontaktalarm beim manuellen starten
	if (_anlage_start && (_kontakt_alarm_oe || _sir_sabo || _cs_sabo || _in_sabo)) {fall = 5;}
		
	//Prio 4: Besttigung beim externen starten
	if (_cse_best) {fall = 4;}
			
	//Prio 3: Error beim externen starten
	if (_cse_error) {fall = 3;}

	//Prio 2: Voralarm
	if (_voralarm) {fall = 2;}	
			
	//Prio 1: Alarm ausgelst
	if (_alarm) {fall = 1;}		
		
	switch (fall)
	{
		//Alarm ausgelst
		case 1:
			//Alarm bereits ausgelst
			if (ausgeloest)
			{
				//falls die Totzeit abgelaufen ist
				if (_t_Sum_tot_s >= T_SUM_TOT_S)
				{
					Summer_Off();
					_t_Sum_tot_s = 0;
				}
			}

			//Alarm noch nicht ausgelst
			else
			{
				ausgeloest = true;
				Summer_On();
			}
		break;
		
		//Kontaktalarm beim manuellen Starten der Anlage
		case 2:
			//Ein-Auszeiten des Summers
			t_an = 100;
			t_aus = 900;
			
			//Summer an
			if(_t_Summer_ms <= t_an)
			{
				Summer_On();
			}
		
			else
			{
				//Summer aus
				if (_t_Summer_ms <= t_an + t_aus)
				{
					Summer_Off();
				}
			
				//Timer zurcksetzten
				else
				{
					_t_Summer_ms = 0;
				}
			}
		break;
		
		//Error beim externen Ansteuern
		case 3:
		
			//Ein-Auszeiten des Summers
			t_an = 100;
			t_aus = 100;
			
			if(zaehler < 4)
			{
				//Summer an
				if(_t_Summer_ms <= t_an)
				{
					Summer_On();
				}
			
				else
				{
					//Summer aus
					if (_t_Summer_ms <= t_an + t_aus)
					{
						Summer_Off();
					}
				
					//Timer zurcksetzten
					else
					{
						_t_Summer_ms = 0;
						zaehler++;
					}
				}
			}
			
			else
			{
				fall = 0;
				_cse_error = false;		//Error wieder lschen, damit das Kommando mehrmals hintereinander ausgefhrt werden kann
			}
		
		break;
		
		//Besttigung beim externen starten
		case 4:
		
			//Ein-Auszeiten des Summers
			t_an = 100;
			t_aus = 100;
			
			if(zaehler < 2)
			{
				//Summer an
				if(_t_Summer_ms <= t_an)
				{
					Summer_On();
				}
			
				else
				{
					//Summer aus
					if (_t_Summer_ms <= t_an + t_aus)
					{
						Summer_Off();
					}
				
					//Timer zurcksetzten
					else
					{
						_t_Summer_ms = 0;
						zaehler++;
					}
				}
			}
			
			else
			{
				fall = 0;
				_cse_best = false;
			}
		
		break;		
		
		//Voralarm ausgelst
		case 5:
			//Ein-Auszeiten des Summers
			t_an = 100;
			t_aus = 900;
			
			//Summer an
			if(_t_Summer_ms <= t_an)
			{
				Summer_On();
			}
						
			else
			{
				//Summer aus
				if (_t_Summer_ms <= t_an + t_aus)
				{
					Summer_Off();
				}
							
				//Timer zurcksetzten
				else
				{
					_t_Summer_ms = 0;
				}
			}
		break;
		
		//Strung ausgelst
		case 6:
		//Ein-Auszeiten des Summers
		t_an = 100;
		t_aus = 59900;
		
		//Summer an
		if(_t_Summer_ms <= t_an)
		{
			Summer_On();
		}
		
		else
		{
			//Summer aus
			if (_t_Summer_ms <= t_an + t_aus)
			{
				Summer_Off();
			}
			
			//Timer zurcksetzten
			else
			{
				_t_Summer_ms = 0;
			}
		}
		break;		
				
		default:
			Summer_Off();
			ausgeloest = false;
			_t_Summer_ms = 0;
			_t_Sum_tot_s = 0;
			zaehler = 0;
		break;
	}
}
void Sirene_Anstrg()		//Sirene ansteuern
{
	uint8_t fall = 0;					//Fallunterscheidung

	/*Fallunterscheidungen*/
	//Prio 1: Alarm ausgelst
	if (_alarm) {fall = 1;}
	
	
	switch (fall)
	{
		//Alarm ausgelst
		case 1:
			Sir_Out1_On();
		break;

		default:
			Sir_Out1_Off();
		break;
	}
}
void Led_Rot_Anstrg()		//LED rot fr allg. Betriebszustnde setzten
{
	static uint16_t t_an = 0;
	static uint16_t t_aus = 0;
	
	//-------------Blinkzeiten festlegen-------------------
	//Blinkzeiten setzten, Reihenfolge beachten
	//Anlage startet gerade
	if (_anlage_start)
	{
		t_an = 500;	//in Milisekunden
		t_aus = 500;
	}

	//Anlage ist scharf
	if (_anlage_scharf)
	{
		t_an = 100;	//in Milisekunden
		t_aus = 2900;
	}

	//Voralarm ist vorhanden
	if (_voralarm)
	{
		t_an = 250;	//in Milisekunden
		t_aus = 250;
	}
	//-----------------------------------------------------

	//------------------LED-Blinker------------------------
	//Wenn einer davon auf True ist, fngt die LED an zu blinken
	if (_voralarm || _anlage_start || _anlage_scharf)
	{

		//Dauerlicht fr Alarm
		if (_alarm)
		{
			Led_Rot_On();
			_t_Led_Rot_ms = 0;	//globale Zeitvariable zurcksetzten
		}

		//..ansonsten Blinken ansetzten
		//LED an
		else if (_t_Led_Rot_ms <= t_an)
		{
			Led_Rot_On();
		}
		//LED aus
		else if (_t_Led_Rot_ms <= t_an + t_aus)
		{
			Led_Rot_Off();
		}
		else
		{
			_t_Led_Rot_ms = 0;
		}
		
	}

	//Anlage abgeschaltet
	else
	{
		Led_Rot_Off();
		_t_Led_Rot_ms = 0;
	}
	//-----------------------------------------------------
}
void Led_Gruen_Anstrg()		//LED grn fr "Alles OK" setzten
{
	static uint16_t t_an = 500;
	static uint16_t t_aus = 500;
	
	//LED einschalten wenn die Anlage nicht scharf ist
	if (!_anlage_scharf)
	{
		
		//Ein Kontakt- oder Sabotagekontakt ist geffnet
		if (_kontakt_alarm || _sir_sabo || _cs_sabo || _in_sabo)
		{
			//LED blinken: AN
			if (_t_Led_Gruen_ms <= t_an)
			{
				Led_Gruen_On();
			}
			//LED blinken_ AUS
			else if (_t_Led_Gruen_ms <= t_an + t_aus)
			{	
				Led_Gruen_Off();
			}
			else
			{
				_t_Led_Gruen_ms = 0;
			}	
		}

		//Alle Kontakte sind geschlossen
		else
		{
			Led_Gruen_On();
			_t_Led_Gruen_ms = 0;
		}
	}

	//bei scharfer Alarmanlage LED ausschalten
	else
	{
		Led_Gruen_Off();
		_t_Led_Gruen_ms = 0;
	}
}
void CS_Exit_Anstrg()		//Anlage von extern Ansteuern
{
	//Kommando wurde empfangen
	if (_kom_bas_alf)
	{
		//berprfung ob die Anlage gerade start oder breits scharf ist
		//Bei der CS-Exit-Anstrg msste direkt der Bit _anlage_start eine 1 bekommen
		if (!_anlage_start && !_anlage_scharf)
		{		
			//berprfen ob angesteuert oder Error signalisiert werden soll
			//Eine Tr ist geffnet oder Sabotagekontakt
			if(_kontakt_alarm || _sir_sabo || _cs_sabo || _in_sabo)
			{
				_cse_error = true;			//Somit Error vorhanden
				_kom_bas_alf = false;		//Bit wieder zurcksetzten
				_protokoll_senden = true;	//Funktion "Protokoll senden" freigeben
			}
		
			//alle Tren sind geschlossen
			else
			{
				_cse_error = false;			//Kein Error vorhanden			
				CS_Exit_On();				//CS-Exit ansteuern
				
				//fr den CS-Code wird nur ein Impuls bentigt
				if(_t_Cs_anstrg_ms >= T_CS_ANSTRG_MS)
				{
					CS_Exit_Off();				//CS-Exit nicht mehr ansteuern				
					_t_Cs_anstrg_ms = 0;		//Zeitzhler zurcksetzten
					_protokoll_senden = true;	//Funktion "Protokoll senden" freigeben
					_cse_best = true;			//Besttigung das angesteuerrt wurde
				}
			}
		}
		
		//CS wird ein weiteres mal angesteuert
		else
		{
			CS_Exit_Off();				//Kann vorzeitig geschlossen werden wenn eine Reaktion gesehen wird			
			_kom_bas_alf = false;		//Bit wieder zurcksetzten			
			_t_Cs_anstrg_ms = 0;		//Zeitzhler zurcksetzten
			_protokoll_senden = true;	//Funktion "Protokoll senden" freigeben			
		}
	}
	
	else
	{
		_t_Cs_anstrg_ms = 0;		//Zeitzhler zurcksetzten
	}
}
int Char_in_int(char zeichen_hb, char zeichen_nb)			//Zeichen in Hex wandeln
{
	int i;			//Laufvariable
	char zeichen;	//Zwischenspeicher fr die bergabeparameter
	
	for(i=0; i<=1; i++)
	{
		if (i == 0) {zeichen = zeichen_hb;}		//HB-Wert zwischenspeichern in der 1ten Schleife
		if (i == 1) {zeichen = zeichen_nb;}		//NB-Wert zwischenspeichern in der 2ten Schleife
		
		//Zahlen 0-9
		if (zeichen >= 48 && zeichen <= 57)
		{
			zeichen = zeichen - 48;
		}
		
		//Buchstaben von A-F
		else if (zeichen >= 65 && zeichen <= 70)
		{
			zeichen = zeichen - 55;		//Char in Int ndern
		}
		
		//Buchstaben von a-f
		else if (zeichen >= 97 && zeichen <= 102)
		{
			zeichen = zeichen - 87;		//Char in Int ndern
		}
		
		if (i == 0) {zeichen_hb = zeichen;}		//HB-Werte zurckschreiben in der 1ten Schleife
		if (i == 1) {zeichen_nb = zeichen;}		//NB-Werte zurckschreiben in der 2ten Schleife
	}

	zeichen_hb = zeichen_hb << 4;				//Bits um 4 Stellen nach links schieben
	return (zeichen_nb + zeichen_hb);			//Kompletter Rckgabewert
}
void RxD_Auswerten()		//Ankommende Daten auswerten
{
	//String komplett und abgeschlossen, Kommando kann ausgewertet werden
	if(_rxd_komplett)
	{
		//BAS-KOM Kommandobit maskieren / erkennen
		int wert;
		wert = Char_in_int(_rxd_string[0], _rxd_string[1]); //Byte0 zwischenspeichern
		bool bas_kom = (wert & 0x80) >> 7;					//falls erkannt, wird eine 1 erzeugt			
		
		//Kommandbit erkannt
		if (bas_kom)
		{				
			//Errorbit zuerst setzten (Kommando nicht erkannt), wird dann durch erkannte Kommandos wieder zurckgesetzt
			_kom_error = true;			
			
			//BAS-ALF Anlage scharf schalten maskieren / erkennen
			_kom_bas_alf = (wert & 0x10) >> 4;
			//Errorbit zurcksetzten
			if(_kom_bas_alf){_kom_error = false;}
						
			//Da bei "Error" keine Aktion stattfindet, muss das Protokoll freigegeben werden
			if(_kom_error)
			{
				_protokoll_senden = true;	//Anlage nur "Protokoll senden" freigeben
			}
		
		}
		
		//Kommando nicht erkannt, nur Status senden
		else
		{
			_kom_error = false;				//Errorbit zurcksetzten
			_protokoll_senden = true;	//Anlage nur "Protokoll senden" freigeben
		}
		//-------------------------------------------------

	memset(_rxd_string, '\0', sizeof(_rxd_string));	//Inhalte vom Array lschen
	_rxd_komplett = false;		//RxD wieder freigeben
	
	}
	
	//Counter reseten, nach Zeitvorgabe
	if(_t_Rxd_tot_ms >= T_RXD_TOT_MS)
	{
		_rxd_count = 0;
		_t_Rxd_tot_ms = 0;
	}	
}
void Haa_Bus_schreiben()	//Haa-bus beschreiben damit zusammengefasst und gesendet wird
{
	BAS_byte_schreiben(_kom_bas_alf, _kom_error, _cs_sabo, _anlage_scharf, _anlage_start, _alarm, _voralarm, _kontakt_alarm);
	SIR_byte_schreiben(_sir_sabo, _sir_fkt, _sir_stoer);//Statusmeldungen der der Sirene
	IN0_byte_schreiben(_inputs);		//als Array bertragen (wird als Zeiger des ersten Arraywertes bergeben)	
	IN1_byte_schreiben(_in_sabo);		//Sabotage der Kontakte
	RES_byte_schreiben();				//Resereve
	SWP_byte_schreiben(SWH, SWN);		//SW-Version
	CHK_byte_schreiben();				//Checksumme bilden
	
	HAA_Bus_Hex();						//setzt die gesammelten Werte in eine HEX um
}
void Protokoll_senden()		//Ausgabe des Protokolls
{		
	//wird nur gesendet wenn ein String empfangen wurde		
	if (_protokoll_senden)		
	{	
		putString(HAAHEX);			//Anfragen zusammenbauen
		_kom_bas_alf = false;		//Kommandoflag wieder zurcksetzten	
		_protokoll_senden = false;	//Antwort-bit wieder zurcksetzten
	}		
}
int main(void)					/***Hauptprogramm***/
{
	initPorts();			//Ports deklarieren
	initUART();				//UART initialisieren
	initTimer();			//Timer initialisieren
	init_Start();			//Power On Prozedur aller Ausgnge
	sei();					//alle Interrups einschalten 

	while(true) //Hauptschleife
	{		
		RxD_Auswerten();	//auf erhaltene Kommandos reagieren
		CS_Exit_Anstrg();	//Anstrg. ber UART 
		Zustand_Anlage();	//Anlagenzustand abfragen
 		Voralarm();			//Abfragen ob ein Voralarm vorhanden ist
 		Alarm();			//Abfragen ob ein Alarm vorhanden ist	 
 		Summer_Anstrg();	//Ausgang des Summers setzten
		Sirene_Anstrg();	//Ausgang der Sirene setzten
  		Led_Gruen_Anstrg();	//Ausgang der grnen LED setzten
 		Led_Rot_Anstrg();	//Ausgang der roten LED setzten
		
		Haa_Bus_schreiben();//Bus fr bertragung und Anzeige beschreiben	
		Protokoll_senden();	//Protokoll ber UART senden
	}
}
ISR (TIMER0_COMPA_vect)		//Time-Compare-Interrupt, wird jede ms ausgelst
{
	static uint16_t millisek = 0;
	
	//Millisekungen zhlen
	_t_Led_Rot_ms ++;
	_t_Led_Gruen_ms ++;
	_t_Summer_ms ++;
	_t_Sirene_ms ++;
	_t_Rxd_tot_ms ++;
	_t_Cs_anstrg_ms ++;
	_t_Impuls_ms ++;

	millisek++;
		
	//Sekunden zhlen
	if (millisek == 999)
	{
		//Funktionszhler
		_t_Sum_tot_s++;
		_t_Zustand_s ++;
		_t_Voralarm_s ++;
		
		millisek = 0;	//Zhler zurcksetzten
	}	
}
ISR (USART_RX_vect)		//Interrupt wird ausgelst sobald neue Daten im USART-Empfangspuffer liegen
{
	_rxd_string[_rxd_count] = UDR0;				//in Array schreiben
	
	//Gesamtlnge wurde empfangen, gleichzeitig kann die max. Arraylnge nicht berschritten werden
	//UDR kann nur einmal gelesen werden, deshalb muss hier aus dem Array geprft werden
	if(_rxd_count == _DATA_LEN -1)	//Wird einer gewissen Zeit automatisch zurckgesetzt, siehe RxD_Auswerten()
	{
		_rxd_string[_rxd_count + 1] = '\0';	//Abschlusszeichen anhngen falls nicht empfangen
		_rxd_count = 0;						//Zhler fr das nchste Mal zurcksetzten
		_rxd_komplett = true;				//Bit setzten damit die Daten ausgewertet werden
	}
	
	else
	{
		_rxd_count++;						//Zhler fr das nchste Mal hochzhlen
	}
}