#pragma once

#include <avr/io.h>
#include "AvrRegister.h"
#include <CombiePin.h>
using namespace Combie::Pin;
OutputPin<2> messPin2; 
OutputPin<3> messPin3; 
OutputPin<4> messPin4; 

// Size of the circular buffer, must be power of 2.
constexpr uint8_t BUFFERSIZE = 16;
// Size of buffer
constexpr uint8_t BUFFER_MASK = BUFFERSIZE - 1;

struct ring
{
  volatile uint8_t fifo[BUFFERSIZE];
  volatile uint8_t head;
  volatile uint8_t tail;
  volatile bool startNewTransfer;
  volatile uint8_t slaveSelect;
  volatile uint8_t lastError;
};


// ------ here begins the template class -----------------------------------

template<uint8_t const csPin, uint8_t const divider>  // slave select pin, SPI clock divider
class SpiSlave  
{
  private:
    OutputPin<csPin> slaveSelectPin;
    
    void setFrequency (void)
    {        
      *ptrStatus  &= ~(static_cast<uint8_t> (bitmask.SPI_SPI2X));
      *ptrControl &= ~(static_cast<uint8_t> (bitmask.SPI_SPR1 | bitmask.SPI_SPR0));
      
      switch (divider)
      {
        case   2:   *ptrStatus  |= static_cast<uint8_t> (bitmask.SPI_SPI2X);
                    break;
        
        case   4:   break;
        
        case   8:   *ptrStatus  |= static_cast<uint8_t> (bitmask.SPI_SPI2X);
                    *ptrControl |= static_cast<uint8_t> (bitmask.SPI_SPR0);
                    break;

        case  16:   *ptrControl |= static_cast<uint8_t> (bitmask.SPI_SPR0);
                    break;
        
        case  32:   *ptrStatus  |= static_cast<uint8_t> (bitmask.SPI_SPI2X);
                    *ptrControl |= static_cast<uint8_t> (bitmask.SPI_SPR1);
                    break;
        
        case  64:   *ptrControl |= static_cast<uint8_t> (bitmask.SPI_SPR1);
                    break;
        
        case 128:   *ptrControl |= static_cast<uint8_t> ((bitmask.SPI_SPR1)|(bitmask.SPI_SPR0));
                    break;

        default :   break; // default divider 4
      }
    }
             
	public:
    // Default Konstruktor
    SpiSlave ()
    { }    
			
    // Variablen and Methoden
    ring buffer;              // create FiFo-Buffer
    
    // Test 'function pointer'
    void (*ptrFunction)() = nullptr;  
      
    void init (void)
    {    
      messPin2.init();
      messPin3.init();
      messPin4.init();
        
      slaveSelectPin.init();
      slaveSelectPin.setHigh();
                   
      buffer.head = 0;
      buffer.tail = 0;
      buffer.startNewTransfer = true;
      buffer.slaveSelect = csPin;
      
      // Test 'function pointer'         
      ptrFunction = slaveSelectHigh;   
           
      // Register content delete
      *ptrStatus  = static_cast<uint8_t> (0);
      *ptrControl = static_cast<uint8_t> (0);
      *ptrData    = static_cast<uint8_t> (0);
              
      // SPI Settings
      *ptrControl |= static_cast<uint8_t> (bitmask.SPI_MSTR);   // set Master
      *ptrControl |= static_cast<uint8_t> (bitmask.SPI_SPE);    // SPI Enable
      
      // read once to delete flag
      while (SPSR & (1<<SPIF)) { ; }
    }
    
    void slaveSelectHigh (void) 
    {	          
      slaveSelectPin.setHigh();
    }
    
    // write one Byte in the buffer for transmitting via SPI
    void fillBuffer(uint8_t const data) 
    {	          
      messPin2.setHigh();
      
      uint8_t const tempHead = (buffer.head + 1) & BUFFER_MASK;
          
      // wait for free space in buffer
      while (tempHead == buffer.tail) { ; } 
      
      buffer.fifo[tempHead] = data;
      buffer.head = tempHead;
      
      messPin2.setLow();
    }

    // write 'first' byte from buffer into SPDR and enable the interrupt
    void transferGoOut() 
    {	          
      messPin3.setHigh();
              
      if (buffer.startNewTransfer) 
      {
        if (buffer.head != buffer.tail)
        { // ab hier 4,4µs 
          buffer.startNewTransfer = false;
          setFrequency();
          slaveSelectPin.setLow();  
          
          // calculate and store new buffer index
          uint8_t const tempTail = (buffer.tail + 1) & BUFFER_MASK;
          buffer.tail = tempTail;     
          
          // get one byte from buffer and write it to SPI
          *ptrData = static_cast<uint8_t> ( buffer.fifo[tempTail] );  
          
          // enable Serial Transfer Complete Interrupt
          *ptrControl |= static_cast<uint8_t> (bitmask.SPI_SPIE);
        }
      }  
      
      messPin3.setLow();
    }
};
