// ********************************************************************
// Demoprogramm
// Inhalt: LCD (DOG-M 081/162/163) mit (ST7036 Controller) ansteuern,
// 5V/3V Variante, mit selbst definierten Zeichen im RAM, SPI-Ansteuerung seriell
// LCD-Controller: ST7036 kompatibel
// Stand: 18.04.2008, Autor: Matthias Kahnt, Url: pro-51.eltra-tec.de
// SDCC-Version: M-IDE-51/SDCC 2.8.0
// AT89C51ED2: --code-loc 0x0000 --iram-size 256 --xram-loc 0x0000 --xram-size 0x0700
// AT89S8253:  --code-loc 0x0000 --iram-size 256
// ********************************************************************
// Fr 8051-Microcontroller, z.B. AT89C51ED2 oder AT89S8253
// Frequenz: 11.0592 MHz
// Code-Speicherbedarf: ca. 1.3kByte, keine XRAM-Verwendung
// Das Programm verwendet nur 8051/52 Standard-SFRs und wurde
// mit dem AT89C51ED2 auf dem PRO-51 System getestet.
// ********************************************************************
// Die Programmfunktionen drfen fr private Nutzung verwendet und
// verndert werden. Fr gewerbliche Nutzung ist die Zustimmung
// des Autors erforderlich. Die Nutzung erfolgt auf eigene Gefahr.
// Fr eventuell entstehende Schden wird keine Haftung bernommen.
// ********************************************************************

#include "at89c51ed2.h"   // Register des AT89C51ED2
//#include "at89s8253.h"   // Register des AT89S8253
#include <string.h>      // Stingfunktionen

// Typendefinition
typedef unsigned char byte;
typedef unsigned int  word;
typedef unsigned long dword;

// Hardware PINs, bitte anpassen
#define LCD_ON    P1_4
#define LCD_RS    P3_7
#define LCD_SCL   P2_6
#define LCD_DATA  P2_7
//#define LCD_CSB (CS-Signal: wird im Bsp. nicht verwendet)

// Displaytyp, bitte anpassen
#define SPALTEN  16
#define ZEILEN   3
#define SPANNUNG 5

// Tabelle fr 8 selbst definierte Sonderzeichen
byte code sonderzeichen[] = {
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F, // 00-07 Zeichen an Adresse 0
  0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x1F, // 08-15 Zeichen an Adresse 1
  0x00,0x00,0x00,0x00,0x00,0x1F,0x1F,0x1F, // 16-23 Zeichen an Adresse 2
  0x00,0x00,0x00,0x00,0x1F,0x1F,0x1F,0x1F, // 32-39 Zeichen an Adresse 3
  0x00,0x00,0x00,0x1F,0x1F,0x1F,0x1F,0x1F, // 40-47 Zeichen an Adresse 4
  0x00,0x00,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F, // 48-55 Zeichen an Adresse 5
  0x00,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F, // 56-63 Zeichen an Adresse 6
  0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F  // 64-71 Zeichen an Adresse 7
};

// Tabelle der DOG-M Initialisierung
byte code lcd_init_code[] = {
  0x31,0x1C,0x51,0x6A,0x74, // 00-04 DOG-M 081 5V
  0x31,0x14,0x55,0x6D,0x7C, // 05-09 DOG-M 081 3V
  0x39,0x1C,0x52,0x69,0x74, // 10-14 DOG-M 162 5V
  0x39,0x14,0x55,0x6D,0x78, // 15-19 DOG-M 162 3V
  0x39,0x1D,0x50,0x6C,0x77, // 20-24 DOG-M 163 5V
  0x39,0x15,0x55,0x6E,0x72  // 25-29 DOG-M 163 3V
};

// Tabelle der DOG-M Zeilenanfangsadressen
byte code lcd_zeilen_adresse[] = {
  0x00,0x00,0x00, // 00-02 DOG-M 081
  0x00,0x40,0x40, // 03-05 DOG-M 162
  0x00,0x10,0x20  // 06-08 DOG-M 163
};

// ********************************************************************
// Verzgerungsschleife fr kurze Zeit
// Zeit: Wartezeit in [zeit]
//       1 = 20us
//       2 = 28us
//       3 = 36us u.s.w. (gilt fr 11.0952MHz)
//     255 = ca. 2ms
// ********************************************************************
void delay_us(byte zeit) {
byte zaehler;
  for ( zaehler = zeit; zaehler; zaehler-- ) {
   _asm nop; _endasm;
  }
}

// ********************************************************************
// Verzgerungsschleife fr lange Zeit
// Zeit: Wartezeit in [zeit]
//       1 = ca. 10ms
//       2 = ca. 20ms
//       3 = ca. 30ms u.s.w.
//     255 = ca. 2,5s (gilt fr 11.0952MHz)
// ********************************************************************
void delay(byte zeit) {
byte zaehler;
  for (zaehler = zeit; zaehler; zaehler--) {
    delay_us(255);  // dauert ca. 2ms
    delay_us(255);  // dauert ca. 2ms
    delay_us(255);  // dauert ca. 2ms
    delay_us(255);  // dauert ca. 2ms
    delay_us(255);  // dauert ca. 2ms
  }
}

// ***********************************************************************
// Schreiben eines Zeichens an das LCD-Modul
// seriell Soft-SPI Mode, 3/4 Draht Interface
// bergabe: lcd_byte : Auszugebendes Zeichen/Steuerzeichen
//           lcd_mode : 0 - Daten
//                      1 - Steuerzeichen
// ***********************************************************************
void write_lcd(byte lcd_byte, bit lcd_mode) {
byte stelle;
  // Pause bevor nchstes Zeichen gesendet werden darf
  delay_us(5); // Pause mind. 26,3us
  // LCD_CSB = 0;
  if (lcd_mode) LCD_RS = 0; // Steuerzeichen
    else LCD_RS = 1;        // Daten
  // Byte senden seriell senden, H-Bit zuerst (Bit 8 7 6 5 4 3 2 1 0)
  for ( stelle = 0x80; stelle; stelle >>= 1 ) {
    LCD_DATA = lcd_byte & stelle;
    LCD_SCL = 0;
    LCD_SCL = 1;
  }
  // LCD_CSB = 1;
}

// ***********************************************************************
// Definieren eines Sonderzeichen
// bergabe: lcd_addr : Adresse
//           lcd_zeichen: Zeiger auf das 1. Byte der Zeichendefinition
// ***********************************************************************
void write_lcd_cg(byte lcd_addr, byte *lcd_zeichen) {
byte lcd_i;
  write_lcd(0x38,1); // Function Set DL=1 N=1 DH=0 IS2=0 IS1=0 IS Table 0
  for(lcd_i = 0; lcd_i < 8; lcd_i++) {
    write_lcd(0x40 + lcd_addr * 8 + lcd_i,1); // CG RAM Adresse Set (01aa azzz)
    write_lcd(*lcd_zeichen,0);  // Data Write 8x Pixelzeile
    lcd_zeichen++;
  }
  write_lcd(0x39,1); // Function Set DL=1 N=1 DH=0 IS2=0 IS1=1 IS Table 1
} 

// ***********************************************************************
// Definieren von 8 Sonderzeichen
// bergabe:  lcd_zeichen: Zeiger auf das Zeichenfeld (8x8 Byte)
// ***********************************************************************
void write_lcd_cg_block(byte *sonderzeichenblock) {
byte lcd_i;
  for(lcd_i = 0; lcd_i < 8; lcd_i++) {
    write_lcd_cg(lcd_i, &sonderzeichenblock[lcd_i * 8]);
  }
} 

// ***********************************************************************
// Lschen des LCD-Display
// und Kursor auf Position 0,0 setzen
// ***********************************************************************
void clear_lcd(void) {
  write_lcd(0x01,1);
  delay(20);
} 

// ***********************************************************************
// Ausgabe eines ASCII-Zeichen positioniert auf dem LCD-Modul
// bergabe: lcd_x : Spalte (0...SPALTEN-1)
//           lcd_y : Zeile  (0...ZEILEN-1)
//           lcd_ascii : ASCII-Zeichen
// *********************************************************************** 
void write_byte_lcd(byte lcd_x, byte lcd_y, byte lcd_ascii) {
byte lcd_offset = 0;
  if (lcd_x > (SPALTEN - 1)) lcd_x = 0;
  if (lcd_y > (ZEILEN  - 1)) lcd_y = 0;
  switch (lcd_y) {
    case 0:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3];     break; // Zeile 1
    case 1:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3 + 1]; break; // Zeile 2 
    case 2:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3 + 2]; break; // Zeile 3
  };
  write_lcd(0x80 + lcd_x + lcd_offset,1); // Kursorposition setzen
  write_lcd(lcd_ascii,0);                 // Ausgabe des ASCII-Zeichen an der Kursorposition
}

// ***********************************************************************
// Ausgabe einer Zeichenkette positioniert auf dem LCD-Modul
// bergabe: lcd_x : Spalte (0...SPALTEN-1)
//           lcd_y : Zeile  (0...ZEILEN-1)
//           lcd_zeichen : Adresse der auszugebenden format.  Zeichenkette
//           clr_line    : Lschen bis Zeilenende
//                         1 - Lschen
//                         0 - kein Lschen
// ***********************************************************************
void printf_lcd(byte lcd_x, byte lcd_y, byte *lcd_zeichen, bit clr_line) {
byte lcd_i;
byte lcd_offset = 0;
  if (lcd_x > (SPALTEN - 1)) lcd_x = 0;
  if (lcd_y > (ZEILEN  - 1)) lcd_y = 0;
  switch (lcd_y) {
    case 0:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3];     break; // Zeile 1
    case 1:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3 + 1]; break; // Zeile 2 
    case 2:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3 + 2]; break; // Zeile 3
  }
  write_lcd(0x80 + lcd_x + lcd_offset,1); // Kursorposition setzen
  // Ausgabe der Zeichenkette ab der Kursorposition
  lcd_offset = strlen(lcd_zeichen); // Lnge der Zeichenkette
  if (lcd_offset > SPALTEN) lcd_offset = SPALTEN;
  for(lcd_i = lcd_offset; lcd_i; lcd_i--) {
    write_lcd(*lcd_zeichen,0); lcd_zeichen++;
  }
  if (clr_line) {
    // Lschen bis Zeilenende
    for(lcd_i = SPALTEN - lcd_offset - lcd_x; lcd_i; lcd_i--)
      write_lcd(' ',0);
  }
}

// ***********************************************************************
// Blinkposition/Cursor auf dem LCD-Displays setzen
// bergabe: lcd_x : Spalte (0...SPALTEN-1)
//           lcd_y : Zeile  (0...ZEILEN-1)
//           lcd_blink :   0 - Blinken/Cursor aus
//                         1 - Blinken an
//                         2 - Cursor  an
// ***********************************************************************
void blink_lcd(byte lcd_x, byte lcd_y, byte lcd_blink) {
byte lcd_offset = 0;
  write_lcd(0x0C,1); // KURSOR ausschalten
  if (lcd_x > (SPALTEN - 1)) lcd_x = 0;
  if (lcd_y > (ZEILEN  - 1)) lcd_y = 0;
  switch (lcd_y) {
    case 0:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3];     break; // Zeile 1
    case 1:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3 + 1]; break; // Zeile 2 
    case 2:  lcd_offset = lcd_zeilen_adresse[(ZEILEN - 1) * 3 + 2]; break; // Zeile 3
  };
  write_lcd(0x80 + lcd_x + lcd_offset,1); // Blinkposition setzen
  if (lcd_blink == 1) write_lcd(0x0D,1);  // Blinken ein
  if (lcd_blink == 2) write_lcd(0x0E,1);  // Cursor ein
}


// ***********************************************************************
// Grundinitialisierung des LCD-Moduls in SPI-Mode (seriell)
// ***********************************************************************
#pragma save
#pragma disable_warning 126
void init_lcd(void) {
byte offset;
  LCD_ON  = 1; // LCD Beleuchtung einschalten
  delay(5); // ca. 50ms Wartezeit nach dem Einschalten 
  offset = (ZEILEN - 1) * 10;     // Offset = 00, 10, 20
  if (SPANNUNG != 5) offset += 5; // Offset = 05, 15, 25
  // Grundinitialisierung (SPI, wie im 8-Bit parallel-Mode)
  write_lcd(lcd_init_code[offset],1);     // Function Set
  write_lcd(lcd_init_code[offset],1);     // Function Set (gleiches Byte nochmal senden)
  write_lcd(lcd_init_code[offset + 1],1); // Bias Set
  write_lcd(lcd_init_code[offset + 2],1); // Power Control + Kontrast Set C5,C4
  write_lcd(lcd_init_code[offset + 3],1); // Follower Control
  write_lcd(lcd_init_code[offset + 4],1); // Kontrast Set C3,C2,C1,C0
  write_lcd(0x0C,1); // Display Set
  write_lcd(0x06,1); // Entry Mode Set
  clear_lcd();       // Display lschen
}
#pragma restore

// ********************************************************************
// Hauptprogramm
// ********************************************************************
void main (void) {
  init_lcd();
  write_lcd_cg_block(sonderzeichen); // Sonderzeichen definieren
  // Ausgabebeispiel fr 3 zeiliges Display
  printf_lcd(0,0,"LCD-Basic's",1); // Einschaltmeldung
  printf_lcd(0,1,"DOG-M 163 ST7036",1);  
  printf_lcd(0,2,"V1 18.04.2008",1);  
  delay(200);  // Pause ca. 2s
  printf_lcd(0,2,"CGRAM:",1); // Sonderzeichen ausgeben
  write_byte_lcd(7,2,0);
  write_byte_lcd(8,2,1);
  write_byte_lcd(9,2,2);
  write_byte_lcd(10,2,3);
  write_byte_lcd(11,2,4);
  write_byte_lcd(12,2,5);
  write_byte_lcd(13,2,6);
  write_byte_lcd(14,2,7);
  delay(200);  // Pause ca. 2s
  printf_lcd(0,2,"SPI seriell Mode",1);
  blink_lcd(15,2,1); 
  while(1);
}
