----------------------------------------------------------------------------------
-- Engineer/Author: Cristian Bureriu (Verilog)
-- Engineer/Author: Thomas T. West (VHDL)
--
-- Create Date:     07:10:15 01/12/2009
-- Design Name:     Sinusoidal Waveform Computator
-- Module Name:     sin.vhd
-- Project Name:    SIN
-- Target Vendors:  Xilinx, Altera, Actel, Lattice
-- Target Devices:  FPGA, CPLD
-- Tool versions:
--
-- Description:     Sin & Cos Waveform Generator
-- Version:         1.0
-- Revision:        1.0-16
--
-- Dependencies:    n/a (stand-alone module)
--
-- License type:    For education purposes only. Portions of this code or it's entirety cannot be used for
--                  commercial purposes without prior written consent from the author
--
--                  Commercial release of this module is available. Contact the author for details.
--                  Highlights in the commercial release: 8-16-24-32 bit selectable width, signed or unsigned output format,
--                  single, bi-phase, tri-phase synchronous output with rigorous timing constraints
--
----------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity sin is
port (
       Clk_IN          : in std_logic;
       nRst_IN         : in std_logic;
       Sin_OUT         : out std_logic_vector(15 downto 0);
       Cos_OUT         : out std_logic_vector(15 downto 0);
       Sin_Sqr_OUT     : out std_logic;
       Cos_Sqr_OUT     : out std_logic
       );
end entity sin;

architecture rtl of sin is

       constant InitCos : signed (15 downto 0) := TO_SIGNED(32640, 16);
       signal iSin : signed(16 downto 0);
       signal iCos : signed(16 downto 0);
       signal Sin1 : signed(15 downto 0);
       signal Cos1 : signed(15 downto 0);

begin

process (nRst_IN, Clk_IN)
begin
       if (nRst_IN = '0') then
               Sin1 <= (others => '0');
               Cos1 <= InitCos;
       elsif (Clk_IN'event and Clk_IN = '1') then
               Sin1 <= iSin(15 downto 0);      -- truncate
               Cos1 <= iCos(15 downto 0); -- truncate
       end if;
end process;

--the computator
iSin <= Sin1 + (Cos1(15) & Cos1(15) & Cos1(15) & Cos1(15) & Cos1(15) &
Cos1(15) & Cos1(15) & Cos1(15) & Cos1(15 downto 7));
iCos <= Cos1 - (iSin(15) & iSin(15) & iSin(15) & iSin(15) & iSin(15) &
iSin(15) & iSin(15) & iSin(15) & iSin(15 downto 7));

--sign
Sin_Sqr_OUT <= not iSin(15);
Cos_Sqr_OUT <= not iCos(15);

-- Outputs can't be read in VHDL, so internal signal is used! (like wire in Verilog)
Sin_OUT <= std_logic_vector(iSin(15 downto 0));
Cos_OUT <= std_logic_vector(iCos(15 downto 0));

end architecture rtl;
