/* 
   Ansteuerung eines HD44780 kompatiblen LCD im 4-Bit-Interfacemodus
   http://www.mikrocontroller.net/articles/HD44780
   http://www.mikrocontroller.net/articles/AVR-GCC-Tutorial/LCD-Ansteuerung

   Die Pinbelegung ist ber defines in lcd.h einstellbar
*/
 
#include <avr/io.h>
#include "lcd.h"
#include <util/delay.h>
 
/////////////////////////////////////////////////////////////////////////////////
// Erzeugt einen Enable-Puls

static void lcd_enable( void ) {
    LCD_PORT_EN |= (1<<LCD_EN);     // Enable auf 1 setzen
    _delay_us( LCD_ENABLE_US );  // kurze Pause
    LCD_PORT_EN &= ~(1<<LCD_EN);    // Enable auf 0 setzen
}
 
////////////////////////////////////////////////////////////////////////////////
// Sendet eine 4-bit Ausgabeoperation an das LCD
static void lcd_out( uint8_t data ) {
    data &= 0xF0;                       // obere 4 Bit maskieren
    LCD_PORT &= ~(0xF0>>(4-LCD_DB));    // Maske lschen
    LCD_PORT |= (data>>(4-LCD_DB));     // Bits setzen
    lcd_enable();
}
 
////////////////////////////////////////////////////////////////////////////////
// Initialisierung: muss ganz am Anfang des Programms aufgerufen werden.
void lcd_init( void ) {
    // verwendete Pins auf Ausgang schalten

    LCD_DDR     |= (0x0F << LCD_DB);
    LCD_PORT_RS |= (1<<LCD_RS);
    LCD_PORT_EN |= (1<<LCD_EN);

 	_delay_ms(LCD_BOOTUP_MS);
	lcd_command(0x20);		// 4 bit data length, 1 line, instruction table 0
 	_delay_ms(LCD_SOFT_RESET_MS1);
	lcd_command(0x20);		// 4 bit data length, 1 line, instruction table 0

	lcd_command(0x29);		// 4 bit data length, 2 line, instruction table 1
	lcd_command(0x14);		// BS: 1/4, 2 line lcd
	lcd_command(0x55);		// booster off, contrast C5, set C4
	lcd_command(0x6D);		// set voltage follower and gain
	lcd_command(0x78);		// set contrast C3, C2, C1
	lcd_command(0x28);		// switch back to instruction table 0
	lcd_command(LCD_SET_DISPLAY | LCD_DISPLAY_ON | LCD_CURSOR_OFF | LCD_BLINKING_OFF);
	lcd_command(LCD_SET_ENTRY | LCD_ENTRY_INCREASE | LCD_ENTRY_NOSHIFT);
    lcd_clear();
    lcd_home();
}
  
////////////////////////////////////////////////////////////////////////////////
// Sendet ein Datenbyte an das LCD
void lcd_data( uint8_t data ) {
    LCD_PORT_RS |= (1<<LCD_RS);    // RS auf 1 setzen
 
    lcd_out( data );            // zuerst die oberen, 
    lcd_out( data<<4 );         // dann die unteren 4 Bit senden
    _delay_us( LCD_WRITEDATA_US );
}
 
////////////////////////////////////////////////////////////////////////////////
// Sendet einen Befehl an das LCD
void lcd_command( uint8_t data ) {
    LCD_PORT_RS &= ~(1<<LCD_RS);    // RS auf 0 setzen
 
    lcd_out( data );             	// zuerst die oberen, 
    lcd_out( data<<4);           	// dann die unteren 4 Bit senden
    _delay_us(LCD_COMMAND_US );
}
 
////////////////////////////////////////////////////////////////////////////////
// Sendet den Befehl zur Lschung des Displays
void lcd_clear( void ) {
    lcd_command( LCD_CLEAR_DISPLAY );
    _delay_ms( LCD_CLEAR_DISPLAY_MS );
}
 
////////////////////////////////////////////////////////////////////////////////
// Sendet den Befehl: Cursor Home
void lcd_home( void ) {
    lcd_command( LCD_CURSOR_HOME );
    _delay_ms( LCD_CURSOR_HOME_MS );
}
 
////////////////////////////////////////////////////////////////////////////////
// Setzt den Cursor in Zeile y (0..3) Spalte x (0..15)
 
void lcd_setcursor( uint8_t x, uint8_t y ) {
    uint8_t data;
 
 	data = LCD_SET_DDADR + x;
    switch (y) {
        case 0: data += LCD_DDADR_LINE1; break; // 1. Zeile
        case 1: data += LCD_DDADR_LINE2; break; // 2. Zeile
        case 2: data += LCD_DDADR_LINE3; break; // 3. Zeile
        case 3: data += LCD_DDADR_LINE4; break; // 4. Zeile 
        default: return; // fr den Fall einer falschen Zeile
    }
    lcd_command( data );
}
 
////////////////////////////////////////////////////////////////////////////////
// Schreibt einen String auf das LCD
 
void lcd_string( const char *data ) {
    while( *data != '\0' )
        lcd_data( *data++ );
}
 
////////////////////////////////////////////////////////////////////////////////
// Schreibt einen String auf das LCD
// String liegt direkt im Flash Speicher
 
void lcd_string_P( PGM_P data ) {
    uint8_t tmp;

    while( (tmp=pgm_read_byte(data++)) ) {        
        lcd_data( tmp );
    }
}


////////////////////////////////////////////////////////////////////////////////
// Schreibt ein Zeichen in den Character Generator RAM
// Daten liegen direkt im Flash-Speicher
 
void lcd_generatechar_P( uint8_t code, PGM_P data ) {
    uint8_t i;

    // Startposition des Zeichens einstellen
    lcd_command( LCD_SET_CGADR | (code<<3) );
    // Bitmuster bertragen
    for ( i=0; i<8; i++ ) {
        lcd_data( pgm_read_byte(data) );
        data++;
    }
}
