#include <avr/io.h>
#include <stdint.h>
#include "aladin.h"
#include "lcd.h"
#include "misc.h"
#include "main.h"
#include "globals.h"
#include "eeprom.h"

// print a 2 digit number to LCD 

void print_number(uint8_t number) {
	lcd_data('0'+ number/10);
	lcd_data('0'+ number%10);
}

void menu_fsm(void) {
	static menu_state_t last_menu = 0xFF;
	static uint8_t cnt;
	uint8_t lcd_refresh=0;
	
	if (last_menu != menu_state) {
		lcd_refresh = 1; 	
		get_key_press(ALL_KEYS);	// clear all key presses to prevent crazy menus
	}
	last_menu = menu_state;

	switch (menu_state) {
		case main_m:
			aladin_cmd = idle_cmd;
			if (lcd_refresh) {
				lcd_setcursor(0, 0);
				lcd_string_P(PSTR("MemoMouse    /32"));
				lcd_setcursor(11, 0);
				print_number(block_down);
				lcd_setcursor(0, 1);
				lcd_string_P(PSTR("           Clear"));
				if (block_down < MAX_BLOCK) {
					lcd_setcursor(6, 1);
					lcd_string_P(PSTR("Down"));
				}
				if (block_down > 0) {
					lcd_setcursor(0, 1);
					lcd_string_P(PSTR("Up"));
				}
			}
			if (get_key_press(1<<KEY1) && block_down > 0) {
				menu_state = upload_m;
				block_up = 0;
			}
			if (get_key_press(1<<KEY2) && block_down < MAX_BLOCK) {
				aladin_cmd = download_cmd;
				menu_state = download_m;
			}
			if (get_key_press (1<<KEY3)) {
				menu_state = clear_m;
				cnt = 0;			
			}
			break;

		case clear_m:
			if (lcd_refresh) {
				lcd_setcursor(0, 0);
				lcd_string_P(PSTR("Clear memory    "));
				lcd_setcursor(0, 1);
				lcd_string_P(PSTR("ESC Press3x Hold"));
			}
			if (get_key_press(1<<KEY1)) menu_state = main_m;
			if (get_key_press(1<<KEY2) && get_key_state(1<<KEY3)) {
				cnt++;
				if (cnt == 3) {
					eeprom_chip_erase();
					block_down = 0;
					menu_state = main_m;
				}
			}
			break;
		
		case upload_m:
			if (lcd_refresh) {
				lcd_setcursor(0, 0);
				lcd_string_P(PSTR("Upload #        "));
				lcd_setcursor(8, 0);
				print_number(block_up+1);		// display index 1-MAX_BLOCK for normal people
				lcd_setcursor(0, 1);
				lcd_string_P(PSTR("ESC  Upload Next"));
			}
			if (get_key_press(1<<KEY1)) menu_state = main_m;
			if (get_key_press(1<<KEY2)) {
				menu_state = upload_run_m;
				cnt = 0;
				id = 0;
				aladin_cmd = upload_cmd;
				lcd_setcursor(5,1);
			}
			if (get_key_press(1<<KEY3)) {
				last_menu = main_m;		// little trick to trigger LCD refresh
				if (block_up < (block_down-1) ) block_up++;
			}
			break;
			
		case upload_run_m:
			if (cnt < ((6*id)>>11)) {
			 	lcd_data('*'); 			// progress bar
				cnt++;
			}
			if (aladin_state == upload_done_s) {
				aladin_cmd = idle_cmd;
				menu_state = upload_m;
			}
			break;

		case download_m:
			if (lcd_refresh) {
				lcd_setcursor(0, 0);
				lcd_string_P(PSTR("Download #      "));
				lcd_setcursor(10, 0);
				print_number(block_down+1);		// display index 1-MAX_BLOCK for normal people
				lcd_setcursor(0, 1);
				lcd_string_P(PSTR("ESC Waiting     "));
			}
			if (get_key_press(1<<KEY1)) menu_state = main_m;
			if (aladin_connect) {
				menu_state = download_connected_m;
			}
			break;
		
		case download_connected_m:
			if (lcd_refresh) {
				lcd_setcursor(0, 1);
				lcd_string_P(PSTR("ESC  Connected  "));
			}
			if (get_key_press(1<<KEY1)) menu_state = main_m;
			if (!aladin_connect) {
				menu_state = download_m;
			}
			if (aladin_state == download_s) {
				menu_state = download_run_m;
				cnt = 0;
				lcd_setcursor(5,1);
			}
			break;

		case download_run_m:
			if (cnt < (id>>8)) {
				lcd_data('*'); 		// progress bar
				cnt++;
			}
			if (aladin_state == download_error_s) {
				menu_state = download_error_m;
			}
			if (aladin_state == download_success_s) {
				if (!check_crc(block_down)) {
					aladin_error = 3;
					menu_state = download_error_m;
				} else {
					block_down++;
					menu_state = download_success_m;
				}
			}
			break;
		
		case download_error_m:
			if (lcd_refresh) {
				lcd_setcursor(0, 1);
				switch (aladin_error) {
					case 1:	lcd_string_P(PSTR("ESC  TO Error!  ")); break;
					case 2:	lcd_string_P(PSTR("ESC  RX Error!  ")); 
						lcd_setcursor(0, 1);
						print_hex(debug>>8);
						print_hex(debug & 0xFF);
						break;
					case 3:	lcd_string_P(PSTR("ESC  CRC Error! ")); break;
				}

			}
			if (get_key_press(1<<KEY1)) {
				eeprom_page_erase(block_down<<11);
				menu_state = main_m;
			}
			break;
			
		case download_success_m:
			if (lcd_refresh) {
				lcd_setcursor(0, 1);
				lcd_string_P(PSTR("ESC  Success!   "));
			}
			if (get_key_press(1<<KEY1)) menu_state = main_m;
			break;
	}
}
