#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/atomic.h>
#include "misc.h"
#include "main.h"
#include "globals.h"
#include "eeprom.h"
#include "lcd.h"

// print a 2 digit hex number to LCD 

void print_hex(uint8_t number) {
	unsigned char x;

	x = number >> 4;
	if (x < 10) x += '0'; else x = x + 'A' - 10;
	lcd_data(x);
	x = number & 0xF;
	if (x < 10) x += '0'; else x = x + 'A' - 10;
	lcd_data(x);
}

uint8_t bitreverse(uint8_t x) {
	uint8_t tmp1=0;

	if (x & (1<<0)) tmp1 |= 1<<7;
	if (x & (1<<1)) tmp1 |= 1<<6;
	if (x & (1<<2)) tmp1 |= 1<<5;
	if (x & (1<<3)) tmp1 |= 1<<4;
	if (x & (1<<4)) tmp1 |= 1<<3;
	if (x & (1<<5)) tmp1 |= 1<<2;
	if (x & (1<<6)) tmp1 |= 1<<1;
	if (x & (1<<7)) tmp1 |= 1<<0;
	return tmp1;
}

///////////////////////////////////////////////////////////////////
//
// check if a key has been pressed. Each pressed key is reported
// only once
//
uint8_t get_key_press( uint8_t key_mask ) {
  	ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {             // read and clear atomic !
  		key_mask &= key_press;                          // read key(s)
  		key_press ^= key_mask;                          // clear key(s)
  	}
  	return key_mask;
}
 
///////////////////////////////////////////////////////////////////
//
// check if a key has been pressed long enough such that the
// key repeat functionality kicks in. After a small setup delay
// the key is reported being pressed in subsequent calls
// to this function. This simulates the user repeatedly
// pressing and releasing the key.
//
uint8_t get_key_rpt( uint8_t key_mask ) {
  	ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {           // read and clear atomic !
  		key_mask &= key_rpt;                      // read key(s)
  		key_rpt  ^= key_mask;                     // clear key(s)
  	}
  	return key_mask;
}

///////////////////////////////////////////////////////////////////
//
// check if a key is pressed right now
//
uint8_t get_key_state( uint8_t key_mask ) {
  	key_mask &= key_state;
  	return key_mask;
}

///////////////////////////////////////////////////////////////////
//
uint8_t get_key_short( uint8_t key_mask ) {
	uint8_t tmp;

  	ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {      // read key state and key press atomic !
  		tmp = get_key_press( ~key_state & key_mask );
	}
	return tmp;
}
 
///////////////////////////////////////////////////////////////////
//
uint8_t get_key_long( uint8_t key_mask ) {
  	return get_key_press( get_key_rpt( key_mask ));
}

// SPI transfer

uint8_t spi_io(uint8_t data) {

	SPDR = data;
	while(!(SPSR & (1<<SPIF)));
	return SPDR;
}

// find first free 2048 byte block
// read first byte of block
// occupied blocks read always 0x00 (Aladin data structure)
// free blocks read always 0xFF
 
uint16_t find_free_block(void) {
	uint16_t i;

	for (i=0; i<MAX_BLOCK; i++) {
		eeprom_start_read(i<<11);
		if (eeprom_read() == 0xFF) break;
		CS_HIGH;	// end page read
	}
	CS_HIGH;
	return i;
}

uint16_t check_crc(int block) {
	uint16_t i, crc, crc_data;

	eeprom_start_read(block<<11);
	eeprom_read(); // skip 0x00
	crc = 0x1FE;
	for (i=0; i<2044; i++) {
		crc += bitreverse(eeprom_read());
	}
	crc_data  = bitreverse(eeprom_read());
	crc_data |= bitreverse(eeprom_read())<<8;
	CS_HIGH;
	
	// TEST
	/*
	lcd_setcursor(0,0);
	print_hex(crc>>8);
	print_hex(crc & 0xFF);
	lcd_data(' ');
	print_hex(crc_data>>8);
	print_hex(crc_data & 0XFF);
	*/

	return (crc == crc_data);
}
