
#include "stdbool.h"
#include "stdlib.h"
#include "DCF77DecHit.h"
#include "math.h"

// "OPTS" : damit beim Debuggen das Programm noch im Ram statt Flash passt
#define OPTS __attribute__((optimize("-Os")))

/*
static const double HVProb[] =
{
  // Wahrscheinlichkeit mit der sich ein HitVal rein zufllig
  // ergibt: Per rand() 512 Bits definieren und anschlieend die
  // gesetzte Bitanzahl zhlen. Fr jede Bitanzahl ergibt sich
  // ein Wahrscheinlichkeitswert HVProb[abs(Bitanzahl-256)].
  3.524464e-002,  //     0  0.035244635485838741
  3.510750e-002,  //     1  0.035107496826360766
  3.469927e-002,  //     2  0.034699270119077502
  3.402940e-002,  //     3  0.034029400039558629
  3.311322e-002,  //     4  0.033113223884647436
  3.197139e-002,  //     5  0.031971388578280281
  3.062908e-002,  //     6  0.030629078370795235
  2.911509e-002,  //     7  0.029115093508360487
  2.746083e-002,  //     8  0.027460826831749097
  2.569919e-002,  //     9  0.025699188884051986
  2.386353e-002,  //    10  0.023863532535191131
  2.198663e-002,  //    11  0.021986625481861491
  2.009971e-002,  //    12  0.020099713593492781
  1.823171e-002,  //    13  0.018231710471420959
  1.640854e-002,  //    14  0.016408539424278865
  1.465264e-002,  //    15  0.014652644061533156
  1.298267e-002,  //    16  0.012982673598637832
  1.141334e-002,  //    17  0.011413339427373919
  9.955431e-003,  //    18  0.009955431106359002
  8.615973e-003,  //    19  0.008615973102957973
  7.398499e-003,  //    20  0.007398498642757390
  6.303414e-003,  //    21  0.006303414006103769
  5.328426e-003,  //    22  0.005328425508756783
  4.469002e-003,  //    23  0.004469002039602463
  3.718848e-003,  //    24  0.003718848125812050
  3.070366e-003,  //    25  0.003070365712414219
  2.515087e-003,  //    26  0.002515086806977605
  2.044063e-003,  //    27  0.002044063482702647
  1.648206e-003,  //    28  0.001648206118094740
  1.318565e-003,  //    29  0.001318564894475792
  1.046553e-003,  //    30  0.001046553255405611
  8.241151e-004,  //    31  0.000824115107044140
  6.438399e-004,  //    32  0.000643839927378234
  4.990316e-004,  //    33  0.000499031639213579
  3.837381e-004,  //    34  0.000383738122567684
  2.927487e-004,  //    35  0.000292748670824831
  2.215666e-004,  //    36  0.000221566631000985
  1.663640e-004,  //    37  0.000166364023277190
  1.239242e-004,  //    38  0.000123924221420764
  9.157790e-005,  //    39  0.000091577899219412
  6.713650e-005,  //    40  0.000067136500441258
  4.882655e-005,  //    41  0.000048826545775460
  3.522721e-005,  //    42  0.000035227205844711
  2.521278e-005,  //    43  0.000025212782778489
  1.790108e-005,  //    44  0.000017901075772727
  1.260807e-005,  //    45  0.000012608066657203
  8.808947e-006,  //    46  0.000008808947234006
  6.105211e-006,  //    47  0.000006105210954262
  4.197333e-006,  //    48  0.000004197332531055
  2.862443e-006,  //    49  0.000002862443168719
  1.936359e-006,  //    50  0.000001936358614134
  1.299316e-006,  //    51  0.000001299315552155
  8.648042e-007,  //    52  0.000000864804182441
  5.709387e-007,  //    53  0.000000570938683553
  3.738728e-007,  //    54  0.000000373872750843
  2.428370e-007,  //    55  0.000000242836963570
  1.564430e-007,  //    56  0.000000156443043838
  9.996361e-008,  //    57  0.000000099963606286
  6.335273e-008,  //    58  0.000000063352731373
  3.982172e-008,  //    59  0.000000039821716863
  2.482556e-008,  //    60  0.000000024825563994
  1.534956e-008,  //    61  0.000000015349560072
  9.412466e-009,  //    62  0.000000009412466082
  5.724196e-009,  //    63  0.000000005724195674
  3.452406e-009,  //    64  0.000000003452405516
  2.064990e-009,  //    65  0.000000002064990215
  1.224886e-009,  //    66  0.000000001224885500
  7.205209e-010,  //    67  0.000000000720520882
  4.203038e-010,  //    68  0.000000000420303848
  2.431296e-010,  //    69  0.000000000243129611
  1.394639e-010,  //    70  0.000000000139463918
  7.932810e-011,  //    71  0.000000000079328100
  4.474298e-011,  //    72  0.000000000044742983
  2.502343e-011,  //    73  0.000000000025023431
  1.387663e-011,  //    74  0.000000000013876630
  7.630050e-012,  //    75  0.000000000007630050
  4.159756e-012,  //    76  0.000000000004159756
  2.248517e-012,  //    77  0.000000000002248517
  1.205044e-012,  //    78  0.000000000001205044
  6.402918e-013,  //    79  0.000000000000640292
  3.372966e-013,  //    80  0.000000000000337297
  1.761549e-013,  //    81  0.000000000000176155
  9.120445e-014,  //    82  0.000000000000091204
  4.681290e-014,  //    83  0.000000000000046813
  2.381951e-014,  //    84  0.000000000000023820
  1.201453e-014,  //    85  0.000000000000012015
  6.007266e-015,  //    86  0.000000000000006007
  2.977362e-015,  //    87  0.000000000000002977
  1.462716e-015,  //    88  0.000000000000001463
  7.122789e-016,  //    89  0.000000000000000712
  3.437878e-016,  //    90  0.000000000000000344
  1.644633e-016,  //    91  0.000000000000000164
  7.797830e-017,  //    92  0.000000000000000078
  3.664310e-017,  //    93  0.000000000000000037
  1.706521e-017,  //    94  0.000000000000000017
  7.876253e-018,  //    95  0.000000000000000008
  3.602491e-018,  //    96  0.000000000000000004
  1.632857e-018,  //    97  0.000000000000000002
  7.334019e-019,  //    98  0.000000000000000001
  3.264155e-019,  //    99  0.000000000000000000
  1.439529e-019,  //   100  0.000000000000000000
  6.290379e-020,  //   101  0.000000000000000000
  2.723488e-020,  //   102  0.000000000000000000
  1.168293e-020,  //   103  0.000000000000000000
  4.965245e-021,  //   104  0.000000000000000000
  2.090629e-021,  //   105  0.000000000000000000
  8.720581e-022,  //   106  0.000000000000000000
  3.603546e-022,  //   107  0.000000000000000000
  1.475078e-022,  //   108  0.000000000000000000
  5.981137e-023,  //   109  0.000000000000000000
  2.402260e-023,  //   110  0.000000000000000000
  9.556675e-024,  //   111  0.000000000000000000
  3.765538e-024,  //   112  0.000000000000000000
  1.469478e-024,  //   113  0.000000000000000000
  5.679335e-025,  //   114  0.000000000000000000
  2.173761e-025,  //   115  0.000000000000000000
  8.239257e-026,  //   116  0.000000000000000000
  3.092483e-026,  //   117  0.000000000000000000
  1.149345e-026,  //   118  0.000000000000000000
  4.229590e-027,  //   119  0.000000000000000000
  1.541101e-027,  //   120  0.000000000000000000
  5.559408e-028,  //   121  0.000000000000000000
  1.985503e-028,  //   122  0.000000000000000000
  7.019984e-029,  //   123  0.000000000000000000
  2.456994e-029,  //   124  0.000000000000000000
  8.512421e-030,  //   125  0.000000000000000000
  2.919181e-030,  //   126  0.000000000000000000
  9.908448e-031,  //   127  0.000000000000000000
  3.328619e-031,  //   128  0.000000000000000000
  1.106658e-031,  //   129  0.000000000000000000
  3.641076e-032,  //   130  0.000000000000000000
  1.185467e-032,  //   131  0.000000000000000000
  3.819158e-033,  //   132  0.000000000000000000
  1.217418e-033,  //   133  0.000000000000000000
  3.839549e-034,  //   134  0.000000000000000000
  1.198018e-034,  //   135  0.000000000000000000
  3.697963e-035,  //   136  0.000000000000000000
  1.129149e-035,  //   137  0.000000000000000000
  3.410374e-036,  //   138  0.000000000000000000
  1.018795e-036,  //   139  0.000000000000000000
  3.010077e-037,  //   140  0.000000000000000000
  8.795188e-038,  //   141  0.000000000000000000
  2.541323e-038,  //   142  0.000000000000000000
  7.260923e-039,  //   143  0.000000000000000000
  2.051211e-039,  //   144  0.000000000000000000
  5.729067e-040,  //   145  0.000000000000000000
  1.581907e-040,  //   146  0.000000000000000000
  4.317859e-041,  //   147  0.000000000000000000
  1.164967e-041,  //   148  0.000000000000000000
  3.106579e-042,  //   149  0.000000000000000000
  8.187289e-043,  //   150  0.000000000000000000
  2.132316e-043,  //   151  0.000000000000000000
  5.487578e-044,  //   152  0.000000000000000000
  1.395374e-044,  //   153  0.000000000000000000
  3.505453e-045,  //   154  0.000000000000000000
  8.699663e-046,  //   155  0.000000000000000000
  2.132684e-046,  //   156  0.000000000000000000
  5.163885e-047,  //   157  0.000000000000000000
  1.234842e-047,  //   158  0.000000000000000000
  2.916013e-048,  //   159  0.000000000000000000
  6.799356e-049,  //   160  0.000000000000000000
  1.565319e-049,  //   161  0.000000000000000000
  3.557544e-050,  //   162  0.000000000000000000
  7.981125e-051,  //   163  0.000000000000000000
  1.767249e-051,  //   164  0.000000000000000000
  3.861922e-052,  //   165  0.000000000000000000
  8.327842e-053,  //   166  0.000000000000000000
  1.771881e-053,  //   167  0.000000000000000000
  3.719279e-054,  //   168  0.000000000000000000
  7.701095e-055,  //   169  0.000000000000000000
  1.572759e-055,  //   170  0.000000000000000000
  3.167617e-056,  //   171  0.000000000000000000
  6.290829e-057,  //   172  0.000000000000000000
  1.231771e-057,  //   173  0.000000000000000000
  2.377604e-058,  //   174  0.000000000000000000
  4.523516e-059,  //   175  0.000000000000000000
  8.481592e-060,  //   176  0.000000000000000000
  1.567038e-060,  //   177  0.000000000000000000
  2.852442e-061,  //   178  0.000000000000000000
  5.114723e-062,  //   179  0.000000000000000000
  9.032883e-063,  //   180  0.000000000000000000
  1.570936e-063,  //   181  0.000000000000000000
  2.689959e-064,  //   182  0.000000000000000000
  4.534328e-065,  //   183  0.000000000000000000
  7.522862e-066,  //   184  0.000000000000000000
  1.228222e-066,  //   185  0.000000000000000000
  1.972936e-067,  //   186  0.000000000000000000
  3.117507e-068,  //   187  0.000000000000000000
  4.844774e-069,  //   188  0.000000000000000000
  7.403250e-070,  //   189  0.000000000000000000
  1.112147e-070,  //   190  0.000000000000000000
  1.642097e-071,  //   191  0.000000000000000000
  2.382507e-072,  //   192  0.000000000000000000
  3.396000e-073,  //   193  0.000000000000000000
  4.754401e-074,  //   194  0.000000000000000000
  6.535983e-075,  //   195  0.000000000000000000
  8.820685e-076,  //   196  0.000000000000000000
  1.168303e-076,  //   197  0.000000000000000000
  1.518279e-077,  //   198  0.000000000000000000
  1.935388e-078,  //   199  0.000000000000000000
  2.419235e-079,  //   200  0.000000000000000000
  2.964490e-080,  //   201  0.000000000000000000
  3.559977e-081,  //   202  0.000000000000000000
  4.188208e-082,  //   203  0.000000000000000000
  4.825544e-083,  //   204  0.000000000000000000
  5.443130e-084,  //   205  0.000000000000000000
  6.008650e-085,  //   206  0.000000000000000000
  6.488823e-086,  //   207  0.000000000000000000
  6.852421e-087,  //   208  0.000000000000000000
  7.073467e-088,  //   209  0.000000000000000000
  7.134183e-089,  //   210  0.000000000000000000
  7.027247e-090,  //   211  0.000000000000000000
  6.756968e-091,  //   212  0.000000000000000000
  6.339160e-092,  //   213  0.000000000000000000
  5.799657e-093,  //   214  0.000000000000000000
  5.171668e-094,  //   215  0.000000000000000000
  4.492339e-095,  //   216  0.000000000000000000
  3.799018e-096,  //   217  0.000000000000000000
  3.125775e-097,  //   218  0.000000000000000000
  2.500620e-098,  //   219  0.000000000000000000
  1.943759e-099,  //   220  0.000000000000000000
  1.466988e-100,  //   221  0.000000000000000000
  1.074154e-101,  //   222  0.000000000000000000
  7.624477e-103,  //   223  0.000000000000000000
  5.241828e-104,  //   224  0.000000000000000000
  3.487287e-105,  //   225  0.000000000000000000
  2.242861e-106,  //   226  0.000000000000000000
  1.393081e-107,  //   227  0.000000000000000000
  8.346975e-109,  //   228  0.000000000000000000
  4.818872e-110,  //   229  0.000000000000000000
  2.677151e-111,  //   230  0.000000000000000000
  1.429280e-112,  //   231  0.000000000000000000
  7.322130e-114,  //   232  0.000000000000000000
  3.593684e-115,  //   233  0.000000000000000000
  1.686831e-116,  //   234  0.000000000000000000
  7.558103e-118,  //   235  0.000000000000000000
  3.226019e-119,  //   236  0.000000000000000000
  1.308730e-120,  //   237  0.000000000000000000
  5.033577e-122,  //   238  0.000000000000000000
  1.830392e-123,  //   239  0.000000000000000000
  6.273520e-125,  //   240  0.000000000000000000
  2.019644e-126,  //   241  0.000000000000000000
  6.083265e-128,  //   242  0.000000000000000000
  1.706728e-129,  //   243  0.000000000000000000
  4.437492e-131,  //   244  0.000000000000000000
  1.062872e-132,  //   245  0.000000000000000000
  2.329003e-134,  //   246  0.000000000000000000
  4.630225e-136,  //   247  0.000000000000000000
  8.268259e-138,  //   248  0.000000000000000000
  1.309823e-139,  //   249  0.000000000000000000
  1.812008e-141,  //   250  0.000000000000000000
  2.144389e-143,  //   251  0.000000000000000000
  2.110619e-145,  //   252  0.000000000000000000
  1.658640e-147,  //   253  0.000000000000000000
  9.756703e-150,  //   254  0.000000000000000000
  3.818670e-152,  //   255  0.000000000000000000
  7.458341e-155,  //   256  0.000000000000000000
};
*/

//################################################### LogGaussHVProb
static int32_t OPTS LogGaussHVProb(int x)
{
  // Rckgabe der Nherung von
  //   -log(HVProb[abs(x)])*DCFDEC_LogScale + Const
  //
  // p(x)  = Nherung von HVProb[]
  //       = 1/sqrt(2*pi*Sigma*Sigma)*exp(-x*x/(2*Sigma*Sigma))
  //         mit Sigma = sqrt(512/4)
  //         Abweichung:
  //               x  | p(x)/HVProb[x]
  //           -------+---------------
  //               0  | 1.00
  //              10  | 1.00
  //              20  | 1.00
  //              30  | 1.00
  //              40  | 0.99
  //              50  | 0.96
  //              60  | 0.90
  //              70  | 0.81
  //              80  | 0.69
  //              90  | 0.54
  //             100  | 0.38
  //             110  | 0.23
  //             120  | 0.12
  // lp(x) = neg. Logarithmus von p(x)
  //       = -(log(1/sqrt(2*pi*Sigma*Sigma))-x*x/(2*Sigma*Sigma))
  //       = 3.345 + x*x/256
  return x*x*DCFDEC_LogScale/256; // ("Const" = -3.345*DCFDEC_LogScale)
}

//################################################### LogAdd
static int32_t OPTS LogAdd(int32_t A, int32_t B)
{
  // Addition "a + b = c" logarithmisch gerechnet - Nherung!
  //
  //   C = -log(exp(-A)+exp(-B))
  //
  // - Parameter
  //   A // = log(a)
  //   B // = log(b)
  //   C // = log(c) = Ergebnis = log(a+b)
  // - ...
  //   C = log(exp(A)+exp(B))
  //     = log(exp(A)  * (1+exp(B)/exp(A))
  //     = log(exp(A)) + log(1+exp(B)/exp(A))
  //     =      A      + log(1+exp( log(exp(B)  /     exp(A))) )
  //     =      A      + log(1+exp( log(exp(B)) - log(exp(A))) )
  //     =      A      + log(1+exp(      B      -      A       )
  //     =      A      + TabellenWert[B-A]
  enum
  {
    TabCnt   = 12,
    TabScale = 256
  };
  // Tab[i] = (uint16_t)(log(1+exp(-1.0*i*TabScale/DCFDEC_LogScale))*DCFDEC_LogScale);
  static const uint16_t Tab[12] = { 277, 169, 98, 54, 29, 15, 8, 4, 2, 1, 0, 0 };
  if(A>B)
  {
    int tmp = A;
    A = B;
    B = tmp;
  }
  int32_t dBA = B-A;
  int32_t p = dBA/TabScale;
  if(p>=TabCnt-2) p=TabCnt-2;
  int32_t c0 = Tab[p+0];
  int32_t c1 = Tab[p+1];
  int32_t dC = c0+(c1-c0)*(dBA-p*TabScale)/TabScale;
  if(dC<0) dC=0;
  // return -log(exp(-A/DCFDEC_LogScale)+exp(-B/DCFDEC_LogScale))*DCFDEC_LogScale;
  return A-dC;
}

//################################################### LogSub
static int32_t OPTS LogSub(int32_t A, int32_t B)
{
  // Subtraktion "a - b = c" logarithmisch gerechnet - spezielle grobe Nherung!
  //
  //   C = -log(exp(-A)-exp(-B))
  //
  //   (mit b<a bzw. A<B)
  //
  enum
  {
    TabCnt   = 12,
    TabScale = 128
  };
  // Tab[i] = (uint16_t)(-log(1-exp(-1.0*(i>0 ? i*TabScale: 40)/DCFDEC_LogScale))*DCFDEC_LogScale);
  static const uint16_t Tab[12] = { 940, 518, 299, 193, 130, 90, 63, 45, 32, 23, 16, 12, };
  int32_t dBA = B-A;
  int32_t p = dBA/TabScale;
  if(p>=TabCnt-2) p=TabCnt-2;
  int32_t c0 = Tab[p+0];
  int32_t c1 = Tab[p+1];
  int32_t dC = c0+(c1-c0)*(dBA-p*TabScale)/TabScale;
  if(dC<0) dC=0;
  int32_t C = A + dC;
  // return -log(exp(-1.0*A/DCFDEC_LogScale)-exp(-1.0*B/DCFDEC_LogScale))*DCFDEC_LogScale;
  return C;
}

//############################################################## LogConv
static inline int32_t __attribute__((always_inline)) LogConv(int32_t v)
{
  // Wahrscheinlichkeitswert uminterpretieren
  // - vorher
  //   -unendlich : 100% falsch
  //   0          : Zufall
  //   +unendlich : 100% korrekt
  // - nachher
  //   0         = -log(1.0)*DCFDEC_LogScale : 100% falsch
  //   ...       = -log(0.5)*DCFDEC_LogScale : Zufall
  //   unendlich = -log(0.0)*DCFDEC_LogScale : 100% korrekt
  return v-LogAdd(0, v);
}

//############################################################## DCF77Dec_DecodeBits
static int32_t OPTS DCF77Dec_DecodeBits(int32_t  nItem,  // quasi Medauer in Minuten
                                        int32_t  nBit,   // Anzahl der Bits der Gruppe
                                        int8_t*  DCFArr, // DCF-Bitwahrscheinlichkeiten
                                        int32_t  DCFOfs, // Position des ersten Bits
                                        int32_t* ResErr) // Ergebnisfehlerwahrscheinlichkeit
{
  // Die Bits einer prfbitgeschtzten Gruppe von DCFBit aus mehreren
  // Messwerten bestimmen. Fr alle Messwerte mssen die DCF-Daten gleich
  // bleiben, da sonst Ergebnis und ResErr undefiniert sind!
  uint32_t Bits = 0;
  int CheckBit = 0;
  uint16_t* ErrArr = (uint16_t*)alloca(nBit*sizeof(uint16_t));
  int iBitEx = 0;
  for(int iBit=0; iBit<nBit; iBit++)
  {
    int ErrSum = 0;
    int p = DCFOfs+iBit;
    for(int iItem=0; iItem<nItem; iItem++)
    {
      if(p>=DCFDEC_MaxSeconds) p-=DCFDEC_MaxSeconds;
      ErrSum += DCFArr[p];
      p += 60;
    }
    if(ErrSum>0) CheckBit^=1;
    if(ErrSum>0) Bits|=1<<iBit;
    int ErrConv = LogConv(abs(ErrSum*128));
    if(ErrConv>0xffff) ErrConv=0xffff;
    ErrArr[iBit] = ErrConv;
    if(ErrConv<ErrArr[iBitEx]) iBitEx=iBit;
  }
  // # Doppelfehlerwahrscheinlichkeit berechnen
  // Der Fehler wird im logarithmischen Bereich berechnet. Linear und mit
  // Fliesskommadatentyp she es so aus:
  //
  //   // solange Gesamtfehlerwahrscheinlichkeit unter 0.01 ist folgende
  //   // Nherung recht genau
  //   double ErrSum = 0;
  //   for(int iItem=0; iItem<nItem; iItem++)
  //   {
  //     ErrSum += ErrArr[iItem];
  //   }
  //   double DblErr = 0;
  //   for(int iItem=0; iItem<nItem; iItem++)
  //   {
  //     DblErr += ErrArr[iItem]*(ErrSum-ErrArr[iItem])/2;
  //   }
  //

  // erst Fehlersumme "ErrSumEx" ohne kleinstem Wert bestimmen
  // (besondere Behandlung nur wegen Rechengenauigkeit)
  int ErrSumEx = 0xffff; // entspricht min. Fehler
  {
    for(int iBit=0; iBit<nBit; iBit++)
    {
      if(iBit!=iBitEx) ErrSumEx=LogAdd(ErrSumEx, ErrArr[iBit]);
    }
  }
  // dann Fehlersumme "ErrSum" mit allen Werten
  int ErrSum = LogAdd(ErrSumEx, ErrArr[iBitEx]);
  // nun Doppelfehler berechnen
  int DblErr = 0xffff;
  {
    for(int iBit=0; iBit<nBit; iBit++)
    {
      int v = iBit==iBitEx ? ErrSumEx : LogSub(ErrSum, ErrArr[iBit]);
      DblErr = LogAdd(DblErr, ErrArr[iBit]+v);
    }
  }
  DblErr += (int32_t)(log(2)*DCFDEC_LogScale); // das lineare '/2'
  if(DblErr<0) DblErr=0;
  if(CheckBit!=0) DblErr=0;
  *ResErr = DblErr;
  return Bits;
}

//############################################################## DCF77Dec_DecodeValue
static int32_t OPTS DCF77Dec_DecodeValue(int32_t  nItem,        // quasi Medauer in Minuten
                                         int32_t  ValCnt,       // Anzahl der mglichen Werte
                                         const uint8_t* ValArr, // Kodierung eines jeden Werts
                                         int32_t  nBit,         // Anzahl der Bits der Kodierung
                                         int8_t*  DCFArr,       // DCF-Bitwahrscheinlichkeiten
                                         int32_t  DCFOfs,       // Position des ersten Bits
                                         int32_t  IncMask,      // je Minute ein Bit ob Val-Inkrement
                                         int32_t  IncVal,       // nderungswert beim Val-Inkrement
                                         int32_t* ResErr)       // Ergebnisfehlerwahrscheinlichkeit
{
  // Den Wert einer Gruppe von DCFBit aus mehreren Messwerten bestimmen.
  // Der Wert darf sich von Gruppe zu Gruppe ndern. Sehr rechenintensiv.
  // Dauert bei ValCnt==60 und nItem==10 rund 1.000.000 STM32-Takte.

  int Val0 = 0;
  *ResErr = 0xffff; // Init Fehler auf (praktisch) 0
  // Durchgang 1: den besten Wert suchen (-> Val0)
  // Durchgang 2: 'Abstand' des besten zum zweitbesten bestimmen
  for(int iLoop=0; iLoop<2; iLoop++)
  {
    for(int Val1=0; Val1<ValCnt; Val1++)
    {
      if(Val1!=Val0)
      {
        int ErrSum = 0;
        int p0 = DCFOfs;
        int i0 = Val0;
        int i1 = Val1;
        for(int iItem=0; iItem<nItem; iItem++)
        {
          int Mask = ValArr[i0]^ValArr[i1]; // nur verschiedene Bits bercksichtigen
          for(int iBit=0; iBit<nBit; iBit++)
          {
            if((Mask&(1<<iBit))!=0)
            {
              int p = p0+iBit;
              if(p>=DCFDEC_MaxSeconds) p-=DCFDEC_MaxSeconds;
              int DCFBitProb = DCFArr[p];
              if((ValArr[i0]&(1<<iBit))==0) DCFBitProb=-DCFBitProb;
              ErrSum += DCFBitProb;
            }
          }
          p0 += 60;
          if(p0>=DCFDEC_MaxSeconds) p0-=DCFDEC_MaxSeconds;
          if((IncMask&(1<<iItem))!=0)
          {
            i0 += IncVal;
            i1 += IncVal;
          }
        }
        if(iLoop==0)
        {
          if(ErrSum<0) Val0=Val1; // Val1 ist besser als Val0
        }
        if(iLoop!=0 && ErrSum<*ResErr)
        {
          *ResErr = ErrSum; // Abstand Val1 zum besten (Val0)
        }
      }
    }
  }
  *ResErr = *ResErr<=0 ? 0 : LogConv(*ResErr*128);
  return Val0;
}

//############################################################## AvrDCFProb
static int32_t OPTS AvrDCFProb(int32_t  nItem,
                               int32_t  nBit,
                               int8_t*  DCFArr,
                               int32_t  DCFOfs)
{
  // Rckgabe: Mittelwert*nItem (das ist sehr sehr grob die Hlfte vom
  // durchschnittlichen Doppelfehlerwert)
  int MinV = 0;
  for(int iLoop=0; ; iLoop++)
  {
    int SumV = 0;
    int SumN = 0;
    for(int iBit=0; iBit<nBit; iBit++)
    {
      int p = DCFOfs+iBit;
      for(int iItem=0; iItem<nItem; iItem++)
      {
        if(p>=DCFDEC_MaxSeconds) p-=DCFDEC_MaxSeconds;
        int v = abs(DCFArr[p]);
        if(v>=MinV)
        {
          SumV += v;
          SumN++;
        }
        p += 60;
      }
    }
    if(iLoop==1) return SumV*128*nItem/SumN;
    MinV = SumV/(SumN*2); // -> Dropouts fr Mittelwert ignorieren
  }
}

//############################################################## OutlierFilter
static void OPTS OutlierFilter(int32_t Avr, int Scale, int32_t* Error)
{
  Avr = Scale*Avr/100;
  if(Avr>0xffff) Avr=0xffff;
  if(Avr>(*Error))
  {
    uint32_t DblErr = *Error; // unsigned - sonst berlauf
    DblErr = DblErr*DblErr/Avr;
    *Error = DblErr;
  }
}

//############################################################## DCF77Dec_Seek
static void OPTS DCF77Dec_Seek(TDCF77DecHitObj* s,
                               int32_t  DCFBit0,
                               int32_t  DCFBit1,
                               int32_t  iStartBit,
                               int32_t* ResOfs,
                               int32_t* ResCnt)
{
  // Bestimmt wo sich im DCFArr[] bestimmte DCFBits befinden und
  // wieviele Minuten zugreifbar sind.
  // - ResOfs : Position des ersten DCFBit 'DCFBit0' in s->DCFArr[]
  // - ResCnt : Anzahl vollstndiger (DCFBit1-DCFBit0+1) Minuten-Elemente
  int p = DCFBit0+iStartBit;
  if(p>=60) p-=60;
  while(s->DCFOfs>p) p+=60;
  int nSec = s->DCFCnt-(p-s->DCFOfs);
  if(p>=DCFDEC_MaxSeconds) p-=DCFDEC_MaxSeconds;
  int n = 0;
  while(nSec>=DCFBit1-DCFBit0+1)
  {
    n++;
    nSec -= 60;
  }
  *ResOfs = p;
  *ResCnt = n;
}

//############################################################## DCF77DecHit_IncTime
static void OPTS DCF77DecHit_IncTime(TDCF77DecHit* o)
{
  // Time_* und Datum_* eine Sekunde weiterstellen
  o->Time_Second++;
  if(o->Time_Second<60) return;
  o->Time_Second = 0;
  o->Time_Minute++;
  if(o->Time_Minute<60) return;
  o->Time_Minute = 0;
  // Achtung: evtl. Zeitzonenumstellung (inkl. Zone_MEZS) wird hier ignoriert
  o->Time_Hour++;
  if(o->Time_Hour<24) return;
  o->Time_Hour = 0;
  o->Date_WeekDay++;
  if(o->Date_WeekDay>=8) o->Date_WeekDay=1;
  o->Date_Day++;
  static const uint8_t DayArr[12] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
  int n = o->Date_Month>=1 && o->Date_Month<=12 ? DayArr[o->Date_Month-1] : 31;
  if((o->Date_Year&3)==0 && n==28) n=29;
  if(o->Date_Day<=n) return;
  o->Date_Day = 1;
  o->Date_Month++;
  if(o->Date_Month<=12) return;
  o->Date_Month = 1;
  o->Date_Year++;
  if(o->Date_Year<100) return;
  o->Date_Year = 0;  // :-)
}

//############################################################## DCF77Dec_Calc_dbErr
static uint8_t OPTS DCF77Dec_Calc_dbErr(int Error1, int Error2)
{
  int Err = LogAdd(Error1, Error2);
  if(Err<0) Err=0;
  int dbErr = Err*((int32_t)(65536*20/log(10)/DCFDEC_LogScale))/65536;
  if(dbErr>255) dbErr=255;
  return dbErr;
}

//############################################################## DCF77DecHit_ProcessVal
void OPTS DCF77DecHit_ProcessVal(TDCF77DecHit* o,
                                 int32_t DCF77_BitCnt,
                                 int32_t DCF77_HitVal)
{
  // (max. 70ms@8MHz)
  enum
  {
    DCFBit_Start0 = 59,
    DCFBit_Start1 = 70,
    DCFBit_Time0  = 16, // erstes  Zeit-Bit (Zonenwechselankndigungsbit)
    DCFBit_Time1  = 35, // letztes Zeit-Bit (Prfbit)
    DCFBit_Date0  = 36, // erstes  Datum-Bit
    DCFBit_Date1  = 58, // letztes Datum-Bit (Prfbit)

    // OutlierFilt_*-Parameter
    // Hohe Werte schwchen Dropout-Fehler (db-Werte) mehr, es erhht jedoch
    // die Meddauer. Die Parameter (50, 120, 120, 170, 90) wurden anhand
    // Simulationsergebnisse festgelegt. Die Werte sind so gewhlt, dass
    // sich unter Normalbedingungen die Meddauer nur wenig erhht.
    OutlierFilt_Start  =  50,
    OutlierFilt_Minute = 120,
    OutlierFilt_Hour   = 120,
    OutlierFilt_Zone   = 170,
    OutlierFilt_Date   =  90
  };

  TDCF77DecHitObj* s = &o->DCF77DecHitObj;

  // neues Bit merken
  {
    // Reset?
    if(DCF77_BitCnt!=s->DCF77_BitCnt+1) DCF77DecHit_Init(o);
    // bei vergessene DCF77Dec_Init() DCFArr[]-Speicherfehler verhindern
    if(s->DCFOfs<0 || s->DCFOfs>=DCFDEC_MaxSeconds) DCF77DecHit_Init(o);
    if(s->DCFCnt<0 || s->DCFCnt> DCFDEC_MaxSeconds) DCF77DecHit_Init(o);
    if(s->DCFCnt<DCFDEC_MaxSeconds) s->DCFCnt++; else s->DCFOfs++;
    if(s->DCFOfs>=DCFDEC_MaxSeconds) s->DCFOfs=0;
    int p = s->DCFOfs+s->DCFCnt-1;
    if(p>=DCFDEC_MaxSeconds) p-=DCFDEC_MaxSeconds;
    s->DCF77_BitCnt = DCF77_BitCnt;

    // mittleren Trefferwert 'AvrHitVal' bestimmen
    int AvrHitVal = 0;
    {
      // - Tiefpass fr mittl. HitVal langsam bis auf Tau=32s erhhen
      int nShift = 0;
      while(nShift<5 && s->DCFCnt>(1<<nShift)) nShift++;
      // - mittleren Absolutwert vom HitVal berechnen
      s->AbsAvrVal += (128*abs(DCF77_HitVal)-s->AbsAvrVal)>>nShift;
      // - mittleren Absolutwert zum echten mittleren HitVal umrechnen
      //   Fr die Berechnung vom echten mittleren HitVal bentigt man den
      //   korrekten DCF-Bitwert. Mittelt man stattdessen die Absolutwerte,
      //   dann verflschen die negativen Anteile den Wert. Je kleiner der
      //   HitVal, desto grer wird dieser falsche Anteil. Dieser kann
      //   aber rausgerechnet werden. Jedoch nur ungenau.
      //
      //   AbsAvr | RealAvr
      //   -------+-------
      //    9.02  |    0
      //    9.06  |    1
      //    9.16  |    2
      //    9.34  |    3
      //    9.58  |    4
      //    9.89  |    5
      //   10.26  |    6
      //   10.70  |    7
      //   11.19  |    8
      //   11.74  |    9
      //   12.34  |   10
      //   12.98  |   11
      //   13.67  |   12
      //   14.40  |   13
      //   15.17  |   14
      //   15.97  |   15
      //   16.80  |   16
      //   17.66  |   17
      //   18.53  |   18
      //   19.43  |   19
      //   20.35  |   20
      //   21.28  |   21
      //   22.22  |   22
      //   23.17  |   23
      int aiAvr = s->AbsAvrVal;
      int riAvr = aiAvr*5-5760;
      if(aiAvr>=1344) riAvr=aiAvr*346/256-832;
      if(riAvr>aiAvr) riAvr=aiAvr;
      if(riAvr<0) riAvr=0;
      AvrHitVal = (riAvr+64)/128;
      if(AvrHitVal>255) AvrHitVal=255;
      o->AvrHitVal = AvrHitVal;
    }
    // Fehlerwahrscheinlichkeit frs Bit berechnen und in s->DCFArr[] ablegen
    {
      // - Referenz
      //   int HitVal = abs(DCF77_BitVal);
      //   double p0 = HVProb[AvrHitVal+HitVal];
      //   double p1 = HVProb[abs(AvrHitVal-HitVal)];
      //   double p2 = HVProb[HitVal]*(2/(100.0/DCFDEC_DropoutPct-1));
      //   double ErrNorm = 1/(p1/p0+1);
      //   double ErrRand = 0.5;
      //   double Err = (p1*ErrNorm+p2*ErrRand)/(p1+p2);
      //   double lErr = log(1/Err-1);
      //   int le = (int)(lErr*DCFDEC_LogScale);
      //   le = (le+64)/128;
      //   if(le>127) le=127;
      //   if(DCF77_BitVal<0) le=-le;
      //   s->DCFArr[p] = le;
      // - Nherung
      //   Die Gesamtabweichung durch die LogGaussHVProb()-Nherung wirkt
      //   sich nur auf seltene (<1%) HitVal-Werte oder bei sehr groen
      //   (>200) dbErr-Werten aus. Die Abweichung fhrt nie zu greren
      //   dbErr-Werten.
      int HitVal = abs(DCF77_HitVal);
      int l0 = 0;
      l0 = l0 - LogGaussHVProb(AvrHitVal-HitVal);
      l0 = l0 + LogGaussHVProb(HitVal          );
      l0 = l0 + (int)((log(2)-log(2/(100.0/DCFDEC_DropoutPct-1)))*DCFDEC_LogScale);
      l0 = -LogAdd(-l0, 0); // 0 = log(1)
      int l1 = 0;
      l1 = l1 - LogGaussHVProb(AvrHitVal-HitVal);
      l1 = l1 + LogGaussHVProb(AvrHitVal+HitVal);
      int le = l0+LogAdd(l1-l0, 0);
      if(le<0) le=0;
      le = (le+64)/128;
      if(le>127) le=127;
      if(DCF77_HitVal<0) le=-le;
      s->DCFArr[p] = le;
    }
  }

  // eine Sekunde weiterstellen
  DCF77DecHit_IncTime(o);

  bool ForceUpdate = false;

  // Startbit suchen
  {
    // Dazu wird das neue Bit mit den frheren Bits verrechnet.
    // Falls nicht alle Bits den gleichen Wert haben ('Mischbits'),
    // ist der Bitwert und vor allem der berechnete Fehler
    // undefiniert. Dies trifft z.B. hufig fr die Minuten-Bits
    // zu. In der Simulation (rund 3 Milliarden Bits des
    // 21. Jahrhundert mit verschiedenen Signalstrken/
    // Konfigurationen) waren Hufigkeit und Fehlerwert
    // unterhalb der Doppelfehlerwerte (s.u.). Das 'undefiniert'
    // scheint praktisch unwesentlich zu sein, weshalb es hier
    // keinen Programmcode zum Schutz vor Fehlern bei Mischbits
    // gibt.
    //
    // Gesucht wird die Bitfolge "011111111110". Auch bei _einem_
    // falsch erkannten Bit ist trotzdem (zumindest bis zum Jahr 2100)
    // gewhrleistet, dass ein Startbit nicht irrtmlich erkannt wird.
    // Fr das Startbit darf deshalb mit der extrem viel besseren
    // Doppel- statt Einzelfehlerwahrscheinlichkeit gerechnet werden.
    // Anders wre fatal, da dieser Fehler zum Time_dbErr, Zone_dbErr
    // und Date_dbErr (alle mit Doppelfehlerw.) addiert werden muss!
    {
      // bestimmt den letzten Bitwert (und nicht immer alle Bits inkl.
      // Gesamtfehler - spart minimal Rechenzeit)
      int LogSum = 0;
      int p = s->DCFOfs+s->DCFCnt-1;
      if(p>=DCFDEC_MaxSeconds) p-=DCFDEC_MaxSeconds;
      for(int i=0; i<s->DCFCnt; i+=60)
      {
        LogSum += s->DCFArr[p];
        p -= 60;
        if(p<0) p+=DCFDEC_MaxSeconds;
      }
      s->StartBits <<= 1;
      if(LogSum>0) s->StartBits|=1;
    }
    if((s->StartBits&0x0fff)==0x07fe || o->Time_Second==11)
    {
      int p = s->DCFOfs+s->DCFCnt+60-11;
      while(p>=60) p-=60;
      int32_t pItem;
      int32_t nItem;
      DCF77Dec_Seek(s, DCFBit_Start0, DCFBit_Start1, p, &pItem, &nItem);
      int32_t Err = 0;
      int DCFBits = DCF77Dec_DecodeBits(nItem, 12, s->DCFArr, pItem, &Err);
      if(DCFBits==0x07fe)
      {
        int32_t Avr = AvrDCFProb(nItem, 12, s->DCFArr, pItem);
        OutlierFilter(Avr, OutlierFilt_Start, &Err);
        if(s->StartOfs!=p) s->ErrStart-=Err/2; // gegen Deadlock bei Schaltsekunde
        if(Err>s->ErrStart)
        {
          ForceUpdate = s->ErrStart<(256*DCFDEC_LogScale*log(10)/20);
          s->ErrStart = Err;
          s->StartOfs = p;
          o->Time_Second = 11;
        }
      }
    }
  }

  // bei neuer Stundenzahl auf Zeitumstellung prfen
  if(s->AssumeZoneChange && o->Time_Minute==0 && o->Time_Second==0)
  {
    // eigentlich bruchte nur der "if(s->AssumeZoneChange)"-Block
    // durchlaufen werden
    ForceUpdate = true;
  }

  // Uhrzeit (und Zeitzone)
  if(ForceUpdate || o->Time_Second==DCFBit_Time1+1)
  {
    int32_t pItem;
    int32_t nItem;
    DCF77Dec_Seek(s, DCFBit_Time0, DCFBit_Time1, s->StartOfs, &pItem, &nItem);
    s->ErrTime = 0;
    o->Time_dbErr = 0;
    o->Zone_dbErr = 0;
    if(nItem>0 && s->ErrStart>0)
    {
      // alle Minutenwerte inkl. Prfbit (7Bit+1Bit)
      static const uint8_t ValArr60[] = // 60+DCFDEC_MaxMinutes-1
      {
        0x00, 0x81, 0x82, 0x03, 0x84, 0x05, 0x06, 0x87,
        0x88, 0x09, 0x90, 0x11, 0x12, 0x93, 0x14, 0x95,
        0x96, 0x17, 0x18, 0x99, 0xa0, 0x21, 0x22, 0xa3,
        0x24, 0xa5, 0xa6, 0x27, 0x28, 0xa9, 0x30, 0xb1,
        0xb2, 0x33, 0xb4, 0x35, 0x36, 0xb7, 0xb8, 0x39,
        0xc0, 0x41, 0x42, 0xc3, 0x44, 0xc5, 0xc6, 0x47,
        0x48, 0xc9, 0x50, 0xd1, 0xd2, 0x53, 0xd4, 0x55,
        0x56, 0xd7, 0xd8, 0x59,
        // (DCFDEC_MaxMinutes-1)-Wiederholungen fr Inkremente
        // beim Minutenwechsel
        0x00, 0x81, 0x82, 0x03, 0x84, 0x05, 0x06, 0x87,
        0x88
      };
      // alle Stundenwerte inkl. Prfbit (6Bit+1Bit)
      static const uint8_t ValArr24[24+1] =
      {
        0x00, 0x41, 0x42, 0x03, 0x44, 0x05, 0x06, 0x47,
        0x48, 0x09, 0x50, 0x11, 0x12, 0x53, 0x14, 0x55,
        0x56, 0x17, 0x18, 0x59, 0x60, 0x21, 0x22, 0x63,
        // eine Wiederholung fr Inkrement beim Minutenberlauf
        0x00
      };
      // Zeitzonen (0:MEZ; 1:MESZ)
      static const uint8_t ValArrZ2[2+1] =
      {
        0x02, 0x01,
        // eine Wiederholung fr Inkrement beim Zeitzonenwechsel
        0x02
      };

      int32_t ErrMinute = 0;
      int32_t ErrHour   = 0;
      int32_t ErrZone   = 0;
      int Minute = DCF77Dec_DecodeValue(nItem, // 37.1ms@8MHz
                                        60,
                                        ValArr60,
                                        8,
                                        s->DCFArr,
                                        pItem+5,
                                        (1<<DCFDEC_MaxMinutes)-1,
                                        1,
                                        &ErrMinute);
      int Hour = DCF77Dec_DecodeValue(nItem, // 12.6ms@8MHz
                                      24,
                                      ValArr24,
                                      7,
                                      s->DCFArr,
                                      pItem+13,
                                      Minute>30 ? (1<<(59-Minute)) : 0,
                                      1,
                                      &ErrHour);
      int Zone = DCF77Dec_DecodeValue(nItem, // 0.3ms@8MHz
                                      2,
                                      ValArrZ2,
                                      2,
                                      s->DCFArr,
                                      pItem+1,
                                      0,
                                      0,
                                      &ErrZone);

      int32_t Avr = AvrDCFProb(nItem, // 1.4ms@8MHz
                               DCFBit_Time1-DCFBit_Time0+1,
                               s->DCFArr,
                               pItem);
      OutlierFilter(Avr, OutlierFilt_Minute, &ErrMinute);
      OutlierFilter(Avr, OutlierFilt_Hour  , &ErrHour);
      OutlierFilter(Avr, OutlierFilt_Zone  , &ErrZone);

      // evtl. Zeitzonenumstellung innerhalb vom Messzeitraum bercksichtigen
      int ZoneHour = -1;
      {
        // Tritt whrend des Messzeitraums eine Zeitzonenumstellung auf,
        // dann wird meist in der Hlfte der Flle der Stundenwert um
        // eins falsch bestimmt. In seltenen Fllen ist er zufllig.
        // Das eigentliche Problem ist, dass die berechnete
        // Fehlerwahrscheinlichkeit in allen Fllen fraglich wird. Bei der
        // Zeitzonenumstellung muss deshalb der Messzeitraum auf vor bzw.
        // nach dem Minutenberlauf getrennt werden. Wegen der schlechteren
        // Erkennungsqualitt sollte das nicht bei jedem berlauf sondern
        // nur bei verdchtigen gemacht werden:
        // Beim Minutenberlauf gibt es fr die Zonen und das Ankndigungs-
        // Bit vier Kombinationen. Welche passt am besten?
        if(Minute+nItem==61)
        {
          static const uint8_t ValArrZ4[8] =
          {
            //         210 Zonenbits (0=A1=Bit16, 1=Z1:Bit17, 2=Z2:Bit18)
            0x04, // 0b100 TestVal=0: Winterzeit
            0x05, // 0b101 TestVal=1: Ankndigung W->S
            0x02, // 0b010 TestVal=2: Sommerzeit
            0x03, // 0b011 TestVal=3: Ankndigung S->W
            // nach Minutenberlauf
            0x04, // 0b100 TestVal=0: Winterzeit (keine nderung)
            0x03, // 0b011 TestVal=1: Wechsel W->S
            0x02, // 0b010 TestVal=2: Sommerzeit (keine nderung)
            0x05, // 0b101 TestVal=3: Wechsel S->W
          };
          int32_t TestDummyErr = 0;
          int TestVal = DCF77Dec_DecodeValue(nItem, // 1.1ms@8MHz
                                             4,
                                             ValArrZ4,
                                             3,
                                             s->DCFArr,
                                             pItem,
                                             1<<(59-Minute),
                                             4,
                                             &TestDummyErr);
          if(TestVal==1 || TestVal==3) s->AssumeZoneChange=true;
        }
        if(Minute+nItem<61) s->AssumeZoneChange=false; // Minutenberlaufende
        if(s->AssumeZoneChange)
        {
          // Es gab mglicherweise eine Zeitumstellung: Mezeitraum auf
          // vor bzw. nach dem berlauf einschrnken und Hour mit
          // Fehlerwert neu bestimmen:
          int nItemSep = nItem;
          if(Minute+nItem==61 && o->Time_Second>=DCFBit_Time1+1)
              {
                // Der nchste o->Time_*-Zeitstempel bezieht sich auf
                // vor dem berlauf. -> Das letzte Element wegnehmen.
                nItemSep--;
              }
            else
              {

                // Der nchste o->Time_*-Zeitstempel bezieht sich auf nach
                // dem berlauf. -> Die ersten Elemente wegnehmen.
                for(int i=0; i<60-Minute; i++)
                {
                  pItem += 60;
                  if(pItem>=DCFDEC_MaxSeconds) pItem-=DCFDEC_MaxSeconds;
                  nItemSep--;
                }
              }
          ZoneHour = DCF77Dec_DecodeValue(nItemSep, // 12.6ms@8MHz
                                          24,
                                          ValArr24,
                                          7,
                                          s->DCFArr,
                                          pItem+13,
                                          0,
                                          0,
                                          &ErrHour);
          Zone = DCF77Dec_DecodeValue(nItemSep, // 0.3ms@8MHz
                                      2,
                                      ValArrZ2,
                                      2,
                                      s->DCFArr,
                                      pItem+1,
                                      0,
                                      0,
                                      &ErrZone);

          int32_t Avr = AvrDCFProb(nItemSep, // 1.4ms@8MHz
                                   DCFBit_Time1-DCFBit_Time0+1,
                                   s->DCFArr,
                                   pItem);
          OutlierFilter(Avr, OutlierFilt_Hour, &ErrHour);
          OutlierFilter(Avr, OutlierFilt_Zone, &ErrZone);
        }
      }

      // Minute-Wert auf Time_*-Bezug umrechnen
      {
        Minute += nItem-1;
        if(Minute>=60)
        {
          Minute -= 60;
          Hour++;
          if(Hour>=24) Hour=0;
        }
        if(o->Time_Second>=DCFBit_Time1+1)
        {
          // die aktuell bestimmte 'Minute' ist noch nicht gltig
          // (DCF-Zeitbits beziehen sich auf die kommende Minutenmarke)
          Minute--;
          if(Minute<0)
          {
            Minute += 60;
            Hour--;
            if(Hour<0) Hour=23;
          }
        }
      }

      ErrMinute = LogAdd(s->ErrStart, ErrMinute);
      s->ErrTime = LogAdd(ErrMinute, ErrHour);

      o->Zone_dbErr  = DCF77Dec_Calc_dbErr(ErrMinute, ErrZone);
      o->Zone_MEZS   = Zone;

      o->Time_dbErr  = DCF77Dec_Calc_dbErr(ErrMinute, ErrHour);
      o->Time_Minute = Minute;
      o->Time_Hour   = ZoneHour<0 ? Hour : ZoneHour;

      ForceUpdate = true; // frs Datum
    }
  }

  // Datum
  if(ForceUpdate || o->Time_Second==DCFBit_Date1+1)
  {
    int32_t pItem;
    int32_t nItem;
    DCF77Dec_Seek(s, DCFBit_Date0, DCFBit_Date1, s->StartOfs, &pItem, &nItem);
    o->Date_dbErr = 0;
    // eine Sekunde vor Tageswechsel hat das letzte Element bereits das
    // Datum vom nchsten Tag -> dieses Element nicht bercksichtigen
    if(o->Time_Hour==23 && o->Time_Minute==59 && o->Time_Second>=DCFBit_Date1+1)
    {
      nItem--;
    }
    // Elemente mit Datum vom vorherigen Tag nicht bercksichtigen
    if(o->Time_Hour==0)
    {
      int nSec = o->Time_Minute*60+o->Time_Second+120-(DCFBit_Date1+1);
      while(nItem*60>nSec)
      {
        pItem += 60;
        if(pItem>=DCFDEC_MaxSeconds) pItem-=DCFDEC_MaxSeconds;
        nItem--;
      }
    }
    if(nItem>0 && s->ErrStart>0)
    {
      int32_t ErrDate = 0;
      int DCFBits = DCF77Dec_DecodeBits(nItem, // 1.4ms@8MHz
                                        23,
                                        s->DCFArr,
                                        pItem,
                                        &ErrDate);
      int32_t Avr = AvrDCFProb(nItem, 23, s->DCFArr, pItem); // 1.6ms@8MHz
      OutlierFilter(Avr, OutlierFilt_Date, &ErrDate);
      int d0 = (DCFBits>>(36-36))&15;
      int d1 = (DCFBits>>(40-36))& 3;
      int wd = (DCFBits>>(42-36))& 7;
      int m0 = (DCFBits>>(45-36))&15;
      int m1 = (DCFBits>>(49-36))& 1;
      int y0 = (DCFBits>>(50-36))&15;
      int y1 = (DCFBits>>(54-36))&15;
      int d = 10*d1+d0;
      int m = 10*m1+m0;
      int y = 10*y1+y0;
      // zustzliche Fehlerprfung (BCD-Digits, Wertebereiche und Wochentag)
      if(DCFDEC_DateCheck!=0)
      {
        bool BCDOk  = true;
        bool DateOk = true;
        bool WDayOk = true;
        if(d0>9) BCDOk=false;
        if(m0>9) BCDOk=false;
        if(y0>9) BCDOk=false;
        if(y1>9) BCDOk=false;
        if(wd<1) DateOk=false;
        if(d<1 || d>31) DateOk=false;
        if(m<1 || m>12) DateOk=false;
        // Wochentag korrekt?
        {
          // Wochentage im 21. Jahrhundet (d:1..31;  m:1..12; y:0..99)
          // mw[m-1] = int(2.6*((i+9)%12+1)-0.2)%7 ("Wikipedia Wochentagsberechnung")
          static const uint8_t mw[12] = { 0, 3, 2, 5, 0, 3, 5, 1, 4, 6, 2, 4 };
          int yy = m<3 ? y-1 : y;
          int w = ((d+mw[m-1]+yy+(yy>>2)+6)%7)+1;
          if(w!=wd) WDayOk=false;
        }
        bool Ok = BCDOk&DateOk&WDayOk; // 90% der Fehler werden so erkannt
        if(!Ok) ErrDate=0;
      }
      o->Date_dbErr   = DCF77Dec_Calc_dbErr(s->ErrTime, ErrDate);
      o->Date_Day     = d;
      o->Date_Month   = m;
      o->Date_WeekDay = wd;
      o->Date_Year    = y;
    }
  }
}

//############################################################## DCF77DecHit_Init
void OPTS DCF77DecHit_Init(TDCF77DecHit* s)
{
  for(int i=0; i<sizeof(*s); i++)
  {
    ((uint8_t*)s)[i] = 0;
  }
  s->DCF77DecHitObj.StartBits = -1;
  s->DCF77DecHitObj.DCF77_BitCnt = -1;
}

//############################################################## DCF77DecHit_Deinit
void OPTS DCF77DecHit_Deinit(TDCF77DecHit* s)
{
}
