
#ifndef DCF77DECHIT_H
#define DCF77DECHIT_H

#include "stdint.h"

//
// Aufwndiger bitfehlertoleranter Dekoder fr phasenmodulierte DCF77-Daten
// anhand von "HitVal"-Werten.
//
// Copyright: Public Domain
//
// C.B., 2021
//

enum
{
  DCFDEC_MaxMinutes = 10, // 1..10; Ringpuffergre; je Minute 60 Byte mehr RAM
  DCFDEC_DateCheck  =  0, // 0/1; Plausibilittsprfung (90% weniger Datumsfehler)
  DCFDEC_DropoutPct = 10, // 1..50; Prozentanteil starker und auergewhnlicher Strungen (s.u.)

  // berechnet / fest
  DCFDEC_LogScale   = 400, // Anpassung einiger Tabellen ntig
  DCFDEC_MaxSeconds = DCFDEC_MaxMinutes*60,
};

typedef struct // privat
{
  int32_t  DCF77_BitCnt;
  int32_t  DCFOfs;
  int32_t  DCFCnt;
  int8_t   DCFArr[DCFDEC_MaxSeconds]; // Wahrscheinlichkeit (max. 4.4e-17)und Bitwert
  int32_t  StartBits; // die letzten 32 dekodierten DCF-Bits
  int32_t  StartOfs;  // Position (0..59) des Startbit BitArr[]
  int32_t  ErrStart;
  int32_t  ErrTime;
  int32_t  AbsAvrVal;
  int32_t  AssumeZoneChange;
} TDCF77DecHitObj;

typedef struct // ffentlich
{
  uint8_t  AvrHitVal;    // 0..127; HitVal-Mittel fr Medauereinschtzung (s.u.)

  uint8_t  Zone_dbErr;   // s.u. "Fehlerwahrscheinlichkeit"
  uint8_t  Zone_MEZS;    // 0..1; 0:Winterzeit

  uint8_t  Time_dbErr;   // s.u. "Fehlerwahrscheinlichkeit"
  uint8_t  Time_Second;  // 0..59
  uint8_t  Time_Minute;  // 0..59
  uint8_t  Time_Hour;    // 0..23

  uint8_t  Date_dbErr;   // s.u. "Fehlerwahrscheinlichkeit"
  uint8_t  Date_Day;     // 1..x
  uint8_t  Date_WeekDay; // 1..7; 1:Montag
  uint8_t  Date_Month;   // 1..12
  uint8_t  Date_Year;    // 0..99; 0:2000

  TDCF77DecHitObj DCF77DecHitObj; // privat

  // - dbErr : mittlerer Fehler unter schlechten Bedingungen
  //     q = 10^(-Date_dbErr/20)
  //   - Beispiele
  //       6 db =       0.5 -> Zufall
  //      20 db =      1/10 -> jeder Zehnte
  //      60 db =    1/1000 -> jeder Tausendste
  //     120 db = 1/1000000 -> jeder Millionste
  //     180 db = 1/1e9     -> jeder Milliardste
  //     255 db = 1/5.62e12 -> rund alle 5000 Milliarden einer
  //   - bei einem dbErr von 0 war der tatschliche dbErr unklar/unbekannt
  //   - falls eine Zeitzonenumstellungsphase (max. 20s pro Jahr) nicht
  //     erkannt wurde, dann ist auch dbErr falsch!
  //   - bei dbErr sind Programmierfehler nicht mit eingerechnet!
  // - Time_Second
  //   Dieser Wert kann unabhngig von Time_dbErr gesehen werden: Aus
  //   'DCF77DecObj.ErrStart' liesse sich ein 'Second_dbErr' errechnen.
  //   Dann wre 'Time_Second' frher und mit besserem Fehler verwendbar.
  // - Wetterdaten
  //   Bei der DCF77-Phasenmodulation werden keine Wetterdaten bertragen.
  //   (Die Bits 1 bis 14 sind dort mit der Minutenkennung belegt.)
  // - Schaltsekunden
  //   Werden nicht bercksichtigt. Fhren bis zu 15 Minuten zu falschen
  //   Ergebnissen.
  // - Zeitdifferenz / exakter Zeitbezug (innerhalb einer Sekunde)
  //   Der nach Aufruf vom DCF77DecHit_ProcessVal() in TDCF77DecHit
  //   angegebene Zeitstempel gilt fr die kommende Sekundenmarke.
  //   Empfnger stellen Bits frhestens 995ms (s. Doku) nach einer
  //   Sekundenmarke bereit. Die Dekoderzeit geht damit um etwa 5ms
  //   vor.
  // - bentigte Messzeit in Abhngigkeit von 'AvrHitVal'
  //   Diese mittleren Zeiten gelten wenn man solange misst, bis dbErr
  //   einen gewnschten Mindestwert erreicht hat.
  //   Beispiel: Es dauert bei einem AvrHitVal von 40 im Mittel 3 Minuten
  //             und eine Sekunde bis die dbErr-Werte fr Zeit und Datum
  //             einen Wert von mindestens 180 erreichen.
  //   (Simulationswerte; bei DCFDEC_MaxMinutes=10 & DCFDEC_DropoutPct=10)
  //   - Messzeit (mm:ss) um die Uhrzeit zu dekodieren
  //           |                        AvrHitVal
  //     dbErr |      10     15     20     25     30     40     60     80
  //     ------+---------------------------------------------------------
  //        60 |   09:57  04:31  02:56  02:16  02:00  01:39  01:01  00:58
  //        80 |   12:13  05:08  03:19  02:36  02:12  01:52  01:05  00:58
  //       100 |   15:44  05:44  03:43  02:55  02:27  01:59  01:10  00:58
  //       120 |   22:44  06:20  04:06  03:12  02:45  02:06  01:18  00:58
  //       180 |   87:13  08:06  05:13  04:04  03:24  02:37  01:46  01:01
  //       240 |   99:59  10:14  06:19  04:56  04:07  03:05  01:59  01:10
  //       255 |   99:59  11:05  06:35  05:08  04:16  03:10  02:00  01:13
  //   - Messzeit fr Uhrzeit und Datum
  //           |                        AvrHitVal
  //     dbErr |      10     15     20     25     30     40     60     80
  //     ------+---------------------------------------------------------
  //        60 |   19:05  05:47  03:38  02:49  02:23  02:02  01:17  01:07
  //        80 |   32:36  06:25  04:02  03:07  02:39  02:09  01:24  01:07
  //       100 |   58:14  07:01  04:25  03:23  02:56  02:17  01:32  01:08
  //       120 |   83:48  07:37  04:46  03:40  03:09  02:26  01:41  01:08
  //       180 |   99:59  09:30  05:52  04:33  03:52  03:01  02:03  01:13
  //       240 |   99:59  12:59  07:01  05:26  04:33  03:25  02:10  01:28
  //       255 |   99:59  15:01  07:18  05:40  04:45  03:34  02:11  01:35
  //
  // - Aktualisierung
  //   DCF-Bit | Aktualisierung
  //   --------+----------------
  //      11   | Zone, Zeit und Datum
  //      35   | Zone, Zeit und Datum
  //      58   | Datum
  //
  // - "DCFDEC_DropoutPct"-Parameter
  //   Die meiste Zeit sind Strungen gleichartig und sind unabhngig von
  //   deren Gre kein Problem. berdeckende strke Strungen zwischen
  //   0.1s bis 5s Dauer knnen sich hingegen sehr negativ auf die
  //   Fehlervorversage ("dbErr") auswirken. Der Decoder erkennt und
  //   bercksichtigt Strungen statistisch. Je grer "DCFDEC_DropoutPct"
  //   desto mehr Strereignisse werden toleriert. Als negativer Nebeneffekt
  //   verringern sich die dbErr-Werte.
  //    100 : sehr sicher (und viel Verschnitt)
  //     30 : sinnvoller?
  //     10 : sinnvoll
  //      1 : unsicher (aber maximale Ausbeute)
  //
  // - Resourcen
  //   -  876 Bytes RAM
  //            652 Bytes TDCF77DecObj bei DCFDEC_MaxMinutes=10
  //          +  60 Bytes fr Stapel (+292 Bytes vom DCF77-Modul, s. "DCF77.h")
  //          -----------------------------------------------------------------
  //          = 876 Bytes insgesamt
  //   - 3944 Bytes FLASH
  //
} TDCF77DecHit;

void DCF77DecHit_Init(TDCF77DecHit* s);

void DCF77DecHit_Deinit(TDCF77DecHit* s);

void DCF77DecHit_ProcessVal(TDCF77DecHit* s,
                            int32_t DCF77_BitCnt,
                            int32_t DCF77_HitVal);

#endif
