

#include "hardware_t.h"
#include "tc_vars.h"
#include "flash_t.h"
#include "display_t.h"

/***************************
  contents: implementation of flash routines
            for the AMD part.
*/
// Class FloatStr
void AMDFlash::Init(void)
{
//printf("AMDFlash::Init\n");
#if defined(nm_cpu_architecture) && (nm_cpu_architecture == nios2)
#define DO_CACHE_INIT 1
#else
#define DO_CACHE_INIT 0
#endif

}

// ---------------------------------------------
// Private Routines
// Wait for the given address to contain 0xFF.
// Return value: 1 on success, 0 on timeout failure.
int AMDFlash::await_erase_complete(int entire_chip, volatile unsigned char *addr)
{
	int iTimeout = 40;

  	// The "typical" time for an erase command is 0.7 seconds.
  	// Empirically (as expected), it takes longer to erase the entire
	// chip than a single sector. Give a generous timeout for a single
  	// sector, and even more for the entire chip.

  	if (entire_chip) iTimeout = 20000;

	while (iTimeout)
	{
		nr_delay(100);      // Wait 100 ms.	
		if (0xFF == *addr)  // done? note: 8bit data
		{
			return 0;     // leave
		}

		iTimeout--;
	}

  	// Timeout error.
  	return -1;
}

// This routine actually takes about 3 us on a 33.333MHz Nios32.
void AMDFlash::wait_at_least_1_us(void)
{
  	volatile unsigned long iTimeout = nasys_clock_freq_1000 / 825; //8250
  	while (iTimeout--)
  	{
    	;
  	}
}

// Unlock bypass mode, enabling fast writes.
void AMDFlash::unlock_bypass_mode(volatile unsigned char *flash_base)
{	// 065d
	flash_base[0xAAA] = 0xAA; // unlock bypass command - cycle 1
	flash_base[0x555] = 0x55; // unlock bypass command - cycle 2
	flash_base[0xAAA] = 0x20; // unlock bypass command - cycle 3
}

// Turn bypass mode off, disabling fast writes and enabling normal function.
void AMDFlash::reset_bypass_mode(volatile unsigned char *flash_base)
{
  	*flash_base = 0x90; // exit unlock bypass reset command - cycle 1
  	*flash_base = 0x0;  // exit unlock bypass reset command - cycle 2
}

// Read the given address until two successive reads yield the
// same value.
// Return value: 0 on success, -1 on timeout failure.
int AMDFlash::await_write_complete(volatile unsigned char *addr, const unsigned char correct_value)
{
        //
        // TPA 2/14/2003: The *Maximum* programming time is 150us.
        //                Waiting several times the maximum time
        //                seems like a much better idea than giving-up
        //                well before the published spec says we
        //                should.
        //
	unsigned long iTimeout = 1200; //1200
	int return_value = -1; // Pessimistic return value.

	while (iTimeout)
	{
		wait_at_least_1_us();

    	// While the flash is working on program data, read accesses return
    	// "status" instead of data.  Status bit 7 is the complement of the
    	// data being written.  When the program operation is complete and
    	// successful, the written data is returned.  So, read the written
    	// address until it equals the data written.

		if (*addr == correct_value) break;
		iTimeout--;
	}

	if (iTimeout) return_value = 0;

	return return_value;
}
      // (entire chip, base)


// Write val to the given flash address, in bypass mode (assumes
// that bypass mode has been enabled already).
// Return value: 0 on success, -1 on failure.
int AMDFlash::nr_flash_write_bypass(volatile unsigned char *flash_base, volatile unsigned char *addr, unsigned char val)
{
	unsigned char us1, us2;
	int iTimeout;
	int result = 1;
	int errors = 1000;

    nm_dcache_invalidate_line(addr);
    nm_icache_invalidate_line(addr);
	
    while(result)
    {
	   *flash_base = 0xA0;   // unlock bypass program command - 1st cycle
	   *addr = val;          // program address and data    - 2nd cycle

	   result = await_write_complete(addr,val);
	
	   errors--;
	
	   if (errors == 0) result = 0;
    }
    //if(result) return result;

	us1 = *addr;

	if (us1 != val) result = -1;

	return result;
}

// ---------------------------------------------
// Public Routines

// Erase the flash sector at sector_address.
// Return value: 0 on success, -1 on failure.

int AMDFlash::nr_flash_erase_sector(unsigned long *flash_base, unsigned long *sector_address)
{
	volatile unsigned char *fb  = (unsigned char *) flash_base;
	volatile unsigned char *sa  = (unsigned char *) sector_address;
	int result;
//printf("AMDFlash::nr_flash_erase_sector\n");	
#if DO_CACHE_INIT
	nr_dcache_init();
	nr_icache_init();
#endif

#ifdef nasys_main_flash
	if (-1 == (int)fb) fb = (unsigned char *) nasys_main_flash;
#endif // nasys_main_flash
//  AM29LV065d
	fb[0xAAA] = 0xAA; // 1st cycle
	fb[0x555] = 0x55; // 2nd cycle
	fb[0xAAA] = 0x80; // 3rd cycle
	fb[0xAAA] = 0xAA; // 4th cycle
	fb[0x555] = 0x55; // 5th cycle

	*sa = 0x30; // 6th cycle

  	// Loop until the data reads as 0xFF.

  	result = await_erase_complete(0, sa);
  	return result;
}


// Write a buffer of data to the flash, using bypass mode.
// Return value: 1 on success, 0 on failure.
// Note: the integer "size" is given as a number of half-words to
// write.   How convenient.  We write this 8-bit-wide flash one byte
// at a time (of course).
int AMDFlash::nr_flash_write_buffer(unsigned long *flash_base, unsigned long *start_address, unsigned long *buffer, int size)
{
	volatile unsigned char *fb  = (unsigned char *) flash_base;
	unsigned char *sa  = (unsigned char *) start_address;
	unsigned char *buf = (unsigned char *) buffer;
	long num_bytes = size * 4;
	long i;
	int result = 0;
//printf("AMDFlash::nr_flash_write_buffer\n");
#ifdef nasys_main_flash
	if (-1 == (int)fb) fb = (unsigned char *) nasys_main_flash;
#endif // nasys_main_flash

	unlock_bypass_mode(fb);
	for (i = 0; i < num_bytes; ++i)
	{
		result = nr_flash_write_bypass(fb, sa + i, buf[i]);
		if(result) break;
	}
	reset_bypass_mode(fb);

	return result;
}

// Write val to the given flash address.
// Return value: 1 on success, 0 on failure.
int AMDFlash::amd29lv065d_flash_write_byte(unsigned char *flash_base, unsigned char *addr, unsigned char val)
{
    volatile unsigned char *fb = (unsigned char *) flash_base;
    volatile unsigned char *a = (unsigned char *) addr;
    unsigned char us1, us2;
    int result = 1;
//printf("AMDFlash::amd29lv065d_flash_write_byte\n");
    nm_dcache_invalidate_line(a);
    nm_icache_invalidate_line(a);

#ifdef nasys_main_flash
    if (-1 == (int)fb) fb = (unsigned char *) nasys_main_flash;
#endif // nasys_main_flash

    while (result)
    {
        fb[0x555] = 0xAA; // 1st cycle  addr = XXX, data = AA
        fb[0x333] = 0x55; // 2nd cycle  addr = XXX, data = 55
        fb[0x555] = 0xA0; // 3rd cycle  addr = XXX, data = A0

        *a = val;     // 4th cycle  addr = PA, data = PD

        result = await_write_complete(a,val);
    }
    //if(result) return result;

    us1 = *a;
    if (us1 != val) result = -1;

    return result;
}

int AMDFlash::nr_flash_write(unsigned long *flash_base, unsigned long *addr, unsigned long val)
{
    unsigned char* fb       = (unsigned char *) flash_base;
    unsigned char* a        = (unsigned char *) addr;
    unsigned char  byte_val = val & 0xff;
    int result;
//printf("AMDFlash::nr_flash_write\n");
    result = amd29lv065d_flash_write_byte (fb, a,   byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 8) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+1, byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 16) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+2, byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 24) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+3, byte_val);

    return result;
}

void AMDFlash::Clear_Data_FLASH(void)
{
	volatile unsigned long fb  = (unsigned long) nasys_main_flash;
	int result = 0;
//printf("AMDFlash::Clear_Data_FLASH\n");		
	unsigned long  Data_Index = 0x0000007F;//0x00000070;
	unsigned long  i;
	
	unsigned long  sectorOffset[] =  // AM29LV065d sector map
	{
		0x000000, 0x010000, 0x020000, 0x030000,
		0x040000, 0x050000, 0x060000, 0x070000,
		0x080000, 0x090000, 0x0A0000, 0x0B0000,
		0x0C0000, 0x0D0000, 0x0E0000, 0x0F0000,
		0x100000, 0x110000, 0x120000, 0x130000,
		0x140000, 0x150000, 0x160000, 0x170000,
		0x180000, 0x190000, 0x1A0000, 0x1B0000,
		0x1C0000, 0x1D0000, 0x1E0000, 0x1F0000,
		0x200000, 0x210000, 0x220000, 0x230000,
		0x240000, 0x250000, 0x260000, 0x270000,
		0x280000, 0x290000, 0x2A0000, 0x2B0000,
		0x2C0000, 0x2D0000, 0x2E0000, 0x2F0000,
		0x300000, 0x310000, 0x320000, 0x330000,
		0x340000, 0x350000, 0x360000, 0x370000,
		0x380000, 0x390000, 0x3A0000, 0x3B0000,
		0x3C0000, 0x3D0000, 0x3E0000, 0x3F0000,
		0x400000, 0x410000, 0x420000, 0x430000,
		0x440000, 0x450000, 0x460000, 0x470000,
		0x480000, 0x490000, 0x4A0000, 0x4B0000,
		0x4C0000, 0x4D0000, 0x4E0000, 0x4F0000,
		0x500000, 0x510000, 0x520000, 0x530000,
		0x540000, 0x550000, 0x560000, 0x570000,
		0x580000, 0x590000, 0x5A0000, 0x5B0000,
		0x5C0000, 0x5D0000, 0x5E0000, 0x5F0000,
		0x600000, 0x610000, 0x620000, 0x630000,
		0x640000, 0x650000, 0x660000, 0x670000,
		0x680000, 0x690000, 0x6A0000, 0x6B0000,
		0x6C0000, 0x6D0000, 0x6E0000, 0x6F0000,
		0x700000, 0x710000, 0x720000, 0x730000,
		0x740000, 0x750000, 0x760000, 0x770000,
		0x780000, 0x790000, 0x7A0000, 0x7B0000,
		0x7C0000, 0x7D0000, 0x7E0000, 0x7F0000,
	};
	
	for (i = Data_Index; i < sizeof(sectorOffset) / sizeof(*sectorOffset); ++i)
	{
#ifdef _Debug_	
		printf("Deleting Sector : %x\n", sectorOffset[i]);
#endif		
		volatile unsigned long *sector = (unsigned long *) sectorOffset[i];

		result = nr_flash_erase_sector((unsigned long *) nasys_main_flash, (unsigned long *) sector);		// Clear Sectors
		
		if(result) break;
	}
}

int AMDFlash::copy_signal_to_flash(unsigned long *flash_address)
{
	unsigned long cx;
	unsigned long buf[0x1000];
	unsigned long *buffer = (unsigned long *) buf;	
	int result = -1;
	int led_value = 0x0000;	
//printf("AMDFlash::copy_signal_to_flash\n");	
	nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address);				// Clear Sectors
	for (cx = 0; cx < 0x1000; cx++)
	{
		buf[cx] = *(SIGNAL1 + (cx * 4) + 0) + (*(SIGNAL1 + (cx * 4) + 1) << 8) + (*(SIGNAL1 + (cx * 4) + 2) << 16) + (*(SIGNAL1 + (cx * 4) + 3) << 24);
	}
	result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address, buf, 0x1000);	// Flash Buffer	
#ifdef _Debug_
    printf("Save Signal1 = %d\n", result);
#endif
	for (cx = 0; cx < 0x1000; cx++)
	{
		buf[cx] = *(SIGNAL2 + (cx * 4) + 0) + (*(SIGNAL2 + (cx * 4) + 1) << 8) + (*(SIGNAL2 + (cx * 4) + 2) << 16) + (*(SIGNAL2 + (cx * 4) + 3) << 24);
	}
	result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address + 0x1000, buf, 0x1000);	// Flash Buffer	
#ifdef _Debug_
    printf("Save Signal2 = %d\n", result);
#endif
	for (cx = 0; cx < 0x1000; cx++)
	{
		buf[cx] = *(SIGNAL3 + (cx * 4) + 0) + *(SIGNAL3 + (cx * 4) + 1) + *(SIGNAL3 + (cx * 4) + 2) + *(SIGNAL3 + (cx * 4) + 3);
	}
	result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address + 0x2000, buf, 0x1000);	// Flash Buffer	
#ifdef _Debug_
    printf("Save Signal3 = %d\n", result);
#endif
	for (cx = 0; cx < 0x1000; cx++)
	{
		buf[cx] = *(SIGNAL4 + (cx * 4) + 0) + *(SIGNAL4 + (cx * 4) + 1) + *(SIGNAL4 + (cx * 4) + 2) + *(SIGNAL4 + (cx * 4) + 3);
	}
	
	if (LED_ON[0] == 1) led_value = led_value | 0x0001;		// Select Channel 1
	if (LED_ON[1] == 1) led_value = led_value | 0x0002;		// Select Channel 2
	if (LED_ON[2] == 1) led_value = led_value | 0x0004;		// Select Channel 3
	if (LED_ON[3] == 1) led_value = led_value | 0x0008;		// Select Channel 4
	if (LED_ON[4] == 1) led_value = led_value | 0x0010;		// Select Channel Math
	if (LED_ON[5] == 1) led_value = led_value | 0x0020;		// General
	if (LED_ON[6] == 1) led_value = led_value | 0x0040;		// Cursor						
	if (LED_ON[7] == 1) led_value = led_value | 0x0080;		// Quick Measure
	if (LED_ON[8] == 1) led_value = led_value | 0x0100;		// Edge
	if (LED_ON[9] == 1) led_value = led_value | 0x0200;		// Pattern - Not Used
	if (LED_ON[10] == 1) led_value = led_value | 0x0400;		// Pulse Width
	if (LED_ON[11] == 1) led_value = led_value | 0x0800;		// More - Not Used
	if (LED_ON[12] == 1) led_value = led_value | 0x1000;		// Run/Stop Green
	if (LED_ON[13] == 1) led_value = led_value | 0x2000;		// Run/Stop Red
	if (LED_ON[14] == 1) led_value = led_value | 0x4000;		// Single Red					
	if (LED_ON[15] == 1) led_value = led_value | 0x8000;		// Single Green	
		
	buf[0x1000 - 0] = (unsigned long) 0x00;
	buf[0x1000 - 1] = (unsigned long) 0xAACC5523;			// Configuration Start Sign
	buf[0x1000 - 2] = (unsigned long) 0x22334455;
	buf[0x1000 - 3] = (unsigned long) 0x00;		
	buf[0x1000 - 4] = (unsigned long) tc_version;			// Quattro Version
	buf[0x1000 - 5] = (unsigned long) tc_subversion;		// SubVerision
	buf[0x1000 - 6] = (unsigned long) tc_dev_version;		// Revision
	buf[0x1000 - 7] = (unsigned long) tc_dev_subversion;		// Compile Status
	buf[0x1000 - 8] = (unsigned long) 0x00;   	
	
	// Timebase
	buf[0x1000 - 10] = (unsigned long) MenuStatus[10][0];
	buf[0x1000 - 11] = (unsigned long) MenuStatus[10][1];
	buf[0x1000 - 12] = (unsigned long) MenuStatus[10][2];
	buf[0x1000 - 13] = (unsigned long) MenuStatus[10][3];
	buf[0x1000 - 14] = (unsigned long) MenuStatus[10][4];
	buf[0x1000 - 15] = (unsigned long) MenuStatus[5][2];	
	buf[0x1000 - 16] = (unsigned long) XY_Mode;
	buf[0x1000 - 17] = (unsigned long) Selected_Timebase;	
	buf[0x1000 - 18] = (unsigned long) SIGNALFaktor_idx;
	buf[0x1000 - 19] = (unsigned long) SIGNAL_StartFr_idx;
	buf[0x1000 - 20] = (unsigned long) Selected_Timebase;	
	buf[0x1000 - 21] = (unsigned long) SIGNALFaktor_idx;
	buf[0x1000 - 22] = (unsigned long) dmode_Selected_Timebase;
	buf[0x1000 - 23] = (unsigned long) dmode_SIGNALFaktor_idx;
	buf[0x1000 - 24] = (unsigned long) dmode_Window_Offset_Pos;
	buf[0x1000 - 25] = (unsigned long) dmode_SIGNAL_StartFr_idx;
//	buf[0x1000 - 26] = (unsigned long) Cursor_Delayed_1;
//	buf[0x1000 - 27] = (unsigned long) Cursor_Delayed_2;
//	buf[0x1000 - 28] = (unsigned long) Cursor_Delayed_Size;
	buf[0x1000 - 29] = (unsigned long) USTB_Mode;							//ultra slow timebase mode
	buf[0x1000 - 30] = (unsigned long) USTB_Mode_bak;
//    		buf[0x1000 - 31] = (unsigned long) ROLLMODE_STARTED;
//  		buf[0x1000 - 32] = (unsigned long) ROLLMODE_Value_rec;	
    	buf[0x1000 - 33] = (unsigned long) Timebase_Offset_Pos;
	buf[0x1000 - 34] = (unsigned long) SwitchesTB;	
	
	// Trigger
	//buf[0x1000 - 35] = (unsigned long) MenuStatus[12][0];                                      // Triggering - Auto
	buf[0x1000 - 36] = (unsigned long) MenuStatus[12][1];                                        // Triggering - Coupling DC
	buf[0x1000 - 37] = (unsigned long) MenuStatus[12][2];                                        // Triggering - Reject
	buf[0x1000 - 38] = (unsigned long) MenuStatus[12][4];                                        // Triggering - External Probe = 1 : 1  ( muß noch als funktion integriert werden )
	buf[0x1000 - 39] = (unsigned long) MenuStatus[14][3];                                        // Triggering - TV

	buf[0x1000 - 40] = (unsigned long) MenuStatus[14][0];                                        // Triggering - positive edge
	buf[0x1000 - 41] = (unsigned long) MenuStatus[14][1];                                        // Triggering - Source = channel 1
	buf[0x1000 - 42] = (unsigned long) MenuStatus[14][2];                                        // Triggering - no externel triggering
	buf[0x1000 - 43] = (unsigned long) MenuStatus[15][0];                                        // Triggering - Source Channel 1
	buf[0x1000 - 44] = (unsigned long) MenuStatus[15][1];                                        // Triggering - Negative Pulse
	buf[0x1000 - 45] = (unsigned long) MenuStatus[15][2];                                        // Triggering - bigger - smaller - then
	buf[0x1000 - 46] = (unsigned long) Selected_Trigger_Source;                                  // Selected Trigger Source is Channel 1
	buf[0x1000 - 47] = (unsigned long) SelectedEdgeExtern;                                       // Trigger Edge by extern triggering
	buf[0x1000 - 48] = (unsigned long) PulseToggle;                                              // Pulse width button selection
	buf[0x1000 - 49] = (unsigned long) Memory_Position;                                          // Memory Position

	//buf[0x1000 - 50] = (unsigned long) Trigger_Offset_Pos; //BF del not used	             // Trigger Offset
	buf[0x1000 - 51] = (unsigned long) Trigger_Pos_CH1;                                          // Trigger Level Channel 1
	buf[0x1000 - 52] = (unsigned long) Trigger_Pos_CH2;                                          // Trigger Level Channel 2
	buf[0x1000 - 53] = (unsigned long) Trigger_Pos_CH3;                                          // Trigger Level Channel 3
	buf[0x1000 - 54] = (unsigned long) Trigger_Pos_CH4;                                          // Trigger Level Channel 4
	buf[0x1000 - 55] = (unsigned long) trg_val_reg;                                              // Trigger Value
	buf[0x1000 - 56] = (unsigned long) trg_hyst_reg;                                             // Trigger Hysterese
	buf[0x1000 - 57] = (unsigned long) trig_range_reg;                                           // Trigger Range
	buf[0x1000 - 58] = (unsigned long) trig_holdoff_reg;                                         // Trigger Holdoff
	buf[0x1000 - 59] = (unsigned long) trig_width_reg;                                           // Trigger Width

	buf[0x1000 - 60] = (unsigned long) ctrl_reg;                                                 // Control register
	buf[0x1000 - 61] = (unsigned long) adc_ctrl_reg;                                             // ADC Control register
	buf[0x1000 - 62] = (unsigned long) pre_reg;                                                  // PreTrigger Value
	//buf[0x1000 - 63] = (unsigned long) channel_Adr_add;                                          // Channel Add Value
	//buf[0x1000 - 64] = (unsigned long) channel_Adr_add2;                                           // ADC Change
	buf[0x1000 - 65] = (unsigned long) triggering;                                               // Triggering
	buf[0x1000 - 66] = (unsigned long) TriggerWay;                                               // Triggerway
	buf[0x1000 - 67] = (unsigned long) Trigger_Pos_CHE;                                          // Trigger extern value	
	
	// Channel 1
	buf[0x1000 - 68] = (unsigned long) MenuStatus[1][0];                                         // Channel - Coupling
	buf[0x1000 - 69] = (unsigned long) MenuStatus[1][1];                                         // Channel - BW Linit

	buf[0x1000 - 70] = (unsigned long) MenuStatus[1][2];                                         // Channel - Invert
	buf[0x1000 - 71] = (unsigned long) MenuStatus[1][3];                                         // Channel - Probe
	buf[0x1000 - 72] = (unsigned long) SwitchesCH1;                                              // Switches for Channel 1 and for the other the AC bit
	buf[0x1000 - 73] = (unsigned long) Selected_Voltage_CH1;                                     // Voltage 5V
	//BF del buf[0x1000 - 74] = (unsigned long) DAC_Correction[0][0];                              // Search Zeors offset
	//BF del buf[0x1000 - 75] = (unsigned long) DAC_Correction[0][1];                              // Search Zeors offset
	//BF del buf[0x1000 - 76] = (unsigned long) DAC_Correction[0][2];                              // Search Zeors offset
	buf[0x1000 - 77] = (unsigned long) CH1_DAC_1;                                                // channel dac 1 offset
	buf[0x1000 - 78] = (unsigned long) CH1_DAC_2;                                                // channel dac 2 offset
	buf[0x1000 - 79] = (unsigned long) CH1_DAC_3;                                                // channel dac 3 offset
	buf[0x1000 - 80] = (unsigned long) CH1_DAC_4;                                                // channel dac 4 offset
	buf[0x1000 - 81] = (unsigned long) signal_zoomed1;                                           // signal zoomed
	buf[0x1000 - 82] = (unsigned long) CH1_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 83] = (unsigned long) ZeroLevelCH1;                                             // ZeroLevel
	buf[0x1000 - 84] = (unsigned long) Virtual_ZeroLevelCH1;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 85] = (unsigned long) alpha[0];                          			     // ADC-alpha correction -> BF changed
    	//buf[0x1000 - 85] = (unsigned long) CH1_Zero_Offset;                                          // Shift offset when stopped

    // Channel 2       	
	buf[0x1000 - 86] = (unsigned long) MenuStatus[2][0];                                         // Channel - Coupling
	buf[0x1000 - 87] = (unsigned long) MenuStatus[2][1];                                         // Channel - BW Linit
	buf[0x1000 - 88] = (unsigned long) MenuStatus[2][2];                                         // Channel - Invert
	buf[0x1000 - 89] = (unsigned long) MenuStatus[2][3];                                         // Channel - Probe

	buf[0x1000 - 90] = (unsigned long) SwitchesCH2;                                              // Switches for Channel
	buf[0x1000 - 91] = (unsigned long) Selected_Voltage_CH2;                                     // Voltage 5V
	//BF del buf[0x1000 - 92] = (unsigned long) DAC_Correction[1][0];                              // Search Zeors offset
	//BF del buf[0x1000 - 93] = (unsigned long) DAC_Correction[1][1];                              // Search Zeors offset
	//BF del buf[0x1000 - 94] = (unsigned long) DAC_Correction[1][2];                              // Search Zeors offset
	buf[0x1000 - 95] = (unsigned long) CH2_DAC_1;                                                // channel dac 1 offset
	buf[0x1000 - 96] = (unsigned long) CH2_DAC_2;                                                // channel dac 2 offset
	buf[0x1000 - 97] = (unsigned long) CH2_DAC_3;                                                // channel dac 3 offset
	buf[0x1000 - 98] = (unsigned long) CH2_DAC_4;                                                // channel dac 4 offset
	buf[0x1000 - 99] = (unsigned long) signal_zoomed2;                                           // signal zoomed

	buf[0x1000 - 100] = (unsigned long) CH2_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 101] = (unsigned long) ZeroLevelCH2;                                             // ZeroLevel
	buf[0x1000 - 102] = (unsigned long) Virtual_ZeroLevelCH2;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 103] = (unsigned long) alpha[1];                                                 // ADC-alpha correction -> BF changed
    	//buf[0x1000 - 103] = (unsigned long) CH2_Zero_Offset;                                          // Shift offset when stopped

    // Channel 3
	buf[0x1000 - 104] = (unsigned long) MenuStatus[3][0];                                         // Channel - Coupling
	buf[0x1000 - 105] = (unsigned long) MenuStatus[3][1];                                         // Channel - BW Linit
	buf[0x1000 - 106] = (unsigned long) MenuStatus[3][2];                                         // Channel - Invert
	buf[0x1000 - 107] = (unsigned long) MenuStatus[3][3];                                         // Channel - Probe
	buf[0x1000 - 108] = (unsigned long) SwitchesCH3;                                              // Switches for Channel
	buf[0x1000 - 109] = (unsigned long) Selected_Voltage_CH3;                                     // Voltage 5V

	//BF del buf[0x1000 - 110] = (unsigned long) DAC_Correction[2][0];                              // Search Zeors offset
	//BF del buf[0x1000 - 111] = (unsigned long) DAC_Correction[2][1];                              // Search Zeors offset
	//BF del buf[0x1000 - 112] = (unsigned long) DAC_Correction[2][2];                              // Search Zeors offset
	buf[0x1000 - 113] = (unsigned long) CH3_DAC_1;                                                // channel dac 1 offset
	buf[0x1000 - 114] = (unsigned long) CH3_DAC_2;                                                // channel dac 2 offset
	buf[0x1000 - 115] = (unsigned long) CH3_DAC_3;                                                // channel dac 3 offset
	buf[0x1000 - 116] = (unsigned long) CH3_DAC_4;                                                // channel dac 4 offset
	buf[0x1000 - 117] = (unsigned long) signal_zoomed3;                                           // signal zoomed
	buf[0x1000 - 118] = (unsigned long) CH3_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 119] = (unsigned long) ZeroLevelCH3;                                             // ZeroLevel
	buf[0x1000 - 120] = (unsigned long) Virtual_ZeroLevelCH3;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 121] = (unsigned long) alpha[2];                                                 // ADC-alpha correction -> BF changed
    	//buf[0x1000 - 121] = (unsigned long) CH3_Zero_Offset;                                          // Shift offset when stopped

    // Channel 4
	buf[0x1000 - 122] = (unsigned long) MenuStatus[4][0];                                         // Channel - Coupling
	buf[0x1000 - 123] = (unsigned long) MenuStatus[4][1];                                         // Channel - BW Linit

	buf[0x1000 - 124] = (unsigned long) MenuStatus[4][2];                                         // Channel - Invert
	buf[0x1000 - 125] = (unsigned long) MenuStatus[4][3];                                         // Channel - Probe
	buf[0x1000 - 126] = (unsigned long) SwitchesCH4;                                              // Switches for Channel
	buf[0x1000 - 127] = (unsigned long) Selected_Voltage_CH4;                                     // Voltage 5V
	//BF del buf[0x1000 - 128] = (unsigned long) DAC_Correction[3][0];                              // Search Zeors offset
	//BF del buf[0x1000 - 129] = (unsigned long) DAC_Correction[3][1];                              // Search Zeors offset
	//BF del buf[0x1000 - 130] = (unsigned long) DAC_Correction[3][2];                              // Search Zeors offset
	buf[0x1000 - 131] = (unsigned long) CH4_DAC_1;                                                // channel dac 1 offset
	buf[0x1000 - 132] = (unsigned long) CH4_DAC_2;                                                // channel dac 2 offset
	buf[0x1000 - 133] = (unsigned long) CH4_DAC_3;                                                // channel dac 3 offset
	buf[0x1000 - 134] = (unsigned long) CH4_DAC_4;                                                // channel dac 4 offset
	buf[0x1000 - 135] = (unsigned long) signal_zoomed4;                                           // signal zoomed
	buf[0x1000 - 136] = (unsigned long) CH4_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 137] = (unsigned long) ZeroLevelCH4;                                             // ZeroLevel
	buf[0x1000 - 138] = (unsigned long) Virtual_ZeroLevelCH4;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 139] = (unsigned long) alpha[3];                                                 // ADC-alpha correction -> BF changed
    	//buf[0x1000 - 139] = (unsigned long) CH4_Zero_Offset;                                          // Shift offset when stopped                   	
     /*
    buf[0x1000 - 140] = (unsigned long) Channel_Math_Active;                                      // Math channel off
    buf[0x1000 - 141] = (unsigned long) MenuStatus[26][1];                                        // standart math fft on
    buf[0x1000 - 142] = (unsigned long) MenuStatus[26][2];                                        // standart math 1*2 of
    buf[0x1000 - 143] = (unsigned long) MenuStatus[26][3];                                        // standart math 1-2 off
    buf[0x1000 - 144] = (unsigned long) MenuStatus[26][4];                                        // standart math 1+2 off
    */
    /* Buffer für Math Settings
    buf[0x1000 - 145] = (unsigned long) fft source
    buf[0x1000 - 146] = (unsigned long) fft Span
    buf[0x1000 - 147] = (unsigned long) fft Center
    buf[0x1000 - 148] = (unsigned long) fft Preset
    buf[0x1000 - 149] = (unsigned long) fft more Scale

    buf[0x1000 - 150] = (unsigned long) fft more Offset
    buf[0x1000 - 151] = (unsigned long) fft more Window
    buf[0x1000 - 152] = (unsigned long) * - + Scale
    buf[0x1000 - 153] = (unsigned long) * - + Offset  */

    	// Cursors    	
	buf[0x1000 - 154] = (unsigned long) MenuStatus[16][0];                                        // Cursor soruce
	buf[0x1000 - 155] = (unsigned long) Cursor_Enabled;
	buf[0x1000 - 156] = (unsigned long) Cursor_Both_Selected;
	buf[0x1000 - 157] = (unsigned long) SelectedCursor;
	buf[0x1000 - 158] = (unsigned long) Cursor_Horizontal_Active;
	buf[0x1000 - 159] = (unsigned long) Cursor_Horizontal_Position_Real1;
	
	buf[0x1000 - 160] = (unsigned long) Cursor_Horizontal_Position_Real2;
	buf[0x1000 - 161] = (unsigned long) Cursor_Vertical_Active;
	buf[0x1000 - 162] = (unsigned long) Cursor_Vertical_Position_Real1;
	buf[0x1000 - 163] = (unsigned long) Cursor_Vertical_Position_Real2;
	
	// Quick Measure
	//buf[0x1000 - 164] = (unsigned long) QM_Enabled;
	buf[0x1000 - 165] = (unsigned long) QM_Type[0];
	buf[0x1000 - 166] = (unsigned long) QM_Type[1];
	buf[0x1000 - 167] = (unsigned long) QM_Type[2];
	buf[0x1000 - 168] = (unsigned long) QM_Channel[0];
	buf[0x1000 - 169] = (unsigned long) QM_Channel[1];
	
	buf[0x1000 - 170] = (unsigned long) QM_Channel[2];
	buf[0x1000 - 171] = (unsigned long) QM_Horizontal_Active;
	buf[0x1000 - 172] = (unsigned long) QM_Vertical_Active;
	buf[0x1000 - 173] = (unsigned long) Quick_Measure_Threshold_btn_select;
	
	// Display
	buf[0x1000 - 174] = (unsigned long) GridColorArrayPos;                                        // Normal Grid
	buf[0x1000 - 175] = (unsigned long) GridColor_Val;
	buf[0x1000 - 176] = (unsigned long) GridColorArrayPos_Delayed;                                // Delayed Grid
	//BF not used buf[0x1000 - 177] = (unsigned long) GridColor_Val_Delayed;
	buf[0x1000 - 178] = (unsigned long) MenuStatus[5][0];                                         // Display - Persist
	buf[0x1000 - 179] = (unsigned long) MenuStatus[5][3];                                         // Display - Vectors

//    buf[0x1000 - 180] = (unsigned long) MenuStatus[0][0];                                         // Aquire - Normal
//    buf[0x1000 - 181] = (unsigned long) MenuStatus[0][1];                                         // Aquire - Averaging
//    buf[0x1000 - 183] = (unsigned long) MenuStatus[0][2];                                         // Aquire - Avgs
	buf[0x1000 - 184] = (unsigned long) MenuStatus[5][4];                                         // Display - Browse
	
	buf[0x1000 - 185] = (unsigned long) MenuStatus[23][1];                                        // Autoscale - Slow TB
	//buf[0x1000 - 186] = (unsigned long) ????;
	
	buf[0x1000 - 187] = (unsigned long) Debug_Mode;
	buf[0x1000 - 188] = (unsigned long) ADC_Debug_Mode;
	
	/////
	buf[0x1000 - 189] = (unsigned long) ((unsigned long) MenuPopupStatus[8][0] | ((unsigned long) MenuPopupStatus[8][1] << 16));
	buf[0x1000 - 190] = (unsigned long) MenuPopupStatus[8][2] = (short) buf[190];
	buf[0x1000 - 191] = (unsigned long) OneShotRestart;
	buf[0x1000 - 192] = (unsigned long) led_value;
	buf[0x1000 - 193] = (unsigned long) Channel_1_Active;
	buf[0x1000 - 194] = (unsigned long) Channel_2_Active;
	buf[0x1000 - 195] = (unsigned long) Channel_3_Active;
	buf[0x1000 - 196] = (unsigned long) Channel_4_Active;
	buf[0x1000 - 197] = (unsigned long) dmode_SIGNAL_OFFSET;
	buf[0x1000 - 198] = (unsigned long) SIGNAL_OFFSET;
	
	buf[0x1000 - 199] = (unsigned long) HoldOff_Value;
	buf[0x1000 - 200] = (unsigned long) HoldOff_Expo;
	buf[0x1000 - 201] = (unsigned long) HoldOff_Steps;
	
	buf[0x1000 - 202] = (unsigned long) Pulse11_Value;
	buf[0x1000 - 203] = (unsigned long) Pulse11_Expo;
	buf[0x1000 - 204] = (unsigned long) Pulse11_Steps;
	
	buf[0x1000 - 205] = (unsigned long) Pulse12_Value;
	buf[0x1000 - 206] = (unsigned long) Pulse12_Expo;
	buf[0x1000 - 207] = (unsigned long) Pulse12_Steps;
	
	buf[0x1000 - 208] = (unsigned long) Pulse21_Value;
	buf[0x1000 - 209] = (unsigned long) Pulse21_Expo;
	buf[0x1000 - 210] = (unsigned long) Pulse21_Steps;
	
	buf[0x1000 - 211] = (unsigned long) Pulse22_Value;
	buf[0x1000 - 212] = (unsigned long) Pulse22_Expo;
	buf[0x1000 - 213] = (unsigned long) Pulse22_Steps;
	
//BF new Math values
	buf[0x1000 - 214] = (unsigned int) Math_Offset;
	buf[0x1000 - 215] = (unsigned int) Math_Mul_Offset;
	buf[0x1000 - 216] = (unsigned int) Math_Sub_Offset;
	buf[0x1000 - 217] = (unsigned int) Math_Add_Offset;
	buf[0x1000 - 218] = (unsigned int) math_mul_scale;
	buf[0x1000 - 219] = (unsigned int) math_mul_scale;
	buf[0x1000 - 220] = (unsigned int) math_sub_scale;
	buf[0x1000 - 221] = (unsigned int) math_add_scale;
//BF end

	buf[0x1000 - 229] = (unsigned long) Trig_Pos_Mem;
	buf[0x1000 - 230] = (unsigned long) Trig_Pos_Display;
	buf[0x1000 - 231] = (unsigned long) Trig_Pos_Zoom;
	buf[0x1000 - 232] = (unsigned long) MemWinStart;
	
	buf[0x1000 - 233] = (unsigned long) Trig_Pos_Mem_old;
	buf[0x1000 - 234] = (unsigned long) Trig_Pos_Display_old;
	buf[0x1000 - 235] = (unsigned long) Trig_Pos_Zoom_old;
	buf[0x1000 - 236] = (unsigned long) MemWinStart_old;
	
	buf[0x1000 - 237] = (unsigned long) ZFT_float;
	buf[0x1000 - 238] = (unsigned long) ZFT_float_old;
	buf[0x1000 - 239] = (unsigned long) ZFT_float_dmode;
	buf[0x1000 - 240] = (unsigned long) ZFT_float_dmode_old;
	
	buf[0x1000 - 241] = (unsigned long) OFT_float;
	buf[0x1000 - 242] = (unsigned long) OFT_float_old;
	buf[0x1000 - 243] = (unsigned long) ZIT_short;
	buf[0x1000 - 244] = (unsigned long) ZIT_short_old;
	
	buf[0x1000 - 245] = (unsigned long) ZIT_short_dmode;
	buf[0x1000 - 246] = (unsigned long) ZIT_short_dmode_old;
	
	buf[0x1000 - 247] = (unsigned long) test_port_reg;

// BF new ADC-correction values
/*
	//channel 1
	buf[0x1000 - 248] = (unsigned int) ADC_Offset[0][0];
	buf[0x1000 - 249] = (unsigned int) ADC_Offset[0][1];
	buf[0x1000 - 250] = (unsigned int) ADC_Offset[0][2];
	buf[0x1000 - 251] = (unsigned int) ADC_Offset[0][3];
	//channel 2
	buf[0x1000 - 252] = (unsigned int) ADC_Offset[1][0];
	buf[0x1000 - 253] = (unsigned int) ADC_Offset[1][1];
	buf[0x1000 - 254] = (unsigned int) ADC_Offset[1][2];
	buf[0x1000 - 255] = (unsigned int) ADC_Offset[1][3];
	//channel 3
	buf[0x1000 - 256] = (unsigned int) ADC_Offset[2][0];
	buf[0x1000 - 257] = (unsigned int) ADC_Offset[2][1];
	buf[0x1000 - 258] = (unsigned int) ADC_Offset[2][2];
	buf[0x1000 - 259] = (unsigned int) ADC_Offset[2][3];
	//channel 4
	buf[0x1000 - 260] = (unsigned int) ADC_Offset[3][0];
	buf[0x1000 - 261] = (unsigned int) ADC_Offset[3][1];
	buf[0x1000 - 262] = (unsigned int) ADC_Offset[3][2];
	buf[0x1000 - 263] = (unsigned int) ADC_Offset[3][3];
*/
// BF new zerolevels for XY
	buf[0x1000 - 264] = (unsigned int) Virtual_ZeroLevelXYCH1;
	buf[0x1000 - 265] = (unsigned int) Virtual_ZeroLevelXYCH2;
	buf[0x1000 - 266] = (unsigned int) Virtual_ZeroLevelXYCH3;
	buf[0x1000 - 267] = (unsigned int) Virtual_ZeroLevelXYCH4;


// BF end
	buf[0x1000 - 299] = 0x23232323;	
	
	
	
	result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address + 0x3000, buf, 0x1000);	// Flash Buffer	
#ifdef _Debug_
    printf("Save Signal4 = %d\n", result);
#endif

	return result;
}

void AMDFlash::load_signal_from_flash(unsigned long *flash_address)
{
	unsigned long lX, bufdata;
	unsigned long buf[300];	
#ifdef _Debug_	
	printf("Restore a Signal  \n");
#endif	
//printf("AMDFlash::load_signal_from_flash\n");
	for (lX = 0; lX < 300; lX++) buf[lX] = flash_address[0x4000 - lX];		
	
	printf("PARAM : ");
	if ((buf[4] == tc_version) && (buf[5] == tc_subversion) && (buf[6] == tc_dev_version) && (buf[1] == 0xAACC5523) && (buf[2] == 0x22334455) && (buf[299] == 0x23232323))
	{
        Continius = 0;
//BF del        UserIface_active = 0;

        SIGNAL_StartFr_idx = SIGNALFaktor_idx;
        Selected_Voltage_CH1_Old = Selected_Voltage_CH1;
        Selected_Voltage_CH2_Old = Selected_Voltage_CH2;
        Selected_Voltage_CH3_Old = Selected_Voltage_CH3;
        Selected_Voltage_CH4_Old = Selected_Voltage_CH4;
                    	
        Hardware::Stop_Record();	
        
        for (lX = 0; lX < 0x1000; lX++)
        {
            bufdata = *(flash_address + lX + 0x0000);
            SIGNAL1[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
            SIGNAL1[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
            SIGNAL1[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
            SIGNAL1[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);

            bufdata = *(flash_address + lX + 0x1000);
            SIGNAL2[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
            SIGNAL2[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
            SIGNAL2[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
            SIGNAL2[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);

            bufdata = *(flash_address + lX + 0x2000);
            SIGNAL3[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
            SIGNAL3[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
            SIGNAL3[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
            SIGNAL3[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);

            bufdata = *(flash_address + lX + 0x2F00);
            SIGNAL4[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
            SIGNAL4[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
            SIGNAL4[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
            SIGNAL4[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);
        }	
	
        printf("Yes we load parameters\n");
        // Timebase      	
        MenuStatus[10][0] = (short) buf[10];
        MenuStatus[10][1] = (short) buf[11];
        MenuStatus[10][2] = (short) buf[12];
        MenuStatus[10][3] = (short) buf[13];
        MenuStatus[10][4] = (short) buf[14];
        MenuStatus[5][2] = (short) buf[15];
        XY_Mode = (char) buf[16];
        Selected_Timebase = (char) buf[17];
        SIGNALFaktor_idx = (volatile short) buf[18];
        SIGNAL_StartFr_idx = (volatile short) buf[19];

        Selected_Timebase = (unsigned long) buf[20];   //?????
        
        if (Selected_Timebase < 4) Selected_Timebase = 4;

        SIGNALFaktor_idx = (char) buf[21];

        dmode_Selected_Timebase = (char) buf[22];
        dmode_SIGNALFaktor_idx = (volatile short) buf[23];
        dmode_Window_Offset_Pos = (int) buf[24];
        dmode_SIGNAL_StartFr_idx = (volatile short) buf[25];

        // werden frisch berrechnet
        //Cursor_Delayed_1 = (int) buf[26];
        //Cursor_Delayed_2 = (int) buf[27];
        //Cursor_Delayed_Size = (int) buf[28];
        USTB_Mode = (volatile short) buf[29];						//ultra slow timebase mode
        USTB_Mode_bak = (volatile short) buf[30];
//  		ROLLMODE_STARTED = (volatile char) buf[31];
//      	ROLLMODE_Value_rec = (volatile char) buf[32];
        Timebase_Offset_Pos = (int) buf[33];
        SwitchesTB = (unsigned short) buf[34];
	
        // Trigger	
        //MenuStatus[12][0] = (short) buf[35];                                        // Triggering - Auto
        MenuStatus[12][1] = (short) buf[36];                                        // Triggering - Coupling
        MenuStatus[12][2] = (short) buf[37];                                        // Triggering - Reject
        MenuStatus[12][4] = (short) buf[38];                                        // Triggering - External Probe = 1 : 1  ( muß noch als funktion integriert werden )
        MenuStatus[14][3] = (short) buf[39];                                        // Triggering - TV
        MenuStatus[14][0] = (short) buf[40];                                        // Triggering - positive edge
        MenuStatus[14][1] = (short) buf[41];                                        // Triggering - Source = channel 1
        MenuStatus[14][2] = (short) buf[42];                                        // Triggering - no externel triggering
        MenuStatus[15][0] = (short) buf[43];                                        // Triggering - Source Channel 1
        //MenuStatus[15][1] = (short) buf[44];                                        // Triggering - Negative Pulse
        MenuStatus[15][2] = (short) buf[45];                                        // Triggering - bigger - smaller - then
        Selected_Trigger_Source = (char) buf[46];                                  // Selected Trigger Source is Channel 1
        SelectedEdgeExtern = (char) buf[47];                                       // Trigger Edge by extern triggering
        PulseToggle = (char) buf[48];                                              // Pulse width button selection                                       //
        Memory_Position = (long) buf[49];                                          // Memory Position
	
        //Trigger_Offset_Pos = (volatile int) buf[50];     //BF del not used      // Trigger Offset
        Trigger_Pos_CH1  = (int) buf[51];                                          // Trigger Level Channel 1
        Trigger_Pos_CH2  = (int) buf[52];                                          // Trigger Level Channel 2
        Trigger_Pos_CH3  = (int) buf[53];                                          // Trigger Level Channel 3
        Trigger_Pos_CH4  = (int) buf[54];                                          // Trigger Level Channel 4
        trg_val_reg      = (unsigned int) buf[55];                                              // Trigger Value
        trg_hyst_reg     = (unsigned int) buf[56];                                             // Trigger Hysterese
        trig_range_reg   = (unsigned int) buf[57];                                           // Trigger Range
        trig_holdoff_reg = (unsigned long) buf[58];                                         // Trigger Holdoff
        trig_width_reg   = (unsigned int) buf[59];                                           // Trigger Width
	
        ctrl_reg = (unsigned int) buf[60];                                                 // Control register
        adc_ctrl_reg = (unsigned int) buf[61];                                             // ADC Control register
        pre_reg = (unsigned int) buf[62];                                                  // PreTrigger Value
//        channel_Adr_add = (unsigned int) buf[63];                                          // Channel Add Value
//        adc_change_reg = (unsigned long) buf[64];                                           // ADC Change
        triggering = (unsigned int) buf[65];                                               // Triggering
        TriggerWay = (char) buf[66];                                               // Triggerway
	    Trigger_Pos_CHE = (int) buf[67];                                                // Trigger extern value
	
        // Channel 1
        MenuStatus[1][0] = (short) buf[68];                                         // Channel - Coupling
        MenuStatus[1][1] = (short) buf[69];                                         // Channel - BW Linit
        MenuStatus[1][2] = (short) buf[70];                                         // Channel - Invert
        MenuStatus[1][3] = (short) buf[71];                                         // Channel - Probe
        SwitchesCH1 = (unsigned short) buf[72];                                              // Switches for Channel 1 and for the other the AC bit
        Selected_Voltage_CH1 = (volatile char) buf[73];                                     // Voltage

        //BF del DAC_Correction[0][0] = (int) buf[74];                              // Search Zeors offset
        //BF del DAC_Correction[0][1] = (int) buf[75];                              // Search Zeors offset
        //BF del DAC_Correction[0][2] = (int) buf[76];                              // Search Zeors offset

        CH1_DAC_1 = (unsigned char) buf[77];                                                // channel dac 1 offset
        CH1_DAC_2 = (unsigned char) buf[78];                                                // channel dac 2 offset
        CH1_DAC_3 = (unsigned char) buf[79];                                                // channel dac 3 offset
        CH1_DAC_4 = (unsigned char) buf[80];                                                // channel dac 4 offset

        signal_zoomed1       = (unsigned char) buf[81];                                           // signal zoomed
        CH1_DAC_Offset       = (int) buf[82];                                           // Channel DAC
        ZeroLevelCH1         = (int) buf[83];                                             // ZeroLevel
        Virtual_ZeroLevelCH1 = (int) buf[84];                                     // Virtual ZeroLevel
        //alpha[0]             = (int) buf[85];                                          // ADC-alpha correction -> BF changed
        //CH1_Zero_Offset    = (int) buf[85];                                          // Shift offset when stopped

	 // Channel 2
        MenuStatus[2][0] = (short) buf[86];                                         // Channel - Coupling
        MenuStatus[2][1] = (short) buf[87];                                         // Channel - BW Linit
        MenuStatus[2][2] = (short) buf[88];                                         // Channel - Invert
        MenuStatus[2][3] = (short) buf[89];                                         // Channel - Probe

        SwitchesCH2 = (unsigned short) buf[90];                                              // Switches for Channel 2
        Selected_Voltage_CH2 = (volatile char) buf[91];                                     // Voltage
        //BF del DAC_Correction[1][0] = (int) buf[92];                              // Search Zeors offset
        //BF del DAC_Correction[1][1] = (int) buf[93];                              // Search Zeors offset
        //BF del DAC_Correction[1][2] = (int) buf[94];                              // Search Zeors offset
        CH2_DAC_1 = (unsigned char) buf[95];                                                // channel dac 1 offset
        CH2_DAC_2 = (unsigned char) buf[96];                                                // channel dac 2 offset
        CH2_DAC_3 = (unsigned char) buf[97];                                                // channel dac 3 offset
        CH2_DAC_4 = (unsigned char) buf[98];                                                // channel dac 4 offset
        signal_zoomed2 = (unsigned char) buf[99];                                           // signal zoomed

        CH2_DAC_Offset       = (int) buf[100];                                           // Channel DAC
        ZeroLevelCH2         = (int) buf[101];                                             // ZeroLevel
        Virtual_ZeroLevelCH2 = (int) buf[102];                                     // Virtual ZeroLevel
        //alpha[1]             = (int) buf[103];                                          // ADC-alpha correction -> BF changed
        //CH2_Zero_Offset    = (int) buf[103];                                          // Shift offset when stopped

        // Channel 3
        MenuStatus[3][0] = (short) buf[104];                                         // Channel - Coupling
        MenuStatus[3][1] = (short) buf[105];                                         // Channel - BW Linit
        MenuStatus[3][2] = (short) buf[106];                                         // Channel - Invert
        MenuStatus[3][3] = (short) buf[107];                                         // Channel - Probe
        SwitchesCH3 = (unsigned short) buf[108];                                              // Switches for Channel 3
        Selected_Voltage_CH3 = (volatile char) buf[109];                                     // Voltage

        //BF del DAC_Correction[2][0] = (int) buf[110];                              // Search Zeors offset
        //BF del DAC_Correction[2][1] = (int) buf[111];                              // Search Zeors offset
        //BF del DAC_Correction[2][2] = (int) buf[112];                              // Search Zeors offset
        CH3_DAC_1 = (unsigned char) buf[113];                                                // channel dac 1 offset
        CH3_DAC_2 = (unsigned char) buf[114];                                                // channel dac 2 offset
        CH3_DAC_3 = (unsigned char) buf[115];                                                // channel dac 3 offset
        CH3_DAC_4 = (unsigned char) buf[116];                                                // channel dac 4 offset
        signal_zoomed3 = (unsigned char) buf[117];                                           // signal zoomed
        CH3_DAC_Offset = (int) buf[118];                                           // Channel DAC
        ZeroLevelCH3 = (int) buf[119];                                             // ZeroLevel
        Virtual_ZeroLevelCH3 = (int) buf[120];                                     // Virtual ZeroLevel
        //alpha[2]             = (int) buf[121];                                          // ADC-alpha correction -> BF changed
        //CH3_Zero_Offset = (int) buf[121];                                          // Shift offset when stopped

        // Channel 4
        MenuStatus[4][0] = (short) buf[122];                                         // Channel - Coupling
        MenuStatus[4][1] = (short) buf[123];                                         // Channel - BW Linit
        MenuStatus[4][2] = (short) buf[124];                                         // Channel - Invert
        MenuStatus[4][3] = (short) buf[125];                                         // Channel - Probe
        SwitchesCH4 = (unsigned short) buf[126];                                              // Switches for Channel 4
        Selected_Voltage_CH4 = (volatile char) buf[127];                                     // Voltage
        //BF del DAC_Correction[3][0] = (int) buf[128];                              // Search Zeors offset
        //BF del DAC_Correction[3][1] = (int) buf[129];                              // Search Zeors offset
        //BF del DAC_Correction[3][2] = (int) buf[130];                              // Search Zeors offset
        CH4_DAC_1 = (unsigned char) buf[131];                                                // channel dac 1 offset
        CH4_DAC_2 = (unsigned char) buf[132];                                                // channel dac 2 offset
        CH4_DAC_3 = (unsigned char) buf[133];                                                // channel dac 3 offset
        CH4_DAC_4 = (unsigned char) buf[134];                                                // channel dac 4 offset
        signal_zoomed4       = (unsigned char) buf[135];                                           // signal zoomed
        CH4_DAC_Offset       = (int) buf[136];                                           // Channel DAC
        ZeroLevelCH4         = (int) buf[137];                                             // ZeroLevel
        Virtual_ZeroLevelCH4 = (int) buf[138];                                     // Virtual ZeroLevel
        //alpha[3]             = (int) buf[139];                                          // ADC-alpha correction -> BF changed
        //CH4_Zero_Offset = (int) buf[139];                                          // Shift offset when stopped
         /*
        Channel_Math_Active = (char) buf[140];                                      // Math channel off
        MenuStatus[26][1] = (short) buf[141];                                        // standart math fft on
        MenuStatus[26][2] = (short) buf[142];                                        // standart math 1*2 of
        MenuStatus[26][3] = (short) buf[143];                                        // standart math 1-2 off
        MenuStatus[26][4] = (short) buf[144];                                        // standart math 1+2 off
         */
    /* Buffer für Math Settings
        buf[145] = (unsigned long) fft source
        buf[146] = (unsigned long) fft Span
        buf[147] = (unsigned long) fft Center
        buf[148] = (unsigned long) fft Preset
        buf[149] = (unsigned long) fft more Scale

        buf[150] = (unsigned long) fft more Offset
        buf[151] = (unsigned long) fft more Window
        buf[152] = (unsigned long) * - + Scale
        buf[153] = (unsigned long) * - + Offset  */

        // Cursors    	
        MenuStatus[16][0] = (short) buf[154];                                        // Cursor soruce
        Cursor_Enabled = (char) buf[155];

        if (Cursor_Enabled) Cursor_Data_First_Draw = 1;

        Cursor_Both_Selected = (char) buf[156];
        SelectedCursor = (char) buf[157];
        Cursor_Horizontal_Active = (char) buf[158];
        Cursor_Horizontal_Position_Real1 = (long) buf[159];

        Cursor_Horizontal_Position_Real2 = (long) buf[160];
        Cursor_Vertical_Active = (char) buf[161];
        Cursor_Vertical_Position_Real1 = (long) buf[162];
        Cursor_Vertical_Position_Real2 = (long) buf[163];

        // Quick Measure
        //QM_Enabled = (char) buf[164];

        if (QM_Enabled) Quick_Measure_First_Draw = 1;

        QM_Type[0] = (char) buf[165];
        QM_Type[1] = (char) buf[166];
        QM_Type[2] = (char) buf[167];
        QM_Channel[0] = (char) buf[168];
        QM_Channel[1] = (char) buf[169];

        QM_Channel[2] = (char) buf[170];
        QM_Horizontal_Active = (char) buf[171];
        QM_Vertical_Active = (char) buf[172];
        Quick_Measure_Threshold_btn_select = (char) buf[173];

        // Display
        GridColorArrayPos = (unsigned char) buf[174];
        GridColor_Val = (unsigned char) buf[175];
        GridColorArrayPos_Delayed = (unsigned char) buf[176];
        //BF not used GridColor_Val_Delayed = (unsigned char) buf[177];
        MenuStatus[5][0] = (short) buf[178];                                         // Display - Persist
        MenuStatus[5][3] = (short) buf[179];                                         // Display - Verctors

//        MenuStatus[0][0] = (short) buf[180];                                         // Aquire - Normal
//        MenuStatus[0][1] = (short) buf[181];                                         // Aquire - Averaging
//        MenuStatus[0][2] = (short) buf[183];                                         // Aquire - Avgs
        MenuStatus[5][4] = (short) buf[184];                                         // Display - Browse

        MenuStatus[23][1] = (short) buf[185];                                        // Autoscale - Slow TB
        //???? = (short) buf[186];

        Debug_Mode = (unsigned char) buf[187];
        ADC_Debug_Mode = (unsigned char) buf[188];

        // vergessene
        MenuPopupStatus[8][0] = (short) (buf[189] & 0x0000FFFF);                    // popup for externeal trigger
        MenuPopupStatus[8][1] = (short) ((buf[189] & 0xFFFF0000) >> 16);
        MenuPopupStatus[8][2] = (short) buf[190];
 
        OneShotRestart = (unsigned char) buf[191];
        // 192 s.o. ledval
        Channel_1_Active = (char) buf[193];
        Channel_2_Active = (char) buf[194];
        Channel_3_Active = (char) buf[195];
        Channel_4_Active = (char) buf[196];
        dmode_SIGNAL_OFFSET = (volatile short) buf[197];
        SIGNAL_OFFSET = (volatile short) buf[198];

        HoldOff_Value = (int) buf[199];
        HoldOff_Expo  = (int) buf[200];
        HoldOff_Steps = (int) buf[201];

        Pulse11_Value = (int) buf[202];
        Pulse11_Expo  = (int) buf[203];
        Pulse11_Steps = (int) buf[204];

        Pulse12_Value = (int) buf[205];
        Pulse12_Expo  = (int) buf[206];
        Pulse12_Steps = (int) buf[207];

        Pulse21_Value = (int) buf[208];
        Pulse21_Expo  = (int) buf[209];
        Pulse21_Steps = (int) buf[210];

        Pulse22_Value = (int) buf[211];
        Pulse22_Expo  = (int) buf[212];
        Pulse22_Steps = (int) buf[213];

//BF new Math values
	Math_Offset     = (int) buf[214]; 
        Math_Mul_Offset = (int) buf[215]; 
        Math_Sub_Offset = (int) buf[216];
        Math_Add_Offset = (int) buf[217];
	math_scale      = (int) buf[218];
        math_mul_scale  = (int) buf[219];
        math_sub_scale  = (int) buf[220];
        math_add_scale  = (int) buf[221];
//BF end

        Trig_Pos_Mem     = (int) buf[229];
        Trig_Pos_Display = (int) buf[230];
        Trig_Pos_Zoom    = (float) buf[231];
        MemWinStart      = (int) buf[232];
        
        Trig_Pos_Mem_old = (int) buf[233];
        Trig_Pos_Display_old = (int) buf[234];
        Trig_Pos_Zoom_old = (float) buf[235];
        MemWinStart_old = (int) buf[236];
        
        ZFT_float = (float) buf[237];
        ZFT_float_old = (float) buf[238];
        ZFT_float_dmode = (float) buf[239];
        ZFT_float_dmode_old = (float) buf[240];
        
        OFT_float = (float) buf[241];
        OFT_float_old = (float) buf[242];
        
        ZIT_short = (short) buf[243];
        ZIT_short_old = (short) buf[244];
        ZIT_short_dmode = (short) buf[245];
        ZIT_short_dmode_old = (short) buf[246];
        
        test_port_reg = (unsigned int) buf[247];

// BF new ADC-correction values
/*
	//channel 1
	ADC_Offset[0][0] = (int)buf[248];
	ADC_Offset[0][1] = (int)buf[249];
	ADC_Offset[0][2] = (int)buf[250];
	ADC_Offset[0][3] = (int)buf[251];
	//channel 2
	ADC_Offset[1][0] = (int)buf[252];
	ADC_Offset[1][1] = (int)buf[253];
	ADC_Offset[1][2] = (int)buf[254];
	ADC_Offset[1][3] = (int)buf[255];
	//channel 3
	ADC_Offset[2][0] = (int)buf[256];
	ADC_Offset[2][1] = (int)buf[257];
	ADC_Offset[2][2] = (int)buf[258];
	ADC_Offset[2][3] = (int)buf[259];
	//channel 4
	ADC_Offset[3][0] = (int)buf[260];
	ADC_Offset[3][1] = (int)buf[261];
	ADC_Offset[3][2] = (int)buf[262];
	ADC_Offset[3][3] = (int)buf[263];
*/

// BF new zerolevels for XY
	Virtual_ZeroLevelXYCH1 = (int)buf[264];
	Virtual_ZeroLevelXYCH2 = (int)buf[265];
	Virtual_ZeroLevelXYCH3 = (int)buf[266];
	Virtual_ZeroLevelXYCH4 = (int)buf[267];



// BF end




	timebase_reg = tb_value[Selected_Timebase];
	Display_Timebase = Selected_Timebase + SIGNALFaktor_idx;

        LED_ON[0] = 0;
        LED_ON[1] = 0;
        LED_ON[2] = 0;
        LED_ON[3] = 0;
        LED_ON[4] = 0;
        //LED_ON[5] = 0;
        LED_ON[6] = 0;
        LED_ON[7] = 0;
        LED_ON[8] = 0;
        LED_ON[9] = 0;
        LED_ON[10] = 0;
        LED_ON[11] = 0;
        //LED_ON[12] = 0;
        //LED_ON[13] = 0;
        LED_ON[14] = 0;
        LED_ON[15] = 0;

        if (((int) buf[192] & 0x0001) == 0x0001) LED_ON[0] = 1;
        if (((int) buf[192] & 0x0002) == 0x0002) LED_ON[1] = 1;
        if (((int) buf[192] & 0x0004) == 0x0004) LED_ON[2] = 1;
        if (((int) buf[192] & 0x0008) == 0x0008) LED_ON[3] = 1;
//        if (((int) buf[192] & 0x0010) == 0x0010) LED_ON[4] = 1;  // Math
        //if (((int) buf[192] & 0x0020) == 0x0020) LED_ON[5] = 1; // Generall
        if (((int) buf[192] & 0x0040) == 0x0040) LED_ON[6] = 1;
        //if (((int) buf[192] & 0x0080) == 0x0080) LED_ON[7] = 1; // Quick Measure
        if (((int) buf[192] & 0x0100) == 0x0100) LED_ON[8] = 1;
        if (((int) buf[192] & 0x0200) == 0x0200) LED_ON[9] = 1;
        if (((int) buf[192] & 0x0400) == 0x0400) LED_ON[10] = 1;
        if (((int) buf[192] & 0x0800) == 0x0800) LED_ON[11] = 1;
        //if (((int) buf[192] & 0x1000) == 0x1000) LED_ON[12] = 1; // Run/Stop Green
        //if (((int) buf[192] & 0x2000) == 0x2000) LED_ON[13] = 1; // Run/Stop Red
//        if (((int) buf[192] & 0x4000) == 0x4000) LED_ON[14] = 1;
//        if (((int) buf[192] & 0x8000) == 0x8000) LED_ON[15] = 1;	


        switch(Pulse11_Expo)
        {
            case 1:
            {
                Pulse11.Write_Value((float) Pulse11_Value / (float) 1000000000);
                break;
            }
            case 2:
            {
                Pulse11.Write_Value((float) Pulse11_Value / (float) 1000000);
                break;
            }
            case 3:
            {
                Pulse11.Write_Value((float) Pulse11_Value / (float) 1000);
                break;
            }
        }
        Pulse11.Write_StepSize(Pulse11_Steps);
        Pulse11.RenderText();

        switch(Pulse12_Expo)
        {
            case 1:
            {
                Pulse12.Write_Value((float) Pulse12_Value / (float) 1000000000);
                break;
            }
            case 2:
            {
                Pulse12.Write_Value((float) Pulse12_Value / (float) 1000000);
                break;
            }
            case 3:
            {
                Pulse12.Write_Value((float) Pulse12_Value / (float) 1000);
                break;
            }
        }
        Pulse12.Write_StepSize(Pulse12_Steps);
        Pulse12.RenderText();

        switch(Pulse21_Expo)
        {
            case 1:
            {
                Pulse21.Write_Value((float) Pulse21_Value / (float) 1000000000);
                break;
            }
            case 2:
            {
                Pulse21.Write_Value((float) Pulse21_Value / (float) 1000000);
                break;
            }
            case 3:
            {
                Pulse21.Write_Value((float) Pulse21_Value / (float) 1000);
                break;
            }
        }
        Pulse21.Write_StepSize(Pulse21_Steps);
        Pulse21.RenderText();

        switch(Pulse22_Expo)
        {
            case 1:
            {
                Pulse22.Write_Value((float) Pulse22_Value / (float) 1000000000);
                break;
            }
            case 2:
            {
                Pulse22.Write_Value((float) Pulse22_Value / (float) 1000000);
                break;
            }
            case 3:
            {
                Pulse22.Write_Value((float) Pulse22_Value / (float) 1000);
                break;
            }
        }
        Pulse22.Write_StepSize(Pulse22_Steps);
        Pulse22.RenderText();


//BF changed
	//Math_Offset_mul.Write_StepSize(1);
//	Math_Offset_mul.Write_Value((float)(Math_Mul_Offset / 50000 ));
//	Math_Offset_mul.RenderText();


	//Math_Offset_sub.Write_StepSize(1);
//	Math_Offset_sub.Write_Value((float)(Math_Sub_Offset / 50000 ));
//	Math_Offset_sub.RenderText();


	//Math_Offset_add.Write_StepSize(0.001);
//	Math_Offset_add.Write_Value((float)(Math_Add_Offset / 50000 ));
//	Math_Offset_add.RenderText();
//BF end	


        // einrücken
        Hardware::UpdateChannel(1, Channel_1_Active);
        Hardware::UpdateChannel(2, Channel_2_Active);
				
        VoltageChangedCh1 = 1;
        VoltageChangedCh2 = 1;				
				
        if (Channel_1_Active) LED_ON[0] = 1;
        else LED_ON[0] = 0;
        if (Channel_2_Active) LED_ON[1] = 1;
        else LED_ON[1] = 0;

        LED_ON[5] = 1;
        LED_ON[6] = 0;
        
        // Run/Stop LED Red
        LED_ON[12] = 0;
        LED_ON[13] = 1;

        Rotary_Steps = 0;
        Hardware::ON_Zero_Channel_1();
        Rotary_Steps = 0;
        Hardware::ON_Zero_Channel_2();

        Hardware::SetSwitches(1, Selected_Voltage_CH1);
        Hardware::SetSwitches(2, Selected_Voltage_CH2);

        if (NumberOfChannels > 2)
        {
            Hardware::UpdateChannel(3, Channel_3_Active);
            Hardware::UpdateChannel(4, Channel_4_Active);

            VoltageChangedCh3 = 1;
            VoltageChangedCh4 = 1;
				
            if (Channel_2_Active) LED_ON[2] = 1;
            else LED_ON[2] = 0;
            if (Channel_4_Active) LED_ON[3] = 1;
            else LED_ON[3] = 0;
				
            Rotary_Steps = 0;
            Hardware::ON_Zero_Channel_3();
            Rotary_Steps = 0;
            Hardware::ON_Zero_Channel_4();

            Hardware::SetSwitches(3, Selected_Voltage_CH3);
            Hardware::SetSwitches(4, Selected_Voltage_CH4);

        }
        else
        {
            Hardware::UpdateChannel(3, 0);
            Hardware::UpdateChannel(4, 0);
        }

        TriggerLevelChanged = 1;
        TriggerModeChanged = 1;	
        TriggerWayChanged = 1;
        TimebaseChanged = 1;
        TimeOffsetChanged = 1;
        MenuStatusChanged = 1;
        MenuAktive = 1;

        Cursor_Data_First_Draw = 0;
        Quick_Measure_First_Draw = 0;

        Hardware::Recalc_Vars();
        Hardware::UpdateTrigger(31);
        Display::CALCPRETRIGGER();
        Display::DRAW_ALL_CURSOR();
				
        // remove the cursor data
        for (lX = 408 * lX; lX < 430 * 20; lX++)
        {
            *(UI_Plane1 + lX) = 0;	// delete
            *(Buffer_UI2Plane + lX) = 0;	// delete Text				
            *(UI_Plane4 + lX) = 0;	// delete
            *(UI_Plane5 + lX) = 0;	// delete
        }				

        Display::DRAWSQUAREBUTTON(2, 1, 14, 16, btnPushed, btnDraw);							// Button
        Display::TEXTOUTvbig("1", 6, 4, 1, Channel_Plane1);									// Draw Voltage Text
        Display::TEXTOUTvbig("1", 6, 4, 1, Buffer_Plane1);									// Draw Voltage Text

        Display::DRAWSQUAREBUTTON(78, 1, 14, 16, btnPushed, btnDraw);						// Button
        Display::TEXTOUTvbig("2", 82, 4, 1, Channel_Plane2);									// Draw Voltage Text
        Display::TEXTOUTvbig("2", 82, 4, 1, Buffer_Plane2);									// Draw Voltage Text

        Display::StatusUpdate();

        if (NumberOfChannels > 2)
        {
            Display::DRAWSQUAREBUTTON(154, 1, 14, 16, btnPushed, btnDraw);						// Button
            Display::TEXTOUTvbig("3", 158, 4, 1, Channel_Plane3);								// Draw Voltage Text
            Display::TEXTOUTvbig("3", 158, 4, 1, Buffer_Plane3);								// Draw Voltage Text

            Display::DRAWSQUAREBUTTON(230, 1, 14, 16, btnPushed, btnDraw);						// Button		
            Display::TEXTOUTvbig("4", 234, 4, 1, Channel_Plane4);								// Draw Voltage Text
            Display::TEXTOUTvbig("4", 234, 4, 1, Buffer_Plane4);								// Draw Voltage Text
        }
                    				
        Display::DRAWSTATUS(6, 0);
				
        Hardware::Send_LED();		

        Continius = false;
        //Signal_Loaded = true;		//BF not used
	}
}

void AMDFlash::copy_flash_to_planes(void)
{
	unsigned long lX;
#ifdef _Debug_	
	printf("Restore the planes \n");
#endif

//printf("AMDFlash::copy_flash_to_planes\n");	
	for (lX = 0; lX < 9600; lX++)
	{	
		UI_Plane1[lX] = *(UI_Plane1_Flash + lX);	
		Buffer_UI2Plane[lX] = *(UI_Plane2_Flash + lX);	
		UI_Plane3[lX] = *(UI_Plane3_Flash + lX);	
		UI_Plane4[lX] = *(UI_Plane4_Flash + lX);	
		UI_Plane5[lX] = *(UI_Plane5_Flash + lX);	
		Channel_Plane1[lX] = *(Channel_Plane1_Flash + lX);	
		Channel_Plane2[lX] = *(Channel_Plane2_Flash + lX);	
		Channel_Plane3[lX] = *(Channel_Plane3_Flash + lX);	
		Channel_Plane4[lX] = *(Channel_Plane4_Flash + lX);	
		Channel_Math_Plane[lX] = *(Channel_Math_Plane_Flash + lX);	
		Buffer_Plane1[lX] = *(Channel_Plane1_Flash + lX);	
		Buffer_Plane2[lX] = *(Channel_Plane2_Flash + lX);	
		Buffer_Plane3[lX] = *(Channel_Plane3_Flash + lX);	
		Buffer_Plane4[lX] = *(Channel_Plane4_Flash + lX);	
		Buffer_PlaneM[lX] = *(Channel_Math_Plane_Flash + lX);			
		Memory_Plane1[lX] = *(Memory_Plane1_Flash + lX);	
		Memory_Plane2[lX] = *(Memory_Plane2_Flash + lX);	
		Memory_Plane3[lX] = *(Memory_Plane3_Flash + lX);	
		Marker_Plane1[lX] = *(Marker_Plane1_Flash + lX);	
		Marker_Plane2[lX] = *(Marker_Plane2_Flash + lX);	
		Grid_Plane[lX] = *(Grid_Plane_Flash + lX);	
	}
	UpdateMenuTextPlane = true;
}

void AMDFlash::copy_flash_to_plane(unsigned long *flash_address, unsigned long *plane)
{
	unsigned long lX;
//printf("AMDFlash::copy_flash_to_plane\n");		
	for (lX = 0; lX < 9600; lX++)
	{	
		plane[lX] = flash_address[lX];	
	}
}
//#####################################################################################################################################################
void AMDFlash::Write_Config_Flash(void)
{
	unsigned long *flash_address;
	unsigned long buf[300];
	int led_value = 0x0000;

	
//	if (AutoTimerOff) return;

/* BF del
	// If UI interrupt occured don't write to flash
	if (UI_request)
	{
		UI_request = 0;
		return;
	}
*/

	if (config_changed == false) return;
	config_changed = false;

printf("AMDFlash::Write_Config_Flash\n");

	if (LED_ON[0] == 1) led_value = led_value | 0x0001;		// Select Channel 1
	if (LED_ON[1] == 1) led_value = led_value | 0x0002;		// Select Channel 2
	if (LED_ON[2] == 1) led_value = led_value | 0x0004;		// Select Channel 3
	if (LED_ON[3] == 1) led_value = led_value | 0x0008;		// Select Channel 4
	if (LED_ON[4] == 1) led_value = led_value | 0x0010;		// Select Channel Math
	if (LED_ON[5] == 1) led_value = led_value | 0x0020;		// General
	if (LED_ON[6] == 1) led_value = led_value | 0x0040;		// Cursor						
	//if (LED_ON[7] == 1) led_value = led_value | 0x0080;		// Quick Measure
	if (LED_ON[8] == 1) led_value = led_value | 0x0100;		// Edge
	if (LED_ON[9] == 1) led_value = led_value | 0x0200;		// Pattern - Not Used
	if (LED_ON[10] == 1) led_value = led_value | 0x0400;		// Pulse Width
	if (LED_ON[11] == 1) led_value = led_value | 0x0800;		// More - Not Used
	if (LED_ON[12] == 1) led_value = led_value | 0x1000;		// Run/Stop Green
	if (LED_ON[13] == 1) led_value = led_value | 0x2000;		// Run/Stop Red
	if (LED_ON[14] == 1) led_value = led_value | 0x4000;		// Single Red					
	if (LED_ON[15] == 1) led_value = led_value | 0x8000;		// Single Green	

	flash_address = (unsigned long) nasys_main_flash + Config_Flash;
	nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address);

	buf[0] = (unsigned long) 0x00;
	buf[1] = (unsigned long) 0xAACC5523;		// Configuration Start Sign
	buf[2] = (unsigned long) 0x22334455;
	buf[3] = (unsigned long) 0x00;		
	buf[4] = (unsigned long) tc_version;		// Quattro Version
	buf[5] = (unsigned long) tc_subversion;		// SubVerision
	buf[6] = (unsigned long) tc_dev_version;		// Revision
	buf[7] = (unsigned long) tc_dev_subversion;	// Compile Status
	buf[8] = (unsigned long) 0x00;   	
	
	// Timebase
	buf[10] = (unsigned long) MenuStatus[MENU_TIMEBASE][0];		//Main
	buf[11] = (unsigned long) MenuStatus[MENU_TIMEBASE][1];		//Delayed
	buf[12] = (unsigned long) MenuStatus[MENU_TIMEBASE][2];		//XY mode
	buf[13] = (unsigned long) MenuStatus[MENU_TIMEBASE][3];		//Roll mode
	buf[14] = (unsigned long) MenuStatus[MENU_TIMEBASE][4];		//Shift mode
	buf[15] = (unsigned long) MenuStatus[MENU_DISPLAY][2];	
	buf[16] = (unsigned long) XY_Mode;
	buf[17] = (unsigned long) Selected_Timebase;	
	buf[18] = (unsigned long) SIGNALFaktor_idx;
	buf[19] = (unsigned long) SIGNAL_StartFr_idx;

	buf[20] = (unsigned long) Selected_Timebase;
	buf[21] = (unsigned long) SIGNALFaktor_idx;
	buf[22] = (unsigned long) dmode_Selected_Timebase;
	buf[23] = (unsigned long) dmode_SIGNALFaktor_idx;
	buf[24] = (unsigned long) dmode_Window_Offset_Pos;
	buf[25] = (unsigned long) dmode_SIGNAL_StartFr_idx;
//	buf[26] = (unsigned long) Cursor_Delayed_1;
//	buf[27] = (unsigned long) Cursor_Delayed_2;
//	buf[28] = (unsigned long) Cursor_Delayed_Size;
	buf[29] = (unsigned long) USTB_Mode;							//ultra slow timebase mode
	buf[30] = (unsigned long) USTB_Mode_bak;
//    		buf[31] = (unsigned long) ROLLMODE_STARTED;
//  		buf[32] = (unsigned long) ROLLMODE_Value_rec;	
    	buf[33] = (unsigned long) Timebase_Offset_Pos;
	buf[34] = (unsigned long) SwitchesTB;	
	
	// Trigger
	//buf[35] = (unsigned long) MenuStatus[12][0];                                        // Triggering - Auto
	buf[36] = (unsigned long) MenuStatus[12][1];                                        // Triggering - Coupling DC
	buf[37] = (unsigned long) MenuStatus[12][2];                                        // Triggering - Reject
	buf[38] = (unsigned long) MenuStatus[12][4];                                        // Triggering - External Probe = 1 : 1  ( muß noch als funktion integriert werden )
	buf[39] = (unsigned long) MenuStatus[14][3];                                        // Triggering - TV

	buf[40] = (unsigned long) MenuStatus[14][0];                                        // Triggering - positive edge
	buf[41] = (unsigned long) MenuStatus[14][1];                                        // Triggering - Source = channel 1
	buf[42] = (unsigned long) MenuStatus[14][2];                                        // Triggering - no externel triggering
	buf[43] = (unsigned long) MenuStatus[15][0];                                        // Triggering - Source Channel 1
	buf[44] = (unsigned long) MenuStatus[15][1];                                        // Triggering - Negative Pulse
	buf[45] = (unsigned long) MenuStatus[15][2];                                        // Triggering - bigger - smaller - then
	buf[46] = (unsigned long) Selected_Trigger_Source;                                  // Selected Trigger Source is Channel 1
	buf[47] = (unsigned long) SelectedEdgeExtern;                                       // Trigger Edge by extern triggering
	buf[48] = (unsigned long) PulseToggle;                                              // Pulse width button selection
	buf[49] = (unsigned long) Memory_Position;                                          // Memory Position

	//buf[50] = (unsigned long) Trigger_Offset_Pos;		//BF del not used           // Trigger Offset
	buf[51] = (unsigned long) Trigger_Pos_CH1;                                          // Trigger Level Channel 1
	buf[52] = (unsigned long) Trigger_Pos_CH2;                                          // Trigger Level Channel 2
	buf[53] = (unsigned long) Trigger_Pos_CH3;                                          // Trigger Level Channel 3
	buf[54] = (unsigned long) Trigger_Pos_CH4;                                          // Trigger Level Channel 4
	buf[55] = (unsigned long) trg_val_reg;                                              // Trigger Value
	buf[56] = (unsigned long) trg_hyst_reg;                                             // Trigger Hysterese
	buf[57] = (unsigned long) trig_range_reg;                                           // Trigger Range
	buf[58] = (unsigned long) trig_holdoff_reg;                                         // Trigger Holdoff
	buf[59] = (unsigned long) trig_width_reg;                                           // Trigger Width

	buf[60] = (unsigned long) ctrl_reg;                                                 // Control register
	buf[61] = (unsigned long) adc_ctrl_reg;                                             // ADC Control register
	buf[62] = (unsigned long) pre_reg;                                                  // PreTrigger Value
//	buf[63] = (unsigned long) channel_Adr_add;                                          // Channel Add Value
	//buf[64] = (unsigned long) adc_change_reg;                                           // ADC Change
	buf[65] = (unsigned long) triggering;                                               // Triggering
	buf[66] = (unsigned long) TriggerWay;                                               // Triggerway
	buf[67] = (unsigned long) Trigger_Pos_CHE;                                          // Trigger extern value
	
	// Channel 1
	buf[68] = (unsigned long) MenuStatus[1][0];                                         // Channel - Coupling
	buf[69] = (unsigned long) MenuStatus[1][1];                                         // Channel - BW Linit
	buf[70] = (unsigned long) MenuStatus[1][2];                                         // Channel - Invert
	buf[71] = (unsigned long) MenuStatus[1][3];                                         // Channel - Probe
	
	buf[72] = (unsigned long) SwitchesCH1;                                              // Switches for Channel 1 and for the other the AC bit
	buf[73] = (unsigned long) Selected_Voltage_CH1;                                     // Voltage 5V
	buf[74] = (unsigned long) DAC_Correction[0][0];                              // Search Zeors offset
	buf[75] = (unsigned long) DAC_Correction[0][1];                              // Search Zeors offset
	buf[76] = (unsigned long) DAC_Correction[0][2];                              // Search Zeors offset
	buf[77] = (unsigned long) CH1_DAC_1;                                                // channel dac 1 offset
	buf[78] = (unsigned long) CH1_DAC_2;                                                // channel dac 2 offset
	buf[79] = (unsigned long) CH1_DAC_3;                                                // channel dac 3 offset
	buf[80] = (unsigned long) CH1_DAC_4;                                                // channel dac 4 offset
	buf[81] = (unsigned long) signal_zoomed1;                                           // signal zoomed
	
	buf[82] = (unsigned long) CH1_DAC_Offset;                                           // Channel DAC
	buf[83] = (unsigned long) ZeroLevelCH1;                                             // ZeroLevel
	buf[84] = (unsigned long) Virtual_ZeroLevelCH1;                                     // Virtual ZeroLevel
        //buf[85] = (unsigned long) alpha[0];                                               // ADC-alpha correction -> BF changed
    	//buf[85] = (unsigned long) CH1_Zero_Offset;                                          // Shift offset when stopped

    // Channel 2       	
	buf[86] = (unsigned long) MenuStatus[2][0];                                         // Channel - Coupling
	buf[87] = (unsigned long) MenuStatus[2][1];                                         // Channel - BW Linit
	buf[88] = (unsigned long) MenuStatus[2][2];                                         // Channel - Invert
	buf[89] = (unsigned long) MenuStatus[2][3];                                         // Channel - Probe

	buf[90] = (unsigned long) SwitchesCH2;                                              // Switches for Channel
	buf[91] = (unsigned long) Selected_Voltage_CH2;                                     // Voltage 5V
	buf[92] = (unsigned long) DAC_Correction[1][0];                              // Search Zeors offset
	buf[93] = (unsigned long) DAC_Correction[1][1];                              // Search Zeors offset
	buf[94] = (unsigned long) DAC_Correction[1][2];                              // Search Zeors offset
	buf[95] = (unsigned long) CH2_DAC_1;                                                // channel dac 1 offset
	buf[96] = (unsigned long) CH2_DAC_2;                                                // channel dac 2 offset
	buf[97] = (unsigned long) CH2_DAC_3;                                                // channel dac 3 offset
	buf[98] = (unsigned long) CH2_DAC_4;                                                // channel dac 4 offset
	buf[99] = (unsigned long) signal_zoomed2;                                           // signal zoomed

	buf[100] = (unsigned long) CH2_DAC_Offset;                                           // Channel DAC
	buf[101] = (unsigned long) ZeroLevelCH2;                                             // ZeroLevel
	buf[102] = (unsigned long) Virtual_ZeroLevelCH2;                                     // Virtual ZeroLevel
        //buf[103] = (unsigned long) alpha[1];                                               // ADC-alpha correction -> BF changed
    //buf[103] = (unsigned long) CH2_Zero_Offset;                                          // Shift offset when stopped

    // Channel 3
	buf[104] = (unsigned long) MenuStatus[3][0];                                         // Channel - Coupling
	buf[105] = (unsigned long) MenuStatus[3][1];                                         // Channel - BW Linit
	buf[106] = (unsigned long) MenuStatus[3][2];                                         // Channel - Invert
	buf[107] = (unsigned long) MenuStatus[3][3];                                         // Channel - Probe
	buf[108] = (unsigned long) SwitchesCH3;                                              // Switches for Channel
	buf[109] = (unsigned long) Selected_Voltage_CH3;                                     // Voltage 5V

	buf[110] = (unsigned long) DAC_Correction[2][0];                              // Search Zeors offset
	buf[111] = (unsigned long) DAC_Correction[2][1];                              // Search Zeors offset
	buf[112] = (unsigned long) DAC_Correction[2][2];                              // Search Zeors offset
	buf[113] = (unsigned long) CH3_DAC_1;                                                // channel dac 1 offset
	buf[114] = (unsigned long) CH3_DAC_2;                                                // channel dac 2 offset
	buf[115] = (unsigned long) CH3_DAC_3;                                                // channel dac 3 offset
	buf[116] = (unsigned long) CH3_DAC_4;                                                // channel dac 4 offset
	buf[117] = (unsigned long) signal_zoomed3;                                           // signal zoomed
	buf[118] = (unsigned long) CH3_DAC_Offset;                                           // Channel DAC
	buf[119] = (unsigned long) ZeroLevelCH3;                                             // ZeroLevel
	buf[120] = (unsigned long) Virtual_ZeroLevelCH3;                                     // Virtual ZeroLevel
       // buf[121] = (unsigned long) alpha[2];                                               // ADC-alpha correction -> BF changed
    //buf[121] = (unsigned long) CH3_Zero_Offset;                                          // Shift offset when stopped

    // Channel 4
	buf[122] = (unsigned long) MenuStatus[4][0];                                         // Channel - Coupling
	buf[123] = (unsigned long) MenuStatus[4][1];                                         // Channel - BW Linit

	buf[124] = (unsigned long) MenuStatus[4][2];                                         // Channel - Invert
	buf[125] = (unsigned long) MenuStatus[4][3];                                         // Channel - Probe
	buf[126] = (unsigned long) SwitchesCH4;                                              // Switches for Channel
	buf[127] = (unsigned long) Selected_Voltage_CH4;                                     // Voltage 5V
	buf[128] = (unsigned long) DAC_Correction[3][0];                              // Search Zeors offset
	buf[129] = (unsigned long) DAC_Correction[3][1];                              // Search Zeors offset
	buf[130] = (unsigned long) DAC_Correction[3][2];                              // Search Zeors offset
	buf[131] = (unsigned long) CH4_DAC_1;                                                // channel dac 1 offset
	buf[132] = (unsigned long) CH4_DAC_2;                                                // channel dac 2 offset
	buf[133] = (unsigned long) CH4_DAC_3;                                                // channel dac 3 offset
	buf[134] = (unsigned long) CH4_DAC_4;                                                // channel dac 4 offset
	buf[135] = (unsigned long) signal_zoomed4;                                           // signal zoomed
	buf[136] = (unsigned long) CH4_DAC_Offset;                                           // Channel DAC
	buf[137] = (unsigned long) ZeroLevelCH4;                                             // ZeroLevel
	buf[138] = (unsigned long) Virtual_ZeroLevelCH4;                                     // Virtual ZeroLevel
       // buf[139] = (unsigned long) alpha[3];                                               // ADC-alpha correction -> BF changed
    //buf[139] = (unsigned long) CH4_Zero_Offset;                                          // Shift offset when stopped                   	


    //Math Settings
    buf[140] = (unsigned long) Channel_Math_Active;                                      // Math channel off
    buf[141] = (unsigned long) MenuStatus[26][1];                                        // standart math fft on
    buf[142] = (unsigned long) MenuStatus[26][2];                                        // standart math 1*2 of
    buf[143] = (unsigned long) MenuStatus[26][3];                                        // standart math 1-2 off
    buf[144] = (unsigned long) MenuStatus[26][4];                                        // standart math 1+2 off

    //FFT Settings
    buf[145] = (unsigned long) FFT_Mode;
    buf[146] = (unsigned long) FFT_Length;
    buf[147] = (unsigned long) FFT_Level;
    buf[148] = (unsigned long) FFT_Windex;
    buf[149] = (unsigned long) FFT_Scale;

    buf[150] = (unsigned long) MenuStatus[27][0]; 
    buf[151] = (unsigned long) MenuStatus[27][1];
    buf[152] = (unsigned long) MenuStatus[27][2];
    buf[153] = (unsigned long) MenuStatus[27][3];

    // Cursors    	
    buf[154] = (unsigned long) MenuStatus[16][0];                                        // Cursor soruce
    buf[155] = (unsigned long) Cursor_Enabled;
    buf[156] = (unsigned long) Cursor_Both_Selected;
    buf[157] = (unsigned long) SelectedCursor;
    buf[158] = (unsigned long) Cursor_Horizontal_Active;
    buf[159] = (unsigned long) Cursor_Horizontal_Position_Real1;

    buf[160] = (unsigned long) Cursor_Horizontal_Position_Real2;
    buf[161] = (unsigned long) Cursor_Vertical_Active;
    buf[162] = (unsigned long) Cursor_Vertical_Position_Real1;
    buf[163] = (unsigned long) Cursor_Vertical_Position_Real2;

    // Quick Measure
    //buf[164] = (unsigned long) QM_Enabled;
    buf[165] = (unsigned long) QM_Type[0];
    buf[166] = (unsigned long) QM_Type[1];
    buf[167] = (unsigned long) QM_Type[2];
    buf[168] = (unsigned long) QM_Channel[0];
    buf[169] = (unsigned long) QM_Channel[1];

    buf[170] = (unsigned long) QM_Channel[2];
    buf[171] = (unsigned long) QM_Horizontal_Active;
    buf[172] = (unsigned long) QM_Vertical_Active;
    buf[173] = (unsigned long) Quick_Measure_Threshold_btn_select;

    // Display
    buf[174] = (unsigned long) GridColorArrayPos;                                        // Normal Grid
    buf[175] = (unsigned long) GridColor_Val;
    buf[176] = (unsigned long) GridColorArrayPos_Delayed;                                // Delayed Grid
    //BF not used buf[177] = (unsigned long) GridColor_Val_Delayed;
    buf[178] = (unsigned long) MenuStatus[5][0];                                         // Display - Persist
    buf[179] = (unsigned long) MenuStatus[5][3];                                         // Display - Verctors

//    buf[180] = (unsigned long) MenuStatus[0][0];                                         // Aquire - Normal
//    buf[181] = (unsigned long) MenuStatus[0][1];                                         // Aquire - Averaging
//    buf[183] = (unsigned long) MenuStatus[0][2];                                         // Aquire - Avgs
    buf[184] = (unsigned long) MenuStatus[5][4];                                         // Display - Browse

    buf[185] = (unsigned long) MenuStatus[23][1];                                       // Autoscale - Slow TB
    //buf[186] = (unsigned long) ?????;	

    	buf[187] = (unsigned long) draw_mode;
    	buf[188] = (unsigned long) dotted_grid;

    /////
	buf[189] = (unsigned long) (MenuPopupStatus[8][0] | (MenuPopupStatus[8][1] << 16));  // popup external trigger
	
	buf[190] = (unsigned long) MenuPopupStatus[8][2];//MenuStatus[14][3];                                        // Triggering - no externel triggering
	buf[191] = (unsigned long) OneShotRestart;
	buf[192] = (unsigned long) led_value;
	buf[193] = (unsigned long) Channel_1_Active;
	buf[194] = (unsigned long) Channel_2_Active;
	buf[195] = (unsigned long) Channel_3_Active;
	buf[196] = (unsigned long) Channel_4_Active;
	buf[197] = (unsigned long) dmode_SIGNAL_OFFSET;
	buf[198] = (unsigned long) SIGNAL_OFFSET;
	
	
	buf[199] = (unsigned long) HoldOff_Value;
	buf[200] = (unsigned long) HoldOff_Expo;
	buf[201] = (unsigned long) HoldOff_Steps;
	
	buf[202] = (unsigned long) Pulse11_Value;
	buf[203] = (unsigned long) Pulse11_Expo;
	buf[204] = (unsigned long) Pulse11_Steps;
	
	buf[205] = (unsigned long) Pulse12_Value;
	buf[206] = (unsigned long) Pulse12_Expo;
	buf[207] = (unsigned long) Pulse12_Steps;
	
	buf[208] = (unsigned long) Pulse21_Value;
	buf[209] = (unsigned long) Pulse21_Expo;
	buf[210] = (unsigned long) Pulse21_Steps;
	
	buf[211] = (unsigned long) Pulse22_Value;
	buf[212] = (unsigned long) Pulse22_Expo;
	buf[213] = (unsigned long) Pulse22_Steps;

//BF new Math values
	buf[214] = (unsigned int) Math_Offset;
	buf[215] = (unsigned int) Math_Mul_Offset;
	buf[216] = (unsigned int) Math_Sub_Offset;
	buf[217] = (unsigned int) Math_Add_Offset;
	buf[218] = (unsigned int) math_mul_scale;
	buf[219] = (unsigned int) math_mul_scale;
	buf[220] = (unsigned int) math_sub_scale;
	buf[221] = (unsigned int) math_add_scale;
//BF end



	buf[229] = (unsigned long) Trig_Pos_Mem;
	buf[230] = (unsigned long) Trig_Pos_Display;
	buf[231] = (unsigned long) Trig_Pos_Zoom;
	buf[232] = (unsigned long) MemWinStart;
	
	buf[233] = (unsigned long) Trig_Pos_Mem_old;
	buf[234] = (unsigned long) Trig_Pos_Display_old;
	buf[235] = (unsigned long) Trig_Pos_Zoom_old;
	buf[236] = (unsigned long) MemWinStart_old;
	
	buf[237] = (unsigned long) ZFT_float;
	buf[238] = (unsigned long) ZFT_float_old;
	buf[239] = (unsigned long) ZFT_float_dmode;
	buf[240] = (unsigned long) ZFT_float_dmode_old;
	
	buf[241] = (unsigned long) OFT_float;
	buf[242] = (unsigned long) OFT_float_old;
	buf[243] = (unsigned long) ZIT_short;
	buf[244] = (unsigned long) ZIT_short_old;
	
	buf[245] = (unsigned long) ZIT_short_dmode;
	buf[246] = (unsigned long) ZIT_short_dmode_old;
	buf[247] = (unsigned long) test_port_reg;


// BF new ADC-correction values
/* -> now stored in protected area

	//channel 1
	buf[248] = (unsigned int) ADC_Offset[0][0];
	buf[249] = (unsigned int) ADC_Offset[0][1];
	buf[250] = (unsigned int) ADC_Offset[0][2];
	buf[251] = (unsigned int) ADC_Offset[0][3];
	//channel 2
	buf[252] = (unsigned int) ADC_Offset[1][0];
	buf[253] = (unsigned int) ADC_Offset[1][1];
	buf[254] = (unsigned int) ADC_Offset[1][2];
	buf[255] = (unsigned int) ADC_Offset[1][3];
	//channel 3
	buf[256] = (unsigned int) ADC_Offset[2][0];
	buf[257] = (unsigned int) ADC_Offset[2][1];
	buf[258] = (unsigned int) ADC_Offset[2][2];
	buf[259] = (unsigned int) ADC_Offset[2][3];
	//channel 4
	buf[260] = (unsigned int) ADC_Offset[3][0];
	buf[261] = (unsigned int) ADC_Offset[3][1];
	buf[262] = (unsigned int) ADC_Offset[3][2];
	buf[263] = (unsigned int) ADC_Offset[3][3];
*/

// BF new zerolevels for XY
	buf[264] = (unsigned int) Virtual_ZeroLevelXYCH1;
	buf[265] = (unsigned int) Virtual_ZeroLevelXYCH2;
	buf[266] = (unsigned int) Virtual_ZeroLevelXYCH3;
	buf[267] = (unsigned int) Virtual_ZeroLevelXYCH4;

	buf[268] = (unsigned int)FFT_Layout;
// BF end

	buf[299] = 0x23232323;

	nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address, buf, 300);	// Flash Buffer

#ifdef _Debug_		
	if (Debug_Mode) printf("Flash written\n");
//    printf("WF : df : %f c1 : %d c2 : %d cs %d st %d sfi %d dst %d dsfi %d dwop %d\n", dmode_factor, Cursor_Delayed_1, Cursor_Delayed_2, Cursor_Delayed_Size, Selected_Timebase, SIGNALFaktor_idx, dmode_Selected_Timebase, dmode_SIGNALFaktor_idx, dmode_Window_Offset_Pos);
#endif

    if (Continius) Hardware::Start_Record();
}
//#####################################################################################################################################################
void AMDFlash::Read_Config_Flash(void)
{
	int x;
	unsigned long *flash_address;
	unsigned long buf[300];
	
	flash_address = (unsigned long) nasys_main_flash + Config_Flash;	
	
	for (x = 0; x < 300; x++) buf[x] = flash_address[x];
	
	if ((buf[4] == tc_version) && (buf[5] == tc_subversion) && (buf[6] == tc_dev_version) && (buf[1] == 0xAACC5523) && (buf[2] == 0x22334455) && (buf[299] == 0x23232323))
	{
printf("AMDFlash::Read_Config_Flash\n");

		// Timebase      	
	        MenuStatus[MENU_TIMEBASE][0] = (short) buf[10];
	        MenuStatus[MENU_TIMEBASE][1] = (short) buf[11];
	        MenuStatus[MENU_TIMEBASE][2] = (short) buf[12];
	        MenuStatus[MENU_TIMEBASE][3] = (short) buf[13];
	        MenuStatus[MENU_TIMEBASE][4] = (short) buf[14];

		MenuStatus[MENU_DISPLAY][2] = (short) buf[15];

		XY_Mode = (char) buf[16];
		Selected_Timebase = (char) buf[17];
		SIGNALFaktor_idx = (volatile short) buf[18];
		SIGNAL_StartFr_idx = (volatile short) buf[19];
		Selected_Timebase = (unsigned long) buf[20];        //????
		
		//if (Selected_Timebase < 0) Selected_Timebase = 0;
		
		SIGNALFaktor_idx = (char) buf[21];
	
		dmode_Selected_Timebase = (char) buf[22];
		dmode_SIGNALFaktor_idx = (volatile short) buf[23];
		dmode_Window_Offset_Pos = (int) buf[24];
		dmode_SIGNAL_StartFr_idx = (volatile short) buf[25];
	
		// werden frisch berrechnet
		//Cursor_Delayed_1 = (int) buf[26];
		//Cursor_Delayed_2 = (int) buf[27];
		//Cursor_Delayed_Size = (int) buf[28];
		USTB_Mode = (volatile short) buf[29];						//ultra slow timebase mode
		USTB_Mode_bak = (volatile short) buf[30];
//      		ROLLMODE_STARTED = (volatile char) buf[31];
//      		ROLLMODE_Value_rec = (volatile char) buf[32];
		Timebase_Offset_Pos = (int) buf[33];
		SwitchesTB = (unsigned short) buf[34];
	
		// Trigger	
		//MenuStatus[12][0] = (short) buf[35];                                        // Triggering - Auto
		MenuStatus[12][1] = (short) buf[36];                                        // Triggering - Coupling
		MenuStatus[12][2] = (short) buf[37];                                        // Triggering - Reject
		MenuStatus[12][4] = (short) buf[38];                                        // Triggering - External Probe = 1 : 1  ( muß noch als funktion integriert werden )
		MenuStatus[14][3] = (short) buf[39];                                        // Triggering - TV
		MenuStatus[14][0] = (short) buf[40];                                        // Triggering - positive edge
		MenuStatus[14][1] = (short) buf[41];                                        // Triggering - Source = channel 1
		MenuStatus[14][2] = (short) buf[42];                                        // Triggering - no externel triggering
		MenuStatus[15][0] = (short) buf[43];                                        // Triggering - Source Channel 1
	//        MenuStatus[15][1] = (short) buf[44];                                        // Triggering - Negative Pulse
		MenuStatus[15][2] = (short) buf[45];                                        // Triggering - bigger - smaller - then
		Selected_Trigger_Source = (char) buf[46];                                  // Selected Trigger Source is Channel 1
		SelectedEdgeExtern = (char) buf[47];                                       // Trigger Edge by extern triggering
		PulseToggle = (char) buf[48];                                              // Pulse width button selection
		Memory_Position = (long) buf[49];                                          // Memory Position
		
		//Trigger_Offset_Pos = (volatile int) buf[50]; 	//BF del not used         // Trigger Offset
		Trigger_Pos_CH1 = (int) buf[51];                                          // Trigger Level Channel 1
		Trigger_Pos_CH2 = (int) buf[52];                                          // Trigger Level Channel 2
		Trigger_Pos_CH3 = (int) buf[53];                                          // Trigger Level Channel 3
		Trigger_Pos_CH4 = (int) buf[54];                                          // Trigger Level Channel 4
		trg_val_reg = (unsigned int) buf[55];                                              // Trigger Value
		trg_hyst_reg = (unsigned int) buf[56];                                             // Trigger Hysterese
		trig_range_reg = (unsigned int) buf[57];                                           // Trigger Range
		trig_holdoff_reg = (unsigned long) buf[58];                                         // Trigger Holdoff
		trig_width_reg = (unsigned int) buf[59];                                           // Trigger Width
		
		ctrl_reg = (unsigned int) buf[60];                                                 // Control register
		adc_ctrl_reg = (unsigned int) buf[61];                                             // ADC Control register
		pre_reg = (unsigned int) buf[62];                                                  // PreTrigger Value
	//       channel_Adr_add = (unsigned int) buf[63];                                          // Channel Add Value
		//adc_change_reg = (unsigned long) buf[64];                                           // ADC Change
		triggering = (unsigned int) buf[65];                                               // Triggering
		TriggerWay = (char) buf[66];                                               // Triggerway
		Trigger_Pos_CHE = (int) buf[67];                                                // Trigger extern value
		
		// Channel 1
		MenuStatus[1][0] = (short) buf[68];                                         // Channel - Coupling
		MenuStatus[1][1] = (short) buf[69];                                         // Channel - BW Linit
		
		MenuStatus[1][2] = (short) buf[70];                                         // Channel - Invert
		MenuStatus[1][3] = (short) buf[71];                                         // Channel - Probe
		SwitchesCH1 = (unsigned short) buf[72];                                              // Switches for Channel 1 and for the other the AC bit
		Selected_Voltage_CH1 = (volatile char) buf[73];                                     // Voltage
		DAC_Correction[0][0] = (int) buf[74];                              // Search Zeors offset
		DAC_Correction[0][1] = (int) buf[75];                              // Search Zeors offset
		DAC_Correction[0][2] = (int) buf[76];                              // Search Zeors offset
		CH1_DAC_1 = (unsigned char) buf[77];                                                // channel dac 1 offset
		CH1_DAC_2 = (unsigned char) buf[78];                                                // channel dac 2 offset
		CH1_DAC_3 = (unsigned char) buf[79];                                                // channel dac 3 offset
		CH1_DAC_4 = (unsigned char) buf[80];                                                // channel dac 4 offset
		signal_zoomed1       = (unsigned char) buf[81];                                           // signal zoomed
		CH1_DAC_Offset 	     = (int) buf[82];                                           // Channel DAC
		ZeroLevelCH1         = (int) buf[83];                                             // ZeroLevel
		Virtual_ZeroLevelCH1 = (int) buf[84];                                     // Virtual ZeroLevel
        	//alpha[0]             = (int) buf[85];                                          // ADC-alpha correction -> BF changed
		//CH1_Zero_Offset = (int) buf[85];                                          // Shift offset when stopped
		
		// Channel 2
		MenuStatus[2][0] = (short) buf[86];                                         // Channel - Coupling
		MenuStatus[2][1] = (short) buf[87];                                         // Channel - BW Linit
		MenuStatus[2][2] = (short) buf[88];                                         // Channel - Invert
		MenuStatus[2][3] = (short) buf[89];                                         // Channel - Probe
	
		SwitchesCH2 = (unsigned short) buf[90];                                              // Switches for Channel 2
		Selected_Voltage_CH2 = (volatile char) buf[91];                                     // Voltage
		DAC_Correction[1][0] = (int) buf[92];                              // Search Zeors offset
		DAC_Correction[1][1] = (int) buf[93];                              // Search Zeors offset
		DAC_Correction[1][2] = (int) buf[94];                              // Search Zeors offset
		CH2_DAC_1 = (unsigned char) buf[95];                                                // channel dac 1 offset
		CH2_DAC_2 = (unsigned char) buf[96];                                                // channel dac 2 offset
		CH2_DAC_3 = (unsigned char) buf[97];                                                // channel dac 3 offset
		CH2_DAC_4 = (unsigned char) buf[98];                                                // channel dac 4 offset
		signal_zoomed2       = (unsigned char) buf[99];                                           // signal zoomed
		CH2_DAC_Offset       = (int) buf[100];                                           // Channel DAC
		ZeroLevelCH2         = (int) buf[101];                                             // ZeroLevel
		Virtual_ZeroLevelCH2 = (int) buf[102];                                     // Virtual ZeroLevel
        	//alpha[1]             = (int) buf[103];                                          // ADC-alpha correction -> BF changed
		//CH2_Zero_Offset = (int) buf[103];                                          // Shift offset when stopped
	
		// Channel 3
		MenuStatus[3][0] = (short) buf[104];                                         // Channel - Coupling
		MenuStatus[3][1] = (short) buf[105];                                         // Channel - BW Linit
		MenuStatus[3][2] = (short) buf[106];                                         // Channel - Invert
		MenuStatus[3][3] = (short) buf[107];                                         // Channel - Probe
		SwitchesCH3 = (unsigned short) buf[108];                                              // Switches for Channel 3
		Selected_Voltage_CH3 = (volatile char) buf[109];                                     // Voltage
	
		DAC_Correction[2][0] = (int) buf[110];                              // Search Zeors offset
		DAC_Correction[2][1] = (int) buf[111];                              // Search Zeors offset
		DAC_Correction[2][2] = (int) buf[112];                              // Search Zeors offset
		CH3_DAC_1 = (unsigned char) buf[113];                                                // channel dac 1 offset
		CH3_DAC_2 = (unsigned char) buf[114];                                                // channel dac 2 offset
		CH3_DAC_3 = (unsigned char) buf[115];                                                // channel dac 3 offset
		CH3_DAC_4 = (unsigned char) buf[116];                                                // channel dac 4 offset
		signal_zoomed3       = (unsigned char) buf[117];                                           // signal zoomed
		CH3_DAC_Offset       = (int) buf[118];                                           // Channel DAC
		ZeroLevelCH3         = (int) buf[119];                                             // ZeroLevel
		Virtual_ZeroLevelCH3 = (int) buf[120];                                     // Virtual ZeroLevel
	        //alpha[2]             = (int) buf[121];                                          // ADC-alpha correction -> BF changed
		//CH3_Zero_Offset = (int) buf[121];                                          // Shift offset when stopped
	
		// Channel 4
		MenuStatus[4][0] = (short) buf[122];                                         // Channel - Coupling
		MenuStatus[4][1] = (short) buf[123];                                         // Channel - BW Linit
		MenuStatus[4][2] = (short) buf[124];                                         // Channel - Invert
		MenuStatus[4][3] = (short) buf[125];                                         // Channel - Probe
		SwitchesCH4 = (unsigned short) buf[126];                                              // Switches for Channel 4
		Selected_Voltage_CH4 = (volatile char) buf[127];                                     // Voltage
		DAC_Correction[3][0] = (int) buf[128];                              // Search Zeors offset
		DAC_Correction[3][1] = (int) buf[129];                              // Search Zeors offset
		DAC_Correction[3][2] = (int) buf[130];                              // Search Zeors offset
		CH4_DAC_1 = (unsigned char) buf[131];                                                // channel dac 1 offset
		CH4_DAC_2 = (unsigned char) buf[132];                                                // channel dac 2 offset
		CH4_DAC_3 = (unsigned char) buf[133];                                                // channel dac 3 offset
		CH4_DAC_4 = (unsigned char) buf[134];                                                // channel dac 4 offset
		signal_zoomed4       = (unsigned char) buf[135];                                           // signal zoomed
		CH4_DAC_Offset       = (int) buf[136];                                           // Channel DAC
		ZeroLevelCH4         = (int) buf[137];                                             // ZeroLevel
		Virtual_ZeroLevelCH4 = (int) buf[138];                                     // Virtual ZeroLevel
	        //alpha[3]             = (int) buf[139];                                          // ADC-alpha correction -> BF changed
		//CH4_Zero_Offset = (int) buf[139];                                          // Shift offset when stopped

        	//Channel_Math_Active = (char) buf[140];                                      // Math channel off
        	//MenuStatus[26][1] = (short) buf[141];                                        // standart math fft deactive
        	//MenuStatus[26][2] = (short) buf[142];                                        // standart math 1*2 deactive
        	//MenuStatus[26][3] = (short) buf[143];                                        // standart math 1-2 off
        	//MenuStatus[26][4] = (short) buf[144];                                        // standart math 1+2 off

    		//FFT Settings
     		//FFT_Mode   = (int)buf[145];
    		FFT_Length = (int)buf[146];
    		FFT_Level  = (int)buf[147];
   		FFT_Windex = (int)buf[148];
    		//FFT_Scale  = (int)buf[149];

		MenuStatus[27][0] = (short)buf[150]; 
		MenuStatus[27][1] = (short)buf[151];
		MenuStatus[27][2] = (short)buf[152];
		MenuStatus[27][3] = (short)buf[153];

		// Cursors    	
		MenuStatus[16][0] = (short) buf[154];                                        // Cursor soruce
		Cursor_Enabled = (char) buf[155];
	
		if (Cursor_Enabled) Cursor_Data_First_Draw = 1;
	
		Cursor_Both_Selected = (char) buf[156];
		SelectedCursor = (char) buf[157];
		Cursor_Horizontal_Active = (char) buf[158];
		Cursor_Horizontal_Position_Real1 = (long) buf[159];
	
		Cursor_Horizontal_Position_Real2 = (long) buf[160];
		Cursor_Vertical_Active = (char) buf[161];
		Cursor_Vertical_Position_Real1 = (long) buf[162];
		Cursor_Vertical_Position_Real2 = (long) buf[163];
	
		// Quick Measure
		//QM_Enabled = (char) buf[164];
	
		if (QM_Enabled) Quick_Measure_First_Draw = 1;
	
		QM_Type[0] = (char) buf[165];
		QM_Type[1] = (char) buf[166];
		QM_Type[2] = (char) buf[167];
		QM_Channel[0] = (char) buf[168];
		QM_Channel[1] = (char) buf[169];
	
		QM_Channel[2] = (char) buf[170];
		QM_Horizontal_Active = (char) buf[171];
		QM_Vertical_Active = (char) buf[172];
		Quick_Measure_Threshold_btn_select = (char) buf[173];
	
		// Display
		GridColorArrayPos = (unsigned char) buf[174];
		GridColor_Val = (unsigned char) buf[175];
		GridColorArrayPos_Delayed = (unsigned char) buf[176];
		//BF not used GridColor_Val_Delayed = (unsigned char) buf[177];
		MenuStatus[MENU_DISPLAY][0] = (short) buf[178];                                         // Display - Persist
		MenuStatus[MENU_DISPLAY][3] = (short) buf[179];                                         // Display - Verctors
	
	//        MenuStatus[0][0] = (short) buf[180];                                         // Aquire - Normal
	//        MenuStatus[0][1] = (short) buf[181];                                         // Aquire - Averaging
		MenuStatus[0][2] = (short) buf[183];                                         // Aquire - Avgs
	
		//MenuStatus[MENU_DISPLAY][4] = (short) buf[184];                            // Display - Browse
	
		MenuStatus[23][1] = (short) buf[185];                                        // Autoscale - Slow TB
		//???? = (short) buf[186]; 
	
		draw_mode = (unsigned char) buf[187];
		dotted_grid = (unsigned char) buf[188];
		// vergessene
		MenuPopupStatus[8][0] = (short) (buf[189] & 0x0000FFFF);                    // popup for externeal trigger
		MenuPopupStatus[8][1] = (short) ((buf[189] & 0xFFFF0000) >> 16);
		MenuPopupStatus[8][2] = (short) buf[190];
	
		//MenuStatus[14][3] = (short) buf[190];                                        // Triggering - no tv triggering
		OneShotRestart = (unsigned char) buf[191];
		// 192 s.o. ledval
		Channel_1_Active = (char) buf[193];
		Channel_2_Active = (char) buf[194];
		Channel_3_Active = (char) buf[195];
		Channel_4_Active = (char) buf[196];
		dmode_SIGNAL_OFFSET = (volatile short) buf[197];
		SIGNAL_OFFSET = (volatile short) buf[198];
	
		HoldOff_Value = (int) buf[199];
		HoldOff_Expo = (int) buf[200];
		HoldOff_Steps = (int) buf[201];
	
		Pulse11_Value = (int) buf[202];
		Pulse11_Expo = (int) buf[203];
		Pulse11_Steps = (int) buf[204];
	
		Pulse12_Value = (int) buf[205];
		Pulse12_Expo = (int) buf[206];
		Pulse12_Steps = (int) buf[207];
	
		Pulse21_Value = (int) buf[208];
		Pulse21_Expo = (int) buf[209];
		Pulse21_Steps = (int) buf[210];
	
		Pulse22_Value = (int) buf[211];
		Pulse22_Expo = (int) buf[212];
		Pulse22_Steps = (int) buf[213];

//BF new Math values
		Math_Offset     = (int) buf[214]; 
		Math_Mul_Offset = (int) buf[215]; 
		Math_Sub_Offset = (int) buf[216];
		Math_Add_Offset = (int) buf[217];
		math_scale      = (int) buf[218];
		math_mul_scale  = (int) buf[219];
		math_sub_scale  = (int) buf[220];
		math_add_scale  = (int) buf[221];
//BF end



		Trig_Pos_Mem = (int) buf[229];
		Trig_Pos_Display = (int) buf[230];
		Trig_Pos_Zoom = (float) buf[231];
		MemWinStart = (int) buf[232];
	
		Trig_Pos_Mem_old = (int) buf[233];
		Trig_Pos_Display_old = (int) buf[234];
		Trig_Pos_Zoom_old = (float) buf[235];
		MemWinStart_old = (int) buf[236];
	
		ZFT_float = (float) buf[237];
		ZFT_float_old = (float) buf[238];
		ZFT_float_dmode = (float) buf[239];
		ZFT_float_dmode_old = (float) buf[240];
	
		OFT_float = (float) buf[241];
		OFT_float_old = (float) buf[242];
		
		ZIT_short = (short) buf[243];
		ZIT_short_old = (short) buf[244];
		ZIT_short_dmode = (short) buf[245];
		ZIT_short_dmode_old = (short) buf[246];
		
		test_port_reg = (unsigned int) buf[247];

// BF new ADC-correction values
/* -> now stored in protected area

		//channel 1
		ADC_Offset[0][0] = (int)buf[248];
		ADC_Offset[0][1] = (int)buf[249];
		ADC_Offset[0][2] = (int)buf[250];
		ADC_Offset[0][3] = (int)buf[251];
		//channel 2
		ADC_Offset[1][0] = (int)buf[252];
		ADC_Offset[1][1] = (int)buf[253];
		ADC_Offset[1][2] = (int)buf[254];
		ADC_Offset[1][3] = (int)buf[255];
		//channel 3
		ADC_Offset[2][0] = (int)buf[256];
		ADC_Offset[2][1] = (int)buf[257];
		ADC_Offset[2][2] = (int)buf[258];
		ADC_Offset[2][3] = (int)buf[259];
		//channel 4
		ADC_Offset[3][0] = (int)buf[260];
		ADC_Offset[3][1] = (int)buf[261];
		ADC_Offset[3][2] = (int)buf[262];
		ADC_Offset[3][3] = (int)buf[263];
*/

// BF new zerolevels for XY
		Virtual_ZeroLevelXYCH1 = (int)buf[264];
		Virtual_ZeroLevelXYCH2 = (int)buf[265];
		Virtual_ZeroLevelXYCH3 = (int)buf[266];
		Virtual_ZeroLevelXYCH4 = (int)buf[267];

		FFT_Layout	= (int)buf[268];
// BF end

	
		timebase_reg = tb_value[Selected_Timebase];
		Display_Timebase = Selected_Timebase + SIGNALFaktor_idx;
	
		LED_ON[0] = 0;
		LED_ON[1] = 0;
		LED_ON[2] = 0;
		LED_ON[3] = 0;
		LED_ON[4] = 0;
		//LED_ON[5] = 0;
		LED_ON[6] = 0;
		LED_ON[7] = 0;
		LED_ON[8] = 0;
		LED_ON[9] = 0;
		LED_ON[10] = 0;
		LED_ON[11] = 0;
		//LED_ON[12] = 0;
		//LED_ON[13] = 0;
		LED_ON[14] = 0;
		LED_ON[15] = 0;
	
		if (((int) buf[192] & 0x0001) == 0x0001) LED_ON[0] = 1;
		if (((int) buf[192] & 0x0002) == 0x0002) LED_ON[1] = 1;
		if (((int) buf[192] & 0x0004) == 0x0004) LED_ON[2] = 1;
		if (((int) buf[192] & 0x0008) == 0x0008) LED_ON[3] = 1;
		//if (((int) buf[192] & 0x0010) == 0x0010) LED_ON[4] = 1;  // Math
		//if (((int) buf[192] & 0x0020) == 0x0020) LED_ON[5] = 1; // Generall
		if (((int) buf[192] & 0x0040) == 0x0040) LED_ON[6] = 1;
		//if (((int) buf[192] & 0x0080) == 0x0080) LED_ON[7] = 1; // Quick Measure
		if (((int) buf[192] & 0x0100) == 0x0100) LED_ON[8] = 1;
		if (((int) buf[192] & 0x0200) == 0x0200) LED_ON[9] = 1;
		if (((int) buf[192] & 0x0400) == 0x0400) LED_ON[10] = 1;
		if (((int) buf[192] & 0x0800) == 0x0800) LED_ON[11] = 1;
		//if (((int) buf[192] & 0x1000) == 0x1000) LED_ON[12] = 1; // Run/Stop Green
		//if (((int) buf[192] & 0x2000) == 0x2000) LED_ON[13] = 1; // Run/Stop Red
	//        if (((int) buf[192] & 0x4000) == 0x4000) LED_ON[14] = 1;
	//        if (((int) buf[192] & 0x8000) == 0x8000) LED_ON[15] = 1;
	
//BF changed		
		//Math_Offset_mul.Write_StepSize(1);
		Math_Offset_mul.Write_Value(float((float)Math_Mul_Offset /  (float)50000));
		Math_Offset_mul.RenderText();

		//Math_Offset_mul.Write_StepSize(1);
		Math_Offset_sub.Write_Value(float((float)Math_Sub_Offset /  (float)50000));
		Math_Offset_sub.RenderText();

		//Math_Offset_mul.Write_StepSize(1);
		Math_Offset_add.Write_Value(float((float)Math_Add_Offset /  (float)50000));
		Math_Offset_add.RenderText();
//BF end
		
		Hardware::Recalc_Vars();
	
		config_loaded = true;

printf("Flash loaded \n");

	}
	else
	{
		config_loaded = false;
printf("Flash not loaded - setting standards \n");
	}

}

void AMDFlash::Read_TestData_Flash(void)
{
	int x, cnt;
//printf("AMDFlash::Read_TestData_Flash\n");	
	unsigned long *flash_address;
	unsigned long buf[256];
	unsigned char buf2[1024];
    	
	flash_address = (unsigned long) nasys_main_flash + TestData_Flash;
	
	for (x = 0; x < 256; x++) buf[x] = flash_address[x];
	
	for (x = 0; x < 256; x++)
    {
        buf2[(x * 4) + 0] = (unsigned char) buf[x] & 0x000000FF;
        buf2[(x * 4) + 1] = (unsigned char) ((buf[x] & 0x0000FF00) >> 8);
        buf2[(x * 4) + 2] = (unsigned char) ((buf[x] & 0x00FF0000) >> 16);
        buf2[(x * 4) + 3] = (unsigned char) ((buf[x] & 0xFF000000) >> 24);
    }                                                       	
	
#ifdef _Debug_
    cnt = 0;
    for (x = 0; x < 48; x++)
    {
        printf("%3d %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x \n", cnt, buf2[(cnt * 21) + 0], buf2[(cnt * 21) + 1], buf2[(cnt * 21) + 2], buf2[(cnt * 21) + 3], buf2[(cnt * 21) + 4], buf2[(cnt * 21) + 5], buf2[(cnt * 21) + 6], buf2[(cnt * 21) + 7], buf2[(cnt * 21) + 8], buf2[(cnt * 21) + 9], buf2[(cnt * 21) + 10], buf2[(cnt * 21) + 11], buf2[(cnt * 21) + 12], buf2[(cnt * 21) + 13], buf2[(cnt * 21) + 14], buf2[(cnt * 21) + 15], buf2[(cnt * 21) + 16], buf2[(cnt * 21) + 17], buf2[(cnt * 21) + 18], buf2[(cnt * 21) + 19], buf2[(cnt * 21) + 20]);

        cnt++;
        nr_delay(10);
    }      	
#endif
}

void AMDFlash::Write_TestData_Flash(void)
{
    int x, cnt;
	unsigned long *flash_address;
	unsigned long buf[300];
//printf("AMDFlash::Write_TestData_Flash\n");	
	flash_address = (unsigned long) nasys_main_flash + TestData_Flash;
#ifdef _Debug_	
	printf("Flashadr : %x = %x\n", flash_address, flash_address[0]);
#endif	
	for (x = 0; x < 300; x++) buf[x] = flash_address[x];
	
	nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address);

	buf[32] = 0x12345678;
	
	nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address, buf, 34);
#ifdef _Debug_	
	if (Debug_Mode) printf("Flash TestData written\n");
#endif	
}

int AMDFlash::EraseSector_Flash(unsigned long *flash_address)
{
    int result;
//printf("AMDFlash::EraseSector_Flash\n");
    result = nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address);
    return result;
}

int AMDFlash::WriteData_Byte_Flash(unsigned long *flash_address, unsigned char add, unsigned char data)
{
    int result = 0;
    int errors = 100;
    unsigned char* fb       = (unsigned char *) nasys_main_flash;
    unsigned char* a        = (unsigned char *) flash_address;
//printf("AMDFlash::WriteData_Byte_Flash\n");

//    result = amd29lv065d_flash_write_byte(fb, a + add, data);

    while (errors)
    {
	   unlock_bypass_mode(fb);
	   result = nr_flash_write_bypass(fb, a + add, data);
	   reset_bypass_mode(fb);
	
	   if (result != -1) errors = 0;
    }
	
    return result;	
}

void AMDFlash::ReadData_Prepare_Flash(unsigned long *flash_address)
{
	int x, cnt;
	
	unsigned long *flash_adr;
	unsigned char result = 0;
//printf("AMDFlash::ReadData_Prepare_Flash\n");    	
	flash_adr = (unsigned long) nasys_main_flash + flash_address;
	
	for (x = 0; x < 256; x++)
    {
        Flash_Read_Buffer[(x * 4) + 0] = (unsigned char) flash_adr[x] & 0x000000FF;
        Flash_Read_Buffer[(x * 4) + 1] = (unsigned char) ((flash_adr[x] & 0x0000FF00) >> 8);
        Flash_Read_Buffer[(x * 4) + 2] = (unsigned char) ((flash_adr[x] & 0x00FF0000) >> 16);
        Flash_Read_Buffer[(x * 4) + 3] = (unsigned char) ((flash_adr[x] & 0xFF000000) >> 24);
//        nr_delay(5);
    }
}


unsigned char AMDFlash::ReadData_Byte_Flash(unsigned int add)
{
	unsigned char result = 0;
//printf("AMDFlash::ReadData_Byte_Flash\n");
    result = Flash_Read_Buffer[add];

    return result;
}
//################################################################################################################################################
void AMDFlash::Read_Protected_Flash(void)
{
	int x;
	
	unsigned long *flash_address;
 //printf("AMDFlash::Read_Protected_Flash\n");   	
	flash_address = (unsigned long) nasys_main_flash + protected_Config_Flash;
	
	for (x = 0; x < 256; x++)
    	{
        	Flash_Protect_Buffer[x] = flash_address[x];
    	}
	
	if (Flash_Protect_Buffer[0] == 0xFF2332FF)
	{
		tc_model           = Flash_Protect_Buffer[1];
		tc_serial          = Flash_Protect_Buffer[2];
		//tc_serial          = 4711;
		tc_production_lot1 = Flash_Protect_Buffer[3];
		tc_production_lot2 = Flash_Protect_Buffer[4];
		tc_shipment_date   = Flash_Protect_Buffer[5];
	
		kill_adc        = (unsigned int) Flash_Protect_Buffer[11];
		m2k_filter_Mode = (unsigned char) Flash_Protect_Buffer[12];
/*	
		Voltage_Correction[0][0] = (int) Flash_Protect_Buffer[13];
		Voltage_Correction[0][1] = (int) Flash_Protect_Buffer[14];
		Voltage_Correction[0][2] = (int) Flash_Protect_Buffer[15];
		Voltage_Correction[0][3] = (int) Flash_Protect_Buffer[16];
		Voltage_Correction[0][4] = (int) Flash_Protect_Buffer[17];
		Voltage_Correction[0][5] = (int) Flash_Protect_Buffer[18];
		Voltage_Correction[0][6] = (int) Flash_Protect_Buffer[19];
		Voltage_Correction[0][7] = (int) Flash_Protect_Buffer[20];
		Voltage_Correction[0][8] = (int) Flash_Protect_Buffer[21];
		Voltage_Correction[0][9] = (int) Flash_Protect_Buffer[22];
		Voltage_Correction[0][10] = (int) Flash_Protect_Buffer[23];
		
		Voltage_Correction[1][0] = (int) Flash_Protect_Buffer[24];
		Voltage_Correction[1][1] = (int) Flash_Protect_Buffer[25];
		Voltage_Correction[1][2] = (int) Flash_Protect_Buffer[26];
		Voltage_Correction[1][3] = (int) Flash_Protect_Buffer[27];
		Voltage_Correction[1][4] = (int) Flash_Protect_Buffer[28];
		Voltage_Correction[1][5] = (int) Flash_Protect_Buffer[29];
		Voltage_Correction[1][6] = (int) Flash_Protect_Buffer[30];
		Voltage_Correction[1][7] = (int) Flash_Protect_Buffer[31];
		Voltage_Correction[1][8] = (int) Flash_Protect_Buffer[32];
		Voltage_Correction[1][9] = (int) Flash_Protect_Buffer[33];
		Voltage_Correction[1][10] = (int) Flash_Protect_Buffer[34];
	
		Voltage_Correction[2][0] = (int) Flash_Protect_Buffer[35];
		Voltage_Correction[2][1] = (int) Flash_Protect_Buffer[36];
		Voltage_Correction[2][2] = (int) Flash_Protect_Buffer[37];
		Voltage_Correction[2][3] = (int) Flash_Protect_Buffer[38];
		Voltage_Correction[2][4] = (int) Flash_Protect_Buffer[39];
		Voltage_Correction[2][5] = (int) Flash_Protect_Buffer[40];
		Voltage_Correction[2][6] = (int) Flash_Protect_Buffer[41];
		Voltage_Correction[2][7] = (int) Flash_Protect_Buffer[42];
		Voltage_Correction[2][8] = (int) Flash_Protect_Buffer[43];
		Voltage_Correction[2][9] = (int) Flash_Protect_Buffer[44];
		Voltage_Correction[2][10] = (int) Flash_Protect_Buffer[45];
	
		Voltage_Correction[3][0] = (int) Flash_Protect_Buffer[46];
		Voltage_Correction[3][1] = (int) Flash_Protect_Buffer[47];
		Voltage_Correction[3][2] = (int) Flash_Protect_Buffer[48];
		Voltage_Correction[3][3] = (int) Flash_Protect_Buffer[49];
		Voltage_Correction[3][4] = (int) Flash_Protect_Buffer[50];
		Voltage_Correction[3][5] = (int) Flash_Protect_Buffer[51];
		Voltage_Correction[3][6] = (int) Flash_Protect_Buffer[52];
		Voltage_Correction[3][7] = (int) Flash_Protect_Buffer[53];
		Voltage_Correction[3][8] = (int) Flash_Protect_Buffer[54];
		Voltage_Correction[3][9] = (int) Flash_Protect_Buffer[55];
		Voltage_Correction[3][10] = (int) Flash_Protect_Buffer[56];
BF end */

		ADC_Offset[0][0] = (int) Flash_Protect_Buffer[57];
		ADC_Offset[0][1] = (int) Flash_Protect_Buffer[58];
		ADC_Offset[0][2] = (int) Flash_Protect_Buffer[59];
		ADC_Offset[0][3] = (int) Flash_Protect_Buffer[60];	
		
		ADC_Offset[1][0] = (int) Flash_Protect_Buffer[61];
		ADC_Offset[1][1] = (int) Flash_Protect_Buffer[62];
		ADC_Offset[1][2] = (int) Flash_Protect_Buffer[63];
		ADC_Offset[1][3] = (int) Flash_Protect_Buffer[64];
	
		ADC_Offset[2][0] = (int) Flash_Protect_Buffer[65];
		ADC_Offset[2][1] = (int) Flash_Protect_Buffer[66];
		ADC_Offset[2][2] = (int) Flash_Protect_Buffer[67];
		ADC_Offset[2][3] = (int) Flash_Protect_Buffer[68];
	
		ADC_Offset[3][0] = (int) Flash_Protect_Buffer[69];
		ADC_Offset[3][1] = (int) Flash_Protect_Buffer[70];
		ADC_Offset[3][2] = (int) Flash_Protect_Buffer[71];
		ADC_Offset[3][3] = (int) Flash_Protect_Buffer[72];
		
		adc_change12_reg = Flash_Protect_Buffer[73];	
		adc_ctr12_reg    = Flash_Protect_Buffer[74];	
		adc_change34_reg = Flash_Protect_Buffer[75];		
		channel_Adr_add  = Flash_Protect_Buffer[76];	
		channel_Adr_add2 = Flash_Protect_Buffer[77];	
/* BF del
		ZeroScaleFactor[0][0] = (float) ((float) (Voltage_Correction[0][0] + 100) / 100);
		ZeroScaleFactor[0][1] = (float) ((float) (Voltage_Correction[0][1] + 100) / 100);
		ZeroScaleFactor[0][2] = (float) ((float) (Voltage_Correction[0][2] + 100) / 100);
		ZeroScaleFactor[0][3] = (float) ((float) (Voltage_Correction[0][3] + 100) / 100);
		ZeroScaleFactor[0][4] = (float) ((float) (Voltage_Correction[0][4] + 100) / 100);
		ZeroScaleFactor[0][5] = (float) ((float) (Voltage_Correction[0][5] + 100) / 100);
		ZeroScaleFactor[0][6] = (float) ((float) (Voltage_Correction[0][6] + 100) / 100);
		ZeroScaleFactor[0][7] = (float) ((float) (Voltage_Correction[0][7] + 100) / 100);
		ZeroScaleFactor[0][8] = (float) ((float) (Voltage_Correction[0][8] + 100) / 100);
		ZeroScaleFactor[0][9] = (float) ((float) (Voltage_Correction[0][9] + 100) / 100);
		ZeroScaleFactor[0][10] = (float) ((float) (Voltage_Correction[0][10] + 100) / 100);
		ZeroScaleFactor[0][11] = (float) ((float) (Voltage_Correction[0][11] + 100) / 100);
	
		ZeroScaleFactor[1][0] = (float) ((float) (Voltage_Correction[1][0] + 100) / 100);
		ZeroScaleFactor[1][1] = (float) ((float) (Voltage_Correction[1][1] + 100) / 100);
		ZeroScaleFactor[1][2] = (float) ((float) (Voltage_Correction[1][2] + 100) / 100);
		ZeroScaleFactor[1][3] = (float) ((float) (Voltage_Correction[1][3] + 100) / 100);
		ZeroScaleFactor[1][4] = (float) ((float) (Voltage_Correction[1][4] + 100) / 100);
		ZeroScaleFactor[1][5] = (float) ((float) (Voltage_Correction[1][5] + 100) / 100);
		ZeroScaleFactor[1][6] = (float) ((float) (Voltage_Correction[1][6] + 100) / 100);
		ZeroScaleFactor[1][7] = (float) ((float) (Voltage_Correction[1][7] + 100) / 100);
		ZeroScaleFactor[1][8] = (float) ((float) (Voltage_Correction[1][8] + 100) / 100);
		ZeroScaleFactor[1][9] = (float) ((float) (Voltage_Correction[1][9] + 100) / 100);
		ZeroScaleFactor[1][10] = (float) ((float) (Voltage_Correction[1][10] + 100) / 100);
		ZeroScaleFactor[1][11] = (float) ((float) (Voltage_Correction[1][11] + 100) / 100);
	
		ZeroScaleFactor[2][0] = (float) ((float) (Voltage_Correction[2][0] + 100) / 100);
		ZeroScaleFactor[2][1] = (float) ((float) (Voltage_Correction[2][1] + 100) / 100);
		ZeroScaleFactor[2][2] = (float) ((float) (Voltage_Correction[2][2] + 100) / 100);
		ZeroScaleFactor[2][3] = (float) ((float) (Voltage_Correction[2][3] + 100) / 100);
		ZeroScaleFactor[2][4] = (float) ((float) (Voltage_Correction[2][4] + 100) / 100);
		ZeroScaleFactor[2][5] = (float) ((float) (Voltage_Correction[2][5] + 100) / 100);
		ZeroScaleFactor[2][6] = (float) ((float) (Voltage_Correction[2][6] + 100) / 100);
		ZeroScaleFactor[2][7] = (float) ((float) (Voltage_Correction[2][7] + 100) / 100);
		ZeroScaleFactor[2][8] = (float) ((float) (Voltage_Correction[2][8] + 100) / 100);
		ZeroScaleFactor[2][9] = (float) ((float) (Voltage_Correction[2][9] + 100) / 100);
		ZeroScaleFactor[2][10] = (float) ((float) (Voltage_Correction[2][10] + 100) / 100);
		ZeroScaleFactor[2][11] = (float) ((float) (Voltage_Correction[2][11] + 100) / 100);
	
		ZeroScaleFactor[3][0] = (float) ((float) (Voltage_Correction[3][0] + 100) / 100);
		ZeroScaleFactor[3][1] = (float) ((float) (Voltage_Correction[3][1] + 100) / 100);
		ZeroScaleFactor[3][2] = (float) ((float) (Voltage_Correction[3][2] + 100) / 100);
		ZeroScaleFactor[3][3] = (float) ((float) (Voltage_Correction[3][3] + 100) / 100);
		ZeroScaleFactor[3][4] = (float) ((float) (Voltage_Correction[3][4] + 100) / 100);
		ZeroScaleFactor[3][5] = (float) ((float) (Voltage_Correction[3][5] + 100) / 100);
		ZeroScaleFactor[3][6] = (float) ((float) (Voltage_Correction[3][6] + 100) / 100);
		ZeroScaleFactor[3][7] = (float) ((float) (Voltage_Correction[3][7] + 100) / 100);
		ZeroScaleFactor[3][8] = (float) ((float) (Voltage_Correction[3][8] + 100) / 100);
		ZeroScaleFactor[3][9] = (float) ((float) (Voltage_Correction[3][9] + 100) / 100);
		ZeroScaleFactor[3][10] = (float) ((float) (Voltage_Correction[3][10] + 100) / 100);
		ZeroScaleFactor[3][11] = (float) ((float) (Voltage_Correction[3][11] + 100) / 100);	

		ZeroLevelCH1_cor_offset = 0;
		ZeroLevelCH2_cor_offset = 0;
		ZeroLevelCH3_cor_offset = 0;
		ZeroLevelCH4_cor_offset = 0;

		for (x = 0; x < 4; x++)
		{
			if (ADC_Offset[0][x] > ZeroLevelCH1_cor_offset) ZeroLevelCH1_cor_offset = ADC_Offset[0][x];
			if (ADC_Offset[1][x] > ZeroLevelCH2_cor_offset) ZeroLevelCH2_cor_offset = ADC_Offset[1][x];
			if (ADC_Offset[2][x] > ZeroLevelCH3_cor_offset) ZeroLevelCH3_cor_offset = ADC_Offset[2][x];
			if (ADC_Offset[3][x] > ZeroLevelCH4_cor_offset) ZeroLevelCH4_cor_offset = ADC_Offset[3][x];
		}
BF end */
	}     	
}
//############################################################################################################################################
void AMDFlash::Write_Protected_Flash(void)
{
    int x, cnt;
	unsigned long *flash_address;
printf("AMDFlash::Write_Protected_Flash\n");	
	flash_address = (unsigned long) nasys_main_flash + protected_Config_Flash;
#ifdef _Debug_	
	printf("Flashadr : %x = %x\n", flash_address, flash_address[0]);
#endif	
	
	nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address);

	//Flash_Protect_Buffer[0] = 0xFF2332FF;              // Kennung			BF del
	//Flash_Protect_Buffer[1] = tc_model;                // model 			BF del
	//Flash_Protect_Buffer[2] = tc_serial;               // serial nr		BF del
	Flash_Protect_Buffer[3] = tc_production_lot1;      // production lot	
	Flash_Protect_Buffer[4] = tc_production_lot2;      // production lot	
	Flash_Protect_Buffer[5] = tc_shipment_date;        // shipment date
	//Flash_Protect_Buffer[6] = 0x00000000;
	//Flash_Protect_Buffer[7] = 0x00000000;
	//Flash_Protect_Buffer[8] = 0x00000000;
	//Flash_Protect_Buffer[9] = 0x00000000;
	//Flash_Protect_Buffer[10] = 0x00000000;
	// spezial config data
	Flash_Protect_Buffer[11] = (unsigned long) kill_adc;	           // kill adc settings for all channels
	Flash_Protect_Buffer[12] = (unsigned long) m2k_filter_Mode;	    // slow tb 2k spike filter settings for all channels	
	
/* BF del	
	Flash_Protect_Buffer[13] = (unsigned long) Voltage_Correction[0][0];
	Flash_Protect_Buffer[14] = (unsigned long) Voltage_Correction[0][1];
	Flash_Protect_Buffer[15] = (unsigned long) Voltage_Correction[0][2];
	Flash_Protect_Buffer[16] = (unsigned long) Voltage_Correction[0][3];
	Flash_Protect_Buffer[17] = (unsigned long) Voltage_Correction[0][4];
	Flash_Protect_Buffer[18] = (unsigned long) Voltage_Correction[0][5];
	Flash_Protect_Buffer[19] = (unsigned long) Voltage_Correction[0][6];
	Flash_Protect_Buffer[20] = (unsigned long) Voltage_Correction[0][7];
	Flash_Protect_Buffer[21] = (unsigned long) Voltage_Correction[0][8];
	Flash_Protect_Buffer[22] = (unsigned long) Voltage_Correction[0][9];
	Flash_Protect_Buffer[23] = (unsigned long) Voltage_Correction[0][10];

	Flash_Protect_Buffer[24] = (unsigned long) Voltage_Correction[1][0];
	Flash_Protect_Buffer[25] = (unsigned long) Voltage_Correction[1][1];
	Flash_Protect_Buffer[26] = (unsigned long) Voltage_Correction[1][2];
	Flash_Protect_Buffer[27] = (unsigned long) Voltage_Correction[1][3];
	Flash_Protect_Buffer[28] = (unsigned long) Voltage_Correction[1][4];
	Flash_Protect_Buffer[29] = (unsigned long) Voltage_Correction[1][5];
	Flash_Protect_Buffer[30] = (unsigned long) Voltage_Correction[1][6];
	Flash_Protect_Buffer[31] = (unsigned long) Voltage_Correction[1][7];
	Flash_Protect_Buffer[32] = (unsigned long) Voltage_Correction[1][8];
	Flash_Protect_Buffer[33] = (unsigned long) Voltage_Correction[1][9];
	Flash_Protect_Buffer[34] = (unsigned long) Voltage_Correction[1][10];

	Flash_Protect_Buffer[35] = (unsigned long) Voltage_Correction[2][0];
	Flash_Protect_Buffer[36] = (unsigned long) Voltage_Correction[2][1];
	Flash_Protect_Buffer[37] = (unsigned long) Voltage_Correction[2][2];
	Flash_Protect_Buffer[38] = (unsigned long) Voltage_Correction[2][3];
	Flash_Protect_Buffer[39] = (unsigned long) Voltage_Correction[2][4];
	Flash_Protect_Buffer[40] = (unsigned long) Voltage_Correction[2][5];
	Flash_Protect_Buffer[41] = (unsigned long) Voltage_Correction[2][6];
	Flash_Protect_Buffer[42] = (unsigned long) Voltage_Correction[2][7];
	Flash_Protect_Buffer[43] = (unsigned long) Voltage_Correction[2][8];
	Flash_Protect_Buffer[44] = (unsigned long) Voltage_Correction[2][9];
	Flash_Protect_Buffer[45] = (unsigned long) Voltage_Correction[2][10];

	Flash_Protect_Buffer[46] = (unsigned long) Voltage_Correction[3][0];
	Flash_Protect_Buffer[47] = (unsigned long) Voltage_Correction[3][1];
	Flash_Protect_Buffer[48] = (unsigned long) Voltage_Correction[3][2];
	Flash_Protect_Buffer[49] = (unsigned long) Voltage_Correction[3][3];
	Flash_Protect_Buffer[50] = (unsigned long) Voltage_Correction[3][4];
	Flash_Protect_Buffer[51] = (unsigned long) Voltage_Correction[3][5];
	Flash_Protect_Buffer[52] = (unsigned long) Voltage_Correction[3][6];
	Flash_Protect_Buffer[53] = (unsigned long) Voltage_Correction[3][7];
	Flash_Protect_Buffer[54] = (unsigned long) Voltage_Correction[3][8];
	Flash_Protect_Buffer[55] = (unsigned long) Voltage_Correction[3][9];
	Flash_Protect_Buffer[56] = (unsigned long) Voltage_Correction[3][10]; 	
BF end */ 
	
	Flash_Protect_Buffer[57] = (unsigned long) ADC_Offset[0][0];
	Flash_Protect_Buffer[58] = (unsigned long) ADC_Offset[0][1];
	Flash_Protect_Buffer[59] = (unsigned long) ADC_Offset[0][2];
	Flash_Protect_Buffer[60] = (unsigned long) ADC_Offset[0][3];	
	
	Flash_Protect_Buffer[61] = (unsigned long) ADC_Offset[1][0];
	Flash_Protect_Buffer[62] = (unsigned long) ADC_Offset[1][1];
	Flash_Protect_Buffer[63] = (unsigned long) ADC_Offset[1][2];
	Flash_Protect_Buffer[64] = (unsigned long) ADC_Offset[1][3];	

	Flash_Protect_Buffer[65] = (unsigned long) ADC_Offset[2][0];
	Flash_Protect_Buffer[66] = (unsigned long) ADC_Offset[2][1];
	Flash_Protect_Buffer[67] = (unsigned long) ADC_Offset[2][2];
	Flash_Protect_Buffer[68] = (unsigned long) ADC_Offset[2][3];	

	Flash_Protect_Buffer[69] = (unsigned long) ADC_Offset[3][0];
	Flash_Protect_Buffer[70] = (unsigned long) ADC_Offset[3][1];
	Flash_Protect_Buffer[71] = (unsigned long) ADC_Offset[3][2];
	Flash_Protect_Buffer[72] = (unsigned long) ADC_Offset[3][3];	
	
	Flash_Protect_Buffer[73] = (unsigned long) adc_change12_reg;  	
	Flash_Protect_Buffer[74] = (unsigned long) adc_ctr12_reg;
	Flash_Protect_Buffer[75] = (unsigned long) adc_change34_reg;
	Flash_Protect_Buffer[76] = (unsigned long) channel_Adr_add;
	Flash_Protect_Buffer[77] = (unsigned long) channel_Adr_add2;
   	
	nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address, Flash_Protect_Buffer, 80);
#ifdef _Debug_	
	if (Debug_Mode) printf("protected flash data written\n");
#endif	
}

//########################################################################################################################################
void AMDFlash::Write_Protected_Config(void)
{
    int x, cnt;
	unsigned long *flash_address;
	printf("AMDFlash::Write_Protected_Config\n");	
	flash_address = (unsigned long) nasys_main_flash + protected_Config_Flash;

	printf("Flashadr : %x = %x\n", flash_address, flash_address[0]);
	
	nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address);

	//Flash_Protect_Buffer[0] = 0xFF2332FF;              // Kennung		BF del
	//Flash_Protect_Buffer[1] = tc_model;                // model		BF del
	//Flash_Protect_Buffer[2] = tc_serial;               // serial nr	BF del
	Flash_Protect_Buffer[3] = tc_production_lot1;      // production lot	
	Flash_Protect_Buffer[4] = tc_production_lot2;      // production lot	
	Flash_Protect_Buffer[5] = tc_shipment_date;        // shipment date
	//Flash_Protect_Buffer[6] = 0x00000000;
	//Flash_Protect_Buffer[7] = 0x00000000;
	//Flash_Protect_Buffer[8] = 0x00000000;
	//Flash_Protect_Buffer[9] = 0x00000000;
	//Flash_Protect_Buffer[10] = 0x00000000;
	// spezial config data
	Flash_Protect_Buffer[11] = (unsigned long) kill_adc;	           // kill adc settings for all channels
	Flash_Protect_Buffer[12] = (unsigned long) m2k_filter_Mode;	   // slow tb 2k spike filter settings for all channels	
	
/* BF del	
	Flash_Protect_Buffer[13] = (unsigned long) Voltage_Correction[0][0];
	Flash_Protect_Buffer[14] = (unsigned long) Voltage_Correction[0][1];
	Flash_Protect_Buffer[15] = (unsigned long) Voltage_Correction[0][2];
	Flash_Protect_Buffer[16] = (unsigned long) Voltage_Correction[0][3];
	Flash_Protect_Buffer[17] = (unsigned long) Voltage_Correction[0][4];
	Flash_Protect_Buffer[18] = (unsigned long) Voltage_Correction[0][5];
	Flash_Protect_Buffer[19] = (unsigned long) Voltage_Correction[0][6];
	Flash_Protect_Buffer[20] = (unsigned long) Voltage_Correction[0][7];
	Flash_Protect_Buffer[21] = (unsigned long) Voltage_Correction[0][8];
	Flash_Protect_Buffer[22] = (unsigned long) Voltage_Correction[0][9];
	Flash_Protect_Buffer[23] = (unsigned long) Voltage_Correction[0][10];

	Flash_Protect_Buffer[24] = (unsigned long) Voltage_Correction[1][0];
	Flash_Protect_Buffer[25] = (unsigned long) Voltage_Correction[1][1];
	Flash_Protect_Buffer[26] = (unsigned long) Voltage_Correction[1][2];
	Flash_Protect_Buffer[27] = (unsigned long) Voltage_Correction[1][3];
	Flash_Protect_Buffer[28] = (unsigned long) Voltage_Correction[1][4];
	Flash_Protect_Buffer[29] = (unsigned long) Voltage_Correction[1][5];
	Flash_Protect_Buffer[30] = (unsigned long) Voltage_Correction[1][6];
	Flash_Protect_Buffer[31] = (unsigned long) Voltage_Correction[1][7];
	Flash_Protect_Buffer[32] = (unsigned long) Voltage_Correction[1][8];
	Flash_Protect_Buffer[33] = (unsigned long) Voltage_Correction[1][9];
	Flash_Protect_Buffer[34] = (unsigned long) Voltage_Correction[1][10];

	Flash_Protect_Buffer[35] = (unsigned long) Voltage_Correction[2][0];
	Flash_Protect_Buffer[36] = (unsigned long) Voltage_Correction[2][1];
	Flash_Protect_Buffer[37] = (unsigned long) Voltage_Correction[2][2];
	Flash_Protect_Buffer[38] = (unsigned long) Voltage_Correction[2][3];
	Flash_Protect_Buffer[39] = (unsigned long) Voltage_Correction[2][4];
	Flash_Protect_Buffer[40] = (unsigned long) Voltage_Correction[2][5];
	Flash_Protect_Buffer[41] = (unsigned long) Voltage_Correction[2][6];
	Flash_Protect_Buffer[42] = (unsigned long) Voltage_Correction[2][7];
	Flash_Protect_Buffer[43] = (unsigned long) Voltage_Correction[2][8];
	Flash_Protect_Buffer[44] = (unsigned long) Voltage_Correction[2][9];
	Flash_Protect_Buffer[45] = (unsigned long) Voltage_Correction[2][10];

	Flash_Protect_Buffer[46] = (unsigned long) Voltage_Correction[3][0];
	Flash_Protect_Buffer[47] = (unsigned long) Voltage_Correction[3][1];
	Flash_Protect_Buffer[48] = (unsigned long) Voltage_Correction[3][2];
	Flash_Protect_Buffer[49] = (unsigned long) Voltage_Correction[3][3];
	Flash_Protect_Buffer[50] = (unsigned long) Voltage_Correction[3][4];
	Flash_Protect_Buffer[51] = (unsigned long) Voltage_Correction[3][5];
	Flash_Protect_Buffer[52] = (unsigned long) Voltage_Correction[3][6];
	Flash_Protect_Buffer[53] = (unsigned long) Voltage_Correction[3][7];
	Flash_Protect_Buffer[54] = (unsigned long) Voltage_Correction[3][8];
	Flash_Protect_Buffer[55] = (unsigned long) Voltage_Correction[3][9];
	Flash_Protect_Buffer[56] = (unsigned long) Voltage_Correction[3][10]; 	
BF end */
	
	Flash_Protect_Buffer[57] = (unsigned long) ADC_Offset[0][0];
	Flash_Protect_Buffer[58] = (unsigned long) ADC_Offset[0][1];
	Flash_Protect_Buffer[59] = (unsigned long) ADC_Offset[0][2];
	Flash_Protect_Buffer[60] = (unsigned long) ADC_Offset[0][3];	
	
	Flash_Protect_Buffer[61] = (unsigned long) ADC_Offset[1][0];
	Flash_Protect_Buffer[62] = (unsigned long) ADC_Offset[1][1];
	Flash_Protect_Buffer[63] = (unsigned long) ADC_Offset[1][2];
	Flash_Protect_Buffer[64] = (unsigned long) ADC_Offset[1][3];	

	Flash_Protect_Buffer[65] = (unsigned long) ADC_Offset[2][0];
	Flash_Protect_Buffer[66] = (unsigned long) ADC_Offset[2][1];
	Flash_Protect_Buffer[67] = (unsigned long) ADC_Offset[2][2];
	Flash_Protect_Buffer[68] = (unsigned long) ADC_Offset[2][3];	

	Flash_Protect_Buffer[69] = (unsigned long) ADC_Offset[3][0];
	Flash_Protect_Buffer[70] = (unsigned long) ADC_Offset[3][1];
	Flash_Protect_Buffer[71] = (unsigned long) ADC_Offset[3][2];
	Flash_Protect_Buffer[72] = (unsigned long) ADC_Offset[3][3];	
	
	Flash_Protect_Buffer[73] = (unsigned long) adc_change12_reg;  	
	Flash_Protect_Buffer[74] = (unsigned long) adc_ctr12_reg;
	Flash_Protect_Buffer[75] = (unsigned long) adc_change34_reg;
	Flash_Protect_Buffer[76] = (unsigned long) channel_Adr_add;
	Flash_Protect_Buffer[77] = (unsigned long) channel_Adr_add2;
   	
	nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address, Flash_Protect_Buffer, 80);
#ifdef _Debug_	
	if (Debug_Mode) printf("Flash Config Data written\n");
#endif	
}


