/*******************************************************
 Author:					Manfred Langemann
 mailto:					Manfred.Langemann t t-online.de
 Begin of project:			04.07.2015
 Latest version generated:	22.05.2022
 Processor:					ATmega8
 Filename:					AuxiDim.c
 Description:				Creates a Pulse Width Modulation (PMW) Signal at the AUXI1 or AUXI2 (or both) output ports.

 This PWM Dimming implementation needs ca. 716 bytes of program memory, i.e. 9% of available program memory !

 The two AUXI output ports, when programmed to the DIM mode, are used for light intensity controlled LEDs.
 To create this PWM, we use the TIMER2 (8-bit) on the basis of an overflow interrupt.
 Setting the prescaler to 1024 and assuming a crystal frequency of 16 MHz, we get a TIMER2 clock frequency of 15.625 kHz.
 Counting from 0 to 256 (generating an overflow) therefore takes 16.384 ms, which is 61 Hz.

 An example of the PWM signal for AUXI1 and AUXI2 outputs is shown below, where:
	AUXI1 = 1/4 High, 3/4 Low
	AUXI2 = 2/3 High, 1/3 Low

                                   PWM Period = 61 Hz
                   |<------------------- 16 ms ------------------------>|
            AUXI1  |                                                    |
             High  |------------                                        |
              Low  |            ----------------------------------------|
                   |                                                    |
            AUXI2  |                                                    |
             High  |-----------------------------------                 |
              Low  |                                   -----------------|
                   |           ^                      ^                 |
                   |           |                      |                 |
       Dim Coeff:  |0         64                     170             255|
 Poss. Min Value:   ^
 Poss. Max Value:                                                      ^


 The LED Intensity is controlled by the Weber-Fechner Law:
 ---------------------------------------------------------
 The problem is, that the human eye recognized intensity of the light is not proportional to the current of the LED,
 where the "current" is implemented by Pulse Width Modulation (PWM) of a dedicated output of the micro controller.
 E.g. running a PWM from 0 to 255 (0% intensity -> PWM = 0, ... , 100% intensity -> PWM = 255),
 the LED becomes very quickly bright and nearly stays at this level until we reach the PWM value of 100% (=255).

 See also https://www.mikrocontroller.net/articles/LED-Fading

 The problem is related to the characteristics of the human eye, which is nearly a logarithmic one.
 This has been found by Weber-Fechner and documented in this law.
 The logarithmic characteristics has been implemented by the tables belows, which are not exactly computed along
 the Weber-Fechner theory, but goes along the following rule for the example of the below table pwmtable_256_65536[256]:

 		rx = Number of steps (resolution) within table (4, 8, 16, 32, 64, 256) --> Note: in this implementation we only use "256" steps
 		ry = Resolution of PWM's (256, 1024, 65536) --> Note: in this implementation we only use the "256" resolution
		b  = Base of exponential function = 30 (found by experiments)
		x  = Input value element of {0, ... ,255}
 		y  = Computed PWM coefficients at position x, element of {0, ... ,65535}

 We then can compute:
 		y = (uint16_t)(pow(b,(float)x/rx)-1.0)/(b-1.0)*ry);

 Because we do not need such a high resolution, we only use the below table pwmtable_64_256[64],
 which has a resolution of 256 (8-bit) with 64 different PWM coefficients.

 ********************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <avr/io.h>
#include <math.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#include "General.h"
#include "Delay.h"
#include "Auxi.h"
#include "AuxiDim.h"
/*
** Definition of the output ports for the Auxi MOSFET Drivers.
** Repeated here from Auxi.c file.
*/
#define AUXI1_PORT		SBIT (PORTB, 1)
#define AUXI2_PORT		SBIT (PORTB, 0)
/*
** Private PROGMEM Constants
**
** These are the exponential tables, as defined above along the Weber-Fechner law.
** The tables are stored in the program memory, in order to reduce the occupied space in the data area.
*/
/*
** In total 4 coefficients, range ry = {0, ... ,255}.
*/
//const uint8_t pwmtable_4_256[4]  PROGMEM =
	//{
	//0, 16, 64, 255
	//};
/*
** In total 8 coefficients, range ry = {0, ... ,255}.
*/
//const uint8_t pwmtable_8_256[8]  PROGMEM =
	//{
	//0, 4, 8, 16, 32, 64, 128, 255
	//};
/*
** In total 16 coefficients, range ry = {0, ... ,255}.
*/
//const uint8_t pwmtable_16_256[16] PROGMEM =
	//{
	//0, 2, 3, 4, 6, 8, 11, 16, 23, 32, 45, 64, 90, 128, 181, 255
	//};
/*
** In total 32 coefficients, range ry = {0, ... ,255}.
*/
//const uint8_t pwmtable_32_256[32] PROGMEM =
	//{
	//0, 1, 1, 2, 2, 3, 3, 4, 5, 6, 7, 8, 9, 11, 13, 16, 19, 23,
	//27, 32, 38, 45, 54, 64, 76, 91, 108, 128, 152, 181, 215, 255
	//};
/*
** --------------------------------------------------------------------------------------
** We only use the table below !!
** In total 64 coefficients, range ry = {0, ... ,255}, 16 coefficients in one line.
** --------------------------------------------------------------------------------------
*/
const uint8_t pwmtable_64_256[64] PROGMEM =
	{
	0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2,
	2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 5, 5, 6, 7, 7, 8,
	9, 10, 11, 12, 14, 15, 17, 19, 21, 24, 26, 29, 33, 37, 41, 45,
	51, 56, 63, 69, 78, 87, 97, 108, 120, 134, 149, 166, 185, 206, 230, 255
	};
/*
** In total 64 coefficients, range ry = {0, ... ,1023}.
*/
//const uint16_t pwmtable_64_1024[64] PROGMEM =
	//{
	//0, 1, 1, 2, 2, 2, 2, 2, 3, 3, 3, 4, 4, 5, 5, 6, 6, 7, 8, 9, 10,
	//11, 12, 13, 15, 17, 19, 21, 23, 26, 29, 32, 36, 40, 44, 49, 55,
	//61, 68, 76, 85, 94, 105, 117, 131, 146, 162, 181, 202, 225, 250,
	//279, 311, 346, 386, 430, 479, 534, 595, 663, 739, 824, 918, 1023
	//};
/*
** In total 256 coefficients, range ry = {0, ... ,65536}, 16 coefficients in one line.
*/
//const uint16_t pwmtable_256_65536[256] PROGMEM =
	//{
	//0, 30, 61, 92, 123, 155, 188, 220, 253, 287, 321, 356, 391, 426, 462, 498,
	//535, 573, 611, 649, 688, 727, 767, 808, 849, 890, 932, 975, 1018, 1062, 1107, 1152,
	//1197, 1244, 1290, 1338, 1386, 1435, 1484, 1534, 1585, 1636, 1689, 1741, 1795, 1849, 1904, 1960,
	//2016, 2073, 2131, 2190, 2250, 2310, 2371, 2433, 2496, 2559, 2624, 2689, 2755, 2822, 2890, 2959,
	//3029, 3100, 3171, 3244, 3318, 3392, 3468, 3544, 3622, 3701, 3780, 3861, 3943, 4026, 4110, 4195,
	//4282, 4369, 4458, 4548, 4639, 4731, 4825, 4919, 5015, 5113, 5211, 5311, 5412, 5515, 5619, 5724,
	//5831, 5939, 6049, 6160, 6273, 6387, 6503, 6620, 6738, 6859, 6981, 7104, 7230, 7357, 7485, 7615,
	//7748, 7881, 8017, 8154, 8294, 8435, 8578, 8723, 8870, 9019, 9170, 9322, 9477, 9634, 9793, 9955,
	//10118, 10283, 10451, 10621, 10794, 10968, 11145, 11324, 11506, 11690, 11877, 12066, 12257, 12451, 12648, 12848,
	//13050, 13254, 13462, 13672, 13885, 14101, 14320, 14542, 14767, 14994, 15225, 15459, 15696, 15936, 16179, 16426,
	//16676, 16929, 17186, 17446, 17709, 17977, 18247, 18521, 18799, 19081, 19366, 19656, 19949, 20246, 20547, 20852,
	//21161, 21474, 21792, 22113, 22439, 22770, 23104, 23444, 23787, 24136, 24489, 24847, 25209, 25577, 25949, 26326,
	//26708, 27096, 27488, 27886, 28290, 28698, 29112, 29532, 29957, 30388, 30825, 31267, 31715, 32170, 32630, 33097,
	//33570, 34049, 34535, 35027, 35525, 36031, 36543, 37062, 37588, 38121, 38661, 39208, 39763, 40325, 40894, 41472,
	//42056, 42649, 43250, 43858, 44475, 45100, 45734, 46376, 47026, 47685, 48353, 49030, 49716, 50411, 51116, 51830,
	//52553, 53286, 54029, 54782, 55545, 56318, 57101, 57895, 58700, 59515, 60341, 61179, 62027, 62887, 63758, 64641
	//};
/*
** Private global variables.
*/
volatile uint8_t	Auxi1Intensity;				// the dim intensity {0, ... ,63}, as set by function AuxiDim_SetIntensity()
volatile uint8_t	Auxi2Intensity;				// the dim intensity {0, ... ,63}, as set by function AuxiDim_SetIntensity()
volatile uint8_t	Auxi1DimCoef;				// the dim coefficient {0, ... ,255} for AUXI1 as read from the program memory area
volatile uint8_t	Auxi2DimCoef;				// the dim coefficient {0, ... ,255} for AUXI2 as read from the program memory area
volatile uint8_t	Auxi1DimOn;					// the dimming status: either TRUE (ON) or FALSE (OFF)
volatile uint8_t	Auxi2DimOn;					// the dimming status: either TRUE (ON) or FALSE (OFF)
volatile uint8_t	AuxiFirst;					// {AUXI1 or AXI2}: the AUXI channel, which is served first in PWM timeline
volatile uint8_t	AuxiDimTimerCounterStart0;	// the TIMER2 counter start value used first in PWM timeline, both AUXI in DIM mode
volatile uint8_t	AuxiDimTimerCounterStart1;	// the TIMER2 counter start value used second in PWM timeline, both AUXI in DIM mode
volatile uint8_t	AuxiDimTimerCounterStart2;	// the TIMER2 counter start value used third in PWM timeline, both AUXI in DIM mode
volatile uint8_t	AuxiDimTimerCounterRun;		// 0 means: next use AuxiDimTimerCounterStart0 value
												// 1 means: next use AuxiDimTimerCounterStart1 value
												// 2 means: next use AuxiDimTimerCounterStart2 value
												// 3 means: set AuxiDimTimerCounterRun back to 0
/*
** Private Functions Declaration.
*/
void AuxiDim_PWM_On (uint8_t iOn);

/*******************************************************
 Function: AuxiDim_Init

 Purpose:
	Init the Auxi Dimming functionalities.

 Input Parameter: None
*******************************************************/
void AuxiDim_Init (void)
	{
/*
** Disable TIMER2 Overflow Interrupt.
** Switch of TIMER2 counter, CS20=0, CS21=0, CS22=0
** Enable TIMER2 Overflow Interrupt.
*/
	TCCR2 = 0;									// Switch TIMER2 OFF
	SET_BIT (TIMSK, TOIE2);						// Set TOIE2 = 1 (Enable TIMER2 Overflow Interrupt)
	Auxi1DimOn = FALSE;							// Set both AUXI DIM stati to OFF = FALSE
	Auxi2DimOn = FALSE;
	Auxi1DimCoef = 0;							// Set both DIM coeff. to 0
	Auxi2DimCoef = 0;
	Auxi1Intensity = 0;							// Set both DIM intensity values to 0
	Auxi1Intensity = 0;
/*
** Source Code to compute the PWM Dim Coefficients.
** Commented out, because defined as constants in PROGMEM area, see above.
*/
	//uint16_t	i, j;
	//float b, rx, ry;
	//b = 30.0;
	//rx = 256.0;
	//ry = 65536.0;
	//j = 0;
	//for (i=0; i<256; i++)
	//{
		//pwmtable_256[i] = (uint16_t)((pow(b,(float)i/rx)-1.0)/(b-1.0)*ry);
		//printf ("%li, ", (int32_t)pwmtable_256[i]);
		//j++;
		//if (j==16)
		//{
			//j=0;
			//printf ("\n");
		//}
	//}

	return;
	}
/*******************************************************
 Function: AuxiDim_Set

 Purpose:
	Switch specified Auxi port to the specified mode: ON or OFF or DIM
	In case of DIM mode, set the specified dim intensity level.

	In case of wrong input parameters the function does nothing.

 Input Parameter:
	- uint8_t	iAuxi	{AUXI1, AUXI2}
	- uint8_t	iMode	{AUXI_OFF, AUXI_ON,	AUXI_DIM)
				iMode == AUXI_OFF  -> switch AUXI port to OFF mode
				iMode == AUXI_ON   -> switch AUXI port to ON mode
				iMode == AUXI_DIM  -> switch AUXI port to DIM mode (PWM)
	- uint8_t	Intensity_In	Optional Dim intensity value : {0, ... ,63}

 Return Value: void
*******************************************************/
void AuxiDim_Set (uint8_t iAuxi, uint8_t iMode, uint8_t Intensity_In)
	{
	uint8_t	temp;
/*
** Check input parameters.
*/
	if (iAuxi > AUXI2) return;
	if (iMode > AUXI_DIM) return;
	if (Intensity_In > 63) return;
/*
** Save the input intensity value, because needed in function AuxiDim_GetIntensity().
*/
	if (iAuxi == AUXI1) Auxi1Intensity = Intensity_In;
	if (iAuxi == AUXI2) Auxi2Intensity = Intensity_In;
/*
** Get the Dim coefficient (used later on as start parameter for TIMER2 counter) from the program memory area
** and save it in local variable for easy access.
** Do not allow interrupts, when reading program memory area.
*/
	cli ();
	temp = pgm_read_byte (&pwmtable_64_256[Intensity_In]);
	sei ();
	if (iAuxi == AUXI1) Auxi1DimCoef = temp;
	if (iAuxi == AUXI2) Auxi2DimCoef = temp;
/*
** Compute the TIMER2 counter start values to run two different Dim values.
** Note: The Auxi Dim coefficiens are in the range of {0, ... ,255}.
*/
/*
	Example Case: Auxi1DimCoef < Auxi2DimCoef --> 64 < 170

                                          PWM Period = 61 Hz
                          |<------------------- 16 ms ------------------------>|
                   AUXI1  |                                                    |
                    High  |------------                                        |
                     Low  |            ----------------------------------------|
                          |                                                    |
                   AUXI2  |                                                    |
                    High  |-----------------------------------                 |
                     Low  |                                   -----------------|
                          |           ^                      ^                 |
                          |           |                      |                 |
               Dim Coeff: |0         64                     170             255|
  AuxiDimTimerCounterRun: |0          1                      2                 |0
                                      ^
                              AuxiFirst = AUXI1
*/
	if (Auxi1DimCoef < Auxi2DimCoef)
		{
		AuxiFirst = AUXI1;
		AuxiDimTimerCounterStart0 = 255 - Auxi1DimCoef;
		AuxiDimTimerCounterStart1 = 255 - (Auxi2DimCoef - Auxi1DimCoef);
		AuxiDimTimerCounterStart2 = Auxi2DimCoef;
		}
/*
	Example Case: Auxi1DimCoef > Auxi2DimCoef --> 170 > 64

                                           PWM Period = 61 Hz
                          |<------------------- 16 ms ------------------------>|
                   AUXI1  |                                                    |
                    High  |-----------------------------------                 |
                     Low  |                                   -----------------|
                          |                                                    |
                   AUXI2  |                                                    |
                    High  |------------                                        |
                     Low  |            ----------------------------------------|
                          |           ^                      ^                 |
                          |           |                      |                 |
               Dim Coeff: |0         64                     170             255|
  AuxiDimTimerCounterRun: |0          1                      2                 |0
                              AuxiFirst = AUXI2
*/
	if (Auxi1DimCoef > Auxi2DimCoef)
		{
		AuxiFirst = AUXI2;
		AuxiDimTimerCounterStart0 = 255 - Auxi2DimCoef;
		AuxiDimTimerCounterStart1 = 255 - (Auxi1DimCoef - Auxi2DimCoef);
		AuxiDimTimerCounterStart2 = Auxi1DimCoef;
		}
/*
	Example Case: Auxi1DimCoef = Auxi2DimCoef --> 64 = 64

                                          PWM Period = 61 Hz
                          |<------------------- 16 ms ------------------------>|
                   AUXI1  |                                                    |
                    High  |------------                                        |
                     Low  |            ----------------------------------------|
                          |                                                    |
                   AUXI2  |                                                    |
                    High  |------------                                        |
                     Low  |            ----------------------------------------|
                          |           ^                                        |
                          |           |                                        |
               Dim Coeff: |0         64                                     255|
  AuxiDimTimerCounterRun: |0          1                                        |0
                                      2
                               AuxiFirst = AUXI1
*/
	if (Auxi1DimCoef == Auxi2DimCoef)
		{
		AuxiFirst = AUXI1;
		AuxiDimTimerCounterStart0 = 255 - Auxi1DimCoef;
		AuxiDimTimerCounterStart1 = 255;
		AuxiDimTimerCounterStart2 = Auxi1DimCoef;
		}
/*
** Set the specified Auxi-x-DimOn variable to either TRUE or FALSE.
*/
	if (iAuxi == AUXI1)
		{
		if (iMode == AUXI_DIM) Auxi1DimOn = TRUE;
		if (iMode != AUXI_DIM) Auxi1DimOn = FALSE;
		}
	if (iAuxi == AUXI2)
		{
		if (iMode == AUXI_DIM) Auxi2DimOn = TRUE;
		if (iMode != AUXI_DIM) Auxi2DimOn = FALSE;
		}
/*
** If both Auxi dimming are OFF, then switch PWM to OFF.
** Otherwise switch PWM to ON.
*/
	if (Auxi1DimOn == FALSE && Auxi2DimOn == FALSE)
		{
		AuxiDim_PWM_On (FALSE);
		}
	else
		{
		AuxiDim_PWM_On (TRUE);
		}

	return;
	}
/*******************************************************
 Function: AuxiDim_GetOn

 Purpose:
	Get actual Auxi Dim ON or OFF status
	In case of wrong input parameter, the function returns the status of AUXI1.

 Input Parameter:
	- uint8_t	iAuxi	{AUXI1, AUXI2}

 Return Value
	if ON, return TRUE
	if OFF, return FALSE
*******************************************************/
uint8_t AuxiDim_GetOn (uint8_t iAuxi)
	{
	if (iAuxi == AUXI2) return Auxi2DimOn;
	return Auxi1DimOn;
	}
/*******************************************************
 Function: AuxiDim_GetIntensity

 Purpose:
	Get the intensity of the Auxi output.
	In case of wrong input parameter, the function returns the intensity value of AUXI1.

 Input Parameters:
	- uint8_t	iAuxi			{AUXI1, AUXI2}

Return Value: uint8_t
	The current Dim Intensity setting
*******************************************************/
uint8_t AuxiDim_GetIntensity (uint8_t iAuxi)
	{
	if (iAuxi == AUXI2) return Auxi2Intensity;
	return Auxi1Intensity;
	}
/*******************************************************
 Function: AuxiDim_PWM_On

 Purpose:
	Switch Auxi PWM to ON or OFF

 Input Parameter:
	- uint8_t	iOn		{TRUE or FALSE)
				if iOn == TRUE -> switch PWM to ON
				if iOn == FALSE -> switch PWM to OFF

Return Value: void
*******************************************************/
void AuxiDim_PWM_On (uint8_t iOn)
	{
	uint8_t		temp = 0;
/*
** Switch PWM to ON or OFF.
*/
	if (iOn == TRUE)
		{
/*
** Set the run counter to 1, which is the state used for the next TIMER2 overflow interrupt.
** Set both Auxi Dim ports to high level.
** In case that only one Auxi Dim is ON (TRUE), use the defined coefficients directly (always ONEs complement to 255).
** Switch TIMER2 counter to ON with prescaler of 1024.
*/
		if (Auxi1DimOn != Auxi2DimOn)					// Case where only one AUXI port is in Dim mode !
			{
			if (Auxi1DimOn == TRUE  && Auxi2DimOn == FALSE) temp = 255 - Auxi1DimCoef;
			if (Auxi1DimOn == FALSE && Auxi2DimOn == TRUE)  temp = 255 - Auxi2DimCoef;
			if (Auxi1DimOn == TRUE) AUXI1_PORT = 1;		// Set output port pins to high level, dependent on Dim-ON-Status
			if (Auxi2DimOn == TRUE) AUXI2_PORT = 1;
			AuxiDimTimerCounterRun = 1;					// use next the AuxiDimTimerCounterStart1 value
			TCNT2 = temp;
			TCCR2 |= (1<<CS20) | (1<<CS21) | (1<<CS22);	// Switch TIMER2 to ON with prescaler to 1024
			}

		if (Auxi1DimOn == TRUE && Auxi2DimOn == TRUE)	// Case where both AUXI port are in Dim mode !
			{
			AUXI1_PORT = 1;								// Set output port pins to high level
			AUXI2_PORT = 1;
			AuxiDimTimerCounterRun = 0;					// use next the AuxiDimTimerCounterStart0 value
			TCNT2 = AuxiDimTimerCounterStart0;
			TCCR2 |= (1<<CS20) | (1<<CS21) | (1<<CS22);	// Switch TIMER2 to ON with prescaler to 1024
			}
		}
	else
		{
/*
** Switch TIMER2 to OFF: stop clock pulses.
*/
		TCCR2 = 0;
		}

	return;
	}
/*******************************************************
 Interrupt TIMER2 Overflow

 Purpose:
	Perform the Auxi Dim functionality.
*******************************************************/
ISR (TIMER2_OVF_vect)
	{
/*
** First handle the cases, where only one Auxi output is in Dim mode.
**
** ----------------------------------------------------------
** Here: Only Auxi1 output port is in Dim mode.
** ----------------------------------------------------------
*/
	if (Auxi1DimOn == TRUE && Auxi2DimOn == FALSE)
		{
		if (AuxiDimTimerCounterRun == 0)
			{
			AUXI1_PORT = 1;
			TCNT2 = 255 - Auxi1DimCoef;
			}
		else
			{
			AUXI1_PORT = 0;
			TCNT2 = Auxi1DimCoef;
			}
/*
** Increment AuxiDimTimerCounterRun.
** Reset it, if value is 2.
*/
		AuxiDimTimerCounterRun++;
		if (AuxiDimTimerCounterRun == 2) AuxiDimTimerCounterRun = 0;
		return;
		}
/*
** ----------------------------------------------------------
** Here: Only Auxi2 output port is in Dim mode.
** ----------------------------------------------------------
*/
	if (Auxi1DimOn == FALSE && Auxi2DimOn == TRUE)
		{
		if (AuxiDimTimerCounterRun == 0)
			{
			AUXI2_PORT = 1;
			TCNT2 = 255 - Auxi2DimCoef;
			}
		else
			{
			AUXI2_PORT = 0;
			TCNT2 = Auxi2DimCoef;
			}
/*
** Increment AuxiDimTimerCounterRun.
** Reset it, if value is 2.
*/
		AuxiDimTimerCounterRun++;
		if (AuxiDimTimerCounterRun == 2) AuxiDimTimerCounterRun = 0;
		return;
		}
/*
** ----------------------------------------------------------
** When we are here, both Auxi output ports are in Dim mode.
** ----------------------------------------------------------
*/
	if (AuxiDimTimerCounterRun == 0)
		{
		AUXI1_PORT = 1;
		AUXI2_PORT = 1;
		TCNT2 = AuxiDimTimerCounterStart0;
		goto next;
		}

	if (AuxiDimTimerCounterRun == 1)
		{
		if (AuxiFirst == AUXI1)
			{
			AUXI1_PORT = 0;
			}
		else
			{
			AUXI2_PORT = 0;
			}
		TCNT2 = AuxiDimTimerCounterStart1;
		}
	else		// AuxiDimTimerCounterRun = 2
		{
		AUXI1_PORT = 0;			// For simplicity we set both to 0, knowing, that one port is already at 0 !
		AUXI2_PORT = 0;
		TCNT2 = AuxiDimTimerCounterStart2;
		}
/*
** Increment AuxiDimTimerCounterRun.
** Reset it, if value is 3 !! --> different as above for only one AUXI port in Dim mode.
*/
next:
	AuxiDimTimerCounterRun++;
	if (AuxiDimTimerCounterRun == 3) AuxiDimTimerCounterRun = 0;

	return;
	}


