/*******************************************************
 Author:					Manfred Langemann
 mailto:					Manfred.Langemann t t-online.de
 Begin of project:			18.01.2022
 Latest version generated:	07.06.2022
 Filename:					Block.c
 Processor:					ATmega8

 Physical Setup:
 ===============

	    |<-----------------------   Block     ----------------------->|
	    |                                                             |
	    |<-----     Block A     ----->|<-----     Block B       ----->|
	    |                             |                               |
	    |Signal A <--                 |                   --> Signal B|
	------------------------------------------------------------------------   Notes:
	....#............x.......O........#......O......x.................#.....   - At points # we insert Current Separations in the Rail
	----------------------------------+-------------------------------------   - At points O we apply either B or B#
	                 ^                ^             ^     		               - At points x we measure the block occupied status
	                 |                |             |						   - At point + we measure the time, where
             Occupied Detector A      |     Occupied Detector B                  the pickup shoe occupies both blocks
	                                  ^
	                   Here we measure the time delta-t,
                  where the pickup shoe occupies both blocks

 General Block Functionality:
 ============================

 Each Block is divided into two subblocks, which are Block A and Block B.
 Each subblock can be applied with Bahnstrom B or Bremsspannung B# (Bremsspannung = -18V).
 Basically we have two cases:

	Case 1: train drives from left to right: A -> B:
	- When the train enters Block A, this is detected by the Occupied Detector in Block A
	- When Signal B is Green or Green/Yellow, the train drives through block B and finally leaves block B
	- When Signal B is Red, Block B is applied with Bremsspannung and the train stops at Signal B

	Case 2: train drives from right to left: B -> A:
	- When the train enters Block B, this is detected by the Occupied Detector in Block B
	- When Signal A is Green or Green/Yellow, the train drives through block A and finally leaves block A
	- When Signal A is Red, Block A is applied with Bremsspannung and the train stops at Signal A

 There is no need for a so called transition area, which is typically necessary to avoid an electrical shortcut
 between Bahnstrom and Bremsspannung. This feature is implemented by an electrical shortcut protection
 on the basis of the Bogobit Classic Bremsmodul, which works fine.

 However, this functionality has the drawback, that the train does not stop right in front of the Signal,
 but much earlier. In the worst case, the train already stops right behind the border of Block A & B and
 not right before the signal at the end of the block (either signal A or B, depending on the drive direction).

 For this purpose, we indirectly measure the velocity of the train. This is accomplished by measuring the time,
 while the pickup shoe of the locomotive is driving over the sectioning point (Trennstelle) between
 block A & B. Also here we use the block occupied sensors, which results in the following sequencing:

 When the locomotive is driving from A -> B and crossing the sectioning point A/B, then the following happens:
	- at the begin, only block A is occupied, then
	- B becomes occupied, and a little bit later
	- A becomes free

 When the locomotive is driving from B -> A and crossing the sectioning point A/B, then the following happens:
	- at the begin, only block B is occupied, then
	- A becomes occupied, and a little bit later
	- B becomes free

 The time delta-t, which is called above as "a little bit later", depends on the drive speed of the train and the physical
 length of the pickup shoe. Typical values are:
	- Lowest possible speed:	delta-t = slightly more than 0.5 seconds
	- Highest possible speed:	delta-t = 0.05 - 0.1 seconds

 Luckily, all "normal" Mrklin locomotives have the same pickup shoe length, so we can assume for this parameter a constant.




 introducing a
 Shortcut detector, which works as follows for the example of driving from left to right, i.e. from A to B:

	- When the train enters Block A and the Signal A is red, the Block B is applied with the Bremsspannung
	- When the Schleifer (current collector) of the train starts to shortcut Block A and Block B:
		- a Timer is started
		- Block B is applied with the regular Bahnstrom, which allows the train to drive with nominal speed
		- The Timer is stopped, when the Schleifer of the train is no longer short-cutting the Block A and B
		- The final value of the Timer is a inversely proportional to the velocity of the train:
			- high Timer value -> low train speed
			- low Timer value  -> high train speed
		- Then the regular Bahnstrom remains applied as long as the value of the inversely timer value.
		- After this Timer value, the Bremsspannung will be applied to the Block A or B and the train starts to brake.
		- This will ensure, that the train nearly stops in front of the Signal.

 This however only works, when the decoder (CV253 - CV255) of all used locomotives are set to "constant brake distance".

 Frage 1: was ist mit geschobenen Wagen ? Kommt der erste geschobene Wagen vor dem Signal zum Halt ?
 Frage 2: was ist, wenn der Schleifer in Fahrtrichtung hinten an der Lok ist ?

 This procedure is implemented in the below source code by a state machine, which has the following sequential states:

	- STATE_INIT			The software performs the initiation of the block functionality
	- STATE_FREE			No train in Block A or B
	- STATE_OCCUPIED		The train has entered either Block A or B, while the signal in drive direction is red
	- STATE_DRIVE_THROUGH	The train shall drive through the block, when the Signal is not red
	- STATE_SHORTCUT		The train is driving over the sectioning point between Block A & B
							and measures the time of crossing the sectioning point, which gives a value of the actual train speed
	- STATE_BRAKE_DELAY,	A delay is introduced before the Bremsspannung will be applied to the related Block part
	- STATE_BRAKE_STOP		The train is braking or has already stopped at the Signal
	- STATE_LEAVING_BLOCK	The train is leaving the Block A or B (dependent on the drive direction),
							because the signal was set to green.
							When the train has left the Block, then the state is set back to STATE_FREE.
	- STATE_ERROR			Both Blocks (A & B) are occupied, which shall not happen, in particular when booting the system

 It shall be noted, that the state machine does not change any of the two signals A and B, associated to this Block.
 The signals have to be set by the user directly.
 ********************************************************/
#include <avr/io.h>
#include <stdio.h>
#include <math.h>
#include <avr/interrupt.h>

#include "General.h"
#include "Signal.h"
#include "EEMEM.h"
#include "Block.h"
#include "Auxi.h"
/*
** Define the input ports for the block occupied lines.
*/
#define BLOCK_A_OCCU_DDR			SBIT (DDRC, 1)
#define BLOCK_A_OCCU_PIN			(PINC & (1<<PC1))
#define BLOCK_B_OCCU_DDR			SBIT (DDRC, 0)
#define BLOCK_B_OCCU_PIN			(PINC & (1<<PC0))
/*
** Define the output ports for the relays for switching between Bahnstrom (B) and Bremsspannung (B#).
*/
#define BLOCK_A_DRIVE_BRAKE_DDR		SBIT (DDRD, 3)
#define BLOCK_A_DRIVE_BRAKE_PORT	SBIT (PORTD, 3)
#define BLOCK_B_DRIVE_BRAKE_DDR		SBIT (DDRD, 0)
#define BLOCK_B_DRIVE_BRAKE_PORT	SBIT (PORTD, 0)


#define MAX_OCCU_COUNTER	20
/*
** Global variables.
*/
volatile uint8_t	iBlockState;						// the current state of the state machine of the block
volatile uint8_t	iBlockStateErrorFlag;				// a flag, indicating that we have already performed once the STATE_ERROR associated routine
volatile uint8_t	iBlockOccupied[2];					// for Block A=[0] and B=[1], Element of {TRUE, FALSE}
volatile uint8_t	iBlockPower[2];						// for Block A=[0] and B=[1], Element of {BLOCK_DRIVE_POWER, BLOCK_BRAKE_POWER}
volatile uint8_t	iBlockTrainDriveDirection;			// Element of {BLOCK_TRAIN_DRIVE_DIR_A2B, BLOCK_TRAIN_DRIVE_DIR_B2A, BLOCK_TRAIN_DRIVE_DIR_VOID}
volatile uint8_t	iBlockBrakeDelayFactorA2B;			// The delay factor for driving from Block A to B, before the Bremsspannung will be applied
volatile uint8_t	iBlockBrakeDelayFactorB2A;			// The delay factor for driving from Block B to A, before the Bremsspannung will be applied
volatile uint8_t	iBlockBrakeDelayIsRunning;			// either FALSE or TRUE
volatile uint8_t	iBlockBrakeShortCutFlag;			// either FALSE when no shortcut or TRUE when shortcut
volatile uint16_t	iBlockBrakeShortCutCounter;			// measures the time, while the train Schleifer shortcuts block A and B
volatile uint16_t	iBlockBrakeShortCutCompareCounter;	// the Brake delay counter compare value
volatile uint8_t	iBlock_A_StatusCounter;				// Used when measuring the "shortcut" duration between block A and B
volatile uint8_t	iBlock_B_StatusCounter;				// Used when measuring the "shortcut" duration between block A and B
volatile uint8_t	iBlock_A_StatusOld;					// Used when measuring the "shortcut" duration between block A and B
volatile uint8_t	iBlock_B_StatusOld;					// Used when measuring the "shortcut" duration between block A and B

volatile uint8_t	iMerkerA;
volatile uint8_t	iMerkerB;
/*******************************************************
 Public Function Block_Init

 Purpose:
	Init Block functionality

 Input Parameter: void

 Return Value: void
 *******************************************************/
void Block_Init (void)
	{
/*
** Set actual Block state.
*/
	iBlockState = STATE_INIT;
	printf ("State => INIT\n");
/*
** Init the Signal functionality, because we use already here the function Signal_Set().
*/
	Signal_Init ();
/*
** Set Block Occupied sensor ports to input.
*/
	BLOCK_A_OCCU_DDR = 0;
	BLOCK_B_OCCU_DDR = 0;
/*
** Set Block Drive (Fahrt) / Brake (Bremsen) ports to output (2 Relays).
*/
	BLOCK_A_DRIVE_BRAKE_DDR = 1;
	BLOCK_B_DRIVE_BRAKE_DDR = 1;
/*
** Read from the EEMEM the Block Brake Delay Factors.
*/
	EEMEM_ReadBrakeDelayBlockA2B (&iBlockBrakeDelayFactorA2B);
	EEMEM_ReadBrakeDelayBlockB2A (&iBlockBrakeDelayFactorB2A);
/*
** Update the block occupied values, for Block A and B.
** Set other parameters to default values.
*/
	iBlock_A_StatusCounter = 0;
	iBlock_B_StatusCounter = 0;
	iBlock_A_StatusOld = BLOCK_IS_FREE;
	iBlock_B_StatusOld = BLOCK_IS_FREE;
	iBlockOccupied[BLOCK_A] = FALSE;
	iBlockOccupied[BLOCK_B] = FALSE;
	Block_UpdateOccupiedStatus ();
	iBlockStateErrorFlag = FALSE;
	iBlockBrakeShortCutFlag = FALSE;
	iBlockBrakeShortCutCounter = 0;
	iBlockBrakeShortCutCompareCounter = 0;
	iBlockBrakeDelayIsRunning = FALSE;
	iBlockTrainDriveDirection = BLOCK_TRAIN_DRIVE_DIR_VOID;
/*
** Set both signals to green as default.
** The function Signal_Set() also sets the block power (Bahnstrom/Bremsspannung):
**	- Signal Red: Bremsspannung (BLOCK_BRAKE_POWER)
**	- Signal Green: Bahnstrom (BLOCK_DRIVE_POWER)
**	- Signal Green/Yellow: Bahnstrom (BLOCK_DRIVE_POWER)
**	- Signal Test (all ON): --> will not change the current block power
** Will be updated, depending on which block part (A or B) is occupied.
*/
	Signal_Set (SIG_A, SIG_GREEN);	// will also set the block power to BLOCK_DRIVE_POWER
	Signal_Set (SIG_B, SIG_GREEN);	// will also set the block power to BLOCK_DRIVE_POWER
/*
** If no Block is occupied:
**	- apply Bahnstrom to Block A and B
**	- set Block state to STATE_FREE
**	- set train drive direction variable to BLOCK_TRAIN_DRIVE_DIR_VOID
**
** If both Blocks are occupied:
**	- this represents an error
**	- apply Bremsspannung to both Blocks
**	- set Block state to STATE_ERROR
**	- set train drive direction variable to BLOCK_TRAIN_DRIVE_DIR_VOID
**
** ##################################
** If both Blocks are shortcut by the Schleifer: ?? Geht das, wenn beide Blocks Bremsspannung haben ??? siehe unten
** ##################################
**	- this represents an error
**	- apply Bremsspannung to both Blocks
**	- set Block state to STATE_ERROR
**	- set train drive direction variable to BLOCK_TRAIN_DRIVE_DIR_VOID
**
** If only one block is occupied, then we assume, that the train is standing in front of the associated Signal:
** ###############################################
** Diese Logik ist noch falsch !!
** ###############################################
**	- Block A is occupied:
**		- assuming train is standing in front of Signal A
**		- apply Bahnstrom to Block B
**		- apply Bremsspannung to Block A
**		- set drive direction from B to A
**		- set starting Block state to STATE_BRAKE_STOP
**		- set Signal A to red
**		- set train drive direction variable to BLOCK_TRAIN_DRIVE_DIR_B2A
**	- Block B is occupied:
**		- assuming train is standing in front of Signal B
**		- apply Bahnstrom to Block A
**		- apply Bremsspannung to Block B
**		- set drive direction from A to B
**		- set starting Block state to STATE_BRAKE_STOP
**		- set Signal B to red
**		- set train drive direction variable to BLOCK_TRAIN_DRIVE_DIR_A2B
**
** -------------------------------------------------------------
** If no Block part is occupied:
*/
	if (iBlockOccupied[BLOCK_A] == FALSE && iBlockOccupied[BLOCK_B] == FALSE)
		{
		Block_SetPower (BLOCK_A, BLOCK_DRIVE_POWER);
		Block_SetPower (BLOCK_B, BLOCK_DRIVE_POWER);
		iBlockState = STATE_FREE;
		printf ("State => FREE\n");
		return;
		}
/*
** If both Block parts are occupied.
*/
	else if (iBlockOccupied[BLOCK_A] == TRUE && iBlockOccupied[BLOCK_B] == TRUE)
		{
		Signal_Set (BLOCK_A, SIG_RED);
		Signal_Set (BLOCK_B, SIG_RED);
		Block_SetPower (BLOCK_A, BLOCK_BRAKE_POWER);
		Block_SetPower (BLOCK_B, BLOCK_BRAKE_POWER);
		iBlockState = STATE_ERROR;
		printf ("State => ERROR: Both blocks are occupied\n");
		return;
		}
/*
** If only Block A is occupied:
** ###############################################
** Diese Logik ist noch falsch !!
** ###############################################
*/	else if (iBlockOccupied[BLOCK_A] == TRUE)
		{
		Block_SetPower (BLOCK_A, BLOCK_BRAKE_POWER);
		Block_SetPower (BLOCK_B, BLOCK_DRIVE_POWER);
		iBlockState = STATE_BRAKE_STOP;
		printf ("State => BRAKE_STOP\n");
		iBlockTrainDriveDirection = BLOCK_TRAIN_DRIVE_DIR_B2A;
		Signal_Set (SIG_B, SIG_GREEN);
		return;
		}
/*
** If only Block B is occupied:
*/
	else
		{
		Block_SetPower (BLOCK_A, BLOCK_DRIVE_POWER);
		Block_SetPower (BLOCK_B, BLOCK_BRAKE_POWER);
		iBlockState = STATE_BRAKE_STOP;
		printf ("State => BRAKE_STOP\n");
		iBlockTrainDriveDirection = BLOCK_TRAIN_DRIVE_DIR_A2B;
		Signal_Set (SIG_A, SIG_GREEN);
		}

	return;
	}
/*******************************************************
 Public Function Block_Processor

 Purpose: This is the state machine processor for the block control unit.
			See description above in header of this file

 Input Parameter: void

 Return Value: void
 *******************************************************/
void Block_Processor (void)
	{
/*
** =================================================================
** State: STATE_INIT
** =================================================================
*/
	if (iBlockState == STATE_INIT)
		{
/*
** Perform the Block Init function once more.
*/
		Block_Init ();
		return;
		}
/*
** =================================================================
** State: STATE_FREE
** =================================================================
*/
	if (iBlockState == STATE_FREE)
		{
/*
** Update the block occupied status.
*/
		Block_UpdateOccupiedStatus ();
/*
** If no Block is occupied, do nothing and return.
*/
		if (iBlockOccupied[BLOCK_A] == FALSE && iBlockOccupied[BLOCK_B] == FALSE) return;
/*
** Check on error: both blocks are occupied.
** In this case set the state to STATE_ERROR.
*/
		if (iBlockOccupied[BLOCK_A] == TRUE && iBlockOccupied[BLOCK_B] == TRUE)
			{
			Signal_Set (BLOCK_A, SIG_RED);
			Signal_Set (BLOCK_B, SIG_RED);
			Block_SetPower (BLOCK_A, BLOCK_BRAKE_POWER);
			Block_SetPower (BLOCK_B, BLOCK_BRAKE_POWER);
			iBlockState = STATE_ERROR;
			printf ("State => ERROR\n");
			return;
			}
/*
** When we are here, at least one Block (A or B) is occupied.
** Determine the train drive direction, based on which Block is now occupied.
**
** If the Signal in drive direction is red, then:
**	- set state to STATE_OCCUPIED
**
** If the Signal in drive direction is NOT red, then:
**	- set state to STATE_DRIVE_THROUGH
**
** Block A is occupied:
** --------------------
*/
		if (iBlockOccupied[BLOCK_A] == TRUE)
			{
			iBlockTrainDriveDirection = BLOCK_TRAIN_DRIVE_DIR_A2B;
			if (Signal_Get (SIG_B) == SIG_RED)
				{
				iBlockState = STATE_OCCUPIED;
				printf ("State => OCCUPIED : A->B\n");
				}
			else
				{
				iBlockState = STATE_DRIVE_THROUGH;
				printf ("State => DRIVE_THROUGH : A->B\n");
				iMerkerA = TRUE;
				iMerkerB = TRUE;
				}
			return;
			}
/*
** Block B is occupied:
** --------------------
*/
		if (iBlockOccupied[BLOCK_B] == TRUE)
			{
			iBlockTrainDriveDirection = BLOCK_TRAIN_DRIVE_DIR_B2A;
			if (Signal_Get (SIG_A) == SIG_RED)
				{
				iBlockState = STATE_OCCUPIED;
				printf ("State => OCCUPIED : B->A\n");
				}
			else
				{
				iBlockState = STATE_DRIVE_THROUGH;
				printf ("State => DRIVE_THROUGH : B->A\n");
				iMerkerA = TRUE;
				iMerkerB = TRUE;
				}
			}
		return;
		}
/*
** =================================================================
** State: STATE_OCCUPIED
** =================================================================
*/
	if (iBlockState == STATE_OCCUPIED)
		{
/*
** We expect in this state that the train starts to drives over the sectioning point (Trennstelle),
** separating block A and B. We call this in the following as a "shortcut", although it isn't a real shortcut.
**
** When the train is driving from A -> B and crossing the sectioning point, then the following happens:
**	- B becomes occupied, and a little bit later
**	- A becomes free
** When the train is driving from B -> A and crossing the sectioning point, then the following happens:
**	- A becomes occupied, and a little bit later
**	- B becomes free
**
** The time delta-t (a little bit later) depends on the drive speed of the train and the physical length of the pick-up device.
** Typical values are:
**	- Lowest possible speed:	t = slightly more than 2 seconds
**	- Highest possible speed:	t = 0.02 - 0.1 seconds
**
** We have to measure the time delta-t by starting the 8-bit TIMER0 and counting the number of overflows,
** which represents the proportional number of the time delta-t.
**
** Depending on the drive direction within the block, see above sequencing, we now do the following:
** When the follow-on block becomes occupied:
**	- reset the iBlockBrakeShortCutCounter to 0
**	- set iBlockBrakeShortCutFlag to TRUE, indicating in ISR (TIMER0_OVF_vect)
**	- set the TIMER0 counter to 0
**	- start the TIMER0 with a prescaler of 64
**	- enable the TIMER0 overflow interrupt, which measures (counts) the duration of the shortcut
**	- set block state to STATE_SHORTCUT
**
** Hence with a 16 MHz crystal and a TIMER0 prescaler of 256 the ISR (INT0_vect) will be called every 0.4 ms.
** Within ISR (INT0_vect) we will count the number of overflows by an uint16_t variable,
** because a uint8_t variable is not sufficient to cover the possible maximum duration of the shortcut of more than 2 seconds.
** Note: with an uint16_t variable we can count for a time delta-t of up to 26 seconds.
*/
		Block_UpdateOccupiedStatus ();
/*
** For the drive direction A -> B
** When Block B becomes occupied, start TIMER0 to measure the duration of the shortcut.
*/
		if (iBlockTrainDriveDirection == BLOCK_TRAIN_DRIVE_DIR_A2B)
			{
			if (iBlockOccupied[BLOCK_B] == TRUE)
				{
				iBlockBrakeShortCutCounter = 0;
				iBlockBrakeShortCutFlag = TRUE;
				iBlockBrakeDelayIsRunning = FALSE;
				TCNT0 = 0;
				TCCR0 = (1<<CS00) | (1<<CS01);
				SET_BIT (TIMSK, TOIE0);
				iBlockState = STATE_SHORTCUT;
				printf ("Start Shortcut A->B\n");
				}
			}
/*
** For the drive direction B -> A
** When Block A becomes occupied, start TIMER0 to measure the duration of the shortcut.
*/
		if (iBlockTrainDriveDirection == BLOCK_TRAIN_DRIVE_DIR_B2A)
			{
			if (iBlockOccupied[BLOCK_A] == TRUE)
				{
				iBlockBrakeShortCutCounter = 0;
				iBlockBrakeShortCutFlag = TRUE;
				iBlockBrakeDelayIsRunning = FALSE;
				TCNT0 = 0;
				TCCR0 = (1<<CS00) | (1<<CS01);
				SET_BIT (TIMSK, TOIE0);
				iBlockState = STATE_SHORTCUT;
				printf ("Start Shortcut B->A\n");
				}
			}
		return;
		}
/*
** =================================================================
** State: STATE_SHORTCUT
** =================================================================
*/
	if (iBlockState == STATE_SHORTCUT)
		{
/*
** We expect in this state the end, where the train pick-up device drives over the sectioning point (Trennstelle),
** separating block A and B. According to the notation as defined in state STATE_OCCUPIED,
** we expect, that the block, which the train is leaving, becomes free.
** We check this on the basis of the actual drive direction.
**
** When this happens, we:
**	- stop TIMER0
**	- set TIMER0 counter back to 0
**	- set prescaler to 1024 --> which starts the counter
**	- set global variable iBlockBrakeDelayIsRunning to TRUE, indicating that the brake delay is now running
**	- set the block state to STATE_BRAKE_DELAY
*/
/*
** For the drive direction A -> B
** When Block A becomes free.
*/
		Block_UpdateOccupiedStatus ();
		if (iBlockTrainDriveDirection == BLOCK_TRAIN_DRIVE_DIR_A2B)
			{
			if (iBlockOccupied[BLOCK_A] == FALSE)	// Block A is now free
				{
				CLEAR_BIT (TIMSK, TOIE0);
				TCCR0 = 0;	// Disable TIMER0 overflow
				CLEAR_BIT (TIMSK, TOIE0);
				iBlockBrakeShortCutFlag = FALSE;
				iBlockBrakeDelayIsRunning = TRUE;
				iBlockBrakeShortCutCompareCounter = 0;
				printf ("BBSCC = %i\n", iBlockBrakeShortCutCounter);
				//iBlockBrakeShortCutCounter = iBlockBrakeShortCutCounter * iBlockBrakeDelayFactorA2B;
				//iBlockBrakeShortCutCounter = 1;
				iBlockBrakeShortCutCounter = Block_ComputeDelayOverflowCounts (iBlockBrakeShortCutCounter);
				iBlockState = STATE_BRAKE_DELAY;
				printf ("BBSCC new = %i\n", iBlockBrakeShortCutCounter);
				TCNT0 = 0;
				TCCR0 = (1<<CS00)| (1<<CS01);	// Set prescaler to 64
				SET_BIT (TIMSK, TOIE0);
				}
			return;
			}
/*
** For the drive direction B -> A
** When Block B becomes free.
*/
		if (iBlockTrainDriveDirection == BLOCK_TRAIN_DRIVE_DIR_B2A)
			{
			if (iBlockOccupied[BLOCK_B] == FALSE)	// Block B is now free
				{
				CLEAR_BIT (TIMSK, TOIE0);
				TCCR0 = 0;	// Disable TIMER0 overflow
				CLEAR_BIT (TIMSK, TOIE0);
				iBlockBrakeShortCutFlag = FALSE;
				iBlockBrakeDelayIsRunning = TRUE;
				iBlockBrakeShortCutCompareCounter = 0;
				printf ("BBSCC = %i\n", iBlockBrakeShortCutCounter);
				//iBlockBrakeShortCutCounter = iBlockBrakeShortCutCounter * iBlockBrakeDelayFactorB2A;
				//iBlockBrakeShortCutCounter = 1;
				iBlockBrakeShortCutCounter = Block_ComputeDelayOverflowCounts (iBlockBrakeShortCutCounter);
				iBlockState = STATE_BRAKE_DELAY;
				printf ("BBSCC new = %i\n", iBlockBrakeShortCutCounter);
				TCNT0 = 0;
				TCCR0 = (1<<CS00)| (1<<CS01);	// Set prescaler to 64
				SET_BIT (TIMSK, TOIE0);
				}
			}
		return;
		}
/*
** =================================================================
** State: STATE_BRAKE_DELAY
** =================================================================
*/
	if (iBlockState == STATE_BRAKE_DELAY)
		{
/*
** In this state we expect that the brake delay has ended, indicated in ISR (TIMER0_OVF_vect).
** If the brake delay has ended, indicated by iBlockBrakeDelayIsRunning == FALSE, then:
**	- apply the Bremsspannung to the associated block part
**	- set state to STATE_BRAKE_STOP
*/
		if (iBlockBrakeDelayIsRunning == FALSE)
			{
			if (iBlockTrainDriveDirection == BLOCK_TRAIN_DRIVE_DIR_A2B) Block_SetPower (BLOCK_B, BLOCK_BRAKE_POWER);
			if (iBlockTrainDriveDirection == BLOCK_TRAIN_DRIVE_DIR_B2A) Block_SetPower (BLOCK_A, BLOCK_BRAKE_POWER);
			iBlockState = STATE_BRAKE_STOP;
			printf ("State => BRAKE_STOP\n");
			}
		return;
		}
/*
** =================================================================
** State: STATE_BRAKE_STOP
** =================================================================
*/
	if (iBlockState == STATE_BRAKE_STOP)
		{
/*
** In this state the train is braking or has already stopped in front of a red Signal.
** This state can only be left by a user setting of the Signal from Red to Green or Green/Yellow.
*/
		if (iBlockTrainDriveDirection == BLOCK_TRAIN_DRIVE_DIR_A2B)
			{
			if (Signal_Get (SIG_B) != SIG_RED)
				{
				Block_SetPower (BLOCK_B, BLOCK_DRIVE_POWER);
				iBlockState = STATE_LEAVING_BLOCK;
				printf ("State => LEAVING_BLOCK : A->B\n");
				}
			}

		if (iBlockTrainDriveDirection == BLOCK_TRAIN_DRIVE_DIR_B2A)
			{
			if (Signal_Get (SIG_A) != SIG_RED)
				{
				Block_SetPower (BLOCK_A, BLOCK_DRIVE_POWER);
				iBlockState = STATE_LEAVING_BLOCK;
				printf ("State => LEAVING_BLOCK : B->A\n");
				}
			}
		return;
		}
/*
** =================================================================
** State: STATE_DRIVE_THROUGH
** =================================================================
*/
	if (iBlockState == STATE_DRIVE_THROUGH)
		{
/*
** Update the block occupied status.
** When train has left the Block (Block A and B is free), we set the state back to STATE_FREE.
** Also invalidate the block drive direction.
*/
		Block_UpdateOccupiedStatus ();
		if (iBlockOccupied[BLOCK_A] == FALSE)
			{
			if (iMerkerA == TRUE)
				{
				iMerkerA = FALSE;
				printf ("A is free\n");
				}
			}
		if (iBlockOccupied[BLOCK_B] == TRUE)
			{
			if (iMerkerB == TRUE)
				{
				iMerkerB = FALSE;
				printf ("B is occupied\n");
				}
			}


		if (iBlockOccupied[BLOCK_A] == FALSE && iBlockOccupied[BLOCK_B] == FALSE)
			{
			iBlockState = STATE_FREE;
			iBlockTrainDriveDirection = BLOCK_TRAIN_DRIVE_DIR_VOID;
			printf ("State => FREE\n");
			}
		return;
		}
/*
** =================================================================
** State: STATE_LEAVING_BLOCK
** =================================================================
*/
	if (iBlockState == STATE_LEAVING_BLOCK)
		{
/*
** In this state the train is leaving the block,
** because the signal has been switched by the user from red to Green or Green/Yellow.
** First update the block occupied status.
*/
		Block_UpdateOccupiedStatus ();
/*
** Having left the block, the two occupied sensors indicate "Block is not occupied".
** In this case we set the block state back to STATE_FREE and invalidate the drive direction.
*/
		if (iBlockOccupied[BLOCK_A] == FALSE && iBlockOccupied[BLOCK_B] == FALSE)
			{
			iBlockState = STATE_FREE;
			printf ("State => FREE\n");
			iBlockTrainDriveDirection = BLOCK_TRAIN_DRIVE_DIR_VOID;
			}

		return;
		}
/*
** =================================================================
** State: STATE_ERROR
** =================================================================
**
** We have detected an error in the state machine.
** Set signals of both block parts to red and apply Bremsspannung to both block parts.
** We can leave this state only by commanding a new state by function Block_SetState().
*/
	if (iBlockState == STATE_ERROR)
		{
		if (iBlockStateErrorFlag == FALSE)
			{
			// ########## to do something --> send error message to Master
			Signal_Set (SIG_A, SIG_RED);
			Signal_Set (SIG_B, SIG_RED);
			Block_SetPower (BLOCK_A, BLOCK_BRAKE_POWER);
			Block_SetPower (BLOCK_B, BLOCK_BRAKE_POWER);
			iBlockStateErrorFlag = TRUE;
			}
		return;
		}
/*
** =================================================================
** State: Wrong block state detected
** =================================================================
** When we are here, then we have a wrong block state, which is an error and should not happen.
** We therefore set the state to STATE_ERROR.
*/
	iBlockState = STATE_ERROR;
	return;
	}
/*******************************************************
 Public Function Block_UpdateOccupiedStatus

 Purpose:
	Update the Block occupied status in iBlockOccupied[] for Block A and Block B:
		Block Occupied		=>	iBlockOccupied[] = TRUE
		Block Not Occupied	=>	iBlockOccupied[] = FALSE

 Input Parameter: void

 Return Value: void

 Software Description:
 =====================
 The measurement of the block status (free or occupied) requires some dedicated hardware and software
 provisions, because the Schleifer (pick-up shoe) of the locomotive generates some random noise signals,
 when entering or leaving a block part. This behavior is depicted in the below scheme, where the loco enters a block part:

  +5V : Block Free        >-------------  ------ ----    -------  ---------
                                       |  |    | |  |    |     |  |       |
                                       |  |    | |  |    |     |  |       |
                                       |  |    | |  |    |     |  |       |
   0V : Block Occupied                 ----    ---  ------     ----       ---------------------------> Time t

 This signal behavior will be corrected by the implemented software to the following signal:

  +5V : Block Free        >---------------------------------------------------------------------
                                                                                               |
                                                                                               |
                                                                                               |
   0V : Block Occupied                                                                         +-----> Time t
                                                                            | <------------->  |
                                                                                  Delta-t

 The mentioned time Delta-t defines, how long the signal must be stable (i.e. no changes in the signal) until we
 define, that the measured signal status is correct. Instead of measuring the time, we use a counter, starting from 0.
 Each time we perform this function Block_UpdateOccupiedStatus(), the counter will be incremented by 1,
 when the signal status is identical with the signal status during the last call of this function.
 If the signal status is different, we reset the counter to 0.

 When the counter has become a maximum value of MAX_OCCU_COUNTER without changes of the signal,
 then it is assumed, that the measured signal status is the correct one, either block is free or occupied.

 On the hardware side an additional RC low-pass filter is implemented at the output side of the opto-coupler transistor,
 which looks like as follows:

              R = 4K7
              ------
       +5V  o-|    |-------------+------------> Output signal to micro-controller port
              ------ |           |
                     |          ---  +  C = 10 uF Capacitor
                     /          ---  -
   Light to Base -> |            |
   of Opto-Coupler   \           |
   Phototrans.       |           |
                    --- GND     --- GND

 When the output signal is low, then the block is occupied. Otherwise the block is free.
 It has been tested with several locos (different pick-ups) and with several loco speeds (from very slow to very quick).
 The implemented hardware & software work very satisfactory and generates no fault signals,
 which is absolutely necessary for this Block Processor Software, running as a state machine.

 A note at the end: it has not been tested, whether the software also works without the RC low-pass filter.

 *******************************************************/
void Block_UpdateOccupiedStatus (void)
	{
	uint8_t		iStatus;		// local occupied status variable, either TRUE or FALSE
/*
** -------------------------------------------------------------------------
** For the Block A.
** -------------------------------------------------------------------------
** Block is occupied when signal at input port is LOW.
** Block is NOT occupied (i.e. free) when signal at input port is HIGH.
*/
	iStatus = BLOCK_IS_FREE;
	if (BLOCK_A_OCCU_PIN == 0) iStatus = BLOCK_IS_OCCU;
/*
** If the actual status is identical with the old status, then increment the block status counter.
** Otherwise reset the block status counter to 0 to start a new measurement period.
*/
	if (iStatus == iBlock_A_StatusOld)
		{
		iBlock_A_StatusCounter++;
		}
	else
		{
		iBlock_A_StatusCounter = 0;
		}
/*
** Save the actual status as old status.
*/
	iBlock_A_StatusOld = iStatus;
/*
** Is the status counter has detected more than MAX_OCCU_COUNTER identical consecutive stati,
** then we apply the actual status of the block in iBlockOccupied[] variable.
** In addition, we decrease the status counter by 1, to prepare for the next call of this function.
*/
	if (iBlock_A_StatusCounter > MAX_OCCU_COUNTER)
		{
		iBlock_A_StatusCounter--;
		iBlockOccupied[BLOCK_A] = iStatus;
		}
/*
** -------------------------------------------------------------------------
** For the Block B.
** -------------------------------------------------------------------------
** Do the same procedure for Block B as above for Block A
**
** Block is occupied when signal at input port is LOW.
** Block is NOT occupied (i.e. free) when signal at input port is HIGH.
*/
	iStatus = BLOCK_IS_FREE;
	if (BLOCK_B_OCCU_PIN == 0) iStatus = BLOCK_IS_OCCU;

	if (iStatus == iBlock_B_StatusOld)
		{
		iBlock_B_StatusCounter++;
		}
	else
		{
		iBlock_B_StatusCounter = 0;
		}

	iBlock_B_StatusOld = iStatus;

	if (iBlock_B_StatusCounter > MAX_OCCU_COUNTER)
		{
		iBlock_B_StatusCounter--;
		iBlockOccupied[BLOCK_B] = iStatus;
		}

	return;
	}
/*******************************************************
 Public Function Block_GetOccupiedStatus

 Purpose:
	Return the occupied status
		Occupied = TRUE
		Not Occupied = FALSE
	In case of wrong input parameter, the function return the status of Block A

 Input Parameter:
	uint8_t	*iBlock		Element of {BLOCK_A, BLOCK_B}

 Return Value: uint8_t
	Element of {TRUE, FALSE}, TRUE = occupied, FALSE not occupied
 *******************************************************/
uint8_t Block_GetOccupiedStatus (uint8_t iBlock)
	{
	if (iBlock == BLOCK_B) return iBlockOccupied[1];
	return iBlockOccupied[0];
	}
/*******************************************************
 Public Function Block_GetState

 Purpose:
	Return the actual state of the Block

 Input Parameter: void

 Return Value: uint8_t
 *******************************************************/
uint8_t Block_GetState (void)
	{
	return iBlockState;
	}
/*******************************************************
 Public Function Block_GetShortCut

 Purpose:
	Return the actual shortcut status of the Block

 Input Parameter: void

 Return Value: uint8_t
 *******************************************************/
uint8_t Block_GetShortCut (void)
	{
	return iBlockBrakeShortCutFlag;
	}
/*******************************************************
 Public Function Block_GetDriveDirection

 Purpose:
	Return the actual drive direction of the Block

 Input Parameter: void

 Return Value: uint8_t
 *******************************************************/
uint8_t Block_GetDriveDirection (void)
	{
	return iBlockTrainDriveDirection;
	}
/*******************************************************
 Public Function Block_SetState

 Purpose:
	Set the state of the Block state machine

 Input Parameter:
	uint8_t	 iState

 Return Value: void
 *******************************************************/
void Block_SetState (uint8_t iState)
	{
/*
** Do nothing, when iState is wrong.
*/
	if (iState > STATE_ERROR) return;
/*
** We are not allowed to set the state to STATE_ERROR.
*/
	if (iState == STATE_ERROR) return;
/*
** Set the new state.
*/
	iBlockState = iState;
/*
** Reset block parameters depending on the actual block state.
**
** ####################################################
** Hier noch TIMSK zurcksetzen, wenn gesetzt.
** D.h. zuerst abfragen, welchen Zustand wir momentan haben.
** Ggf. noch andere parameter zurck setzen !
** ####################################################
*/
	if (iBlockState == STATE_BRAKE_DELAY && iState != STATE_BRAKE_DELAY)
		{
		CLEAR_BIT (TIMSK, TOIE0);
		iBlockBrakeDelayIsRunning = FALSE;
		iBlockBrakeShortCutFlag = FALSE;
		}

	if (iBlockState == STATE_ERROR) iBlockStateErrorFlag = FALSE;

	return;
	}
/*******************************************************
 Public Function Block_SetPower

 Purpose:
	Set the specified power to the Block

 Input Parameter:
	uint8_t	iBlock		Element of {BLOCK_A, BLOCK_B}
	uint8_t	iPower		Element of {BLOCK_DRIVE_POWER, BLOCK_BRAKE_POWER}

 Return Value: void
 *******************************************************/
void Block_SetPower (uint8_t iBlock, uint8_t iPower)
	{
	if (iBlock > BLOCK_B) return;
	if (iPower > BLOCK_BRAKE_POWER) return;

	iBlockPower[iBlock] = iPower;
	if (iBlock == BLOCK_A) BLOCK_A_DRIVE_BRAKE_PORT = iPower;
	if (iBlock == BLOCK_B) BLOCK_B_DRIVE_BRAKE_PORT = iPower;

	return;
	}
/*******************************************************
 Public Function Block_GetPower

 Purpose:
	Get the power to the specified Block
	In case of wrong input parameter, the function returns the block A power

 Input Parameter:
	uint8_t	iBlock		Element of {BLOCK_A, BLOCK_B}

 Return Value: uint8_t
	Element of {BLOCK_DRIVE_POWER, BLOCK_BRAKE_POWER}
 *******************************************************/
uint8_t Block_GetPower (uint8_t iBlock)
	{
	if (iBlock == BLOCK_B) return iBlockPower[BLOCK_B];
	return iBlockPower[BLOCK_A];
	}
/*******************************************************
 Public Function Block_SetBrakeDelay

 Purpose:
	Set the brake delay, before the brake is activated.
	In case of wrong input parameter, the function performs nothing.
	The delay factor depends on the drive direction:
	- BLOCK_A = Driving from Block A to Block B = A2B
	- BLOCK_B = Driving from Block B to Block A = B2A
	On wrong input parameter iBlock, set delay to iBlockBrakeDelayFactorA2B.

 Input Parameter:
	uint8_t	iBlock		{BLOCK_A, BLOCK_B}
	uint8_t	iDelay		The delay value

 Return Value: void
 *******************************************************/
void Block_SetBrakeDelay (uint8_t iBlock, uint8_t iDelay)
	{
	if (iBlock == BLOCK_B) {iBlockBrakeDelayFactorB2A = iDelay; return;}
	iBlockBrakeDelayFactorA2B = iDelay;
	return;
	}
/*******************************************************
 Public Function Block_GetBrakeDelay

 Purpose:
	Get the brake delay, before the brake is activated
	In case of wrong input parameter, the function returns the delay factor for Block A2B

 Input Parameter:
	uint8_t	iBlock		{BLOCK_A, BLOCK_B}

 Return Value:
 	uint8_t	iBlockBrakeDelayFactor
 *******************************************************/
uint8_t Block_GetBrakeDelay (uint8_t iBlock)
	{
	if (iBlock == BLOCK_B) return iBlockBrakeDelayFactorB2A;
	return iBlockBrakeDelayFactorA2B;
	}
/*******************************************************
 Public Function Block_ComputeDelayOverflowCounts

 Purpose:
	Compute the number of TIMER0 delay counts as function of the shortcut duration of the pickup shoe between block A and B.
	The returned iCountsDelay are the TIMER0 overflow counts, before we apply the Bremsspannung to the block part.
	This value is equivalent to milli-seconds.

	The principle is described below:

	The test setup is based on an track oval (as shown below as a square) with 3 electrical brakes (X and Y).
	To Block A or B we can apply Bahnstrom or Bremsspannung. The nominal drive direction was A->B.
	The signal SB is positioned at the end of block part B, where the locomotive shall come to a stop,
	independent of the actual speed of the locomotive.

                   <------ Train Drive direction Block A->B

                   Block B                     Block A
              -----------------------X------------------------
              |                                              |
              |                                              |
              |                                              |
              --------------Y-----------------Y---------------
                          SB         ^
                                     |
                              Normal Bahnstrom

 Then we performed a measurement test sequence beginning with the highest possible locomotive speed and
 ending with a lowest drive speed. For each speed we measured:
	- The duration Ns of the shortcut of the locomotive pickup shoe between block A and B
	  (given as TIMER0 overflow -> 1 overflow = 1ms)
	- When the pickup shoe only occupied block B, then we applied Bremsspannung to block B
	- When the locomotive has stopped, we measured the distance Ds in cm from the position X to the position,
	  where the locomotive came to a stop

 With:
	- Ns = number of TIMER0 overflows during shortcut between A and B
	- Ds = stop distance in cm from point X up to the position, where the locomotive stopped

 we can compute the delay (in terms of TIMER0 overflow counts), before we apply the Bremsspannung to the block,
 which shall guarantee, that the locomotive shall stop exactly before the Signal B (SB).

 Note:
	- Test-Number 1 is the highest locomotive speed
	- Test-Number 15 is the lowest locomotive speed

                     Ns          Ds            1/Ds
                   TIMER0   Stop Distance   Inverse of
                                           Stop Distance
  Test-Number     Overflows      cm            1/cm
      1              95         21.2          0.047
      2             100         18.8          0.053
      3             103         15.9          0.063
      4             109         14.5          0.069
      5             113         13.5          0.074
      6             122         11.4          0.088
      7             134          9.8          0.102
      8             146          8.5          0.118
      9             160          7.2          0.139
     10             199          5.6          0.179
     11             218          4.9          0.204
     12             266          3.9          0.255
     13             335          3.0          0.333
     14             405          2.4          0.417
     15             490          1.9          0.526

 We can now approximate the above relations between Ns and Ds by an exponential function f():

		Ds = f(Ns) with Ds = 10923.98283 * Ns^-1.41534

 Because it takes too long to compute an exponential function f() by a micro-controller,
 we did a trick by using the function 1/Ds = f(Ns) with f(Ns) = a * Ns + b.

 Then we can compute Ds by the inverse of the linear function f() with:

		Ds = 1 / f(Ns)
		Ds = 1 / (0.001187892 * Ns - 0.059317897)

 where f(Ns) comes with a regression value of 0.99913, i.e. its quite accurate.

 Here we only have 2 float multiplications and one float addition, which can be computed relatively quickly.

 The next step is to compute the distance Db from the shortcut point X in cm within the Block B
 up to the point, where we apply the Bremsspannung. Here we use the formula:

	Db = Bl - Ds

 where Bl is the length of Block B in cm. Because Bl is a known constant and Ds has just been computed,
 we can directly compute Db. The final step is to derive from Db the number of TIMER0 overflow counts Nb,
 before we apply Bremsspannung.

 Assuming that the length of the pickup-shoe is Lp (in cm), we can compute the number of
 required TIMER0 overflow counts before we apply Bremsspannung as follows:

	Nb = Db * Ns / Lp

 That's all :-)

 Input Parameter:
	uint16_t	iNs		The number of Overflows of the TIMER0, generated during the pickup shoe shortcut between Block A and B
						Based on:
							- 16 MHz crystal frequency
							- 8-Bit TIMER0 Prescaler = 64
							- Overflow after 256 clocks
						i.e. the duration between two overflows is identical with 1/(16000000/64/256) = 1 milli-second (ms)

 Return Value: uint16_t
 	iNb = Number of TIMER0 overflow counts before we apply Bremsspanung, after the pickup shoe shortcut has ended
 *******************************************************/
uint16_t Block_ComputeDelayOverflowCounts (uint16_t iNs)
	{
	float		Lp = 3.5;		// Length of pick-up shoe in cm
	float		Bl = 131.9;		// Length of Block part cm
	float		Ds;				// cm
	float		Db;				// cm
	double		x;
	uint16_t	iNb;			// Number of TIMER0 overflow counts before we apply Bremsspannung

	//Ds = 1.0 / (0.001187892 * (float)iNs - 0.059317897);	// BR 216
	//Ds = 1.0 / (0.000248122517 * (float)iNs - 0.010189500594);	// ICE 2
	Ds = 1.0 / (0.000249477948 * (float)iNs - 0.010470972116);	// ICE 2
	Ds = 1.0 / (0.000120373107 * (float)iNs - 0.002318017758);	// BR 86
	
	//a =  0,000247802281
	//= 0,000249477948x - 0,010470972116
	// 0,000120373107x - 0,002318017758

	Db = (Bl-Lp) - Ds;
	x = (Db * (float)iNs / Lp);
	iNb = (uint16_t)x;

	return iNb;
	}
/*******************************************************
 TIMER0 Overflow Interrupt Handler

 Purpose:
	This function is the TIMER0 Overflow Interrupt Service Routine (ISR)
	which is called every 16 ms with prescaler = 1024 and 16 MHz crystal clock.

	This TIMER0 overflow interrupt function performs two different activities:

	1. When iBlockBrakeShortCutFlag == TRUE:
		-> measure the duration, where the pickup shoe shortcuts Block A and B.
		The TIMER0 overflow interrupt will be initiated within the ISR INT0 interrupt function.

	2. When iBlockBrakeDelayIsRunning == TRUE:
		-> performing a delay before switching ON the Bremsspannung
 *******************************************************/
ISR (TIMER0_OVF_vect)
	{
/*
** Here for case 1: Measure duration of shortcut of pickup shoe between block A and B.
** =================================================================================
*/
	if (iBlockBrakeShortCutFlag == TRUE)
		{
/*
** Increment the shortcut counter.
** Do this only, when the counter is NOT at its limit.
*/
		if (iBlockBrakeShortCutCounter != 65535) iBlockBrakeShortCutCounter++;
		return;
		}
/*
** Here for case 2: Perform a brake delay.
** =======================================
*/
	if (iBlockBrakeDelayIsRunning == TRUE)
		{
/*
** Increment the global brake shortcut compare counters.
*/
		iBlockBrakeShortCutCompareCounter++;
/*
** If the compare counter value is larger than the iBlockBrakeShortCutCounter value, then the brake delay has ended:
**	- set the global variable iBlockBrakeDelayIsRunning to FALSE
**	- stop TIMER0 by selecting "No clock source" (CS00=0, CS01=0, CS02=0)
**	- disable the TIMER0 interrupt overflow bit in the TIMSK register
*/
		if (iBlockBrakeShortCutCompareCounter > iBlockBrakeShortCutCounter)
			{
			iBlockBrakeDelayIsRunning = FALSE;
			TCCR0 = 0;
			CLEAR_BIT (TIMSK, TOIE0);
			}
		}
	return;
	}
