/*******************************************************
 Author:					Manfred Langemann
 mailto:					Manfred.Langemann t t-online.de
 Begin of project:			21.01.2022
 Latest version generated:	11.03.2022
 Filename:					Coms.c
 Processor:					ATmega8
 Description:				This code performs the command communication of this slave with a master.
							Due to electrical design with the TWI bus driver PCA9600,
							the TWI bit rate of the Master shall not be higher than 50 Kbit/sec.
 ********************************************************/
#include <avr/io.h>
#include <stdio.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>

#include "General.h"
#include "Signal.h"
#include "Weiche.h"
#include "Auxi.h"
#include "AuxiDim.h"
#include "Block.h"
#include "TWI_Slave.h"
#include "Coms.h"
#include "EEMEM.h"
/*
** Enums for TWI commands.
** This enum list MUST be identical with:
**	- the one in the Electronic Block Control Units (ATmega8 Micro-Controllers)
**	- the one in the Train-Controller (ATmega32 Micro-Controllers)
**	- the one in the PC Train-Controller software
*/
enum COMMANDS
	{
/*
** These are the commands for the Electronic Block Controller Units (EBCU), which shall be sent via TWI to the Slave-Controller.
** The naming of the commands is related to the view of the Master-Controller.
*/
	COMMAND_PING,					// Receive and confirm a PING				return 0 data byte
	COMMAND_BOOT,					// Perform a boot of the slave controller	return 0 data byte

	COMMAND_SET_SIGNAL,				// 2 data bytes: Set Signal for block A or B to red or green or green / yellow
	COMMAND_SET_WEICHE,				// 2 data bytes: Set Weiche 1 or 2 to gerade or abzweig
	COMMAND_SET_AUXI,				// 2 or 3 data bytes: Set AUXI 1 or 2 to ON or OFF or DIM, in case of DIM also the Dim value
	COMMAND_SET_BLOCK_POWER,		// 2 data bytes: Set Block A or B to B or B#
	COMMAND_SET_BLOCK_BRAKE_DELAY,	// 2 data byte : Set Brake Delay value for A->B or B->A
	COMMAND_SET_BLOCK_STATE,		// 1 data byte : Set State of Block State Machine
	COMMAND_SET_SLAVE_ADDRESS,		// 1 data bytes: Set new Slave Address in EEMEM of slave controller

	COMMAND_GET_SIGNAL,				// Get Signal A and B status				return 2 data bytes
	COMMAND_GET_WEICHE,				// Get Weiche 1 and 2 status				return 2 data bytes
	COMMAND_GET_AUXI,				// Get AUXI 1 and 2 status					return 4 data bytes
	COMMAND_GET_BLOCK_POWER,		// Get the Block power for Block A and B	return 2 data bytes
	COMMAND_GET_BLOCK_BRAKE_DELAY,	// Get Brake Delay values					return 2 data byte

	COMMAND_GET_BLOCK_STATE,		// Get state of Block state machine			return 1 data byte
	COMMAND_GET_BLOCK_OCCUPIED,		// Get occupied status of block A and B		return 2 data bytes
	COMMAND_GET_BLOCK_DRIVE_DIR,	// Get current train drive direction		return 1 data byte
	COMMAND_GET_BLOCK_SHORTCUT_AB,	// Get B-B# shortcut indicator				return 1 data byte
/*
** These are the commands, which shall be processed directly by the train-controller
** and not by this Electronic Block Control Unit.
** These are included here only for incompleteness on the commands, which will be issued by the PC unit.
*/
	COMMAND_TC_EEMEM_RESET,			// Reset the serial EEMEM memory page pointers to 0
	COMMAND_TC_EEMEM_WRITE,			// Write the received 128 bytes to the serial EEMEM
	COMMAND_TC_EEMEM_READ,			// Read 128 bytes from the serial EEMEM and send them back to PC
	COMMAND_TC_PING,				// A ping command from the PC to the Train-Controller
	COMMAND_TC_BOOT,				// Boot this Train-Controller
	COMMAND_TC_GET_EBCU_TABLE,		// Get the EBCU availability table from the Train-Controller
	COMMAND_MAXIMUM					// Maximum number of commands
	};
/*
** When positively confirming a command, we set the highest bit of the command byte to 1 and send it back to the Master.
** In case of receiving a wrong command, don't set this confirm bit, indicating to the Master an error in transmission.
*/
#define CONFIRM_COMMAND		0x80

/*
** This is the struct of a TWI command, having:
**	- a command byte and
**	- up to five data bytes
*/
typedef struct
	{
	uint8_t	iCommand;
	uint8_t	iData[5];
	} COM_BUF;
/*
** Private Functions Declaration.
*/
void Coms_ConfirmCommand (uint8_t iNumBytes, uint8_t iConfirm);
/*
** Global variables.
*/
COM_BUF	ComBuf;

/*******************************************************
 Public Function Coms_Init

 Purpose:
	Init Coms functionality

 Input Parameter: void

 Return Value: void
 *******************************************************/
void Coms_Init (void)
	{
	uint8_t		iSlaveAddress;
/*
** Init the TWI interface with the correct slave address, which is stored in the EEMEM area.
** If slave address is wrong, take the default one.
*/
	EEMEM_ReadSlaveAddress (&iSlaveAddress);
	if (iSlaveAddress < 1 || iSlaveAddress > MAX_EBCU_ADDRESS) EEMEM_GetDefaultSlaveAddress (&iSlaveAddress);
	TWIS_Init (iSlaveAddress);

	return;
	}
/*******************************************************
 Public Function Coms_Processor

 Purpose:
	Perform command communication over TWI (I2C) with Master

 Input Parameter: void

 Return Value: void
 *******************************************************/
void Coms_Processor (void)
	{
	uint8_t		iNumBytes;
/*
** Check TWI interface on a received command:
**	- If nothing received, return
**	- Otherwise perform and confirm the command
*/
	if (!TWIS_ReadDataFromTWI (&iNumBytes, sizeof (ComBuf), (uint8_t*)&ComBuf)) return;
	//printf ("Doing Command\n");
/*
** When we have received more than sizeof(ComBuf) bytes, then we have received a wrong command.
** Send it back without confirmation.
*/
	if (iNumBytes > sizeof (ComBuf))
		{
		Coms_ConfirmCommand (0, FALSE);
		return;
		}
/*
** Analyse and perform the command.
**
** COMMAND_SET_SIGNAL -----------------
**		ComBuf.iData[0] = {SIG_A, SIG_B}
**		ComBuf.iData[1] = {SIG_RED, SIG_GREEN, SIG_GREEN_YELLOW, SIG_TEST}
*/
	if (ComBuf.iCommand == COMMAND_SET_SIGNAL)
		{
		Coms_ConfirmCommand (0, TRUE);
		Signal_Set (ComBuf.iData[0], ComBuf.iData[1]);
		//printf ("COMMAND_SET_SIGNAL\n");
		return;
		}
/*
** COMMAND_SET_WEICHE -----------------
**		ComBuf.iData[0] = {WEI1, WEI2}
**		ComBuf.iData[1] = {W_GERADE, W_ABZWEIG, W_LEFT, W_MID, W_RIGHT}
**		Note:
**		If ComBuf.iData[1] = {W_LEFT, W_MID, W_RIGHT}, then ComBuf.iData[0] is always WEI1
*/
	if (ComBuf.iCommand == COMMAND_SET_WEICHE)
		{
		Coms_ConfirmCommand (0, TRUE);
		//printf ("Weiche Set Command = %i %i\n", ComBuf.iData[0], ComBuf.iData[1]);
		Weiche_Set (ComBuf.iData[0], ComBuf.iData[1]);
		//printf ("COMMAND_SET_WEICHE\n");
		return;
		}
/*
** COMMAND_SET_AUXI -----------------
**		ComBuf.iData[0] = {AUXI1, AUXI2}
**		ComBuf.iData[1] = {AUXI_OFF, AUXI_ON, AUXI_DIM}
**		ComBuf.iData[2] = the Dim value in case of AUXI_DIM {0, ... , 63}
*/
	if (ComBuf.iCommand == COMMAND_SET_AUXI)
		{
		Coms_ConfirmCommand (0, TRUE);
		Auxi_Set (ComBuf.iData[0], ComBuf.iData[1], ComBuf.iData[2]);
		//printf ("COMMAND_SET_AUXI\n");
		return;
		}
/*
** COMMAND_SET_BLOCK_POWER -----------------
**		ComBuf.iData[0] = {BLOCK_A, BLOCK_B}
**		ComBuf.iData[1] = {BLOCK_DRIVE, BLOCK_BRAKE}
*/
	if (ComBuf.iCommand == COMMAND_SET_BLOCK_POWER)
		{
		Coms_ConfirmCommand (0, TRUE);
		Block_SetPower (ComBuf.iData[0], ComBuf.iData[1]);
		//printf ("COMMAND_SET_BLOCK_POWER\n");
		return;
		}
/*
** COMMAND_SET_BLOCK_BRAKE_DELAY -----------------
**		ComBuf.iData[0] = {BLOCK_A, BLOCK_B}
**		ComBuf.iData[1] = {0, ... 255}
*/
	if (ComBuf.iCommand == COMMAND_SET_BLOCK_BRAKE_DELAY)
		{
		Coms_ConfirmCommand (0, TRUE);
		if (ComBuf.iData[0] == BLOCK_A) EEMEM_WriteBrakeDelayBlockA2B (ComBuf.iData[1]);
		if (ComBuf.iData[0] == BLOCK_B) EEMEM_WriteBrakeDelayBlockB2A (ComBuf.iData[1]);
		Block_SetBrakeDelay (ComBuf.iData[0], ComBuf.iData[1]);
		//printf ("COMMAND_SET_BLOCK_BRAKE_DELAY\n");
		return;
		}
/*
** COMMAND_SET_BLOCK_STATE -----------------
**		ComBuf.iData[0] = iState : element of {STATE_INIT, STATE_FREE, ... ,STATE_LEAVING_BLOCK}
*/
	if (ComBuf.iCommand == COMMAND_SET_BLOCK_STATE)
		{
		Coms_ConfirmCommand (0, TRUE);
		Block_SetState (ComBuf.iData[0]);
		//printf ("COMMAND_SET_BLOCK_STATE\n");
		return;
		}
/*
** COMMAND_SET_SLAVE_ADDRESS -----------------
** ComBuf.iData[0] = the new TWI (I2C) slave address {1, ... ,MAX_EBCU_ADDRESS} -> We do not allow the address 0 !
** Having defined the new slave address, perform the TWIS_Init() function,
** which sets the new address in the TWAR register.
*/
	if (ComBuf.iCommand == COMMAND_SET_SLAVE_ADDRESS)
		{
		if (ComBuf.iData[0] > MAX_EBCU_ADDRESS || ComBuf.iData[0] == 0)
			{
			Coms_ConfirmCommand (0, FALSE);
			return;
			}
		Coms_ConfirmCommand (0, TRUE);
		EEMEM_WriteSlaveAddress (ComBuf.iData[0]);
		TWIS_Init (ComBuf.iData[0]);
		////printf ("COMMAND_SET_SLAVE_ADDRESS\n");
		return;
		}
/*
** COMMAND_GET_SIGNAL -----------------
**		ComBuf.iData[0] = {SIG_RED, SIG_GREEN, SIG_GREEN_YELLOW, SIG_TEST} for Signal A
**		ComBuf.iData[1] = {SIG_RED, SIG_GREEN, SIG_GREEN_YELLOW, SIG_TEST} for Signal B
*/
	if (ComBuf.iCommand == COMMAND_GET_SIGNAL)
		{
		ComBuf.iData[0] = Signal_Get (SIG_A);
		ComBuf.iData[1] = Signal_Get (SIG_B);
		Coms_ConfirmCommand (2, TRUE);
		//printf ("COMMAND_GET_SIGNAL\n");
		return;
		}
/*
** COMMAND_GET_WEICHE -----------------
**		ComBuf.iData[0] = {W_GERADE, W_ABZWEIG, W_LEFT, W_MID, W_RIGHT} for Weiche 1
**		ComBuf.iData[1] = {W_GERADE, W_ABZWEIG} for Weiche 2
**		Note:
**		The values {W_LEFT, W_MID, W_RIGHT} for a 3-way Weiche are only applied to Weiche 1 !
*/
	if (ComBuf.iCommand == COMMAND_GET_WEICHE)
		{
		ComBuf.iData[0] = Weiche_Get (WEI1);
		ComBuf.iData[1] = Weiche_Get (WEI2);
		Coms_ConfirmCommand (2, TRUE);
		//printf ("COMMAND_GET_WEICHE\n");
		return;
		}
/*
** COMMAND_GET_AUXI -----------------
**		ComBuf.iData[0] = {AUXI_OFF, AUXI_ON, AUXI_DIM} for Aux 1
**		ComBuf.iData[1] = {AUXI_OFF, AUXI_ON, AUXI_DIM} for Aux 2
**		ComBuf.iData[2] = the Dim value of AUX 1 {0, ... , 63} in case of AUXI_DIM
**		ComBuf.iData[3] = the Dim value of AUX 2 {0, ... , 63} in case of AUXI_DIM
*/
	if (ComBuf.iCommand == COMMAND_GET_AUXI)
		{
		ComBuf.iData[0] = Auxi_Get (AUXI1);
		ComBuf.iData[1] = Auxi_Get (AUXI2);
		ComBuf.iData[2] = AuxiDim_GetIntensity (AUXI1);
		ComBuf.iData[3] = AuxiDim_GetIntensity (AUXI2);
		Coms_ConfirmCommand (4, TRUE);
		//printf ("COMMAND_GET_AUXI\n");
		return;
		}
/*
** COMMAND_GET_BLOCK_POWER -----------------
**		ComBuf.iData[0] = {BLOCK_DRIVE, BLOCK_BRAKE} for Block A
**		ComBuf.iData[1] = {BLOCK_DRIVE, BLOCK_BRAKE} for Block B
*/
	if (ComBuf.iCommand == COMMAND_GET_BLOCK_POWER)
		{
		ComBuf.iData[0] = Block_GetPower (BLOCK_A);
		ComBuf.iData[1] = Block_GetPower (BLOCK_B);
		Coms_ConfirmCommand (2, TRUE);
		//printf ("COMMAND_GET_BLOCK_POWER\n");
		return;
		}
/*
** COMMAND_GET_BLOCK_OCCUPIED -----------------
**		ComBuf.iData[0] = the occupied status of Block A {TRUE, FALSE}
**		ComBuf.iData[1] = the occupied status of Block B {TRUE, FALSE}
*/
	if (ComBuf.iCommand == COMMAND_GET_BLOCK_OCCUPIED)
		{
		ComBuf.iData[0] = Block_GetOccupiedStatus (BLOCK_A);
		ComBuf.iData[1] = Block_GetOccupiedStatus (BLOCK_B);
		Coms_ConfirmCommand (2, TRUE);
		//printf ("COMMAND_GET_BLOCK_OCCUPIED\n");
		return;
		}
/*
** COMMAND_GET_BLOCK_STATE -----------------
**		ComBuf.iData[0] = the actual block state, element of {STATE_FREE, ..., STATE_ERROR}
*/
	if (ComBuf.iCommand == COMMAND_GET_BLOCK_STATE)
		{
		ComBuf.iData[0] = Block_GetState ();
		Coms_ConfirmCommand (1, TRUE);
		//printf ("COMMAND_GET_BLOCK_STATE\n");
		return;
		}
/*
** COMMAND_GET_BLOCK_DRIVE_DIR -----------------
**		ComBuf.iData[0] = the actual drive direction,
**						element of {BLOCK_TRAIN_DRIVE_DIR_A2B, BLOCK_TRAIN_DRIVE_DIR_B2A, BLOCK_TRAIN_DRIVE_DIR_VOID}
*/
	if (ComBuf.iCommand == COMMAND_GET_BLOCK_DRIVE_DIR)
		{
		ComBuf.iData[0] = Block_GetDriveDirection ();
		Coms_ConfirmCommand (1, TRUE);
		//printf ("COMMAND_GET_BLOCK_DRIVE_DIR\n");
		return;
		}
/*
** COMMAND_GET_BLOCK_BRAKE_DELAY -----------------
**		ComBuf.iData[0] = the Brake Delay value for driving from Block A to B {0, ... , 255}
**		ComBuf.iData[1] = the Brake Delay value for driving from Block B to A {0, ... , 255}
*/
	if (ComBuf.iCommand == COMMAND_GET_BLOCK_BRAKE_DELAY)
		{
		ComBuf.iData[0] = Block_GetBrakeDelay (BLOCK_A);
		ComBuf.iData[1] = Block_GetBrakeDelay (BLOCK_B);
		Coms_ConfirmCommand (2, TRUE);
		//printf ("COMMAND_GET_BLOCK_BRAKE_DELAY\n");
		return;
		}
/*
** COMMAND_GET_BLOCK_SHORTCUT_AB -----------------
**		ComBuf.iData[0] = the status of the shortcut between Block A and B
*/
	if (ComBuf.iCommand == COMMAND_GET_BLOCK_SHORTCUT_AB)
		{
		Coms_ConfirmCommand (0, FALSE);		// Not implemented, therefore return FALSE
		//printf ("COMMAND_GET_BLOCK_SHORTCUT_AB\n");
		return;
		}
/*
** COMMAND_PING -----------------
*/
	if (ComBuf.iCommand == COMMAND_PING)
		{
		Coms_ConfirmCommand (0, TRUE);
		//printf ("COMMAND_PING\n");
		return;
		}
/*
** COMMAND_BOOT -----------------
*/
	if (ComBuf.iCommand == COMMAND_BOOT)
		{
		Coms_ConfirmCommand (0, TRUE);
		//printf ("COMMAND_BOOT\n");
		cli ();
		CLEAR_BIT (GICR, INT0);
		wdt_enable (WDTO_15MS);
		while (TRUE) {};
		}
/*
** When we are here, then we have received a wrong command.
** Send the Command back without confirmation.
*/
	Coms_ConfirmCommand (0, FALSE);

	return;
	}
/*******************************************************
 Public Function Coms_ConfirmCommand

 Purpose:
	Confirm a command and send it back to the master.

 Input Parameter:
	uint8_t	iNumBytes		Number of data bytes to be transmitted via TWI,
							excluding the confirmed command byte, which also has to be transmitted.
	uint8_t	iConfirm		Confirm the command:
							TRUE: Confirm the command
							FALSE: Do NOT confirm the command

 Return Value: void
 *******************************************************/
void Coms_ConfirmCommand (uint8_t iNumBytes, uint8_t iConfirm)
	{
	if (iConfirm == TRUE) ComBuf.iCommand = ComBuf.iCommand | CONFIRM_COMMAND;
	TWIS_WriteDataToTWI (iNumBytes+1, (uint8_t*)&ComBuf);
	return;
	}
/*******************************************************
 Public Function Coms_CheckByte

 Purpose:
	Perform a Check Byte computation.

 Input Parameter:
	uint8_t	iNumBytes		Number of bytes to check in iBuf
	uint8_t	*iBuf			The buffer where to check the bytes

 Return Value:
	uint8_t	iCheck			The computed check byte, which shall be NULL
 *******************************************************/
uint8_t Coms_CheckByte (uint8_t iNumBytes, uint8_t *iBuf)
	{
	uint8_t	i;
	uint8_t	iCheck = 0;

	for (i=0; i<iNumBytes; i++)
		{
		iCheck = iCheck ^ iBuf[i];
		}

	return iCheck;
	}