/*******************************************************
 Author:					Manfred Langemann
 mailto:					Manfred.Langemann t t-online.de
 Begin of project:			22.01.2022
 Latest version generated:	27.05.2022
 Processor:					ATmega8
 Filename:					EEMEM.c
 Description:				To read and write EEMEM
							When reading from or writing to the EEMEM, do not allow interrupts !
  *******************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <avr/io.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>

#include "General.h"
#include "EEMEM.h"

#define EEMEM_DATA_VALID				0x55
#define	EEMEM_DEFAULT_SLAVE_ADDRESS		31

/*
** Private Function Prototypes
*/
void EEMEM_ReadDataValidByte (uint8_t *Byte);
void EEMEM_WriteDataValidByte (uint8_t Byte);
/*
** Global variables.
*/
uint8_t		EEMEM_DataValid EEMEM;	// When content is not 0x55, then configure the EEMEM data by function EEMEM_Configure()
uint8_t		EEMEM_SlaveAddress EEMEM;
uint8_t		EEMEM_BrakeDelayBlockA2B EEMEM;
uint8_t		EEMEM_BrakeDelayBlockB2A EEMEM;

/*******************************************************
 Public Function: EEMEM_Configure

 Purpose:
	Configure the EEMEM area with predefined values, when 'data valid byte' is not valid:

	- write data valid byte = 0x55 hex = 85 decimal
	- write to EEMEM_SlaveAddress the value EEMEM_DEFAULT_SLAVE_ADDRESS
	- write default brake delay values

 Input parameters: void

 Return value: void
 *******************************************************/
void EEMEM_Configure (void)
	{
	uint8_t	iValid;
	uint8_t	iSaveAddress;
/*
** Check whether the 'EEMEM_DataValid' byte is not correct, i.e. processor flashed for the first time.
** If not correct, then:
**	- write the value EEMEM_DATA_VALID to EEMEM_DataValid Byte and
**	- write EEMEM_DEFAULT_SLAVE_ADDRESS value to EEMEM_SlaveAddress Byte and
**	- write a 1 value to EEMEM_BrakeDelayBlockxxx Byte (for Block drive direction A2B and B2A)
**
** It shall be noted, that all EEMEM area bytes hold the value 0xFF = 255
** when the micro controller has never been programmed, i.e. a new one.
** Note that the EEMEM area is deleted during programming (flashing) when EESAVE fuse bit is not checked !
*/
	EEMEM_ReadDataValidByte (&iValid);
	EEMEM_ReadSlaveAddress (&iSaveAddress);
	printf ("Slave address = %i\n", iSaveAddress);
/*
** Configure the EEMEM area, if the EEMEM_DATA_VALID byte is not correct.
*/
	if (iValid != EEMEM_DATA_VALID)
		{
		EEMEM_WriteDataValidByte (EEMEM_DATA_VALID);
		iSaveAddress = EEMEM_DEFAULT_SLAVE_ADDRESS;
		EEMEM_WriteSlaveAddress (iSaveAddress);
		EEMEM_WriteBrakeDelayBlockA2B (1);
		EEMEM_WriteBrakeDelayBlockB2A (1);
		}
/*
** The slave address is only allowed between 1 and MAX_EBCU_ADDRESS.
** If not correct, set it to the default address.
*/
	if (iSaveAddress < 1 || iSaveAddress > MAX_EBCU_ADDRESS) EEMEM_WriteSlaveAddress (EEMEM_DEFAULT_SLAVE_ADDRESS);

	return;
	}
/*******************************************************
 Public Function: EEMEM_ReadDataValidByte

 Purpose:
	Read the Data Valid Byte from the EEMEM area.

 Input / Output parameter:
	- uint8_t	*Byte

 Return value: void
 *******************************************************/
void EEMEM_ReadDataValidByte (uint8_t *Byte)
	{
	cli ();
	*Byte = eeprom_read_byte (&EEMEM_DataValid);
	sei ();
	return;
	}
/*******************************************************
 Public Function: EEMEM_WriteDataValidByte

 Purpose:
	Write the Data Valid Byte to the EEMEM area.

 Input parameter:
	- uint8_t	Byte

 Return value: void
 *******************************************************/
void EEMEM_WriteDataValidByte (uint8_t Byte)
	{
	cli ();
	eeprom_write_byte (&EEMEM_DataValid, Byte);
	sei ();

	return;
	}
/*******************************************************
 Public Function: EEMEM_ReadSlaveAddress

 Purpose:
	Read the TWI Slave Address from the EEMEM area.

 Input / Output parameter:
	- uint8_t	*Byte

 Return value: void
 *******************************************************/
void EEMEM_ReadSlaveAddress (uint8_t *Byte)
	{
	cli ();
	*Byte = eeprom_read_byte (&EEMEM_SlaveAddress);
	sei ();

	return;
	}
/*******************************************************
 Public Function: EEMEM_WriteSlaveAddress

 Purpose:
	Write the TWI Slave Address to the EEMEM area.
	Note: slave addresses are only allowed up to MAX_EBCU_ADDRESS.

 Input parameter:
	- uint8_t	Byte

 Return value: void
 *******************************************************/
void EEMEM_WriteSlaveAddress (uint8_t Byte)
	{
	if (Byte > MAX_EBCU_ADDRESS) return;

	cli ();
	eeprom_write_byte (&EEMEM_SlaveAddress, Byte);
	sei ();

	return;
	}
/*******************************************************
 Public Function: EEMEM_ReadBrakeDelayBlockA2B

 Purpose:
	Read the Block A2A Brake Delay value from the EEMEM area.

 Input / Output parameter:
	- uint8_t	*Byte

 Return value: void
 *******************************************************/
void EEMEM_ReadBrakeDelayBlockA2B (volatile uint8_t *Byte)
	{
	cli ();
	*Byte = eeprom_read_byte (&EEMEM_BrakeDelayBlockA2B);
	sei ();

	return;
	}
/*******************************************************
 Public Function: EEMEM_ReadBrakeDelayBlockB2A

 Purpose:
	Read the Block B2A Brake Delay value from the EEMEM area.

 Input / Output parameter:
	- uint8_t	*Byte

 Return value: void
 *******************************************************/
void EEMEM_ReadBrakeDelayBlockB2A (volatile uint8_t *Byte)
	{
	cli ();
	*Byte = eeprom_read_byte (&EEMEM_BrakeDelayBlockB2A);
	sei ();

	return;
	}
/*******************************************************
 Public Function: EEMEM_WriteBrakeDelayBlockA2B

 Purpose:
	Write the Block A2B Brake Delay value to the EEMEM area.

 Input parameter:
	- uint8_t	Byte

 Return value: void
 *******************************************************/
void EEMEM_WriteBrakeDelayBlockA2B (uint8_t Byte)
	{
	cli ();
	eeprom_write_byte (&EEMEM_BrakeDelayBlockA2B, Byte);
	sei ();

	return;
	}
/*******************************************************
 Public Function: EEMEM_WriteBrakeDelayBlockB2A

 Purpose:
	Write the Block B2A Brake Delay value to the EEMEM area.

 Input parameter:
	- uint8_t	Byte

 Return value: void
 *******************************************************/
void EEMEM_WriteBrakeDelayBlockB2A (uint8_t Byte)
	{
	cli ();
	eeprom_write_byte (&EEMEM_BrakeDelayBlockB2A, Byte);
	sei ();

	return;
	}
/*******************************************************
 Public Function: EEMEM_GetDefaultSlaveAddress

 Purpose:
	Get the default slave address.

 Input parameter:
	uint8_t		*iValue		Pointer, where to return the default slave address

 Return value: void
 *******************************************************/
void EEMEM_GetDefaultSlaveAddress (uint8_t *iValue)
	{
	*iValue = EEMEM_DEFAULT_SLAVE_ADDRESS;
	return;
	}
