/*******************************************************
 Author:					Manfred Langemann
 mailto:					Manfred.Langemann t t-online.de
 Begin of project:			04.01.2008
 Latest version generated:	30.04.2015
 Filename:					TWI_Slave.c
 Description:    			TWI slave functions

 -------------------------------------------------------
 Program Example
 -------------------------------------------------------
#include "General.h"
#include "TWI_Slave.h"

int main (void)
	{
	uint8_t		i=0;
	uint8_t		j=0;
	uint8_t		byte[8];
	uint8_t		TWIS_ResponseType;

 // Clear any interrupt
	cli ();

 // Wait 0.5 second for POR
	Delay_ms (500);

 // Initiate the TWI slave interface with TWI Address 2
	TWIS_Init (2);

 // Endless loop
	while (1)
		{

 // Is something to do for the TWI slave interface ?
		if (TWIS_ResponseRequired (&TWIS_ResponseType))
			{
			switch (TWIS_ResponseType)
				{
 // TWI requests to read bytes from the master.
 // It is implicitely assumed, that the master
 // sends 8 bytes.

				case TWIS_ReadBytes:
					for (i=0;i<7;i++)
						{
						byte[0] = TWIS_ReadAck ();
						}
					byte[7] = TWIS_ReadNack ();
					TWIS_Stop ();
					break;

 // TWI requests to write bytes to the master.
 // It is implicitely assumed, that the master
 // is prepared to receive 8 bytes.

				case TWIS_WriteBytes:
					for (i=0;i<8;i++)
						{                   
						TWIS_Write (j++);
						}
				    TWIS_Stop ();
					break;
				}
			}
 // Do something else ....... e.g.:
 		i++;
		}
	return 0;
	}

 Siehe auch: https://www.mikrocontroller.net/articles/AVR_TWI

 Bekannte Bugs:
 -------------
 Bei einigen ATMEGA Varianten (z.B. ATMEGA8) hat die TWI Implementierung einen Bug.
 Nach dem Aufwachen aus einem Sleep Modus funktioniert der TWI nicht mehr. Erst ein Reset mit

	TWCR &= ~((1 << TWSTO) | (1 << TWEN));
	TWCR |= (1 << TWEN);

 nach dem Aufwachen lst das Problem.

 Nach dem senden eines "Stop" mu vor erneutem "Start" das TWSTO Bit im TWCR Register auf "0" abgefragt werden.
 Wenn dies noch eins enthlt ist der Stop nicht am Bus wirksam.
 Ein erneuter Start innerhalb dieser Situation bringt statt der "Start" Rckmeldung "$08" die "Restart" Rckmeldung "$10".
	
 ********************************************************/
#include <stdio.h>
#include <avr/interrupt.h>

#include "General.h"
#include "TWI_Slave.h"

/****************************************************************************
  Private TWI Slave functions
****************************************************************************/
void	TWIS_Stop (void);
void	TWIS_Write (uint8_t byte);
uint8_t	TWIS_ReadAck (void);
uint8_t	TWIS_ReadNack (void);
uint8_t	TWIS_Read (uint8_t Ack);
uint8_t	TWIS_ResponseRequired (uint8_t *TWI_ResponseType);

/****************************************************************************
  TWI State codes
****************************************************************************/
// General TWI Master status codes
#define TWI_START					0x08  // START has been transmitted  
#define TWI_REP_START				0x10  // Repeated START has been transmitted
#define TWI_ARB_LOST				0x38  // Arbitration lost

// TWI Master Transmitter status codes
#define TWI_MTX_ADR_ACK				0x18  // SLA+W has been transmitted and ACK received
#define TWI_MTX_ADR_NACK			0x20  // SLA+W has been transmitted and NACK received 
#define TWI_MTX_DATA_ACK			0x28  // Data byte has been transmitted and ACK received
#define TWI_MTX_DATA_NACK			0x30  // Data byte has been transmitted and NACK received 

// TWI Master Receiver status codes
#define TWI_MRX_ADR_ACK				0x40  // SLA+R has been transmitted and ACK received
#define TWI_MRX_ADR_NACK			0x48  // SLA+R has been transmitted and NACK received
#define TWI_MRX_DATA_ACK			0x50  // Data byte has been received and ACK transmitted
#define TWI_MRX_DATA_NACK			0x58  // Data byte has been received and NACK transmitted

// TWI Slave Transmitter status codes
#define TWI_STX_ADR_ACK				0xA8  // Own SLA+R has been received; ACK has been returned
#define TWI_STX_ADR_ACK_M_ARB_LOST	0xB0  // Arbitration lost in SLA+R/W as Master; own SLA+R has been received; ACK has been returned
#define TWI_STX_DATA_ACK			0xB8  // Data byte in TWDR has been transmitted; ACK has been received
#define TWI_STX_DATA_NACK			0xC0  // Data byte in TWDR has been transmitted; NOT ACK has been received
#define TWI_STX_DATA_ACK_LAST_BYTE	0xC8  // Last data byte in TWDR has been transmitted (TWEA = 0); ACK has been received

// TWI Slave Receiver status codes
#define TWI_SRX_ADR_ACK				0x60  // Own SLA+W has been received ACK has been returned
#define TWI_SRX_ADR_ACK_M_ARB_LOST	0x68  // Arbitration lost in SLA+R/W as Master; own SLA+W has been received; ACK has been returned
#define TWI_SRX_GEN_ACK				0x70  // General call address has been received; ACK has been returned
#define TWI_SRX_GEN_ACK_M_ARB_LOST	0x78  // Arbitration lost in SLA+R/W as Master; General call address has been received; ACK has been returned
#define TWI_SRX_ADR_DATA_ACK		0x80  // Previously addressed with own SLA+W; data has been received; ACK has been returned
#define TWI_SRX_ADR_DATA_NACK		0x88  // Previously addressed with own SLA+W; data has been received; NOT ACK has been returned
#define TWI_SRX_GEN_DATA_ACK		0x90  // Previously addressed with general call; data has been received; ACK has been returned
#define TWI_SRX_GEN_DATA_NACK		0x98  // Previously addressed with general call; data has been received; NOT ACK has been returned
#define TWI_SRX_STOP_RESTART		0xA0  // A STOP condition or repeated START condition has been received while still addressed as Slave

// TWI Miscellaneous status codes
#define TWI_NO_STATE				0xF8  // No relevant state information available; TWINT = 0
#define TWI_BUS_ERROR				0x00  // Bus error due to an illegal START or STOP condition

/*******************************************************
 Public Function: TWIS_ReadDataFromTWI

 Purpose: Read the Bytes from the TWI Interface
 
  Input / Output Parameter:
 	- uint8_t	*iNumBytes		The number of bytes received from TWI interface
 	- uint8_t	iMaxNumBytes	The maximum number of bytes to store in TWIS_Buf
	- uint8_t	*TWIS_Buf		The buffer where to write the received bytes

 Return Value: uint8_t
 	- TRUE:		Data read correctly
 	- FALSE:	Nothing to receive or Error in reading data
*******************************************************/
uint8_t TWIS_ReadDataFromTWI (uint8_t *iNumBytes, uint8_t iMaxNumBytes, uint8_t *TWIS_Buf)
	{
	uint8_t		iAck;
	uint8_t		i;
	uint8_t		TWIS_ResponseType;
	uint8_t		NumBytesM1;
	uint8_t		temp;
/*
** Check whether something is to do for the TWI slave interface.
** If not, then return FALSE.
*/
	if (!TWIS_ResponseRequired (&TWIS_ResponseType)) return FALSE;
/*
** If response type is not TWIS_ReadBytes, return FALSE.
*/
	if (TWIS_ResponseType != TWIS_ReadBytes) return FALSE;
/*
** Get the number of data bytes to receive.
*/
	*iNumBytes = TWIS_ReadAck ();
	NumBytesM1 = *iNumBytes-1;
/*
** Get the data bytes.
** We only allow to store up to iMaxNumBytes bytes into the TWIS_Buf[] array.
*/
	iAck = TRUE;
	for (i=0; i<*iNumBytes; i++)
		{
		if (i == NumBytesM1) iAck = FALSE;
		temp = TWIS_Read (iAck);			// Do a TWIS_ReadNack() or TWIS_ReadAck()
		if (i < iMaxNumBytes) TWIS_Buf[i] = temp;
		}
/*
** Stop TWI reading.
*/
	TWIS_Stop ();

	return TRUE;
	}
/*******************************************************
 Public Function: TWIS_WriteDataToTWI

 Purpose: Write data to the TWI interface

 Input Parameter:
   Input / Output Parameter:
 	- uint8_t	*iNumBytes		The number of data bytes to send to the TWI master
	- uint8_t	*TWIS_Buf		The buffer with the data bytes to send to the TWI master

 Return Value: uint8_t
 	- TRUE:		Data written correctly
 	- FALSE:	Error in writing data
*******************************************************/
uint8_t TWIS_WriteDataToTWI (uint8_t iNumBytes, uint8_t *TWIS_Buf)
	{
	uint8_t		i;
	uint8_t		TWIS_ResponseType;
/*
** Wait until master requests to send data.
** Besser:
	do
		{
		while (!TWIS_ResponseRequired (&TWIS_ResponseType));
		} while (TWIS_ResponseType != 0xF8);
*/
Once_More:
	while (!TWIS_ResponseRequired (&TWIS_ResponseType));
/*
** If TWI_ResponseType is "No relevant state info available, TWINT=0",
** which is TWI_NO_STATE = 0xF8 then repeat above sequence once more
*/
	if (TWIS_ResponseType == 0xF8) goto Once_More;
/*
** Is master requesting to write to master ?
** If not, then there was an error from the master.
*/
	if (TWIS_ResponseType != TWIS_WriteBytes) return FALSE;
/*
** Write the number of data bytes to send.
*/
	TWIS_Write (iNumBytes);
/*
** Send the data bytes to the master.
*/
	for (i=0; i<iNumBytes; i++)
		{
		TWIS_Write (TWIS_Buf[i]);
		}
/*
** Stop transmission.
*/
	TWIS_Stop ();

	return TRUE;
	}
/*******************************************************
 Public Function: TWIS_Init

 Purpose: Initialise the TWI Slave Interface

 Input Parameter:
  	- uint8_t	Slave address, element of {1, ... , 127}

 Return Value: None
 *******************************************************/
void TWIS_Init (uint8_t Address)
	{
/*
** Set the TWI slave address.
*/
	TWAR = (Address << 1);
//	TWAR = (Address << 1) | (1<<TWGCE);	// will enable General Calls.
/*
** Activate TWI interface.
*/
	TWCR = (1<<TWEN)|(1<<TWEA);

	return;
	}
/*******************************************************
 Public Function: TWIS_Stop

 Purpose: Stop the TWI Slave Interface

 Input Parameter: None

 Return Value: None
*******************************************************/
void TWIS_Stop (void)
	{
	TWCR = (1<<TWINT)|(1<<TWEN)|(1<<TWSTO)|(1<<TWEA);
	}
/*******************************************************
 Public Function: TWIS_Write

 Purpose: Write a byte to the master

 Input Parameter:
 	- uint8_t	Byte to be sent

 Return Value: None
*******************************************************/
void TWIS_Write (uint8_t byte)
	{
	TWDR = byte;
	TWCR = (1<<TWINT)|(1<<TWEN)|(1<<TWEA);
	while (!(TWCR & (1<<TWINT)));
	}
/*******************************************************
 Public Function: TWIS_ReadAck

 Purpose: Read a byte from the master and request next byte

 Input Parameter: None

 Return Value: uint8_t
  	- uint8_t	Read byte
*******************************************************/
uint8_t	TWIS_ReadAck (void)
	{
	TWCR = (1<<TWINT)|(1<<TWEN)|(1<<TWEA);
	while (!(TWCR & (1<<TWINT)));
	return TWDR;
	}
/*******************************************************
 Public Function: TWIS_ReadNack

 Purpose: Read the last byte from the master

 Input Parameter: None

 Return Value: uint8_t
  	- uint8_t	Read byte
*******************************************************/
uint8_t	TWIS_ReadNack (void)
	{
	TWCR = (1<<TWINT)|(1<<TWEN);
	while (!(TWCR & (1<<TWINT)));
	return TWDR;
	}
/*******************************************************
 Public Function: TWIS_Read

 Purpose: Read a byte from the master

 Input Parameter:
  	- uint8_t	Ack
		Ack == TRUE  --> send ACK  --> Master expects to write more bytes
		Ack == FALSE --> send NACK --> Master sends last byte

 Return Value: uint8_t
  	- uint8_t	Read byte
*******************************************************/
uint8_t	TWIS_Read (uint8_t Ack)
	{
	uint8_t		i;

	i = (1<<TWINT)|(1<<TWEN);
	if (Ack) i = i|(1<<TWEA);
	TWCR = i;
	while (!(TWCR & (1<<TWINT)));

	return TWDR;
	}
/*******************************************************
 Public Function: TWIS_ResponseRequired

 Purpose: Get the response type to be performed by slave

 Input Parameter:
  	- uint8_t*	Pointer to response type
	on return:
		TWIS_ReadBytes	--> Read byte(s) from master
		TWIS_WriteBytes	--> Write byte(s) to master

 Return Value: uint8_t
  	Response required
		TRUE: Yes, response required
		FALSE: No response required
*******************************************************/
uint8_t	TWIS_ResponseRequired (uint8_t *TWI_ResponseType)
	{
	*TWI_ResponseType = TWSR;
	return TWCR & (1<<TWINT);
	}

