/*******************************************************
 Author:					Manfred Langemann
 mailto:					Manfred.Langemann t t-online.de
 Begin of project:			18.01.2022
 Latest version generated:	20.05.2022
 Filename:					Weiche.c
 Processor:					ATmega8

Used Hardware:
 =============
  The 2 electronic Weichen switches for the electric turnout mechanisms are based on MOSFETs of type IRLML0040 (2 for each Weiche),
  which allow a pulsed current of up to 3 Ampere. However, the coil of the Mrklin electric turnout mechanism has
  an internal resistance of 13.5 Ohm, only drawing a current of 1.1 Ampere at 15V=.

 Software:
 =========
  The program is able to set 2 Weichen to the direction "straight ahead" or "turnout".
  During software initiation with Weiche_Init(), the Weichen will be set to direction "straight ahead".

  The Weiche is set to the required position by a short pulse of 50 ms (W_PULSE_DURATION) with a voltage of 15V=, see Weiche.c
  This pulse driven feature avoids thermal heating of the coils of the Mrklin electrical turnout mechanism.
  It therefore works without the feature of the end-switches, which are integral part of the Mrklin electrical turnout mechanism.

  Numerous Internet forums report, that these end-switches mostly fail because the contacts of the switches
  will be damaged after some time due to demolition sparks.
  In such a case, this implemented pulse driven design allows to short circuit these end-switches
  without any damage of the coil of the electrical turnout mechanism. However, it shall be considered that
  a software error, caused by whatever reason, the coils of the electrical turnout mechanism may be damaged, if the pulse
  will not be reset after 50 ms !!

  The design still allows to manually set a Weiche without any damage or conflict with the electronic Weichen settings.

 ********************************************************/
#include <avr/io.h>
#include <stdio.h>
#include <avr/interrupt.h>

#include "General.h"
#include "Delay.h"
#include "RS232.h"
#include "Weiche.h"
/*
** Definition of the ports for the Weichen MOSFET Drivers.
** For a 3-way Weiche, both drivers are used, i.e. W1 and W2.
*/
#define W1_Gerade_DDR		SBIT (DDRC, 3)	// Weiche 1 gerade		(straight ahead)
#define W1_Gerade_PORT		SBIT (PORTC, 3)
#define W1_Abzweig_DDR		SBIT (DDRC, 2)	// Weiche 1 abzweig		(turnout)
#define W1_Abzweig_PORT		SBIT (PORTC, 2)

#define W2_Gerade_DDR		SBIT (DDRD, 4)	// Weiche 2 gerade
#define W2_Gerade_PORT		SBIT (PORTD, 4)
#define W2_Abzweig_DDR		SBIT (DDRB, 5)	// Weiche 2 abzweig
#define W2_Abzweig_PORT		SBIT (PORTB, 5)
/*
** Define +15V= Pulse duration in milli-second (ms) for setting a Weiche (Magnet of the electric turnout mechanism).
** The pulse duration shall be as short as possible to avoid demolition sparks at the end switches !
** Remark: Shorter pulse durations will not correctly set the Weiche to its required position (somewhere in between).
*/
#define W_PULSE_DURATION	50		// value in milli seconds
/*
** Global variables.
*/
typedef  struct
	{
	uint8_t	iSetPulse;				// either TRUE or FALSE
	uint8_t iPulseRunning;			// either TRUE or FALSE
	} WEICHE_PULSE;

volatile uint8_t		WeichenDirectionStatus[2];		// Element of {W_GERADE, W_ABZWEIG}
volatile uint16_t		TCNT1_LoadWeichePulseCounter;	// TIMER1 Constant used as start value for the TIMER1 counter register
volatile WEICHE_PULSE	WeichePulse[2];					// see above struct definition
/*******************************************************
 Public Function Weiche_Init

 Purpose:
	Init Weichen functionality

 Input Parameter: void

 Return Value: void
 *******************************************************/
void Weiche_Init (void)
	{
/*
** Reset the Weichen Pulse status variables.
*/
	WeichePulse[WEI1].iSetPulse = FALSE;
	WeichePulse[WEI1].iPulseRunning = FALSE;

	WeichePulse[WEI2].iSetPulse = FALSE;
	WeichePulse[WEI2].iPulseRunning = FALSE;
/*
** Configure TIMER1 for overflow interrupt for performing a Weichen Pulse.
** We use a prescaler value of 256 in TCCR1B with CS10=0, CS11=0, CS12=1
** We start and stop the TIMER1 as follows:
** - Stop TIMER1  -> CS12=0
** - Start TIMER1 -> CS12=1
** But do NOT start here already the TIMER1, we therefore keep the CS12 to low level !
** Set the TIMER1 Overflow Interrupt Enable Bit in TIMSK register to 1 to enable the TIMER1_OVF_vect interrupt.
** Set interrupt frequency: Interrupt shall occur after W_PULSE_DURATION ms.
** Timer counts up from TCNT1_LoadWeichePulseCounter to 65535+1.
** For 16 MHz crystal and W_PULSE_DURATION = 50 ms: TCNT1_LoadWeichePulseCounter shall be 65536 - 3125 = 62411 = 0xF3CD
** This results in 0.016 ms * 3125 = 50 ms
** Note: Check for correct F_CPU value in file "General.h" with crystal on board, here 16 MHz.
*/
	SET_BIT (TIMSK, TOIE1);
	TCNT1_LoadWeichePulseCounter = 65536 - (F_CPU / 256 * W_PULSE_DURATION * 1e-3);
/*
** Set Weichen-Driver ports to output.
*/
	W1_Gerade_DDR = 1;
	W1_Abzweig_DDR = 1;
	W2_Gerade_DDR = 1;
	W2_Abzweig_DDR = 1;
/*
** Set both Weichen to W_GERADE.
** ############################################################################################
** Das kann zu einem Problem fhren, wenn eine Lok oder Wagen gerade auf einer Weiche steht !
** ############################################################################################
*/
	Weiche_Set (WEI1, W_GERADE);
	Weiche_Set (WEI2, W_GERADE);

	return;
	}
/*******************************************************
 Public Function Weiche_Set

 Purpose:
	Set a Weiche

 Input Parameter:
	uint8_t iWeiche		Element of {WEI1, ... ,WEI2}
	uint8_t iDirection	Element of {W_GERADE, W_ABZWEIG, W_LEFT, W_MID, W_RIGHT}
						Note:
						The directions W_LEFT, W_MID, W_RIGHT are for 3-Way Weichen !
						In case of a 3-Way Weiche the received value of iWeiche is always WEI1
						Note:
						In this function, we only indicate, that a Weiche shall be set.
						The real setting of the Weiche is performed in the function Weiche_Processor(),
						which is called in the main() function in the endless loop.

 Return Value: void
 *******************************************************/
void Weiche_Set (uint8_t iWeiche, uint8_t iDirection)
	{
/*
** Check on correct input values.
*/
	if (iWeiche > WEI2) return;
	if (iDirection > W_RIGHT) return;
/*
** If we have already set a pulse or a pulse is running for the specified Weiche, do nothing.
*/
	if (WeichePulse[iWeiche].iPulseRunning == TRUE || WeichePulse[iWeiche].iSetPulse == TRUE) return;
/*
** Indicate, that the Weiche shall be set by a Pulse.
** The real setting of the Weiche by a pulse is done by the function Weiche_Processor().
** Save the new Weichen direction status only when we really set the Weichen pulse.
**
** Here for the case W_GERADE and W_ABZWEIG.
** We only indicate here, that the selected Weiche (iWeiche) shall be set.
*/
	if (iDirection == W_GERADE || iDirection == W_ABZWEIG)
		{
		WeichenDirectionStatus[iWeiche] = iDirection;
		WeichePulse[iWeiche].iSetPulse = TRUE;			// Next call to Weiche_Processor() will start the pulse
		WeichePulse[iWeiche].iPulseRunning = FALSE;		// Indicate, that pulse is still not running
		//printf ("Prep Pulse W%i\n", iWeiche+1);
		return;
		}
/*
** Here for the case of a 3-Way Weiche with W_LEFT, W_MID and W_RIGHT.
** We have to check here explicitly also that for Weiche 2 a pulse is NOT set and NOT running, because iWeiche = WEI1.
** We only indicate here, that both Weiche shall be set.
*/
	if (iDirection == W_LEFT || iDirection == W_MID || iDirection == W_RIGHT)
		{
		if (WeichePulse[WEI2].iPulseRunning == TRUE || WeichePulse[WEI2].iSetPulse == TRUE) return;
		WeichenDirectionStatus[iWeiche] = iDirection;
		WeichePulse[WEI1].iSetPulse = TRUE;				// Next call to Weiche_Processor() will start the pulse
		WeichePulse[WEI1].iPulseRunning = FALSE;		// Indicate, that pulse is still not running
		WeichePulse[WEI2].iSetPulse = TRUE;				// Next call to Weiche_Processor() will start the pulse
		WeichePulse[WEI2].iPulseRunning = FALSE;		// Indicate, that pulse is still not running
		//printf ("Prep Pulse 3WAY\n");
		}

	return;
	}
/*******************************************************
 Function: Weiche_Get

 Purpose: Get the current Weichen direction.
 In case of wrong input parameter, the function returns W_GERADE.

 Input parameters:
	uint8_t iWeiche		Element of {WEI1, WEI2}

 Return value: 	uint8_t
	Element of {W_GERADE, W_ABZWEIG, W_LEFT, W_MID, W_RIGHT}
 *******************************************************/
uint8_t Weiche_Get (uint8_t iWeiche)
	{
	if (iWeiche > WEI2) return W_GERADE;
	return WeichenDirectionStatus[iWeiche];
	}
/*******************************************************
 Function: Weiche_Processor

 Purpose: This function shall be called from main() in the endless loop.
 It will start a Weichen pulse, if required.

 For a 3-way Weiche we define the following direction settings:
	- W_LEFT
		WEI1 --> W_GERADE
		WEI2 --> W_GERADE
	- W_MID
		WEI1 --> W_GERADE
		WEI2 --> W_ABZWEIG
	- W_RIGHT
		WEI1 --> W_ABZWEIG
		WEI2 --> W_ABZWEIG

 This means, the combination:
 		WEI1 --> W_ABZWEIG
		WEI2 --> W_GERADE
 shall NEVER be used.

 Input parameters: void

 Return value: void
 *******************************************************/
void Weiche_Processor (void)
	{
	uint8_t		iWDS1;
	uint8_t		iWDS2;
/*
** Start the pulse only for ONE Weiche, never for both,
** i.e. when a pulse for any Weiche is already running, do nothing.
*/
	if (WeichePulse[WEI1].iPulseRunning == TRUE || WeichePulse[WEI2].iPulseRunning == TRUE) return;
/*
** Start a Weichen pulse, if required.
** Either for Weiche 1 or Weiche 2, never for both.
** Here start the pulse for Weiche 1
*/
	if (WeichePulse[WEI1].iSetPulse == TRUE)
		{
		//printf ("Start Pulse W1\n");
		iWDS1 = WeichenDirectionStatus[WEI1];
		if (iWDS1 == W_GERADE) W1_Gerade_PORT = 1;			// Starts the pulse
		if (iWDS1 == W_ABZWEIG) W1_Abzweig_PORT = 1;		// Starts the pulse
		if (iWDS1 == W_LEFT) W1_Gerade_PORT = 1;			// Starts the pulse
		if (iWDS1 == W_MID) W1_Gerade_PORT = 1;				// Starts the pulse
		if (iWDS1 == W_RIGHT) W1_Abzweig_PORT = 1;			// Starts the pulse
		sei ();												// for security, we enable globally all interrupts
		WeichePulse[WEI1].iPulseRunning = TRUE;				// Indicate, that pulse is running
		TCNT1 = TCNT1_LoadWeichePulseCounter;				// Load the TIMER1 counter for W_PULSE_DURATION ms overflow interrupt
															// -> 3125 clocks until overflow
		SET_BIT (TCCR1B, CS12);								// Start TIMER1 counter (prescaler = 256)
		return;												// Return here, never start two Weichen pulses !
		}
/*
** Here start the pulse for Weiche 2
*/
	if (WeichePulse[WEI2].iSetPulse == TRUE)
		{
		//printf ("Start Pulse W2\n");
		iWDS1 = WeichenDirectionStatus[WEI1];
		iWDS2 = WeichenDirectionStatus[WEI2];
		if (iWDS1 != W_GERADE && iWDS1 != W_ABZWEIG)
			{
			if (iWDS1 == W_LEFT) W2_Gerade_PORT = 1;		// Starts the pulse
			if (iWDS1 == W_MID) W2_Abzweig_PORT = 1;		// Starts the pulse
			if (iWDS1 == W_RIGHT) W2_Abzweig_PORT = 1;		// Starts the pulse
			}
		else
			{
			if (iWDS2 == W_GERADE) W2_Gerade_PORT = 1;		// Starts the pulse
			if (iWDS2 == W_ABZWEIG) W2_Abzweig_PORT = 1;	// Starts the pulse
			}
		sei ();												// for security, we enable globally all interrupts
		WeichePulse[WEI2].iPulseRunning = TRUE;				// Indicate, that pulse is running
		TCNT1 = TCNT1_LoadWeichePulseCounter;				// Load the TIMER1 counter for W_PULSE_DURATION ms overflow interrupt
															// -> 3125 clocks until overflow
		SET_BIT (TCCR1B, CS12);								// Start TIMER1 counter (prescaler = 256)
		return;
		}
/*
** When we are here, no Weiche pulse shall be started.
*/
	return;
	}
/*******************************************************
 TIMER1 Overflow Interrupt Handler

 Purpose:
	This function is the Interrupt Service Routine (ISR)
	and is called by TIMER1 on overflow.

	This function will reset a running Weichen pulse.
 *******************************************************/
ISR (TIMER1_OVF_vect)
	{
/*
** Stop TIMER1 counter.
*/
	CLEAR_BIT (TCCR1B, CS12);
/*
** Check, for which Weiche {WEI1 or WEI2} we have completed the pulse.
** Reset the Weichen pulse output port to 0.
** We reset both Weiche output ports to 0, because we do not know here, which port (Gerade or Abzweig) was set to 1.
** Reset the Weichen pulse status parameters to FALSE, indicating that pulse is completed.
*/
	if (WeichePulse[WEI1].iPulseRunning == TRUE)
		{
		W1_Gerade_PORT = 0;		// Stops the pulse
		W1_Abzweig_PORT = 0;	// Stops the pulse
		WeichePulse[WEI1].iPulseRunning = FALSE;
		WeichePulse[WEI1].iSetPulse = FALSE;
		//printf ("Stop Pulse W1\n\n");
		}

	if (WeichePulse[WEI2].iPulseRunning == TRUE)
		{
		W2_Gerade_PORT = 0;		// Stops the pulse
		W2_Abzweig_PORT = 0;	// Stops the pulse
		WeichePulse[WEI2].iPulseRunning = FALSE;
		WeichePulse[WEI2].iSetPulse = FALSE;
		//printf ("Stop Pulse W2\n\n");
		}
	}