/*************************************************************************
 *  Dateiname    : LCD Test.c
 *
 *  Beschreibung : Pollin TECDIS C521001 an
 *                 Mega 16 auf SDK500
 *
 *  Display CN1 (Daten)  1 = Datenbit 7 / 8 = Datenbit 0
 *
 *  Display CN2 (Steuer) 1 = /RE  (Reset)
 *                       2 = /WE  (Write)
 *                       3 = /CS1 (Enable IC2)
 *                       4 = /CS2 (Enable IC1)
 *                       5 = READY
 *                       6 = SYNC
 *                       7 = Vss  (-)
 *                       8 = Vdd  (+5V)
 *                       9/10 gehen auf Kontakte
 *
 *                       Vop+ und Vop- ist die Kontrastspannung (2.5V)
 *                       (erzeugt das Board)
 *
 *************************************************************************/
#include "f_cpu.h"
/*************************************************************************
 *                         einbinden der System header
 *************************************************************************/
#include "crtl.h"
/*************************************************************************
 *                      einbinden der Projekt header
 *************************************************************************/
#include "globdef.h"
/*************************************************************************
 *                            Definitionen
 *************************************************************************/
#define LCD_DATA_DDR            DDRD        // Daten an Port D
#define LCD_DATA_PORT	        PORTD       // ...
#define LCD_DATA_PIN            PIND      

#define LCD_CNT_DDR             DDRA        // Steuerport = SDK500 Port A
#define LCD_CNT_PORT            PORTA       // ...
#define LCD_CNT_PIN             PINA

#define LED_DDR			        DDRB        // SDK500 LEDs an Port B
#define LED_PORT		        PORTB       //..

#define LCD_nRE			0           // Ausgang
#define LCD_nWE			1           // Ausgang
#define LCD_nCS1		2           // Ausgang (IC2)
#define LCD_nCS2		3           // Ausgang (IC1)
#define LCD_READY		4           // Eingang
#define LCD_SYNC		5           // Ausgang
/*************************************************************************
 *
 *  Name         :
 *
 *  Beschreibung :
 *
 *  Parameter    :
 *
 *
 *  Rckgabe     :
 *
 *************************************************************************/
#define DRIVE_MODE  0b10001111   // 1/4 Duty, Ready im Port Mode, Extern Power,
#define DISPL_MODE  0b00000011   // 1/4 Duty, Display on
#define HIDE_DATA	0b01000000   // Segmente ausschalten
#define SHOW_DATA   0b01100000   // Segmente einschalten
/*************************************************************************
 *
 *  Name         : T_Byte
 *
 *  Beschreibung : Bit's in einem Byte umdrehen 0->7 / 7->0
 *                 Der Stecker am Display war so einfacher an das SDK zu
 *                 bringen
 *
 *  Parameter    :
 *
 *
 *  Rckgabe     :
 *
 *************************************************************************/
uint8_t T_Byte( uint8_t Byte )
{

 uint8_t Ret;

 Ret  = Byte & 0b10000000 ? 0b00000001 : 0;
 Ret |= Byte & 0b01000000 ? 0b00000010 : 0;
 Ret |= Byte & 0b00100000 ? 0b00000100 : 0;
 Ret |= Byte & 0b00010000 ? 0b00001000 : 0;
 Ret |= Byte & 0b00001000 ? 0b00010000 : 0;
 Ret |= Byte & 0b00000100 ? 0b00100000 : 0;
 Ret |= Byte & 0b00000010 ? 0b01000000 : 0;
 Ret |= Byte & 0b00000001 ? 0b10000000 : 0;

 return Ret;

}
/*************************************************************************
 *
 *  Name         :
 *
 *  Beschreibung : Config kennt noch kein READY Bit.
 *
 *  Parameter    :
 *
 *
 *  Rckgabe     :
 *
 *************************************************************************/
static void WriteConfig(uint8_t Byte1, uint8_t Byte2)
{
 LCD_DATA_PORT = T_Byte(Byte1);
 CBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   low
 SBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   high

 LCD_DATA_PORT = T_Byte(Byte2);
 CBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   low
 SBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   high
 _delay_us(50);
}
/*************************************************************************
 *
 *  Name         :
 *
 *  Beschreibung : Command schon!
 *
 *  Parameter    :
 *
 *
 *  Rckgabe     :
 *
 *************************************************************************/
static void WriteCommand(uint8_t Byte1, uint8_t Byte2)
{
 LCD_DATA_PORT = T_Byte(Byte1);
 CBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   low
 SBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   high

 LCD_DATA_PORT = T_Byte(Byte2);
 CBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   low
 SBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   high
 loop_until_bit_is_clear(LCD_CNT_PIN,LCD_READY);
}
/*************************************************************************
 *
 *  Name         :
 *
 *  Beschreibung :
 *
 *  Parameter    :
 *
 *
 *  Rckgabe     :
 *
 *************************************************************************/
int main(void)
{

 uint8_t Com        = 0;
 uint8_t Chip       = 0;

 uint8_t AnAus      = SHOW_DATA;    // Segmente anzeigen/ausblenden
 uint8_t Dsp_Com    = 0;            // COM0 bis 3
 uint8_t Dsp_Seg    = 0;            // Segmentzhler

 /* Hardware einrichten */
 LED_DDR 		= 0xff;             // LEDs auf Ausgang
 LCD_DATA_DDR 	= 0xff;             // LCD Datenport  auf Ausgang
 LCD_CNT_DDR	= 0xff;	            // LCD Steuerport auf Ausgang

 CBI(LCD_CNT_DDR ,_BV(LCD_READY));  // READY auf Eingang
 SBI(LCD_CNT_PORT,_BV(LCD_READY));  // PullUp Ein

 _delay_ms(1);                      // POR

 /* Steuerleitungen vorbelegen */
 SBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  high
 SBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  high
 SBI(LCD_CNT_PORT,_BV(LCD_nRE));	// nRE   high
 SBI(LCD_CNT_PORT,_BV(LCD_nWE));	// nWE   high
 CBI(LCD_CNT_PORT,_BV(LCD_SYNC));	// SYNC  low
 _delay_ms(1);

 /* Reset */
 CBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  low
 CBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  low
 CBI(LCD_CNT_PORT,_BV(LCD_nRE));	// nRE   low
 _delay_ms(1);
 SBI(LCD_CNT_PORT,_BV(LCD_nRE));	// nRE   high
 SBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  high
 SBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  high
 _delay_ms(1);

 /* Mode Set das erste mal */
 CBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  low
 CBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  low
 WriteConfig(DRIVE_MODE,DISPL_MODE);
 SBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  high
 SBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  high
 _delay_ms (1);

 /* Sync beider Controller */
 SBI(LCD_CNT_PORT,_BV(LCD_SYNC));	// SYNC  high
 _delay_us(50);
 CBI(LCD_CNT_PORT,_BV(LCD_SYNC));	// SYNC  low

 /* noch mal Mode Set */
 _delay_ms (1);
 CBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  low
 CBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  low
 WriteConfig(DRIVE_MODE,DISPL_MODE);
 SBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  high
 SBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  high
 _delay_ms (1);


 while(1)
 {

  Dsp_Com = AnAus|Com;

  LED_PORT = ~Com;                  // Controller #


  if(Chip == 0)
  {
   CBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  low
   WriteCommand(Dsp_Com,Dsp_Seg);
   SBI(LCD_CNT_PORT,_BV(LCD_nCS1));	// nCS1  high
   _delay_ms (5);                   // damit man was zu sehen hat!
  }
  else
  {
   CBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  low
   WriteCommand(Dsp_Com,Dsp_Seg);
   SBI(LCD_CNT_PORT,_BV(LCD_nCS2));	// nCS2  high
   _delay_ms (5);                   // damit man was zu sehen hat!
  }

  Dsp_Seg++;

  if(Dsp_Seg > 50 ) // 50 Segmente ...
  {
   Dsp_Seg = 0;
   Com ++;
  }


  if(Com > 3)      // ... auf 4 COM-Leiungen  ...
  {
   Com = 0;

   if( AnAus == HIDE_DATA )
   {
    Chip++;
    if(Chip > 1)   //   * 2 Controller
     Chip = 0;
   }

   AnAus &= SHOW_DATA;

   if( AnAus == SHOW_DATA )
   {
    AnAus = HIDE_DATA;
    _delay_ms(1000);
   }
   else
    AnAus = SHOW_DATA;
  }
 }
 return (0);
}

