;################################################################################
;#										#
;# avr-chipbasic2 - single chip basic computer with ATmega644			#
;# expression parser								#
;# copyright (c) 2006-2009 Joerg Wolfram (joerg@jcwolfram.de)			#
;#										#
;#										#
;# This program is free software; you can redistribute it and/or		#
;# modify it under the terms of the GNU General Public License			#
;# as published by the Free Software Foundation; either version 3		#
;# of the License, or (at your option) any later version.			#
;#										#
;# This program is distributed in the hope that it will be useful,		#
;# but WITHOUT ANY WARRANTY; without even the implied warranty of		#
;# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the GNU		#
;# General Public License for more details.					#
;#										#
;# You should have received a copy of the GNU General Public			#
;# License along with this library; if not, write to the			#
;# Free Software Foundation, Inc., 59 Temple Place - Suite 330,			#
;# Boston, MA 02111-1307, USA.							#
;#										#
;################################################################################
; ereg			= error code
;		 0	= OK
;		 1	= break
;		 2	= overflow (add,sub,mul)
;		 3	= division by zero
;		 4	= square root from negative value
;		 5	= constant overflow (basic expression parser)
;		 6	= garbage in expression
;		 7	= syntax error (basic expression parser)
;
;		expar_ros	= address of stack root
;##############################################################################

expar_end:	rjmp	expar_bp_e2
;##############################################################################
; the basic parser
; Z= pointer in ram
;##############################################################################
expar:		ldi	YL,LOW(expar_ros)	;set stack pointer
		ldi	YH,HIGH(expar_ros)	;
		st	-Y,const_0		;write low value
		st	-Y,const_0		;write high value
		ldi	ereg,0x00		;no error
		push	const_0			;stack stopper
		clr	XL			;OP L
		clr	XH			;OP H
		clr	tempreg1		;clear status (last=op)

;------------------------------------------------------------------------------
; get character from string
;------------------------------------------------------------------------------
expar_bp_01:	ld	tempreg4,Z+		;get char
		cpi	tempreg4,0x20		;space
		breq	expar_bp_01
expar_bp_02:	cpi	tempreg4,0x80
		brcc	expar_end
		cpi	tempreg4,':'
		breq	expar_end

		cpi	tempreg4,'~'
		breq	expar_sysv
		rjmp	expar_pcr_01

expar_sysv:	ld	tempreg4,Z+
		cpi	tempreg4,'S'
		brne	expar_sysv_l
		movw	XL,r8
		rjmp	expar_pcr_02b

expar_sysv_l:	cpi	tempreg4,'L'
		brne	expar_sysv_p
		mov	XL,r12
		inc	XL
		clr	XH
		rjmp	expar_pcr_02b
		
expar_sysv_p:	cpi	tempreg4,'P'
		brne	expar_sysv_v
		lds	XL,libmio_prog
		inc	XL
		clr	XH
		rjmp	expar_pcr_02b

expar_sysv_v:	cpi	tempreg4,'V'
		brne	expar_sysv_x
		lds	XH,libmio_config
		ldi	XL,57
		sbrs	XH,2
		ldi	XL,7
		ldi	XH,1
		rjmp	expar_pcr_02b

expar_sysv_x:	cpi	tempreg4,'X'
		brne	expar_sysv_y
		lds	XL,libmio_clipx2
		clr	XH
		rjmp	expar_pcr_02b

expar_sysv_y:	cpi	tempreg4,'Y'
		brne	expar_sysv_r
		lds	XL,libmio_clipy2
		clr	XH
		rjmp	expar_pcr_02b

expar_sysv_r:	cpi	tempreg4,'R'
		brne	expar_sysv_n
		lds	XL,bas_values
		lds	XH,bas_values+1
		rjmp	expar_pcr_02b

expar_sysv_n:	cpi	tempreg4,'N'
		brne	expar_sysv_1
		lds	XL,bas_values+12
		clr	XH
		rjmp	expar_pcr_02b

expar_sysv_1:	cpi	tempreg4,'('
		brne	expar_mo_05
		ldi	tempreg4,0x1c
		rjmp	expar_pcr_05a

;------------------------------------------------------------------------------
; get precompressed decimal 8 bit
;------------------------------------------------------------------------------
expar_pcr_01:	cpi	tempreg4,0x1c		;token
		breq	expar_pcr_02a

;------------------------------------------------------------------------------
; get precompressed hexadecimal 8 bit
;------------------------------------------------------------------------------
expar_pcr_02:	cpi	tempreg4,0x1e		;token
		brne	expar_pcr_03
expar_pcr_02a:	ld	XL,Z+
		clr	XH
expar_pcr_02b:	ld	tempreg4,Z+		;get next char
		rjmp	expar_bp_n3

;------------------------------------------------------------------------------
; get precompressed decimal 16 bit
;------------------------------------------------------------------------------
expar_pcr_03:	cpi	tempreg4,0x1d		;token
		breq	expar_pcr_04a

;------------------------------------------------------------------------------
; get precompressed hexadecimal 16 bit
;------------------------------------------------------------------------------
expar_pcr_04:	cpi	tempreg4,0x1f		;token
		brne	expar_pcr_05
expar_pcr_04a:	ld	XH,Z+
		ld	XL,Z+
		ld	tempreg4,Z+		;get next char
		rjmp	expar_bp_n3

;-----------------------------------------------------------------------------
; get precompressed function
;------------------------------------------------------------------------------
expar_pcr_05:	cpi	tempreg4,0x1c		;upper limit for functions
		brcc	expar_bp_n1
expar_pcr_05a:	mov	tempreg2,tempreg4	;copy OP
		movw	tempreg5,YL		;save registers
		movw	YL,ZL			;copy RAM pointer
expar_mo_pcr:	push	tempreg2		;push OP
		mov	tempreg1,tempreg2	;set status
		clr	XL			;clear operand
		clr	XH
		movw	ZL,YL
		movw	YL,tempreg5
		rjmp	expar_bp_01		;next char
expar_mo_05:	ldi	ereg,7			;syntax error
		rjmp	expar_bp_err		;error handling

;------------------------------------------------------------------------------
; check for a decimal number
;------------------------------------------------------------------------------
expar_bp_n1:	cpi	tempreg4,48		;"0"
		brcs	expar_bp_m1		;next check
		cpi	tempreg4,58		;"9"+1
		brcc	expar_bp_m1		;next check
		cpi	tempreg1,0x80		;last was op?
		brcs	expar_bp_n2
		ldi	ereg,6			;missing operator
		rjmp	expar_bp_err		;error handling

expar_bp_n2:	mov	XL,tempreg4
		subi	XL,0x30
		clr	XH
		ld	tempreg4,Z+
		
expar_bp_n3:	cpi	tempreg1,0x7f		;inverter?
		brne	expar_bp_n4
		rcall	expar_inva		;invert value
expar_bp_n4:	st	-Y,XL			;write low value
		st	-Y,XH			;write high value
		ldi	tempreg1,0x80		;last element is a value
		rjmp	expar_bp_02		;next

expar_bp_n5:	ldi	ereg,5			;constant overflow
		rjmp	expar_bp_err		;error handling

;------------------------------------------------------------------------------
; multichar ops AR(, ABS(...
;------------------------------------------------------------------------------
expar_bp_m1:	cpi	tempreg4,65
		brcs	expar_bp_i1
		cpi	tempreg4,91
		brcc	expar_bp_i1

;------------------------------------------------------------------------------
; a variable "A...Z"
;------------------------------------------------------------------------------
expar_bp_v1:	cpi	tempreg1,0x80		;last was op?
		brcs	expar_bp_v2
		ldi	ereg,6			;missing operator
		rjmp	expar_bp_err		;error handling

expar_bp_v2:	subi	tempreg4,65		;-"A"
		lsl	tempreg4		;*2
		push	ZL			;save Z-reg
		push	ZH
		ldi	ZL,LOW(varspace)	;pointer to vars
		ldi	ZH,HIGH(varspace)
		add	ZL,tempreg4		;set actual pointer
		adc	ZH,const_0
		ld	XL,Z+
		ld	XH,Z+
		pop	ZH			;restore Z-reg
		pop	ZL
		ld	tempreg4,Z+		;get next char
		rjmp	expar_bp_n3

;------------------------------------------------------------------------------
; an inverter
;------------------------------------------------------------------------------
expar_bp_i1:	cpi	tempreg4,45		;"-"
		brne	expar_bp_p1		;no
		cpi	tempreg1,0x7f		;is already inverted
		brne	expar_bp_i2		;no
		clr	tempreg1		;
		rjmp	expar_bp_01		;thats all
		
expar_bp_i2:	brcc	expar_bp_i3		;last was not an operator
		cpi	tempreg1,0x20
		brcs	expar_bp_i3
		ldi	tempreg1,0x7f		;set inverter
		rjmp	expar_bp_01		;next
expar_bp_i3:	rjmp	expar_bp_c1		

;------------------------------------------------------------------------------
; check for parantheses
;------------------------------------------------------------------------------
expar_bp_p1:	cpi	tempreg4,40		;"("
		brne	expar_bp_p2
		ldi	tempreg4,0x01		;code for left paranthesis
		push	tempreg4		;push to stack	
		clr	XL			;clear operand
		clr	XH
		mov	tempreg1,tempreg4
		rjmp	expar_bp_01		;next character

expar_bp_p2:	cpi	tempreg4,41		;")"
		breq	expar_bp_p3
		rjmp	expar_bp_l1

expar_bp_p3:	pop	tempreg2		;get operator
		cpi	tempreg2,0x1d		;left paranthesis
		brcc	expar_bp_p7		;no fct
		ld	XH,Y+			;get high value of TOS
		ld	XL,Y+			;get low value of TOS
		push	ZH			;save Y+Z pointer
		push	ZL
		push	YH
		push	YL
		ldi	r16,LOW(expar_fret)
		push	r16
		ldi	r16,HIGH(expar_fret)
		push	r16
		ldi	r16,LOW(expar_jtab1)
		ldi	r17,HIGH(expar_jtab1)
		add	r16,tempreg2
		adc	r17,const_0
		push	r16			;this is like an icall to r16/r17
		push	r17
		ret
		
expar_jtab1:	ldi	ereg,6			;00 no open
		ret				;01 open paranthesis
		rjmp	expar_absa		;02 ABS
		rjmp	expar_sgn		;03 SGN
		rjmp	expar_sqr		;04 SQR
		rjmp	expar_cp_not		;05 NOT
		rjmp	expar_rnd		;06 RND
		rjmp	expar_adc		;07 ADC
		rjmp	expar_din		;08 DIN
		rjmp	expar_xpeek		;09 XPEEK
		rjmp	expar_temp		;0a TEMP
		rjmp	expar_lo		;0b LO
		rjmp	expar_hi		;0c HI
		rjmp	expar_key		;0d KEY
		rjmp	expar_epeek		;0e EPEEK
		rjmp	expar_geterr		;0f ERR
		rjmp	expar_ar		;10 AR
		rjmp	expar_sin		;11 SIN
		rjmp	expar_cos		;12 COS
		rjmp	expar_ftype		;13 FTYPE
		rjmp	expar_fsize		;14 FSIZE
		rjmp	expar_pstat		;15 PSTAT
		rjmp	expar_spi		;16 SPI
		rjmp	expar_menu		;17 MENU
		rjmp	expar_vpeek		;18 VBYTE
		rjmp	expar_xrec		;19 XREC
		rjmp	expar_ffind		;1a FFIND
		rjmp	expar_dbit		;1b DBIT
		rjmp	expar_cval		;1c ~(

expar_cp_not:	com	XH			;invert
		com	XL			
		ret				;goto end

expar_bp_p7:	rcall	expar_bp_x
		cpi	ereg,0x00
		breq	expar_bp_p3x
		rjmp	expar_bp_err
expar_bp_p3x:	rjmp	expar_bp_p3		;jump extender

expar_bp_p8:	ldi	ereg,6			;error code

expar_fret:	nop
		pop	YL
		pop	YH
		pop	ZL
		pop	ZH
expar_bp_pa:	cpi	ereg,0
		breq	expar_bp_pa1
		ret
expar_bp_pa1:	st	-Y,XL			;write result to stack
		st	-Y,XH			;

expar_bp_p9:	clr	XL			;clear operand
		clr	XH
		ldi	tempreg1,0x02		;code for right paranthesis
		rjmp	expar_bp_01		;next character

;------------------------------------------------------------------------------
; check if last was an operand
;------------------------------------------------------------------------------
expar_bp_l1:	cpi	tempreg1,0x7f		;upper border for operators
		brcc	expar_bp_c1		;last was no operator
		cpi	tempreg1,0x0f		;lower border for operators
		brcs	expar_bp_c1
		ldi	ereg,6			;missing operand
		rjmp	expar_bp_err

;------------------------------------------------------------------------------
; a comparsion
;------------------------------------------------------------------------------
expar_bp_c1:	ld	tempreg3,Z		;get next char
		ldi	r19,0x22		;code for ne
		cpi	tempreg4,60		;"<"
		brne	expar_bp_c4
		cpi	tempreg3,62		;">"
		breq	expar_bp_d01		;is ne
		ldi	r19,0x23		;code for le
		cpi	tempreg3,61		;"="
		breq	expar_bp_d01		;no
		ldi	r19,0x21		;code for less
		rjmp	expar_bp_d02

expar_bp_c4:	cpi	tempreg4,62		;">"
		brne	expar_bp_c6
		ldi	r19,0x25		;code for ge
		cpi	tempreg3,61		;"="
		breq	expar_bp_d01		;no
		ldi	r19,0x24		;code for greater
		rjmp	expar_bp_d02

expar_bp_c6:	ldi	r19,0x26		;code for equal
		cpi	tempreg4,61		;"="
		breq	expar_bp_d02

;------------------------------------------------------------------------------
; an arithmetic operator
;------------------------------------------------------------------------------
		ldi	r19,0x37		;code for +
		cpi	tempreg4,43		;"+"
		breq	expar_bp_d02
		ldi	r19,0x38		;code for -
		cpi	tempreg4,45		;"-"
		breq	expar_bp_d02
		ldi	r19,0x39		;code for or
		cpi	tempreg4,35		;"#"
		breq	expar_bp_d02

;next level
		ldi	r19,0x4a		;code for mult
		cpi	tempreg4,42		;"*"
		breq	expar_bp_d02
		ldi	r19,0x4b		;code for div
		cpi	tempreg4,47		;"/"
		breq	expar_bp_d02
		ldi	r19,0x4c		;code for mod
		cpi	tempreg4,37		;"%"
		breq	expar_bp_d02
		ldi	r19,0x4d		;code for and
		cpi	tempreg4,38		;"&"
		breq	expar_bp_d02

;------------------------------------------------------------------------------
; we are at the end of expression
;------------------------------------------------------------------------------
expar_bp_e2:	pop	tempreg2		;get last operand
		cpi	tempreg2,0x00		;end?
		breq	expar_bp_e3		;->end
		cpi	tempreg2,0x1f
		brcs	expar_bp_e5
		rcall	expar_bp_x
		rjmp	expar_bp_e2
expar_bp_e3:	ld	XH,Y+			;get high result
		ld	XL,Y+			;get low result
expar_bp_e4:	sbiw	ZL,1			;corect Z-pointer to next char
		ret
expar_bp_e5:	ldi	ereg,6
expar_bp_e6:	pop	tempreg2		;get last operand
		cpi	tempreg2,0x00		;end?
		breq	expar_bp_e3		;->end
		rjmp	expar_bp_e6

;------------------------------------------------------------------------------
; clear stack on error
;------------------------------------------------------------------------------
expar_bp_err:	pop	tempreg1		;get operator
		cpi	tempreg1,0		;stack ist empty?
		brne	expar_bp_err		;loop
expar_bp_er1:	ret				;thats all

;------------------------------------------------------------------------------
; do operator function 
;------------------------------------------------------------------------------
expar_bp_d01:	adiw	ZL,1			;correct Z register
expar_bp_d02:	mov	tempreg4,r19		;copy value
expar_bp_d0:	pop	tempreg2		;get last operator from stack

		mov	XL,tempreg4		;recent operator
		mov	XH,tempreg2		;last operator
		andi	XL,0xf0			;we need only priority
		andi	XH,0xf0			;
		cp	XH,XL			;compare last with recent
		brcs	expar_bp_d2		;branch if last is not less

expar_bp_d1:	rcall	expar_bp_x		;do last operation
		cpi	ereg,0x00		;error?
		brne	expar_bp_err		;yes
		rjmp	expar_bp_d0

;write last and recent operator on stack

expar_bp_d2:	push	tempreg2		;put last on stack
		push	tempreg4		;put recent on stack
		clr	XL			;clear operand
		clr	XH
		mov	tempreg1,tempreg4	;set status
		rjmp	expar_bp_01		;get next char

;------------------------------------------------------------------------------
; execute function with 2 parameters 
;------------------------------------------------------------------------------
expar_bp_x:	push	r21
		push	r20
		ld	r17,Y+			;get high value of TOS
		ld	r16,Y+			;get low value of TOS
		ld	XH,Y+			;get high value of SEC
		ld	XL,Y+			;get low value of SEC

		ldi	r18,LOW(expar_jtab2)
		mov	r19,tempreg2
		andi	r19,0x0f
		lsl	r19
		lsl	r19
		add	r18,r19
		ldi	r19,HIGH(expar_jtab2)
		adc	r19,const_0
		push	r18			;this is like an ijmp to r16/r17
		push	r19
		ret

expar_jtab2:	rjmp	expar_bp_xy		;0
		nop				;only filling
		nop
		nop

		rcall	expar_gt		;<
		rjmp	expar_bp_xy
		nop
		nop

		rcall	expar_eq		;<>
		subi	XL,1
		andi	XL,1
		rjmp	expar_bp_xy		

		rcall	expar_lt		;<=
		subi	XL,1
		andi	XL,1
		rjmp	expar_bp_xy		

		rcall	expar_lt		;>
		rjmp	expar_bp_xy		
		nop
		nop

		rcall	expar_gt		;>=
		subi	XL,1
		andi	XL,1
		rjmp	expar_bp_xy		

		rcall	expar_eq		;=
		rjmp	expar_bp_xy
		nop
		nop

		rcall	expar_add		;+
		rjmp	expar_bp_xy
		nop
		nop

		rcall	expar_sub		;-
		rjmp	expar_bp_xy
		nop
		nop

		or	XL,r16			;or
		or	XH,r17
		rjmp	expar_bp_xy
		nop

		rcall	expar_mul		;*
		rjmp	expar_bp_xy		;put TOS
		nop
		nop

		rcall	expar_div		;call division
		rjmp	expar_bp_xy		;put TOS
		nop
		nop

		rcall	expar_div		;call division
		mov	XL,r18			;get remainder
		mov	XH,r19
		rjmp	expar_bp_xy		;put TOS

		and	XL,r16			;and
		and	XH,r17
		rjmp	expar_bp_xy
		nop

		rjmp	expar_bp_xy		;e
		nop				;only filling
		nop
		nop

expar_bp_xy:	st	-Y,XL			;write result to stack
		st	-Y,XH
		pop	r20
		pop	r21
		ret

;-------------------------------------------------------------------------------
; absolute value of op1
;-------------------------------------------------------------------------------
expar_absa:	sbrs	XH,7			;skip if negative
		ret				;return if op was positive
expar_inva:	com	XL			;switch sign
		com	XH			;
		add	XL,const_1		;
		adc	XH,const_0		;
		ret				;thats all

expar_abs1:	st	-Y,XL
		st	-Y,XH
		ret

;-------------------------------------------------------------------------------
; absolute value of op2
;-------------------------------------------------------------------------------
expar_absb:	sbrs	r17,7			;skip if negative
		ret				;return if op was positive
expar_invb:	com	r16			;switch sign
		com	r17			;
		add	r16,const_1		;
		adc	r17,const_0		;
		ret				;thats all

;-------------------------------------------------------------------------------
; signum
; TOS <= sign(TOS) (-1,0,1)
;-------------------------------------------------------------------------------
expar_sgn:	or	XL,XH			;or to check if zero
		breq	expar_sgn_1		;branch if not zero
		bst	XH,7			;write sign to T
		ldi	XL,0x01			;set to 1
		clr	XH
		brtc	expar_sgn_1
		sbiw	XL,2
expar_sgn_1:	ret				;put result to stack

;-------------------------------------------------------------------------------
; 16 Bit subtraction
; (TOS) <= (SEC)-(TOS)
;-------------------------------------------------------------------------------
expar_sub:	rcall	expar_invb		;-(TOS)

;-------------------------------------------------------------------------------
; 16 Bit addition
; (TOS) <= (TOS)+(SEC)
;-------------------------------------------------------------------------------
expar_add:	mov	r18,XH			;High 1
		eor	r18,r17			;exclusive or with high 2
		add	XL,r16			;add low bytes
		adc	XH,r17			;add high bytes
		sbrc	r18,7			;skip if signs equal
		ret				;result is ok
		mov	r18,XH			;High result
		eor	r18,r17			;High OP2
		sbrc	r18,7			;skip if signs equal
		ldi	ereg,2			;set error
		ret				;thats all

;-------------------------------------------------------------------------------
; 16 Bit multiplication
; (TOS) <= (TOS) * (SEC)
;-------------------------------------------------------------------------------
expar_mul:	rcall	mul16
expar_mul_1:	movw	XL,r18			;result
		andi	XH,0x7f			;set temp result positive
		brtc	expar_mul_2		;result is positive
		rcall	expar_inva		;invert result
expar_mul_2:	sbrc	r19,7			;skip if no overflow
		rjmp	expar_mul_e		;overflow in bit 15 -> error
		or	r20,r21
		brne	expar_mul_e		;overflow
		ret
expar_mul_e:	ldi	ereg,2			;set error
		ret				;thats all

;-------------------------------------------------------------------------------
; 16*16 Bit signed multiplication
;-------------------------------------------------------------------------------
mul16:		rcall	expar_2abs
mul16a:		rcall	expar_absa		;absolute val 2
mul16b:		mul	XH,r17			;high bytes
		movw	r20,r0
		mul	XL,r16			;low bytes
		movw	r18,r0
		mul	XL,r17			;first cross multiplication
		add	r19,r0
		adc	r20,r1
		adc	r21,const_0
		mul	XH,r16			;second cross multiplication
		add	r19,r0
		adc	r20,r1
		adc	r21,const_0
		ret

;-------------------------------------------------------------------------------
; shift value XH,XL to 0...359 
;-------------------------------------------------------------------------------
expar_shan:	ldi	r16,LOW(360)		;interval
		ldi	r17,HIGH(360)
expar_shan1:	cpi	XH,0x80			;sign
		brcs	expar_shan2		;branch if positive
		add	XL,r16			;+360
		adc	XH,r17
		rjmp	expar_shan1	
expar_shan2:	cp	XL,r16			;compare with 360
		cpc	XH,r17
		brcs	expar_shan4		;OK
expar_shan3:	sub	XL,r16
		sbc	XH,r17
		rjmp	expar_shan2
expar_shan4:	ret

;-------------------------------------------------------------------------------
; calc sinus 
;-------------------------------------------------------------------------------
expar_sin:	ldi	r16,90			;shift
		sub	XL,r16
		sbc	XH,const_0

;-------------------------------------------------------------------------------
; calc cosinus 
;-------------------------------------------------------------------------------
expar_cos:	rcall	expar_shan		;shift to 0..359
		ldi	ZL,LOW(bas_sintab*2)
		ldi	ZH,HIGH(bas_sintab*2)
		ldi	r16,LOW(90)
		ldi	r17,HIGH(90)
		cp	XL,r16
		cpc	XH,r17
		brcc	expar_cos_2		;not Q1
		add	ZL,r16
		adc	ZH,r17
		sub	ZL,XL
		sbc	ZH,XH
expar_cos_1:	lpm	XL,Z
		clr	XH
		ret

expar_cos_2:	ldi	r16,LOW(180)
		ldi	r17,HIGH(180)
		cp	XL,r16
		cpc	XH,r17
		brcc	expar_cos_4		;not Q2
		ldi	r16,LOW(90)
		ldi	r17,HIGH(90)
		sub	ZL,r16
		sbc	ZH,r17
		add	ZL,XL
		adc	ZH,XH
expar_cos_3:	lpm	XL,Z
		clr	XH
		com	XL			;switch sign
		com	XH			;
		add	XL,const_1		;
		adc	XH,const_0		;
		ret

expar_cos_4:	ldi	r16,LOW(270)
		ldi	r17,HIGH(270)
		cp	XL,r16
		cpc	XH,r17
		brcc	expar_cos_5		;->Q4
		add	ZL,r16
		adc	ZH,r17
		sub	ZL,XL
		sbc	ZH,XH
		rjmp	expar_cos_3
		
expar_cos_5:	ldi	r16,LOW(270)
		ldi	r17,HIGH(270)
		sub	ZL,r16
		sbc	ZH,r17
		add	ZL,XL
		adc	ZH,XH
		rjmp	expar_cos_1

;-------------------------------------------------------------------------------
; 16 Bit signed division
; r16,r17 / XL,XH => XH,XL + remainder r18,r19
;-------------------------------------------------------------------------------
expar_div:	mov	r18,r17			;HIGH divisor
		or	r18,r16			;or LOW divisor
		brne	expar_div_1		;jump if no zero
		ldi	ereg,3			;set error
		ret				;return (error)

expar_div_1:	rcall	expar_2abs
		clr	r18			;clear remainder L
		clr	r19			;clear remainder H
		clr	r20			;clear carry source
		ldi	r21,17			;loop counter

expar_div_2:	rol	r20
		rol	XL			;lshift dividend
		rol	XH			;
		dec	r21			;loop counter
		breq	expar_div_3		;end of loop
		rol	r18			;shift into remainder
		rol	r19			;
		ldi	r20,0xff		;set carry source
		sub	r18,r16			;sub divisor
		sbc	r19,r17			;
		brcc	expar_div_2		;branch if no borrow
		add	r18,r16			;add divisor to restore
		adc	r19,r17			;
		ldi	r20,0x00		;clear carry source
		rjmp	expar_div_2		;goto loop
expar_div_3:	brtc	expar_div_4		;jump if result is positive
		rjmp	expar_inva		;invert result
expar_div_4:	ret				;thats all

;-------------------------------------------------------------------------------
; get sign and absolute value 
;-------------------------------------------------------------------------------
expar_2abs:	mov	r18,XH			;High dividend
		eor	r18,r17			;exclusive or with high divisor
		bst	r18,7			;store sign of result in T
		rcall	expar_absa		;absolute val 1
		rjmp	expar_absb		;absolute val 2

;-------------------------------------------------------------------------------
; 16 Bit equal (return 1 if TOS == SEC)
;-------------------------------------------------------------------------------
expar_eq:	push	r16
		push	r17
		eor	r16,XL			;compare low
		eor	r17,XH			;compare high
		or	r16,r17			;diff bits
		breq	expar_true
		rjmp	expar_false

;-------------------------------------------------------------------------------
; 16 Bit greater (return 1 if TOS > SEC)
;-------------------------------------------------------------------------------
expar_gt:	push	r16
		push	r17
		subi	XH,0x80			;invert sign-bit of OP1
		subi	r17,0x80		;invert sign-bit of OP2
		sub	XL,r16			;SEC-TOS
		sbc	XH,r17
		brcs	expar_true		;no carry
		rjmp	expar_false

;-------------------------------------------------------------------------------
; 16 Bit less (return 1 if TOS < SEC)
;-------------------------------------------------------------------------------
expar_lt:	push	r16
		push	r17
		subi	XH,0x80			;invert sign-bit of OP1
		subi	r17,0x80		;invert sign-bit of OP2
		sub	r16,XL			;TOS-SEC
		sbc	r17,XH
		brcs	expar_true

;-------------------------------------------------------------------------------
; result is true
;-------------------------------------------------------------------------------
expar_false:	clr	XL			;set result to 0
		rjmp	expar_true_0

expar_true:	ldi	XL,0x01
expar_true_0:	ldi	XH,0x00
		pop	r17
		pop	r16
		ret

;-------------------------------------------------------------------------------
; 16 Bit square root
; TOS <= sqrt(TOS)
; t is set, if operand is negative
;-------------------------------------------------------------------------------
expar_sqr:	ldi	ereg,4			;set error
		sbrc	XH,7			;skip if positive
		ret
expar_sqr_u:	clr	ereg			;clear error
		mov	r16,XL			;copy value
		mov	r17,XH		
		ldi	XH,2			;offset for tempreg
		clr	XL			;clear result
		mov	r18,const_1		;set tempreg to 1
		clr	r19
expar_sqr_1:	sub	r16,r18			;sub tempreg
		sbc	r17,r19
		brcs	expar_sqr_2		;overflow
		inc	XL			;result+1
		add	r18,XH			;tempreg=tempreg+2
		adc	r19,const_0		;
		rjmp	expar_sqr_1		;loop
expar_sqr_2:	clr	XH			;
expar_sqr_e:	ret

;-------------------------------------------------------------------------------
; 16 Bit RND
;-------------------------------------------------------------------------------
expar_rnd:	push	r20
		push	r21
		movw	r16,XL
		libmio_random			;calc a new value
		rcall	mul16b			;mult
		mov	XL,r20
		mov	XH,r21
		pop	r21
		pop	r20
		ret

;-------------------------------------------------------------------------------
; player status
;-------------------------------------------------------------------------------
expar_pstat:	cpi	XL,0xFF			;infinit loops
		breq	expar_pstat_2
		sts	libmio_seqstat,XL

expar_pstat_2:	lds	XL,libmio_seqstat
		clr	XH
		ret				;end

;-------------------------------------------------------------------------------
; SPI IO
;-------------------------------------------------------------------------------
expar_spi:	out	SPDR0,XL		;put out LOW value
		call	spi_wait
		in	XL,SPDR0
		clr	XH
		ret				;goto end

;-------------------------------------------------------------------------------
; menu
;-------------------------------------------------------------------------------
expar_menu:	lds	r17,libmio_vidmode	;get video mode
		cpi	r17,0x00
		breq	expar_menu_1
		ldi	ereg,28			;not in graphics mode
		ret

expar_menu_1:	ldi	r16,LOW(708)
		ldi	r17,HIGH(708)
		cp	XL,r16
		cpc	XH,r17
		brcs	expar_menu_2
		ldi	ereg,18			;out of array
		ret

expar_menu_2:	ldi	ZH,HIGH(bas_array)
		ldi	ZL,LOW(bas_array)
		add	ZL,XL
		adc	ZH,XH
		set				;read menu data from RAM
		libmio_menu			;show menu

		mov	XL,tempreg1		;set result to 0
		clr	XH
		ret				;goto end

;-------------------------------------------------------------------------------
; arrayelement value
;-------------------------------------------------------------------------------
expar_ar:	sts	bas_ram+11,XH		;store area
		sts	bas_ram+10,XL		;store offset
		call	arr_read
		ret				;goto end

;-------------------------------------------------------------------------------
; filetype
;-------------------------------------------------------------------------------
expar_ftype:	cpi	XH,0xff
		breq	expar_ftype_2		;-1
		sts	bas_partab,XL		;file number
		call	fsys_gettype		;get file type
		cpi	ereg,0x00
		breq	expar_ftype_1
expar_ftype_0:	ldi	XL,0xff			;-1
		ldi	XH,0xff
		clr	ereg
		ret				;goto end

expar_ftype_1:	mov	XL,tempreg1
		clr	XH
		ret				;goto end

expar_ftype_2:	call	fsys_size
		clr	ereg
		mov	XH,tempreg1
		clr	XL
		ret				;goto end

;-------------------------------------------------------------------------------
; flash/file-size
; -1 free files
; -2 free pages
; other
;-------------------------------------------------------------------------------
expar_fsize:	cpi	XH,0xff
		brne	expar_fsize_4		;
		sbrc	XL,0
		rjmp	expar_fsize_2
;free data pages
		call	fsys_free		;calc pages
		cpi	ereg,0x00
		breq	expar_fsize_1
		clr	ereg
		clr	YL
		clr	YH
expar_fsize_1:	movw	XL,YL			;copy value
expar_fsize_e:	ret

expar_fsize_2:	call	fsys_free		;calc pages
		cpi	ereg,0x00
		breq	expar_fsize_3
		clr	ereg
		clr	ZL
		clr	ZH
expar_fsize_3:	movw	XL,ZL			;copy value
		rjmp	expar_fsize_e		;goto end

expar_fsize_4:	sts	bas_partab,XL		;file number
		call	fsys_gettype		;get type byte (dummy) and set page
		cpi	ereg,0x00
		breq	expar_fsize_6
expar_fsize_5:	clr	ereg
		clr	XL			;result is 0
		clr	XH		
		rjmp	expar_fsize_e		;goto end

expar_fsize_6:	push	tempreg4
		clr	tempreg4
expar_fsize_7:	call	fsys_gword		;get data word
		cpi	YH,0xff
		brne	expar_fsize_9
		cpi	YL,0xff
		brne	expar_fsize_9
expar_fsize_8:	mov	XL,tempreg4
		clr	XH
		pop	tempreg4		;restore registers
		rjmp	expar_fsize_e		;goto end

expar_fsize_9:	inc	tempreg4
		cpi	tempreg4,0x80
		brne	expar_fsize_7
		rjmp	expar_fsize_8

;-------------------------------------------------------------------------------
; analog input
;-------------------------------------------------------------------------------
expar_adc:	andi	XL,0x07
		ori	XL,0xc0
		sts	ADMUX,XL
		lds	XL,ADCSRA
		ori	XL,0x40			;set ADSC
		sts	ADCSRA,XL
expar_adc1:	lds	XL,ADCSRA
		sbrc	XL,6
		rjmp	expar_adc1
		lds	XL,ADCL
		lds	XH,ADCH
		ret

;-------------------------------------------------------------------------------
; video RAM peek
;-------------------------------------------------------------------------------
expar_vpeek:	cpi	XH,11
		brcc	expar_vpeek_e
		cpi	XH,10
		brcs	expar_vpeek_1
		cpi	XL,0xc8
		brcs	expar_vpeek_1
expar_vpeek_e:	ldi	XL,0xff			;return -1
		ldi	XH,0xff
		ret				;

expar_vpeek_1:	ldi	YL,LOW(libmio_vram)
		ldi	YH,HIGH(libmio_vram)
		add	YL,XL
		adc	YH,XH
		ld	XL,Y
		clr	XH
		ret

;-------------------------------------------------------------------------------
; xmodem receive
;-------------------------------------------------------------------------------
expar_xrec:	cpi	XH,3
		brcc	expar_xrec_e
		cpi	XH,2
		brcs	expar_xrec_1
		sbrc	XL,7
expar_xrec_e:	rjmp	expar_bp_err
expar_xrec_1:	push	YH
		push	YL
		ldi	YL,LOW(bas_array)
		ldi	YH,HIGH(bas_array)
		add	YL,XL
		adc	YH,XH
		libmio_recvx
		mov	XL,tempreg3
		clr	XH
		ret

;-------------------------------------------------------------------------------
; digital input
;-------------------------------------------------------------------------------
expar_din:	cpi	XH,0x04			;sys
		brne	expar_din_1		;no
		clr	XH
		ld	XL,X
expar_din_e:	clr 	XH
		ret				;goto end

expar_din_1:	cpi	XL,255
		brne	expar_din_2
		in	XL,PINA
		rjmp	expar_din_e

expar_din_2:	andi	XL,0x07
		inc	XL
		in	XH,PINA
expar_din_3:	lsr	XH
		dec	XL
		brne	expar_din_3
		clr	XL
		rol	XL
		rjmp	expar_din_e

;-------------------------------------------------------------------------------
; xpeek
;-------------------------------------------------------------------------------
expar_xpeek:	lds	tempreg4,libmio_sysconf1;chip address
		andi	tempreg4,0x07
		libi2c_read			;get byte
		clr	XH
		mov	XL,tempreg2
		cpi	ereg,0
		breq	expar_xpeek_1
		ldi	ereg,15
expar_xpeek_1:	ret				;goto end

;-------------------------------------------------------------------------------
; epeek
;-------------------------------------------------------------------------------
expar_epeek:	movw	YL,XL
		cpi	YH,0x0c
		brcc	expar_fpeek
		libeep_read			;get byte
		clr	XH
		mov	XL,tempreg1
		cpi	ereg,0
		breq	expar_epeek_1
		ldi	ereg,34
expar_epeek_1:	ret

;get value from flash
expar_fpeek:	ldi	ZL,LOW(bas_programs*2-3072)
		ldi	ZH,HIGH(bas_programs*2-3072)
		add	ZL,YL
		adc	ZH,YH
		lpm	XL,Z
		clr	XH
		ret				;goto end

;-------------------------------------------------------------------------------
; temp
;-------------------------------------------------------------------------------
expar_temp:	mov	tempreg4,XL
		libi2c_rlm75
		cpi	ereg,0
		breq	expar_temp_1
		ldi	ereg,15
expar_temp_1:	ret				;goto end

;-------------------------------------------------------------------------------
; hi/lo
;-------------------------------------------------------------------------------
expar_hi:	mov	XL,XH
expar_lo:	clr	XH
		ret				;goto end

;-------------------------------------------------------------------------------
; key
;-------------------------------------------------------------------------------
expar_key:	mov	YL,XL
		clr	XL			;neutral value
		clr	XH
		cpi	YL,0			;getkey
		brne	expar_key_01		;no
		lds	XL,libmio_keycode
		rjmp	expar_key_end

expar_key_01:	cpi	YL,1			;waitkey
		brne	expar_key_02
expar_key_01a:	lds	XL,libmio_keycode	;get keycode
		cpi	XL,0x00			;no key
		breq	expar_key_01a
		rjmp	expar_key_end

expar_key_02:	cpi	YL,2			;waitnokey
		brne	expar_key_03
expar_key_02a:	lds	XL,libmio_keycode	;get keycode
		cpi	XL,0x00			;no key
		brne	expar_key_02a
		lds	XL,libmio_lastkey
		rjmp	expar_key_end

expar_key_03:	cpi	YL,3
		brne	expar_key_04
		lds	XL,libmio_kbdstate	;keyboard status
		rjmp	expar_key_end

expar_key_04:	cpi	YL,4			;left s/c
		brne	expar_key_05		;jump if no lshift
		lds	YL,libmio_kbdstate
		sbrc	YL,0			;left shift
		adiw	XL,1
		sbrc	YL,2			;left ctrl
		sbiw	XL,1
		rjmp	expar_key_end

expar_key_05:	cpi	YL,5
		brne	expar_key_06		;jump if no rshift
		lds	YL,libmio_kbdstate
		sbrc	YL,1			;right shift
		adiw	XL,1
		sbrc	YL,3			;right ctrl
		sbiw	XL,1
		rjmp	expar_key_end

expar_key_06:	cpi	YL,6
		brne	expar_key_07		;jump if no xcursor
		lds	YL,libmio_keycode
		cpi	YL,0xe2		;left
		brne	expar_key_06a
		sbiw	XL,1
expar_key_06a:	cpi	YL,0xe3		;right
		brne	expar_key_end
		adiw	XL,1
		rjmp	expar_key_end

expar_key_07:	cpi	YL,7
		brne	expar_key_08
		lds	YL,libmio_keycode
		cpi	YL,0xe4		;up
		brne	expar_key_07a
		adiw	XL,1
expar_key_07a:	cpi	YL,0xe5		;down
		brne	expar_key_end
		sbiw	XL,1
		rjmp	expar_key_end

expar_key_08:	lds	YH,libmio_keycode
		cp	YL,YH
		brne	expar_key_08
		mov	XL,YL
		clr	XH
expar_key_end:	ret

;-------------------------------------------------------------------------------
; err(
;-------------------------------------------------------------------------------
expar_geterr:	clr	XH
		cpi	XL,0x01			;line
		brne	expar_geterr1		;no
		lds	XL,bas_ram+5
		inc	XL			;+1
		ret				;goto end

expar_geterr1:	cpi	XL,0x02			;statement
		brne	expar_geterr2		;no
		lds	XL,bas_ram+6
		inc	XL			;+1
		ret				;goto end

expar_geterr2:	lds	XL,bas_ram+7		;errno
		ret				;goto end

;-------------------------------------------------------------------------------
; ffind
;-------------------------------------------------------------------------------
expar_ffind:	push	tempreg7
		push	tempreg8
		push	tempreg1
		libdfl_maxpage			;calc pages
		cpi	tempreg1,0		;
		breq	expar_ffind_2
		mov	r19,tempreg1
		inc	r19
		movw	YL,XL			;copy array pointer
		call	arr_read1		;get array value
		cpi	ereg,0
		brne	expar_ffind_ex
		adiw	YL,1			;next array byte
		movw	tempreg7,YL
		mov	r17,XL
		clr	r16			;first file
expar_ffind_1:	mov	XL,r16			
		libdfl_gettype1
		cp	r17,tempreg1		;found?
		breq	expar_ffind_3
expar_ffind_7:	inc	r16
		cp	r16,r19
		brne	expar_ffind_1
expar_ffind_2:	rjmp	expar_ffind_er

expar_ffind_3:	movw	YL,tempreg7
		call	arr_read1		;get array value
		cpi	ereg,0
		brne	expar_ffind_ex
		cpi	XL,0			;only header
		breq	expar_ffind_ok
		mov	r18,XL
		mov	XL,r16
		libdfl_nread
		movw	YL,tempreg7
		adiw	YL,1
		ldi	ZL,LOW(bas_inbuf)
		ldi	ZH,HIGH(bas_inbuf)
expar_ffind_4:	call	arr_read1		;get array value
		cpi	ereg,0
		brne	expar_ffind_ex
		ld	XH,Z+
		cp	XL,XH
		brne	expar_ffind_7
		adiw	YL,1
		dec	r18
		brne	expar_ffind_4
expar_ffind_ok:	mov	XL,r16
		clr	XH
		rjmp	expar_ffind_e
expar_ffind_er:	ldi	XL,0xff			;-1 not found
		ldi	XH,0xff
expar_ffind_e:	clr	ereg
expar_ffind_ex:	pop	tempreg1
		pop	tempreg8
		pop	tempreg7
		ret				;goto end

;-------------------------------------------------------------------------------
; dbit
;-------------------------------------------------------------------------------
expar_dbit:	lsl	XL			;ignore bit 6+7
		lsl	XL
		mov	XH,XL			;copy in value
		clr	XL			;clear result
		ldi	YL,0x00			;start value is 0
		ldi	YH,6			;number of bits to do
expar_dbit_1:	eor	YL,XH			;bit 7 is now difference
		lsl	YL			;shift out into result
		rol	XL
		mov	YL,XH			;
		lsl	XH			;shift left for next bit
		dec	YH
		brne	expar_dbit_1
		lsl	XL			;expand
		lsl	XL
		clr	XH
		ret

;-------------------------------------------------------------------------------
; cval
;-------------------------------------------------------------------------------
expar_cval:	dec	XL			;-1
		cpi	XL,5
		brcc	expar_cval_e
		mov	r0,XL
		ldi	XL,LOW(bas_values+2)
		ldi	XH,HIGH(bas_values+2)
		add	XL,r0
		add	XL,r0
		ld	r0,X+
		ld	r1,X+
		movw	XL,r0
		ret				;goto end

expar_cval_e:	clr	XL
		clr	XH
		ret				;goto end

;-------------------------------------------------------------------------------
; scale
;-------------------------------------------------------------------------------
math_scale:	ldi	YL,LOW(bas_partab)	;use ldd to save code words
		ldi	YH,HIGH(bas_partab)
		ld	r16,Y			;Y0
		ldd	r17,Y+1
		ldd	XL,Y+2			;Y1
		ldd	XH,Y+3
		rcall	expar_sub		;calc Y1-Y0
		bst	XH,7			;store sign
		rcall	expar_absa		;we use absolute |Y1-Y0|
		movw	tempreg5,XL		;store this to TR5/6
		ldd	r16,Y+4			;X0
		ldd	r17,Y+5
		ldd	XL,Y+6			;X
		ldd	XH,Y+7
		rcall	expar_sub		;calc X-X0
		movw	r16,tempreg5		;get for mult
		rcall	mul16a			;resultat is now in r18-r21
		ldd	r16,Y+4			;Y0
		ldd	r17,Y+5
		ldd	XL,Y+8			;Y1
		ldd	XH,Y+9
		movw	ZL,r18
		rcall	expar_sub		;calc Y1-Y0
		rcall	expar_absa		;we use absolute

;-------------------------------------------------------------------------------
; 32/16 Bit signed division
; r21,r20,ZH,ZL / XH,XL => XH,XL
;-------------------------------------------------------------------------------
		movw	tempreg5,XL		;add 1/2 divisor to dividend
		lsr	tempreg6		;for rounding
		ror	tempreg5
		add	ZL,tempreg5
		adc	ZH,tempreg6
		adc	tempreg1,const_0
		adc	tempreg2,const_0
		mov	tempreg5,const_0	;set temp regs to zero
		mov	tempreg6,const_0
		movw	tempreg7,tempreg5
		ldi	r16,32			;loop counter

expar_div32_1:	lsl	ZL
		rol	ZH
		rol	tempreg1
		rol	tempreg2
		rol	tempreg5
		rol	tempreg6
		rol	tempreg7
		rol	tempreg8
		cp	tempreg5,XL
		cpc	tempreg6,XH
		cpc	tempreg7,const_0
		cpc	tempreg8,const_0
		brcs	expar_div32_2
		sub	tempreg5,XL
		sbc	tempreg6,XH
		sbc	tempreg7,const_0
		sbc	tempreg8,const_0
		inc	ZL
expar_div32_2:	dec	r16
		brne	expar_div32_1
		movw	XL,ZL
expar_div32_3:	brtc	expar_div32_4		;jump if result is positive
		rjmp	expar_inva		;invert result
expar_div32_4:	ld	r16,Y			;Y0
		ldd	r17,Y+1
		rjmp	expar_add
