/*
 * 	Doku, siehe http://www.mikrocontroller.net/articles/AVR_FAT32
 *	Autor: Daniel R.
 */

#include <avr/io.h>

#include "config.h"
#include "mmc.h"
#include "uart.h"


//############################################################################
//Routine zur Initialisierung der MMC/SD-Karte (SPI-MODE)
unsigned char mmc_init (void){

  unsigned char a;
  unsigned int Timeout = 0;
   
  //Konfiguration des Ports an der die MMC/SD-Karte angeschlossen wurde
  MMC_Direction_REG &=~(1<<SPI_DI);         //Setzen von Pin MMC_DI auf Input
  MMC_Direction_REG |= (1<<SPI_Clock);      //Setzen von Pin MMC_Clock auf Output
  MMC_Direction_REG |= (1<<SPI_DO);         //Setzen von Pin MMC_DO auf Output
  MMC_Direction_REG |= (1<<MMC_Chip_Select);//Setzen von Pin MMC_Chip_Select auf Output
  MMC_Direction_REG |= (1<<SPI_SS);   
  MMC_Write |= (1<<MMC_Chip_Select);        //Setzt den Pin MMC_Chip_Select auf High Pegel

  for(a=0;a<200;a++){
	nop();
	};      //Wartet eine kurze Zeit
 
   //Aktiviren des SPI - Bus, Clock = Idel LOW
   //SPI Clock teilen durch 128
   SPCR = (1<<SPE)|(1<<MSTR)|(1<<SPR0)|(1<<SPR1); //Enable SPI, SPI in Master Mode     
   
  //Initialisiere MMC/SD-Karte in den SPI-Mode
  for (a = 0;a<0x0f;a++){ //Sendet min 74+ Clocks an die MMC/SD-Karte
    mmc_write_byte(0xff);
	}
   
  //Sendet Commando CMD0 an MMC/SD-Karte
  unsigned char CMD[] = {0x40,0x00,0x00,0x00,0x00,0x95};

  while(mmc_write_command (CMD) !=1){
	if (Timeout++ > 200){
	 MMC_Disable();
     return FALSE; 	//Abbruch bei Commando1 (Return Code1)
     }
  }
  //Sendet Commando CMD1 an MMC/SD-Karte
  Timeout = 0;
  CMD[0] = 0x41;//Commando 1
  CMD[5] = 0xFF;
  while( mmc_write_command (CMD) !=0){
	if (Timeout++ > 400){
	  MMC_Disable();
      return FALSE; 	//Abbruch bei Commando2 (Return Code2)
      }
	}

  CMD[0]=0x50;					// cmd16 (set block len), response r1
  CMD[3]=0x02;					// argument CMD[1,4]: block länge. hier 512 (0x00000200)
  mmc_write_command (CMD);

	#if (MAX_SPEED==TRUE)
	  //SPI Bus auf max Geschwindigkeit
	  SPCR &= ~((1<<SPR0) | (1<<SPR1));
	  SPSR |= (1<<SPI2X);
	#endif

  //set MMC_Chip_Select to high (MMC/SD-Karte Inaktiv)
  MMC_Disable();
  return TRUE;
}

#if (MULTI_BLOCK==TRUE && OVER_WRITE == FALSE)

//############################################################################
// stopt multiblock read operation
unsigned char mmc_multi_block_stop_read (void){

	unsigned char response;

	unsigned char cmd[] = {0x4C,0x00,0x00,0x00,0x00,0xFF};	// CMD12  -> response R1b (kein fehler, dann 0)

	response = mmc_write_command (cmd);		// r1 antwort auf cmd12

	response = mmc_read_byte();				// dummy byte nach cmd12

	return response;
}


//############################################################################
// stop von multiblock schreiben
unsigned char mmc_multi_block_stop_write (void){

	unsigned char cmd[] = {0x4D,0x00,0x00,0x00,0x00,0xFF};	// CMD13	check
	unsigned char response;

	mmc_write_byte(0xFD);					// stop token

	mmc_wait_ready();

	response=mmc_write_command (cmd);		// cmd13, alles ok?

	mmc_wait_ready();

	return response;
}


//############################################################################
// starten von multi block read. ab sektor addr wird der reihe nach gelesen. also addr++ usw...
unsigned char mmc_multi_block_start_read (unsigned long int addr){

	unsigned char response;

	// commando ist erstes byte. berechnung: 0x40 + cmd nummer
	unsigned char cmd[] = {0x52,0x00,0x00,0x00,0x00,0xFF};	// CMD18  -> response R1

	addr = addr << 9; //addr = addr * 512, um auf bytes zu kommen

	cmd[1] = ((addr & 0xFF000000) >>24 );
	cmd[2] = ((addr & 0x00FF0000) >>16 );
	cmd[3] = ((addr & 0x0000FF00) >>8 );

	//mmc_wait_ready ();

	response=mmc_write_command (cmd);		// commando senden und response speichern

	while (mmc_read_byte() != 0xFE){		// warten auf start byte
		nop();
	};

	return response;
}


//############################################################################
//multi block lesen von sektoren. bei aufruf wird immer ein sektor gelesen und immer der reihe nach
void mmc_multi_block_read_sector (unsigned char *Buffer){

	unsigned short i=512;				// einfacher zähler für bytes eines sektors
	unsigned char tmp;					// hilfs variable zur optimierung


	SPDR = 0xff;							// dummy byte
	do{										// 512er block lesen
		loop_until_bit_is_set(SPSR,SPIF);
		tmp=SPDR;
		SPDR = 0xff;						// dummy byte
		*Buffer=tmp;
		Buffer++;

	}while(--i);

	mmc_read_byte();						// crc byte
	mmc_read_byte();						// crc byte


	while (mmc_read_byte() != 0xFE){		// warten auf start byte 0xFE, damit fängt jede datenübertragung an...
		nop();
		}
}


//############################################################################
// starten von multi block write. ab sektor addr wird der reihe nach geschrieben. also addr++ usw...
unsigned char mmc_multi_block_start_write (unsigned long int addr){

	unsigned char response;

	// commando ist erstes byte. berechnung: 0x40 + cmd nummer
	unsigned char cmd[] = {0x59,0x00,0x00,0x00,0x00,0xFF};	// CMD59  -> response R1

	addr = addr << 9; //addr = addr * 512, um auf bytes zu kommen

	cmd[1] = ((addr & 0xFF000000) >>24 );
	cmd[2] = ((addr & 0x00FF0000) >>16 );
	cmd[3] = ((addr & 0x0000FF00) >>8 );

	response=mmc_write_command (cmd);		// commando senden und response speichern

	// senden von clocks zur karte
	//mmc_write_byte(0xFF); // dummy byte

	return response;
}


//############################################################################
//multi block schreiben von sektoren. bei aufruf wird immer ein sektor geschrieben immer der reihe nach
unsigned char mmc_multi_block_write_sector (unsigned char *Buffer){

	unsigned short a;			// einfacher zähler für bytes eines sektors
	//unsigned char b;
	unsigned char tmp;			// hilfs variable zur optimierung
	unsigned char response;

	mmc_write_byte(0xFC);

	a=512;				// do while konstrukt weils schneller geht
	tmp=*Buffer;		// holt neues byte aus ram in register
	Buffer++;			// zeigt auf nächstes byte
	do{
		SPDR = tmp;    	//Sendet ein Byte
		tmp=*Buffer;	// holt schonmal neues aus ram in register
		Buffer++;
		loop_until_bit_is_set(SPSR,SPIF);
	//	SPDR = tmp;    	//Sendet ein Byte
	}while(--a);

	//CRC-Bytes schreiben
	mmc_write_byte(0xFF); //Schreibt Dummy CRC
	mmc_write_byte(0xFF); //CRC Code wird nicht benutzt

	response=mmc_read_byte();

	mmc_wait_ready();

	if ((response&0x1F) == 0x05 ){			// daten von der karte angenommen, alles ok.
		return TRUE;
	}

	return FALSE;							// daten nicht angenommen... hiernach muss stop token gesendet werden !

}


//############################################################################
// wartet darauf, dass die mmc karte in idle geht
unsigned char mmc_wait_ready (void){

	unsigned char response;
	unsigned short ticks=100;

	do{
		response = mmc_read_byte();
		ticks--;
	}while ( (response != 0xFF) && (ticks>0) );

	return ticks;
}

#endif

//############################################################################
//Sendet ein Commando an die MMC/SD-Karte
unsigned char mmc_write_command (unsigned char *cmd){
	//unsigned char a;
    unsigned char tmp = 0xff;
    unsigned short Timeout = 0;

    //set MMC_Chip_Select to high (MMC/SD-Karte Inaktiv)
    MMC_Disable();

	//sendet 8 Clock Impulse
	mmc_write_byte(0xFF);

	//set MMC_Chip_Select to low (MMC/SD-Karte Aktiv)
	MMC_Enable();

	mmc_write_byte(cmd[0]);
	mmc_write_byte(cmd[1]);
	mmc_write_byte(cmd[2]);
	mmc_write_byte(cmd[3]);
	mmc_write_byte(cmd[4]);
	mmc_write_byte(cmd[5]);

	//Wartet auf ein gültige Antwort von der MMC/SD-Karte
	while (tmp == 0xff){
		tmp = mmc_read_byte();
		if (Timeout++ > 500){
			break; //Abbruch da die MMC/SD-Karte nicht Antwortet
		}
	}
	return(tmp);
}


//############################################################################
//Routine zum Empfangen eines Bytes von der MMC-Karte 
unsigned char mmc_read_byte (void){
  SPDR = 0xff;
  loop_until_bit_is_set(SPSR,SPIF);
  return (SPDR);
}


//############################################################################
//Routine zum Senden eines Bytes zur MMC-Karte
void mmc_write_byte (unsigned char Byte){
   SPDR = Byte;    //Sendet ein Byte
   loop_until_bit_is_set(SPSR,SPIF);
}


//############################################################################
//Routine zum schreiben eines Blocks(512Byte) auf die MMC/SD-Karte
unsigned char mmc_write_sector (unsigned long addr,unsigned char *Buffer){

   unsigned short a;
   unsigned char b;
   unsigned char tmp;

   //Commando 24 zum schreiben eines Blocks auf die MMC/SD - Karte
   unsigned char cmd[] = {0x58,0x00,0x00,0x00,0x00,0xFF}; 		//CMD24 noch ohne adresse
   
   // Die Adressierung der MMC/SD-Karte wird in Bytes angegeben,
   // addr wird von Blocks zu Bytes umgerechnet danach werden
   // diese in das Commando eingefuegt
     
   addr = addr << 9; //addr = addr * 512
   
   cmd[1] = ((addr & 0xFF000000) >>24 );
   cmd[2] = ((addr & 0x00FF0000) >>16 );
   cmd[3] = ((addr & 0x0000FF00) >>8 );

   //Sendet Commando cmd24 an MMC/SD-Karte (Write 1 Block/512 Bytes)
   tmp = mmc_write_command (cmd);

   if (tmp != 0)  return FALSE;
         
   //Wartet einen Moment und sendet einige Clocks an die MMC/SD-Karte
   for (b=0;b<100;b++)
      {
      mmc_read_byte();
      }
   
   //Sendet Start Byte an MMC/SD-Karte
   mmc_write_byte(0xFE);

   a=512;				// do while konstrukt weils schneller geht
   tmp=*Buffer;			// holt neues byte aus ram in register
   Buffer++;			// zeigt auf nächstes byte
   do{
	   SPDR = tmp;    //Sendet ein Byte
	   tmp=*Buffer;	// holt schonmal neues aus ram in register
	   Buffer++;
	   loop_until_bit_is_set(SPSR,SPIF);
	}while(--a);

   //CRC-Byte schreiben
   mmc_write_byte(0xFF); //Schreibt Dummy CRC
   mmc_write_byte(0xFF); //CRC Code wird nicht benutzt
   
   //Fehler beim schreiben? (Data Response XXX00101 = OK)
   if((mmc_read_byte()&0x1F) != 0x05) return FALSE;

   //Wartet auf MMC/SD-Karte Bussy
   while (mmc_read_byte() != 0xff){
	   nop();
   }
   
   //set MMC_Chip_Select to high (MMC/SD-Karte Inaktiv)
   MMC_Disable();
   
   return TRUE;
}


//############################################################################
//Routine zum lesen eines Blocks(512Byte) von der MMC/SD-Karte
unsigned char mmc_read_sector (unsigned long addr,unsigned char *Buffer){   

   unsigned short a=512;
   unsigned char tmp;

   //Commando 17 zum lesen eines Blocks von der MMC/SD - Karte
   unsigned char cmd[] = {0x51,0x00,0x00,0x00,0x00,0xFF}; 
   
   //Die Adressierung der MMC/SD-Karte wird in Bytes angegeben,
   //addr wird von Blocks zu Bytes umgerechnet danach werden
   //diese in das Commando eingefügt
     
   addr = addr << 9; //addr = addr * 512

   cmd[1] = ((addr & 0xFF000000) >>24 );
   cmd[2] = ((addr & 0x00FF0000) >>16 );
   cmd[3] = ((addr & 0x0000FF00) >>8 );

   //Sendet Commando cmd an MMC/SD-Karte
   if(0!=mmc_write_command (cmd)){
	   return FALSE;
   }

   //Wartet auf Start Byte von der MMC/SD-Karte (0xFE/Start Byte)
   while (mmc_read_byte() != 0xFE){};

   //Lesen des Bolcks (normal 512Bytes) von MMC/SD-Karte
   SPDR = 0xff;								// dummy byte
   	do{										// 512er block lesen
   		loop_until_bit_is_set(SPSR,SPIF);
   		tmp=SPDR;
   		SPDR = 0xff;						// dummy byte
   		*Buffer=tmp;
   		Buffer++;

   	}while(--a);

   //CRC-Byte auslesen
   mmc_read_byte();//CRC - Byte wird nicht ausgewertet
   mmc_read_byte();//CRC - Byte wird nicht ausgewertet

   //set MMC_Chip_Select to high (MMC/SD-Karte Inaktiv)
   MMC_Disable();

   return TRUE;
}




