#include <util\atomic.h>			// need "--std=c99"
#include <avr\power.h>
#include <avr\interrupt.h>
#include "humidity.h"
#include "getkey.h"


enum {
	MEASURE,
	WET,
	NORMAL,
	DRY,
	LOWBATT,
	OFF
};


uint8_t state = MEASURE;


static inline void init( void )
{
  LED_WET_DDR = 1;
  LED_NORMAL_DDR = 1;
  LED_DRY_DDR = 1;
  LED_LOWBATT_DDR = 1;
  MEAS_PWR_DDR = 1;
  
  get_key_init();
  clock_prescale_set( clock_div_8 );		// 1MHz
}


static inline void power_on_off( void )
{
  if( get_key_press( 1<<KEY_ON_OFF )){
    if( state == OFF )
      state = MEASURE;
    else
      state = OFF;
  }
}


uint16_t read_adc( uint8_t channel )
{
  uint16_t val = 0;

  ADMUX = channel;
  for( uint8_t i = 16; i; i-- ){
    ADCSRA = 1<<ADEN			// ADC on
	   | 1<<ADSC			// ADC start
	   | 1<<ADIF			// clear ADIF
	   | 1<<ADPS1 | 1<<ADPS0;	// 1MHz / 8 = 125kHz (50..200kHz)
    while( !(ADCSRA & 1<<ADIF) );	// wait until conversion done
    val += ADC;
  }
  return val >> 4;			// average over 16 samples
}


static inline void low_bat( void )
{
					// Bandgap need dummy measurement !!!
      read_adc( 1<<REFS0			// VCC as reference
	      | 0<<ADLAR			// ADC = 10 bit value
	      | 14 );				// input 14 = 1.1V Bandgap

  if( read_adc( 1<<REFS0			// VCC as reference
	      | 0<<ADLAR			// ADC = 10 bit value
	      | 14 ) > THRESHOLD_LOWBATT )	// input 14 = 1.1V Bandgap
    state = LOWBATT;
}


static inline void meas_humidity( void )
{
  MEAS_PWR = 1;					// sensor power on
  switch( read_adc( 1<<REFS0			// VCC as reference
		  | 0<<ADLAR			// ADC = 10 bit value
		  | MEAS_IN )){			// input ADC1

    case 0 ... THRESHOLD_WET1:
			state = WET;		// wet
			break;

    case THRESHOLD_WET1 + 1 ... THRESHOLD_WET2:
			if( state == WET )	// hysteresis (stay wet/normal)
			  break;

    case THRESHOLD_WET2 + 1 ... THRESHOLD_DRY1:
			state = NORMAL;		// normal
			break;

    case THRESHOLD_DRY1 + 1 ... THRESHOLD_DRY2:
			if( state == NORMAL )	// hysteresis (stay normal/dry)
			  break;

    default:
	 		state = DRY;		// dry
  }
  MEAS_PWR = 0;      				// sensor power off
}


int main( void )
{
  init();
  sei();

  for(;;){

    power_on_off();
    switch( state ){

      case MEASURE:
      case WET:
      case NORMAL:
      case DRY:
		meas_humidity();
		low_bat();
    }
    switch( state ){

      case WET:
		LED_WET     = LED_ON ^ !!(flasher & 0x20);	// 1s
		LED_NORMAL  = LED_OFF;
		LED_DRY     = LED_OFF;
		LED_LOWBATT = LED_OFF;
		break;

      case NORMAL:
		LED_WET     = LED_OFF;
		LED_NORMAL  = LED_ON;
		LED_DRY     = LED_OFF;
		LED_LOWBATT = LED_OFF;
		break;

      case DRY:
		LED_WET     = LED_OFF;
		LED_NORMAL  = LED_OFF;
		LED_DRY     = LED_ON ^ !!(flasher & 0x10);	// 0.5s
		LED_LOWBATT = LED_OFF;
		break;

      case LOWBATT:
		LED_WET     = LED_OFF;
		LED_NORMAL  = LED_OFF;
		LED_DRY     = LED_OFF;
		LED_LOWBATT = LED_ON ^ !!(flasher & 0xF0);	// 4s
		break;

      case OFF:
		LED_WET     = LED_OFF;
		LED_NORMAL  = LED_OFF;
		LED_DRY     = LED_OFF;
		LED_LOWBATT = LED_OFF;
		break;
    }
  }
}
