#include "debounce.h"

volatile uint8_t key_state;
volatile uint8_t key_press;
volatile uint8_t key_rpt;

// 10ms ISR, must be adapted for Non-AVR Arduinos    
ISR( TIMER1_COMPA_vect ) {
  static uint8_t ct0 = 0xFF, ct1 = 0xFF, rpt;
  uint8_t i, keys=0;

#ifdef KEY0_IN  
  if (!digitalRead(KEY0_IN)) keys |= KEY0;
#endif
#ifdef KEY1_IN  
  if (!digitalRead(KEY1_IN)) keys |= KEY1;
#endif
#ifdef KEY2_IN  
  if (!digitalRead(KEY2_IN)) keys |= KEY2;
#endif
#ifdef KEY3_IN  
  if (!digitalRead(KEY3_IN)) keys |= KEY3;
#endif
#ifdef KEY4_IN  
  if (!digitalRead(KEY4_IN)) keys |= KEY4;
#endif
#ifdef KEY5_IN  
  if (!digitalRead(KEY5_IN)) keys |= KEY5;
#endif
#ifdef KEY6_IN  
  if (!digitalRead(KEY6_IN)) keys |= KEY6;
#endif
#ifdef KEY7_IN  
  if (!digitalRead(KEY7_IN)) keys |= KEY7;
#endif

  i = key_state ^ keys;                           // key changed ?
  ct0 = ~( ct0 & i );                             // reset or count ct0
  ct1 = ct0 ^ (ct1 & i);                          // reset or count ct1
  i &= ct0 & ct1;                                 // count until roll over ?
  key_state ^= i;                                 // then toggle debounced state
  key_press |= key_state & i;                     // 0->1: key press detect

  if ( (key_state & REPEAT_MASK) == 0 ) {          // check repeat function
     rpt = REPEAT_START;                          // start delay
  }
  if ( --rpt == 0 ) {
    rpt = REPEAT_NEXT;                            // repeat delay
    key_rpt |= key_state & REPEAT_MASK;
  }
}

void init_debounce(void) {
  
#ifdef KEY0_IN    
  pinMode(KEY0_IN, INPUT_PULLUP);
#endif
#ifdef KEY1_IN    
  pinMode(KEY1_IN, INPUT_PULLUP);
#endif
#ifdef KEY2_IN    
  pinMode(KEY2_IN, INPUT_PULLUP);
#endif
#ifdef KEY3_IN    
  pinMode(KEY3_IN, INPUT_PULLUP);
#endif
#ifdef KEY4_IN    
  pinMode(KEY4_IN, INPUT_PULLUP);
#endif
#ifdef KEY5_IN    
  pinMode(KEY5_IN, INPUT_PULLUP);
#endif
#ifdef KEY6_IN    
  pinMode(KEY6_IN, INPUT_PULLUP);
#endif
#ifdef KEY7_IN    
  pinMode(KEY7_IN, INPUT_PULLUP);
#endif

  // Timer for 10ms ISR, must be adapted for Non-AVR Arduinos    
  // Timer 1, CTC mode 4, prescaler 256  
  TCCR1A  = 0;
  TCCR1B  = (1<<WGM12) | (1<<CS12);
  OCR1A   = (F_CPU / (256L * F_TIMER1))-1;
  TIMSK1 |= (1<<OCIE1A);                   // enable timer interrupt
}

/*
 * check if a key has been pressed. Each pressed key is reported
 * only once
 */

uint8_t get_key_press( uint8_t key_mask ) {
  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    key_mask &= key_press;                          // read key(s)
    key_press ^= key_mask;                          // clear key(s)
  }
  return key_mask;
}

/*
 * check if a key has been pressed long enough such that the
 * key repeat functionality kicks in. After a small setup delay
 * the key is reported being pressed in subsequent calls
 * to this function. This simulates the user repeatedly
 * pressing and releasing the key.
 */

uint8_t get_key_rpt( uint8_t key_mask ) {
  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    key_mask &= key_rpt;                            // read key(s)
    key_rpt ^= key_mask;                            // clear key(s)
  }
  return key_mask;
}

/*
 * check if a key is pressed right now
 */
 
uint8_t get_key_state( uint8_t key_mask ) {
  key_mask &= key_state;
  return key_mask;
}

/*
 * Get short key press
 */
 
uint8_t get_key_short( uint8_t key_mask ) {
  uint8_t i;
  ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
    i = get_key_press( ~key_state & key_mask );
  }
  return i;
}

/*
 * Get long key press
 */

uint8_t get_key_long( uint8_t key_mask ) {
  return get_key_press( get_key_rpt( key_mask ));
}
