; File:		ir-rx-pulse-dist.asm
; Device:	AVR
; Version:	2023-10-24
; Author:	Johannes Fechner
;			https://www.mikrocontroller.net/user/show/jofe

; == The register ir_status ==
; Bit positions are assigned as follows:
.equ IR_STATUS_PREV = 0 ; previous state of IR RX input
.equ IR_STATUS_DATA = 1 ; event flag, received valid data frame
.if IR_RECOGNIZE_REPETITION
.equ IR_STATUS_REPETITION = 2 ; event flag, received repetition frame
.endif
.if IR_DEBUG
.equ IR_STATUS_DISCARDED = 3 ; event flag, discarded reception
.endif

; == Possible values of register ir_pulseCntr ==
; Pulse/pause # within a frame, incremented at each falling edge (= begin of pulse):
; 0 = no edge received yet, waiting for begin of start pulse; or pulse already too long -> to be discarded
; 1 = after start edge, start pulse or pause ongoing
; 2 = first data pulse or pause ongoing, after reception of start pulse+pause
; 3 = second data pulse or pause ongoing, after reception of first data pulse+pause
; .
; .
; .
; 33 = last data pulse or pause ongoing
; 34 = stop pulse ongoing
.equ IR_DATA_START = 3 ; value of ir_pulseCntr when first data pulse+pause has been received
.equ IR_REPETITION_PAUSE_RECEIVED = 200 ; = repetition pause received, possible repetition frame stop bit ongoing

; == Debug message codes ==
.equ IR_PULSE_TOO_SHORT = $01
.equ IR_PULSE_TOO_LONG = $02
.equ IR_PULSE_TIMEOUT = $03
.equ IR_PAUSE_TOO_SHORT = $04
.equ IR_PAUSE_BETWEEN_0_1 = $05
.equ IR_PAUSE_TOO_LONG = $06
.equ IR_PAUSE_START_TOO_SHORT = $07
.equ IR_PAUSE_START_BETWEEN_R_D = $08
.equ IR_PAUSE_START_TOO_LONG = $09
.equ IR_PULSE_START_TOO_SHORT = $0A
.equ IR_PULSE_START_TOO_LONG = $0B
.equ IR_PULSE_COUNT_INVALID = $0C
.equ IR_DATA_INVALID = $0D

; == IR code definitions ==
; Time values must be in ascending order, otherwise the corresponding comparisons must be modified.
.if IR_PROTOCOL == IR_NEC
.equ IR_PULSE = 560 ; µs
.equ IR_PAUSE_0 = 560 ; µs
.equ IR_PAUSE_1 = 1690 ; µs
.equ IR_PAUSE_START_REPETITION = 2250 ; µs
.equ IR_PAUSE_START_DATA = 4500 ; µs
.equ IR_PULSE_START = 9000 ; µs
.equ IR_PULSE_COUNT = 34 ; including start and stop bit
.elif IR_PROTOCOL == IR_SAMSUNG32
.equ IR_PULSE = 550 ; µs
.equ IR_PAUSE_0 = 550 ; µs
.equ IR_PAUSE_1 = 1650 ; µs
.equ IR_PAUSE_START_DATA = 4500 ; µs
.equ IR_PULSE_START = 4500 ; µs
.equ IR_PULSE_COUNT = 34 ; including start and stop bit
.endif

; == Calculations ==
#define IR_TIMER_OCR round(1.0*F_CPU/IR_TIMER_PRESC/IR_INTR_FREQ-1)
; IR_DELAY = resulting delay between IR interrupts in µs
#define IR_DELAY (1.0e6*IR_TIMER_PRESC*(IR_TIMER_OCR+1)/F_CPU)
#define IR_TIMEOUT_INTR round(1.0e3*IR_TIMEOUT_MS/IR_DELAY)
; Calculate the approximate interrupt counts of time constants (T is time in µs):
#define IR_MIN_INTR(T) round((100.0-IR_TOLERANCE)*(T)/IR_DELAY/100.0)
#define IR_MAX_INTR(T) round((100.0+IR_TOLERANCE)*(T)/IR_DELAY/100.0)+1

; == Routines ==
; === IR register initialization routine ===
ir_init:
	clr		ir_status
; Set IR_STATUS_PREV bit (low-active):
	set
	bld		ir_status, IR_STATUS_PREV
	clr		ir_intrCntrL
	clr		ir_intrCntrH
	clr		ir_pulseCntr
	ret

; === IR receiver interrupt routine ===
ir_intr:
; Save temporary registers and SREG:
	push	temp0
	push	temp1
.if IR_DEBUG
	push	temp2
.endif
	xin		temp0, SREG
	push	temp0
.if TINY_1_SERIES
; Clear interrupt flag:
	ldi		temp0, 1
	xout	TCA0_SINGLE_INTFLAGS, temp0
.endif
; Increment interrupt counter:
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
	subi	temp0, low(-1) ; increment temp1:temp0 ...
	sbci	temp1, high(-1) ; ... 16-bit
	mov		ir_intrCntrL, temp0
	mov		ir_intrCntrH, temp1
; Check whether timeout is reached:
	subi	temp0, low(IR_TIMEOUT_INTR)
	sbci	temp1, high(IR_TIMEOUT_INTR)
	brlo	ir_intr_noTimeout
	rjmp	ir_intr_timeout
ir_intr_noTimeout:
; Detect whether IR_RX value has changed:
	bst		ir_status, IR_STATUS_PREV
	bld		temp0, IR_RX
	xin		temp1, IR_RX_PIN
	eor		temp0, temp1
	sbrs	temp0, IR_RX
	rjmp	ir_intr_end								; no change
; IR_RX has changed.
	bst		temp1, IR_RX							; store current IR state into T flag
	bld		ir_status, IR_STATUS_PREV				; update previous state
	brtc	ir_intr_fallingEdge
; Rising edge.
	cpi		ir_pulseCntr, 0
	breq	ir_risingEdge_end					; do nothing if timeout occurred during previous pulse
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
	cpi		ir_pulseCntr, 1
	breq	ir_risingEdge_1st
; Data or stop rising edge.
.if IR_DEBUG
	ldi		temp2, IR_PULSE_TOO_SHORT
	sts		ir_disc_code, temp2
.endif
	subi	temp0, low(IR_MIN_INTR(IR_PULSE))
	sbci	temp1, high(IR_MIN_INTR(IR_PULSE))
	brlo	ir_risingEdge_discard				; pulse was too short
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
.if IR_DEBUG
	ldi		temp2, IR_PULSE_TOO_LONG
	sts		ir_disc_code, temp2
.endif
	subi	temp0, low(IR_MAX_INTR(IR_PULSE))
	sbci	temp1, high(IR_MAX_INTR(IR_PULSE))
	brsh	ir_risingEdge_discard				; pulse was too long
; Data or stop pulse was within bounds.
	rjmp	ir_risingEdge_end
ir_risingEdge_1st:
.if IR_DEBUG
	ldi		temp2, IR_PULSE_START_TOO_SHORT
	sts		ir_disc_code, temp2
.endif
	subi	temp0, low(IR_MIN_INTR(IR_PULSE_START))
	sbci	temp1, high(IR_MIN_INTR(IR_PULSE_START))
	brlo	ir_risingEdge_discard				; start pulse was too short
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
.if IR_DEBUG
	ldi		temp2, IR_PULSE_START_TOO_LONG
	sts		ir_disc_code, temp2
.endif
	subi	temp0, low(IR_MAX_INTR(IR_PULSE_START))
	sbci	temp1, high(IR_MAX_INTR(IR_PULSE_START))
	brsh	ir_risingEdge_discard				; start pulse was too long
; Start pulse was within bounds.
	rjmp	ir_risingEdge_end
ir_risingEdge_discard:
.if IR_DEBUG
	sts		ir_disc_intrCntrL, ir_intrCntrL
	sts		ir_disc_intrCntrH, ir_intrCntrH
	sts		ir_disc_pulseCntr, ir_pulseCntr
	mov		temp2, ir_status
	ori		temp2, (1<<IR_STATUS_DISCARDED)
	mov		ir_status, temp2
.endif
	clr		ir_pulseCntr
ir_risingEdge_end:
	clr		ir_intrCntrL
	clr		ir_intrCntrH
; ### DEVICE-DEPENDENT ###
.if IR_TIMER_PRESC > 1
; Reset Timer2 prescaler, ATmega88A
	ldi		temp0, 1<<PSRASY
	xout	GTCCR, temp0
.endif
; ### END OF DEVICE-DEPENDENT ###
	rjmp	ir_intr_end
ir_intr_fallingEdge:
	inc		ir_pulseCntr
	cpi		ir_pulseCntr, 1
	breq	ir_fallingEdge_1st					; first falling edge, jump to end
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
	cpi		ir_pulseCntr, 2
	breq	ir_fallingEdge_2nd					; second falling edge (after start pause)
; Falling edge after data pause.
.if IR_DEBUG
	ldi		temp2, IR_PAUSE_TOO_SHORT
	sts		ir_disc_code, temp2
.endif
	subi	temp0, low(IR_MIN_INTR(IR_PAUSE_0))
	sbci	temp1, high(IR_MIN_INTR(IR_PAUSE_0))
	brlo	ir_fallingEdge_discard				; pause was too short
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
	subi	temp0, low(IR_MAX_INTR(IR_PAUSE_0))
	sbci	temp1, high(IR_MAX_INTR(IR_PAUSE_0))
	brlo	ir_fallingEdge_received0			; 0-bit received, jump to end
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
.if IR_DEBUG
	ldi		temp2, IR_PAUSE_BETWEEN_0_1
	sts		ir_disc_code, temp2
.endif
	subi	temp0, low(IR_MIN_INTR(IR_PAUSE_1))
	sbci	temp1, high(IR_MIN_INTR(IR_PAUSE_1))
	brlo	ir_fallingEdge_discard
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
	subi	temp0, low(IR_MAX_INTR(IR_PAUSE_1))
	sbci	temp1, high(IR_MAX_INTR(IR_PAUSE_1))
	brlo	ir_fallingEdge_setDataBit			; '1' received
.if IR_DEBUG
	ldi		temp2, IR_PAUSE_TOO_LONG
	sts		ir_disc_code, temp2
.endif
	rjmp	ir_fallingEdge_discard				; pause was too long
ir_fallingEdge_1st:
ir_fallingEdge_received0:
	rjmp	ir_fallingEdge_end
ir_fallingEdge_2nd:
.if IR_DEBUG
	ldi		temp2, IR_PAUSE_START_TOO_SHORT
	sts		ir_disc_code, temp2
.endif
.if IR_RECOGNIZE_REPETITION
	subi	temp0, low(IR_MIN_INTR(IR_PAUSE_START_REPETITION))
	sbci	temp1, high(IR_MIN_INTR(IR_PAUSE_START_REPETITION))
	brlo	ir_fallingEdge_discard
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
	subi	temp0, low(IR_MAX_INTR(IR_PAUSE_START_REPETITION))
	sbci	temp1, high(IR_MAX_INTR(IR_PAUSE_START_REPETITION))
	brlo	ir_fallingEdge_afterRepPause
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
.if IR_DEBUG
	ldi		temp2, IR_PAUSE_START_BETWEEN_R_D
	sts		ir_disc_code, temp2
.endif
.endif
	subi	temp0, low(IR_MIN_INTR(IR_PAUSE_START_DATA))
	sbci	temp1, high(IR_MIN_INTR(IR_PAUSE_START_DATA))
	brlo	ir_fallingEdge_discard
	mov		temp0, ir_intrCntrL
	mov		temp1, ir_intrCntrH
.if IR_DEBUG
	ldi		temp2, IR_PAUSE_START_TOO_LONG
	sts		ir_disc_code, temp2
.endif
	subi	temp0, low(IR_MAX_INTR(IR_PAUSE_START_DATA))
	sbci	temp1, high(IR_MAX_INTR(IR_PAUSE_START_DATA))
	brsh	ir_fallingEdge_discard
; Start pause was within bounds.
; Clear the data buffer:
	sts		ir_data, reg00
	sts		ir_data+1, reg00
	sts		ir_data+2, reg00
	sts		ir_data+3, reg00
	rjmp	ir_fallingEdge_end
ir_fallingEdge_discard:
.if IR_DEBUG
	sts		ir_disc_intrCntrL, ir_intrCntrL
	sts		ir_disc_intrCntrH, ir_intrCntrH
	sts		ir_disc_pulseCntr, ir_pulseCntr
	mov		temp2, ir_status
	ori		temp2, (1<<IR_STATUS_DISCARDED)
	mov		ir_status, temp2
.endif
	clr		ir_pulseCntr
	rjmp	ir_fallingEdge_end
ir_fallingEdge_setDataBit:
; Get the current data bit index:
	mov		temp0, ir_pulseCntr
	subi	temp0, IR_DATA_START
	mov		temp1, temp0				; Save a copy for later usage.
; Determine the data byte index:
	lsr		temp0						; Divide ...
	lsr		temp0
	lsr		temp0						; ... by 8.
	cpi		temp0, 4					; Make sure that ...
	brsh	ir_fallingEdge_end			; ... data byte index is within bounds.
; Load the Z pointer with data byte address:
	ldiz	ir_data
	addz	temp0
; Load the data byte:
	ld		temp0, Z
; Prepare the necessary bit mask for setting the bit:
	andi	temp1, 0x07
	ldi		temp2, 1					; becomes the bit mask
	tst		temp1
ir_setDataBit_loop:
	breq	ir_setDataBit_afterLoop
	lsl		temp2
	dec		temp1
	rjmp	ir_setDataBit_loop
ir_setDataBit_afterLoop:
; Set the bit in the data byte:
	or		temp0, temp2
; Write back:
	st		Z, temp0
.if IR_RECOGNIZE_REPETITION
	rjmp	ir_fallingEdge_end
ir_fallingEdge_afterRepPause:
	ldi		ir_pulseCntr, IR_REPETITION_PAUSE_RECEIVED
.endif
ir_fallingEdge_end:
	clr		ir_intrCntrL
	clr		ir_intrCntrH
; ### DEVICE-DEPENDENT ###
.if IR_TIMER_PRESC > 1
; Reset Timer2 prescaler, ATmega88A
	ldi		temp0, 1<<PSRASY
	xout	GTCCR, temp0
.endif
; ### END OF DEVICE-DEPENDENT ###
	rjmp	ir_intr_end
ir_intr_timeout:
.if IR_DEBUG
	ldi		temp0, IR_PULSE_TIMEOUT
	sts		ir_disc_code, temp0
.endif
	sbrs	ir_status, IR_STATUS_PREV			; If timeout occurred during pulse ...
	rjmp	ir_timeout_discard					; ... discard current reception.
; Timeout occurred during pause.
	tst		ir_pulseCntr
	breq	ir_timeout_end
.if IR_RECOGNIZE_REPETITION
	cpi		ir_pulseCntr, IR_REPETITION_PAUSE_RECEIVED
	breq	ir_timeout_repetition
.endif
.if IR_DEBUG
	ldi		temp0, IR_PULSE_COUNT_INVALID
	sts		ir_disc_code, temp0
.endif
	cpi		ir_pulseCntr, IR_PULSE_COUNT
	brne	ir_timeout_discard					; invalid pulse count
; Complete data frame received.
.if IR_PROTOCOL == IR_NEC
; Check integrity of address and command:
.if IR_DEBUG
	ldi		temp0, IR_DATA_INVALID
	sts		ir_disc_code, temp0
.endif
	lds		temp0, ir_data
	lds		temp1, ir_data+1
	com		temp1
	cp		temp0, temp1
	brne	ir_timeout_discard
	lds		temp0, ir_data+2
	lds		temp1, ir_data+3
	com		temp1
	cp		temp0, temp1
	brne	ir_timeout_discard
.endif
	mov		temp0, ir_status
	ori		temp0, (1<<IR_STATUS_DATA)
	mov		ir_status, temp0
	rjmp	ir_timeout_clr
ir_timeout_discard:
.if IR_DEBUG
	sts		ir_disc_intrCntrL, ir_intrCntrL
	sts		ir_disc_intrCntrH, ir_intrCntrH
	sts		ir_disc_pulseCntr, ir_pulseCntr
	mov		temp0, ir_status
	ori		temp0, (1<<IR_STATUS_DISCARDED)
	mov		ir_status, temp0
.endif
.if IR_RECOGNIZE_REPETITION
	rjmp	ir_timeout_clr
ir_timeout_repetition:
	mov		temp0, ir_status
	ori		temp0, (1<<IR_STATUS_REPETITION)
	mov		ir_status, temp0
.endif
ir_timeout_clr:
	clr		ir_pulseCntr
ir_timeout_end:
	clr		ir_intrCntrL
	clr		ir_intrCntrH
ir_intr_end:
; Restore modified registers:
	pop		temp0
	xout	SREG, temp0
.if IR_DEBUG
	pop		temp2
.endif
	pop		temp1
	pop		temp0
	reti
