; File:		timer-m48-lc204\main.asm
; Device:	ATmega48
; Created:	2023-10-09
; Version:	2023-10-24
; Author:	Johannes Fechner
;			https://www.mikrocontroller.net/user/show/jofe

.include <m48def.inc>	; automatically included by Microchip Studio
.equ EEARH = 0x22		; missing in m48def.inc (version 2.35)

.org 0
	rjmp	reset
.org OC2Aaddr				; Timer2 Compare Match A
	rjmp	ir_intr			; IR receiver
.org OC1Aaddr				; Timer1 Compare Match A
	rjmp	intr_seconds	; seconds tick
.org OC0Aaddr				; Timer0 Compare Match A
	rjmp	intr_display	; LED display multiplexing
.org INT_VECTORS_SIZE

.include "ir-rx-pulse-dist.config.asm"
.include "definitions.inc"
.include "macros.asm"
.include "ir-rx-pulse-dist.asm"
.include "remote-control-noname-defs.asm"

; === Register displayMode ===
; Bit positions of register displayMode:
.equ DISPLAY_MODE_EN = 0	; display enable (0: off, 1: on)
.equ DISPLAY_MODE_TEST = 1	; PSU test (all display segments and dots driven; 0: normal, 1: test)

; === Register timerState ===
;   Value		Description
;		0		display shows remaining time
;		1		set tens of minutes
;		2		set units of minutes
;		3		set tens of seconds
;		4		set units of seconds

; === Convenience macros ===
.macro clearSecondsTick
	ldi		temp0, (1<<PSRSYNC)
	xout	GTCCR, temp0			; Reset Timer0/1 prescaler.
	xout	TCNT1H, reg00			; Clear Timer1 counter registers.
	xout	TCNT1L, reg00
.endmacro

.macro startSecondsTick
	xin		temp0, TCCR1B
	ori		temp0, (1<<CS11) | (1<<CS10) ; prescaling 1/64
	xout	TCCR1B, temp0
.endmacro

.macro stopSecondsTick
	xin		temp0, TCCR1B
	andi	temp0, ~((1<<CS11) | (1<<CS10))
	xout	TCCR1B, temp0
.endmacro
	
reset:
; Initialize stack pointer:
	ldi		temp0, high(RAMEND)
	xout	SPH, temp0
	ldi		temp0, low(RAMEND)
	xout	SPL, temp0
; Initialize outputs:
	ldi		temp0, 0
	xout	LED_S_PORT, temp0
	ldi		temp0, 0xFF
	xout	LED_S_DDR, temp0
	ldi		temp0, (1<<LED_C_DIGIT3) | (1<<LED_C_DIGIT2) | (1<<LED_C_DIGIT1) | (1<<LED_C_DIGIT0) | (1<<LED_C_DOTS)
	xout	LED_C_DDR, temp0
	xout	LED_C_PORT, temp0
	xin		temp0, RELAY_PORT
	andi	temp0, ~(1<<RELAY)
	xout	RELAY_PORT, temp0
	xin		temp0, RELAY_DDR
	ori		temp0, (1<<RELAY)
	xout	RELAY_DDR, temp0
; Initialize Timer0 (LED display multiplexing):
	ldi		temp0, 250-1							; 50 Hz multiplex frequency
	xout	OCR0A, temp0
	ldi		temp0, (1<<WGM01)						; CTC mode
	xout	TCCR0A, temp0
	ldi		temp0, (1<<OCIE0A)						; enable compare match A interrupt
	xout	TIMSK0, temp0
; Initialize Timer1 (seconds tick):
	ldi		temp0, high(F_CPU/64-1)					; delay of 1s between compare matches
	xout	OCR1AH, temp0
	ldi		temp0, low(F_CPU/64-1)
	xout	OCR1AL, temp0
	ldi		temp0, (1<<WGM12)						; CTC mode
	xout	TCCR1B, temp0
	ldi		temp0, (1<<OCIE1A)						; enable compare match A interrupt
	xout	TIMSK1, temp0
; Initialize Timer2 (IR receiver):
	ldi		temp0, IR_TIMER_OCR
	xout	OCR2A, temp0
	ldi		temp0, (1<<WGM21)						; CTC mode
	xout	TCCR2A, temp0
	ldi		temp0, (1<<OCIE2A)						; enable compare match A interrupt
	xout	TIMSK2, temp0
; Initialize general registers:
	clr		reg00
	clr		timerState
	clr		displayState
	clr		displayMode
; Initialize IR registers:
	rcall	ir_init
; Initialize timer:
	rcall	loadSavedTime
; == Enable interrupts ==
	sei
; Start Timer0 and Timer2:
	ldi		temp0, (1<<CS01) | (1<<CS00)			; prescaling 1/64
	xout	TCCR0B, temp0
	ldi		temp0, (1<<CS20)						; no prescaling
	xout	TCCR2B, temp0
; Main loop:
loop:
	sbrs	ir_status, IR_STATUS_DATA
	rjmp	loop
; Valid IR data frame received.
	cli
; Check the RC address, ignore reception if it does not match:
	lds		temp0, ir_data
	cpi		temp0, RC_ADDRESS
	breq	loop_addrOK
	rjmp	loop_end
loop_addrOK:
; Copy the received command:
	lds		temp0, ir_data+2
; Convert command to ID:
	ldiz	2*rcCmdIdTable
	addz	temp0
	lpm		temp0, Z
; Check for POWER button:
	cpi		temp0, RC_CMD_POWER
	brne	loop_1
	ldi		temp0, 1<<DISPLAY_MODE_EN
	eor		displayMode, temp0			; toggle DISPLAY_MODE_EN
	rjmp	loop_end
loop_1:
; Check for TEST button:
	cpi		temp0, RC_CMD_TEST
	brne	loop_2
	ldi		temp0, 1<<DISPLAY_MODE_TEST
	eor		displayMode, temp0			; toggle DISPLAY_MODE_TEST
	rjmp	loop_end
loop_2:
; Make sure timerState is in allowed range:
	cpi		timerState, 5				; total count of states (highest state index +1)
	brlo	loop_3
; timerState is out of bounds.
; Get the timer into initial state:
	clr		timerState
	cbi		RELAY_PORT, RELAY			; Switch off relay.
	stopSecondsTick
	rcall	loadSavedTime
	rjmp	loop_end
loop_3:
; Branch according to timerState:
	ldiz	loop_jmpTbl
	addz	timerState
	ijmp
loop_jmpTbl:
	rjmp	loop_state0
	rjmp	loop_state1
	rjmp	loop_state2
	rjmp	loop_state3
	rjmp	loop_state4
loop_state0:
; Branch according to whether Timer1 is running or not:
	xin		temp1, TCCR1B
	sbrc	temp1, CS10
	rjmp	loop_state0_tr				; Timer1 is running.
; Timer1 is not running.
	cpi		temp0, RC_CMD_MENU
	brne	loop_state0_1
	rcall	timeSetup
	rjmp	loop_end
loop_state0_1:
	cpi		temp0, RC_CMD_PLUS
	brne	loop_state0_2
	sbi		RELAY_PORT, RELAY			; Switch on relay.
	rjmp	loop_end
loop_state0_2:
	cpi		temp0, RC_CMD_MINUS
	brne	loop_state0_3
	cbi		RELAY_PORT, RELAY			; Switch off relay.
	rjmp	loop_end
loop_state0_3:
	cpi		temp0, RC_CMD_PLAY
	brne	loop_state0_4
	sbi		RELAY_PORT, RELAY			; Switch on relay.
	clearSecondsTick
	startSecondsTick
	rjmp	loop_end
loop_state0_4:
	cpi		temp0, RC_CMD_BACK
	brne	loop_end
	rcall	loadSavedTime
	rjmp	loop_end
loop_state0_tr:
	cpi		temp0, RC_CMD_CANCEL
	brne	loop_end
	cbi		RELAY_PORT, RELAY			; Switch off relay.
	stopSecondsTick
	rjmp	loop_end
loop_state1:
	cpi		temp0, 10
	brsh	loop_state1_inv				; invalid command
; Button 0..9 was pressed.
	ldi		temp1, 10
	mul		temp0, temp1
	mov		minutes, r0
	rcall	to7segm
	mov		digit3, temp1
	inc		timerState
loop_state1_inv:
	rjmp	loop_end
loop_state2:
	cpi		temp0, 10
	brsh	loop_state2_inv				; invalid command
; Button 0..9 was pressed.
	add		minutes, temp0
	rcall	to7segm
	mov		digit2, temp1
	inc		timerState
loop_state2_inv:
	rjmp	loop_end
loop_state3:
	cpi		temp0, 6
	brsh	loop_state3_inv				; invalid command
; Button 0..5 was pressed.
	ldi		temp1, 10
	mul		temp0, temp1
	mov		seconds, r0
	rcall	to7segm
	mov		digit1, temp1
	inc		timerState
loop_state3_inv:
	rjmp	loop_end
loop_state4:
	cpi		temp0, 10
	brsh	loop_end					; invalid command
; Button 0..9 was pressed.
	add		seconds, temp0
	rcall	to7segm
	mov		digit0, temp1
; Store chosen time to EEPROM:
	rcall	storeTime
	clr		timerState
loop_end:
; Clear IR_STATUS_DATA flag:
	mov		temp0, ir_status
	andi	temp0, ~(1<<IR_STATUS_DATA)
	mov		ir_status, temp0
; Re-enable interrupts:
	sei
	rjmp	loop

; === Routine: start time setup ===
timeSetup:
	push	temp0
	ldi		timerState, 1
	clr		seconds
	clr		minutes
	ldi		temp0, 0b0000_0001		; '-' (segment g)
	mov		digit0, temp0
	mov		digit1, temp0
	mov		digit2, temp0
	mov		digit3, temp0
	ldi		temp0, 0b0001_1000		; ':' (dots D4, D5)
	mov		dots, temp0
	clt								; display leading zero
	pop		temp0
	ret

; === Routine: load saved time from EEPROM ===
; Interrupts must be globally disabled before calling this routine.
loadSavedTime:
	push	temp0
; Wait for completion of previous EEPROM write:
loadSavedTime_wait:
	sbic	EECR, EEPE
	rjmp	loadSavedTime_wait
; Set up address:
	xout	EEARH, reg00
	ldi		temp0, ee_seconds
	xout	EEARL, temp0
; Start EEPROM read:
	sbi		EECR, EERE
	xin		seconds, EEDR
; Set up address:
	xout	EEARH, reg00
	ldi		temp0, ee_minutes
	xout	EEARL, temp0
; Start EEPROM read:
	sbi		EECR, EERE
	xin		minutes, EEDR
; Display minutes:
	mov		temp0, minutes
	set							; suppress leading zero
	rcall	to7segm
	mov		digit3, temp2
	mov		digit2, temp1
; Display seconds:
	mov		temp0, seconds
	clt							; display leading zero
	rcall	to7segm
	mov		digit1, temp2
	mov		digit0, temp1
; Display colon separator:
	ldi		temp0, 0b0001_1000	; ':'
	mov		dots, temp0
	pop		temp0
	ret

; === Routine: store time setting to EEPROM ===
; Interrupts must be globally disabled before calling this routine.
storeTime:
	push	temp0
; Wait for completion of previous EEPROM write:
storeTime_wait0:
	sbic	EECR, EEPE
	rjmp	storeTime_wait0
; Set up address:
	xout	EEARH, reg00
	ldi		temp0, ee_seconds
	xout	EEARL, temp0
; Load data into EEPROM data register:
	xout	EEDR, seconds
; Start EEPROM write:
	sbi		EECR, EEMPE
	sbi		EECR, EEPE
; Wait for completion of previous EEPROM write:
storeTime_wait1:
	sbic	EECR, EEPE
	rjmp	storeTime_wait1
; Set up address:
	xout	EEARH, reg00
	ldi		temp0, ee_minutes
	xout	EEARL, temp0
; Load data into EEPROM data register:
	xout	EEDR, minutes
; Start EEPROM write:
	sbi		EECR, EEMPE
	sbi		EECR, EEPE
	pop		temp0
	ret

; === Interrupt routine: display multiplex ===
intr_display:
	push	temp0
	push	temp1
	xin		temp0, SREG
	push	temp0
; Branch according to displayState
	ldiz	intr_display_jmpTbl
	addz	displayState
	ijmp
intr_display_jmpTbl:
	rjmp	intr_display_state0
	rjmp	intr_display_state1
	rjmp	intr_display_state2
	rjmp	intr_display_state3
	rjmp	intr_display_state4
intr_display_state0:
; Drive DIGIT3
	mov		temp0, digit3
	sbrc	displayMode, DISPLAY_MODE_TEST
	ldi		temp0, 0xFF
	sbrs	displayMode, DISPLAY_MODE_EN
	clr		temp0
	xout	LED_S_PORT, temp0
	ldi		temp0, (1<<LED_C_DIGIT2) | (1<<LED_C_DIGIT1) | (1<<LED_C_DIGIT0) | (1<<LED_C_DOTS)
	xout	LED_C_PORT, temp0
	rjmp	intr_display_inc
intr_display_state1:
; drive DIGIT2
	mov		temp0, digit2
	sbrc	displayMode, DISPLAY_MODE_TEST
	ldi		temp0, 0xFF
	sbrs	displayMode, DISPLAY_MODE_EN
	clr		temp0
	xout	LED_S_PORT, temp0
	ldi		temp0, (1<<LED_C_DIGIT3) | (1<<LED_C_DIGIT1) | (1<<LED_C_DIGIT0) | (1<<LED_C_DOTS)
	xout	LED_C_PORT, temp0
	rjmp	intr_display_inc
intr_display_state2:
; drive DIGIT1
	mov		temp0, digit1
	sbrc	displayMode, DISPLAY_MODE_TEST
	ldi		temp0, 0xFF
	sbrs	displayMode, DISPLAY_MODE_EN
	clr		temp0
	xout	LED_S_PORT, temp0
	ldi		temp0, (1<<LED_C_DIGIT3) | (1<<LED_C_DIGIT2) | (1<<LED_C_DIGIT0) | (1<<LED_C_DOTS)
	xout	LED_C_PORT, temp0
	rjmp	intr_display_inc
intr_display_state3:
; drive DIGIT0
	mov		temp0, digit0
	sbrc	displayMode, DISPLAY_MODE_TEST
	ldi		temp0, 0xFF
	sbrs	displayMode, DISPLAY_MODE_EN
	clr		temp0
	xout	LED_S_PORT, temp0
	ldi		temp0, (1<<LED_C_DIGIT3) | (1<<LED_C_DIGIT2) | (1<<LED_C_DIGIT1) | (1<<LED_C_DOTS)
	xout	LED_C_PORT, temp0
	rjmp	intr_display_inc
intr_display_state4:
; drive DOTS
	mov		temp0, dots
	sbic	RELAY_PORT, RELAY						; If relay is on ...
	ori		temp0, 0b0000_0111						; ... left-hand-side dots (D1, D2, D3) on.
	xin		temp1, TCCR1B							; If Timer1 (seconds tick) ...
	sbrc	temp1, CS10								; ... is running, ...
	ori		temp0, 0b1010_0000						; ... right-hand-side dots (D7, D8) on.
	sbrc	displayMode, DISPLAY_MODE_TEST
	ldi		temp0, 0xFF
	sbrs	displayMode, DISPLAY_MODE_EN
	clr		temp0
	xout	LED_S_PORT, temp0
	ldi		temp0, (1<<LED_C_DIGIT3) | (1<<LED_C_DIGIT2) | (1<<LED_C_DIGIT1) | (1<<LED_C_DIGIT0)
	xout	LED_C_PORT, temp0
intr_display_inc:
; Update displayState
	inc		displayState
	cpi		displayState, 5
	brlo	intr_display_end
	clr		displayState
intr_display_end:
	pop		temp0
	xout	SREG, temp0
	pop		temp1
	pop		temp0
	reti
	
; === Routine: convert register to 2-digits decimal 7-segment code ===
; Input:	temp0:	Number to be converted, will be destroyed.
;					If T bit is set, the leading zero will be suppressed.
; Output:	temp1:	units in 7-segment code
;			temp2:	tens in 7-segment code
to7segm:
	ldi		temp2, 0
to7segm_tens:
	subi	temp0, 10
	brcs	to7segm_units
	inc		temp2
	rjmp	to7segm_tens
to7segm_units:
; temp2 now contains the tens.
	subi	temp0, -10						; Add 10, because the previous loop subtracted 10 once too much.
; temp0 now contains the units, convert to 7-segment code:
	ldiz	2*sevenSegmentTable
	addz	temp0
	lpm		temp1, Z
	brtc	to7segm_convTens				; Skip tens testing if T bit is cleared.
	tst		temp2
	breq	to7segm_end						; Skip converting if tens are zero.
to7segm_convTens:
; Convert the tens to 7-segment code:
	ldiz	2*sevenSegmentTable
	addz	temp2
	lpm		temp2, Z
to7segm_end:
	ret
	
; === Interrupt routine: seconds "tick" ===
intr_seconds:
	push	temp0
	push	temp1
	push	temp2
	xin		temp0, SREG
	push	temp0
	tst		minutes
	breq	intr_seconds_min0
	tst		seconds
	breq	intr_seconds_secUndfl
intr_seconds_decSec:
	dec		seconds
	rjmp	intr_seconds_convert
intr_seconds_min0:
	cpi		seconds, 1
	breq	intr_seconds_stop
	rjmp	intr_seconds_decSec
intr_seconds_secUndfl:
	dec		minutes
	ldi		seconds, 59
	rjmp	intr_seconds_convert
intr_seconds_stop:
	clr		seconds
	cbi		RELAY_PORT, RELAY		; Switch off relay.
	stopSecondsTick					; Stop Timer1 (seconds tick).
intr_seconds_convert:
; Convert minutes:
	mov		temp0, minutes
	set								; suppress leading zero
	rcall	to7segm
	mov		digit3, temp2
	mov		digit2, temp1
; Convert seconds:
	mov		temp0, seconds
	clt								; display leading zero
	rcall	to7segm
	mov		digit1, temp2
	mov		digit0, temp1
; Blink colon separator:
	ldi		temp0, 0b0001_1000
	eor		dots, temp0
; Restore SREG, temp2, temp1, temp0:
	pop		temp0
	xout	SREG, temp0
	pop		temp2
	pop		temp1
	pop		temp0
	reti

; === 7-Segment Code Table ===
; Bit assignment:
; bit no. | 76543210
; segment | c adbfeg
; dot     | 86745231
sevenSegmentTable:
.db 0b1011_1110, 0b1000_1000 ; 0, 1
.db 0b0011_1011, 0b1011_1001 ; 2, 3
.db 0b1000_1101, 0b1011_0101 ; 4, 5
.db 0b1011_0111, 0b1010_1000 ; 6, 7
.db 0b1011_1111, 0b1011_1101 ; 8, 9
