#![allow(dead_code)]
#![allow(non_snake_case)] // for ISRs
use core::cell::RefCell;
use core::intrinsics::bitreverse;

use attiny_hal::{
    pac::{EXINT, TC0, USI},
    port::{
        mode::{Input, Output, PullUp},
        Pin, PB0, PB1, PB2, PB3,
    },
};
use avr_device;
use avr_device::interrupt;
use avr_device::interrupt::{free, Mutex};

/*
 * USI UART Defines
 */

#[repr(u32)]
pub enum Baudrate {
    _115200 = 115200,
    _57600 = 57600,
    _38400 = 38400,
    _28800 = 28800,
    _19200 = 19200,
    _14400 = 14400,
    _9600 = 9600,
}

#[repr(u8)]
pub enum StopBits {
    _1 = 1,
    _2 = 2,
}

#[derive(PartialEq)]
enum State {
    Available,
    First,
    Second,
    Receive,
}

pub struct Pins {
    pub tx: Pin<Output, PB1>,        // Transmit pin   / ISP MISO / CH1 tx/red
    pub rx: Pin<Input<PullUp>, PB0>, // Receive pin0 / ISP MOSI / CH2 rx/green
    pub rw: Pin<Output, PB3>,        // nRE/DI pin             (CH3/black?)
    pub yellow: Pin<Output, PB2>,    //  debug pin / ISP SCK  / CH4 yellow
}

struct Timing {
    full_bit_ticks: u8,
    half_bit_ticks: u8,
    need_scaling: bool,
    start_delay: u8,
    timer_start_delay: u8,
}

struct USIUart {
    exint: EXINT, // external interrupts
    timer0: TC0,  // Access to the timer 0 device
    usi: USI,     // Access to the Universal Serial Interface
    pins: Pins,   // Access to the devices pins
    tx_data: u8,
    rx_data: Option<u8>, // Received byte, if any.
    state: State,
    timing: Timing,
    stop_bits: u8,
}

static USI_UART: Mutex<RefCell<Option<USIUart>>> = Mutex::new(RefCell::new(None));

pub fn init(
    exint: EXINT,
    timer0: TC0,
    usi: USI,
    pins: Pins,
    systemclock: u32,
    baudrate: Baudrate,
    stop_bits: StopBits,
) {
    free(|cs| {
        USI_UART.borrow(cs).replace(Some(USIUart {
            exint,
            timer0,
            usi,
            pins,
            state: State::Available,
            timing: calculate_timing(systemclock, baudrate),
            tx_data: 0xff,
            rx_data: None,
            stop_bits: stop_bits as u8,
        }));
    });
}

fn calculate_timing(systemclock: u32, baudrate: Baudrate) -> Timing {
    let cycles_per_bit: u32 = systemclock / baudrate as u32;

    let mut divisor: u8 = 1;
    let mut need_scaling = false;

    if cycles_per_bit > 255 {
        divisor = 8;
        need_scaling = true;
    }

    let full_bit_ticks = (cycles_per_bit / divisor as u32).try_into().unwrap();
    let start_delay: u8 = 65 + 42; // TODO: get correct value
    Timing {
        full_bit_ticks,
        half_bit_ticks: full_bit_ticks / 2,
        need_scaling,
        start_delay,
        timer_start_delay: start_delay / divisor,
    }
}

pub fn initialize_receive() {
    let mut ready = false;

    while !ready {
        free(|cs| {
            if let Some(usi_uart) = USI_UART.borrow(cs).borrow_mut().as_mut() {
                ready = usi_uart.state == State::Available;
                if ready {
                    usi_uart.pins.yellow.set_high();
                    usi_uart.state = State::Receive;
                    usi_uart.pins.rw.set_low(); // nRE low (read mode, MAX485)
                    usi_uart.pins.tx.set_high();
                    usi_uart.usi.usicr.reset(); // disable USI
                    usi_uart.exint.gifr.write(|w| w.pcif().set_bit()); // Clear pin change interrupt flag

                    // enable pin change interrupt on PB0 = rx
                    usi_uart.exint.gimsk.modify(|_, w| w.pcie().set_bit());
                    usi_uart.exint.pcmsk.modify(|_, w| w.bits(1))
                }
            }
        });
    }
}

pub fn send_byte(data: u8) {
    let mut ready = false;

    while !ready {
        free(|cs| {
            if let Some(usi_uart) = USI_UART.borrow(cs).borrow_mut().as_mut() {
                ready = usi_uart.state == State::Available;
            }
        });
    }

    free(|cs| {
        if let Some(usi_uart) = USI_UART.borrow(cs).borrow_mut().as_mut() {
            usi_uart.state = State::First;
            usi_uart.tx_data = bitreverse(data); // uart protocol

            usi_uart.timer0.tccr0a.write(|w| w.wgm0().ctc()); // clear timer on compare

            // select clock scale
            if usi_uart.timing.need_scaling {
                usi_uart.timer0.tccr0b.write(|w| w.cs0().prescale_8());
            } else {
                usi_uart.timer0.tccr0b.write(|w| w.cs0().direct());
            }
            usi_uart.timer0.gtccr.modify(|_, w| w.psr0().set_bit()); // reset prescaler

            // Set timer0 cycles
            usi_uart
                .timer0
                .ocr0a
                .write(|w| w.bits(usi_uart.timing.full_bit_ticks));
            usi_uart.timer0.tcnt0.reset();

            // Write start bit and first 7 data bits to usi dr
            usi_uart
                .usi
                .usidr
                .write(|w| w.bits(0 | usi_uart.tx_data >> 1));

            usi_uart.usi.usicr.write(|w| {
                w.usioie()
                    .set_bit()
                    .usiwm()
                    .three_wire()
                    .usics()
                    .tc0()
                    .usiclk()
                    .clear_bit()
            });
            // USI_OVF is executed after usicnt bits have been counted by tc0
            usi_uart
                .usi
                .usisr
                .write(|w| w.usioif().set_bit().usicnt().bits(16 - 8));
        }
    });
}

// USI overflow interrupt indicates we have sent a buffer
#[interrupt(attiny85)]
fn USI_OVF() {
    free(|cs| {
        if let Some(usi_uart) = USI_UART.borrow(cs).borrow_mut().as_mut() {
            if usi_uart.state == State::First {
                usi_uart.state = State::Second;
                usi_uart
                    .usi
                    .usidr
                    // Send last 1 bit of data and stop bits (high)
                    .write(|w| w.bits(usi_uart.tx_data << 7 | 0b0111_1111));
                usi_uart
                    .usi
                    .usisr
                    // Clear USI overflow interrupt flag,
                    // set USI counter to send last data bit and stop bit(s)
                    .write(|w| {
                        w.usioif()
                            .set_bit()
                            .usicnt()
                            .bits(16 - (1 + usi_uart.stop_bits))
                    });
            } else if usi_uart.state == State::Receive {
                let received_byte: u8 = usi_uart.usi.usibr.read().bits();
                usi_uart.usi.usicr.reset(); // all to default 0, i.e. disable USI
                usi_uart.rx_data = Some(bitreverse(received_byte));
                usi_uart.exint.gifr.write(|w| w.pcif().set_bit()); // Clear pin change interrupt flag
                usi_uart.pins.yellow.set_low();
            } else {
                usi_uart.pins.tx.set_high();
                usi_uart.usi.usicr.reset(); // all to default 0, i.e. disable USI
                usi_uart.usi.usisr.modify(|_, w| w.usioif().set_bit()); // clear interrupt flag

                usi_uart.state = State::Available;
            }
        }
    });
}

// The pin change interrupt is used to detect USI_UART reception.
// It is here the USI is configured to sample the UART signal.
#[interrupt(attiny85)]
fn PCINT0() {
    free(|cs| {
        if let Some(usi_uart) = USI_UART.borrow(cs).borrow_mut().as_mut() {
            if usi_uart.pins.rx.is_low() {
                usi_uart.pins.yellow.set_low();
                // We received a start bit.
                // Don't trigger interrupt for data bits
                usi_uart.exint.gimsk.modify(|_, w| w.pcie().clear_bit());

                usi_uart.timer0.tccr0a.write(|w| w.wgm0().ctc()); // clear timer on compare

                // select clock scale
                if usi_uart.timing.need_scaling {
                    usi_uart.timer0.tccr0b.write(|w| w.cs0().prescale_8());
                } else {
                    usi_uart.timer0.tccr0b.write(|w| w.cs0().direct());
                }
                usi_uart.timer0.gtccr.modify(|_, w| w.psr0().set_bit()); // reset prescaler

                // Set timer0 cycles
                usi_uart.timer0.ocr0a.write(|w| {
                    w.bits(usi_uart.timing.half_bit_ticks + usi_uart.timing.timer_start_delay)
                });
                usi_uart.timer0.tcnt0.reset(); //start counting at 0
                usi_uart.timer0.tifr.modify(|_, w| w.ocf0a().set_bit()); // Clear output compare flag
                usi_uart.timer0.timsk.modify(|_, w| w.ocie0a().set_bit()); // Enable timer0 output compare match A interrupt
            }
        }
    });
}

// Timer0 COMPA interrupt will be called when the middle of bit 0 has arrived
#[interrupt(attiny85)]
fn TIMER0_COMPA() {
    free(|cs| {
        if let Some(usi_uart) = USI_UART.borrow(cs).borrow_mut().as_mut() {
            usi_uart.pins.yellow.set_high();
            usi_uart.timer0.timsk.modify(|_, w| w.ocie0a().clear_bit()); // Disable this interrupt
            usi_uart.timer0.tcnt0.reset(); //start counting at 0
                                           // Set timer0 cycles
            usi_uart
                .timer0
                .ocr0a
                .write(|w| w.bits(usi_uart.timing.full_bit_ticks));
            usi_uart.timer0.tcnt0.reset();

            // Set up usi to use timer0 compare/match, no automatic wire mode
            usi_uart.usi.usicr.write(|w| {
                w.usioie()
                    .set_bit()
                    .usiwm()
                    .disabled()
                    .usics()
                    .tc0()
                    .usiclk()
                    .clear_bit()
            });
            // USI_OVF is executed after usicnt bits have been counted by tc0
            // We want to read 8 bits
            usi_uart
                .usi
                .usisr
                .write(|w| w.usioif().set_bit().usicnt().bits(8));
        }
    });
}
