/* ------------------------------------------------------------------
                              tm16xx.h

     Header zum Ansprechen eines TM16xx, Treiberbaustein fuer
     4 und 6 stellige 7-Segmentanzeigen mit gemeinsamer Anode und
     zusaetzlichem Tasteninterface.

     Geeignet fuer: TM1650, TM1637

     MCU   : STM8S
     F_CPU :

     Pinbelegung

       STM8S     7-Segment Anzeigentreiber
     --------------------------------------
        PC6                 CLK
        PC5                 DIO

     CLK und DIO sind mit jeweils 2,2 kOhm Pop-Up Wider-
     staenden nach +5V zu versehen

     05.12.2024 R. Seelig
   ------------------------------------------------------------------ */

/*

               F   DIG1  B    A   DIG0
               _    _    _    _    _
            __|_|__|_|__|_|__|_|__|_|__
           |                           |
           |         DIG1  DIG0        |
           |          __    __         |
           |         |__|  |__|        |
           |         |__|. |__|.       |
           |                           |
           |           2821BS          |
           |___________________________|
              |_|  |_|  |_|  |_|  |_|

               E    D    C    G   dp


             DIG3  A   F   DIG2  DIG1  B
             _    _    _    _    _    _
          __|_|__|_|__|_|__|_|__|_|__|_|__
         |                                |
         |     DIG3  DIG2   DIG1  DIG0    |
         |      __    __     __    __     |
         |     |__|  |__|   |__|  |__|    |
         |     |__|. |__|.  |__|. |__|.   |
         |                                |
         |           KYX-3641BS           |
         |________________________________|
            |_|  |_|  |_|  |_|  |_|  |_|

             E    D   dp    C    G   DIG0

    Segmentbelegung der Anzeige:

        a
       ---
    f | g | b            Segment |  a  |  b  |  c  |  d  |  e  |  f  |  g  | Doppelpunkt (nur fuer POS1) |
       ---               ---------------------------------------------------------------------------------
    e |   | c            Bit-Nr. |  0  |  1  |  2  |  3  |  4  |  5  |  6  |              7              |
       ---
        d

    Grundsaetzlicher Hinweis:

    An den Anschluss GRID1 des TM16xx wird die hoechstwertigste Anzeige angeschlossen.
    Bei einer 6-stelligen Anzeige ist das DIG5, bei einer 4-stelligen ist das DIG3.
    An einen TM1651 koennen nur max. 4 Digits angeschlossen werden

    -------------------------------------------

    Bsp. Anschluesse fuer TM1650 (gemeinsame Kathode)

    TM1651 (Pin)          7-Segmentanzeige (Pin)
    --------------------------------------------

    DIG4     7   ---------------  DIG0       6
    DIG3     6   ---------------  DIG1       8
    DIG2     5   ---------------  DIG2       9
    DIG1     1   ---------------  DIG3      12

    SEG1/K1  8   ---------------    A       11
    SEG2/K2  9   ---------------    B        7
    SEG3/K3 11   ---------------    C        4
    SEG4/K4 12   ---------------    D        2
    SEG5/K5 13   ---------------    E        1
    SEG6/K6 14   ---------------    F       10
    SEG7/K7 15   ---------------    G        5
      DP/KP 16   ---------------   DP        3

    VDD     10
    GND      4
    CLK      2
    DIO      3


    ----------------------------------------------------------

    Bsp. Anschluesse fuer TM1637 (gemeinsame Anode)

    TM1637 (Pin)          7-Segmentanzeige (Pin)
                                4 stellig           6 stellig
    ----------------------------------------------------------

    GRID6   10   ---------------                  |   DIG0
    GRID5   11   ---------------                  |   DIG1
    GRID4   12   ---------------  DIG0       6    |   DIG2
    GRID3   13   ---------------  DIG1       8    |   DIG3
    GRID2   14   ---------------  DIG2       9    |   DIG4
    GRID1   15   ---------------  DIG3      12    |   DIG5

    SEG1     2   ---------------    A       11    |
    SEG2     3   ---------------    B        7    |
    SEG3     4   ---------------    C        4    |
    SEG4     5   ---------------    D        2    |
    SEG5     6   ---------------    E        1    |
    SEG6     7   ---------------    F       10    |
    SEG7     8   ---------------    G        5    |
    SEG8     9   ---------------   dp        3    |

    VDD     16
    GND      1
    K2      20 (Keyscan 2)
    K1      19 (Keyscan 1)
    CLK     18
    DIO     17

*/

#ifndef in_tm16xx
  #define in_tm16xx

  #include "stm8s.h"
  #include "stm8_init.h"
  #include "stm8_gpio.h"

  /* ------------------------------------------------------------------
                            Controllerauswahl

       es kann (und muss) nur ein einziger Controller ausgewaehlt sein.
     ------------------------------------------------------------------ */

  #define  tm1637_ctrl              1
  #define  tm1650_ctrl              0


  /* ------------------------------------------------------------------
                        Auswahl einzubindender Funktionen
     ------------------------------------------------------------------ */


  #define enable_6dezdigit          1         // Code fuer 6-stellige dezimale Anzeige einbinden
  #define enable_6dezdigit_nonull   1         // Code fuer 6-stellige dezimale Anzeige mit
                                              // Ausblendung fuehrender Nullen einbinden
  #define enable_6hexdigit          0         // Code fuer 6-stellige hexadezimale Anzeige einbinden


  /* ----------------------------------------------------------
            Anschluss CLK und DIO  an den Controller
     ---------------------------------------------------------- */
  // ----------------------------------------------------------------
  //   Anmerkung zum Setzen von 1 und 0 auf den Pins
  //
  //   Die Pins sind in der Grundkonfiguration als Eingang geschaltet.
  //   Beim Setzen einer 1 wird nur die Konfiguration des Pins als
  //   Eingang benoetigt, da dieser dann hochohmig ist und die Leitung
  //   ueber den Pull-Up Widerstand auf 1 gelegt wird.
  //   Bei der Ausgabe einer 0 wird der Pin als Ausgang konfiguriert
  //   und dieser Pin mit einer 0 beschrieben
  // ----------------------------------------------------------------

  // CLK nach PC6

  #define scl_init()     PC6_input_init()
  #define bb_scl_lo()    { PC6_output_init(); PC6_clr(); }
  #define bb_scl_hi()    scl_init()

  // SDA nach PC5

  #define sda_init()     PC5_input_init()
  #define bb_sda_lo()    { PC5_output_init(); PC5_clr(); }
  #define bb_sda_hi()    sda_init()
  #define bb_is_sda()    is_PC5()

  #define puls_us        10
  #define puls_len()     delay_us(puls_us)

  /* ----------------------------------------------------------
                       Globale Variable
     ---------------------------------------------------------- */

  extern uint8_t hellig;                  // beinhaltet Wert fuer die Helligkeit (erlaubt: 0x00 .. 0x07)

  extern uint8_t  led7sbmp[16];           // Bitmapmuster fuer Ziffern von 0 .. F

  /* ----------------------------------------------------------
                           tm16_xofs
       Position, um die eine Anzeigenausgabe verschoben werden
       kann. Wird hier bspw. 2 angegeben, so erscheint die
       Ausgabe fuer Digit 0 nicht an Digit 0, sondern an
       Digit 2, Digit 1 an Digit 3 etc. Dieses dient vor allem
       dafuer, wenn ein 4-stelliger Wert auf einem 6-stelligen
       Display rechts buendig ausgegeben werden soll.
     ---------------------------------------------------------- */
   extern int8_t tm16_xofs;


  /* ----------------------------------------------------------
                           PROTOTYPEN
     ---------------------------------------------------------- */

  void tm16_start(void);
  void tm16_stop(void);
  void tm16_write (uint8_t value);
  uint8_t tm16_read(uint8_t ack);

  void tm16_init(void);
  void tm16_clear(void);
  void tm16_selectpos(int8_t nr);
  void tm16_setbright(uint8_t value);
  void tm16_setbmp(uint8_t pos, uint8_t value);
  void tm16_setzif(uint8_t pos, uint8_t zif);
  void tm16_setseg(uint8_t pos, uint8_t seg);
  void tm16_setdez(int value, uint8_t dpanz);
  void tm16_setdez2(uint8_t value, uint8_t pos, uint8_t dpanz);
  void tm16_sethex(uint16_t value);
  uint8_t tm16_readkey(void);

  void tm16_setdez6digit(uint32_t value, uint8_t dpanz);
  void tm16_setdez6digit_nonull(int32_t value, uint8_t dpanz);
  void tm16_sethex6digit(uint32_t value);

#endif
