/* -------------------------------------------------------------
                        my_time.c

     Eigene Zeitberechnung, aehnlich time.h der
     Standardbibliothek, diese jedoch abgespeckt.

     Es werden die Sekunden gezaehlt, die seit dem
     01.01.1970  (angegeben in yearofs) vergangen
     sind.

     Mittels der Struktur < mydate_time > kann das
     Datum fuer Jahr, Monat, Tag, Stunde, Minude und
     Sekunde angegeben werden und daraus ein Zeit-
     stempel errechnet werden.

     Zudem kann "rueckwaerts" aus einem Zeitstempel
     das Datum berechnet werden

     30.11.2020 R. Seelig
   ------------------------------------------------------------- */

#include <stdint.h>
#include "my_time.h"

mydate_time mydate;

/* -------------------------------------------------------------
                      is_schaltjahr

     Stellt fest, ob es ein Schaltjahr ist oder nicht.
     Es erfolgt keine Ueberpruefung fuer 100 und 400
     jaehrige Zyklen und somit nur gueltig von 1900
     bis 2099

     Rueckgabe:   1 : Schaltjahr
                  0 : kein Schaltjahr
   ------------------------------------------------------------- */
uint8_t is_schaltjahr(int year)
{
  if ((year % 4) == 0) return 1; else return 0;
}

/* -------------------------------------------------------------
                      month_len

     Ermittelt die Anzahl Tage eines Monats fuer ein
     gegebenes Jahr

     Rueckgabe:   Anzahl Tage
   ------------------------------------------------------------- */
uint8_t month_len(int year, uint8_t month)
{
  if (month == 2) return 28 + is_schaltjahr(year);

  // Juli und August haben beide 31 Tage ( Knoechel einer Hand Regel :-) )
  if (month > 7) month++;          // hierdurch wird ab August jeder gerade Monat
                                   // zu einem ungeraden Monat

  return 30 + (month & 1);         // ungerade Monate haben 31 Tage
}

/* -------------------------------------------------------------
                            mydate_get

     Ermittelt aus einem Zeitstempel (Sekunden die seit dem
     01.01. des Jahres, die in yearofs angegeben sind, vergangen
     sind) das Datum.

     Uebergabe:   stamp : Anzahl Sekunden seit dem 01.01.yearofs

     Rueckgabe:   keine, veraenderte globale Struktur mydate
   ------------------------------------------------------------- */
void mydate_get(uint32_t stamp)
{

  mydate.year= yearofs;

  // Jahre abzaehlen
  do
  {
     if (stamp >= (sec_per_year + (sec_per_day * is_schaltjahr(mydate.year))))
     {
       stamp -=   (sec_per_year + (sec_per_day * is_schaltjahr(mydate.year)));
       mydate.year++;
     }
  } while (stamp >= (sec_per_year + (sec_per_day * is_schaltjahr(mydate.year))));

  // Monate abzaehlen
  mydate.month= 1;
  do
  {
    if (stamp >= (sec_per_day * month_len(mydate.year, mydate.month)))
    {
      stamp -= (sec_per_day * month_len(mydate.year, mydate.month));
      mydate.month++;
    }
  } while (stamp >= (sec_per_day * month_len(mydate.year, mydate.month)));

  // Tage abzaehlen
  mydate.day= 1;
  do
  {
    if (stamp >= sec_per_day)
    {
      stamp -= sec_per_day;
      mydate.day++;
    }
  } while (stamp >= sec_per_day);

  // Stunden abzaehlen
  mydate.hour= 0;
  do
  {
    if (stamp >= sec_per_hour)
    {
      stamp -= sec_per_hour;
     mydate. hour++;
    }
  } while (stamp >= sec_per_hour);

  // Minuten abzaehlen
  mydate.min= 0;
  do
  {
    if (stamp >= sec_per_min)
    {
      stamp -= sec_per_min;
      mydate.min++;
    }
  } while (stamp >= sec_per_min);

  mydate.sec= stamp;

}


/* -------------------------------------------------------------
                            mydate_getstamp

     Ermittelt aus dem Kalenderdatum der Strukturvariable mydate
     den Zeitstempel (vergangene Sekunden seit dem 01.01. des
     Jahres, das in yearofs angegeben ist)

     Rueckgabe:   Zeitstempel in Sekunden
   ------------------------------------------------------------- */
uint32_t mydate_getstamp(void)
{
  uint32_t cnt1, cnt2;

  // Sekunden der Jahre in cnt2 aufaddieren
  cnt1= 0; cnt2= 0;
  while (cnt1 + yearofs < mydate.year)
  {
    cnt2 += sec_per_year + (sec_per_day * is_schaltjahr(cnt1 + yearofs));
    cnt1++;
  }

  // Sekunden der Monate aufaddieren
  cnt1 = 1;
  while (cnt1 < mydate.month)
  {
    cnt2 += month_len(mydate.year, cnt1) * sec_per_day;
    cnt1++;
  }

  // Sekunden der Tage aufaddieren
  cnt1 = 1;
  while (cnt1 < mydate.day)
  {
    cnt2 += sec_per_day;
    cnt1++;
  }

  // Stunden seit Tagesbeginn aufaddieren
  cnt1 = 0;
  while (cnt1 < mydate.hour)
  {
    cnt2 += sec_per_hour;
    cnt1++;
  }

   // Minuten der aktuellen Stunde aufaddieren
  cnt1 = 0;
  while (cnt1 < mydate.min)
  {
    cnt2 += sec_per_min;
    cnt1++;
  }

  cnt2 += mydate.sec;
  return cnt2;
}

/* --------------------------------------------------
                    mydate_getwtag

     Berechnet zu einem bestimmten Datum den
     Wochentag (nach Carl Friedrich Gauss):

     Rueckgabe: der Tag der Woche, beginnend ab
                Sonntag. Sonntag entspricht 0.

     Bsp.:      Mo: 01.07.2024 06:03.08
     Rueckgabe: 1

   -------------------------------------------------- */
char mydate_getwtag(void)
{
  int w_tag;

  if (mydate.month < 3)
  {
     mydate.month = mydate.month + 12;
     mydate.year--;
  }
  w_tag = (mydate.day + 2 * mydate.month + (3 * mydate.month+3) / 5 +                       \
           mydate.year + mydate.year/4 - mydate.year/100 + mydate.year/400 + 1) % 7 ;
  return w_tag;
}
