{* ------------------------------------------------------------------------
                          stm8_bootflash_rts.pas

     Hostprogramm zum Upload von Intel-Hex Dateien in einen STM8S103F3
     Microcontroller.

     Betriebssystem  : Linux
     Compiler        : FreePascal 2.6.4 oder hoeher

     Proprietaeres Uebertragungsprotokoll:

          Host sendet       STM8 Bootloader antwortet
       ----------------------------------------------
             "u"                      "o"
       ----------------------------------------------
            32-Bit
         Resetvektor
         Userprogramm
         Binaerformat
       ----------------------------------------------
             Anzahl
             Byte Bloecke
             (64 bei low density
              pagesize bei high density)
        (1 Byte binaer)
       ----------------------------------------------
                            Echo der Resetvektor-
                            adresse und der Anzahl
                            der Bloecke
       ----------------------------------------------
           n-Anzahl
           Byte Bloecke
                            nach jedem empfangenen
                            Block wird ein "o"
                            gesendet
       ----------------------------------------------
                            Start des Userprogramms
       ----------------------------------------------

     19.04.2017  R. Seelig
   ------------------------------------------------------------------------ *}
program stm8_bootflash;

uses
  sysutils,synaser;

const
  dtime_clear           = 10;
  dtime_hi              = 25;
  dtime_lo              = 5;

  maxadress_lodensity   = $9e3f;
  maxadress_hidensity   = $be3f;

var
  ser         : tblockserial;
  showtxbar   : boolean;
  filename    : string;
  portname    : string;
  devicename  : string;
  binarray    : array[0..65535] of byte;

  pagesize    : byte;                       // nimmt die Anzahl der Byte per Page
                                            // auf und ist abhaengig vom verwendeten Controller
                                            // STM8S103F3 = low density = 64 Bytes
                                            // STM8S105K4 = high density = pagesize Bytes

  maxadress   : word;


procedure txpbarscala(pbaranzahl : byte);
var
  b, b2, b3       : byte;

begin
  write(' Writing |');
  b2:= pbaranzahl;
  for b:= 0 to b2 do
  begin
    b3:= b mod 10;
    if (b3= 0) then
    begin
      write('|');
    end else
    begin
      write('-');
    end;
  end;
end;

procedure txpbar(startzeit : comp; maxwert : word; value : word; ch : char; pbaranzahl : byte);
var
  r          : real;
  zanz, b    : byte;
  ts         : ttimestamp;
  lz2        : comp;
  prozent    : word;
  newstep    : word;
  xpos       : word;

begin
  newstep:= maxwert div pbaranzahl;
  if newstep= 0 then newstep:= 1;
  if (((value mod newstep)= 0) and (value> 0)) then
  begin
    write(chr(13));
    write(chr(27),'[',10,'C');
    r:= pbaranzahl;
    r:= (r / maxwert);
    zanz:= trunc(r * value);
    for b:= 0 to zanz do write(ch);

    xpos:= pbaranzahl + 1 + 10;
    write(chr(13));
    write(chr(27),'[',xpos,'C');

    prozent:= (value*100) div maxwert;
    ts:= DateTimeToTimeStamp(Now);
    lz2:= TimeStampToMSecs(ts);
    write (' ',prozent,'% ',((lz2 - startzeit) / 1000):3:2,'s  ');
  end;
end;

procedure txpbar100(startzeit : comp; pbaranzahl : byte);
var
  ts              : ttimestamp;
  lz2             : comp;
  prozent         : word;
  xpos            : word;
begin
  xpos:= pbaranzahl + 1 + 9;
  write(chr(13));
  write(chr(27),'[',xpos,'C');

  prozent:= 100;
  ts:= DateTimeToTimeStamp(Now);
  lz2:= TimeStampToMSecs(ts);
  write (' ',prozent,'% ',((lz2 - startzeit) / 1000):3:2,'s  ');
end;


function word2hex(val16 : word) : string;
var
  ub  : byte;
  w   : word;
  hs  : string[6];
begin
  w:= val16;
  ub:= (w shr 12);
  ub:= ub and $0f;
  if (ub< 10) then ub:= ub+ 48 else ub:= ub+55;                  // umwandeln nach Hex-Ziffer
  hs:= chr(ub);

  w:= val16;
  ub:= (w shr 8);
  ub:= ub and $0f;
  if (ub< 10) then ub:= ub+ 48 else ub:= ub+55;                  // umwandeln nach Hex-Ziffer
  hs:= hs+chr(ub);

  w:= val16;
  ub:= (w shr 4);
  ub:= ub and $0f;
  if (ub< 10) then ub:= ub+ 48 else ub:= ub+55;                  // umwandeln nach Hex-Ziffer
  hs:= hs+chr(ub);

  w:= val16;
  ub:= w and $0f;
  if (ub< 10) then ub:= ub+ 48 else ub:= ub+55;                  // umwandeln nach Hex-Ziffer
  hs:= hs+chr(ub);

  word2hex:= hs;

end;

function byte2hex(val8 : word) : string;
var
  ub  : byte;
  b   : word;
  hs  : string[4];
begin

  b:= val8;
  ub:= (b shr 4);
  ub:= ub and $0f;
  if (ub< 10) then ub:= ub+ 48 else ub:= ub+55;                  // umwandeln nach Hex-Ziffer
  hs:= chr(ub);

  b:= val8;
  ub:= b and $0f;
  if (ub< 10) then ub:= ub+ 48 else ub:= ub+55;                  // umwandeln nach Hex-Ziffer
  hs:= hs+chr(ub);

  byte2hex:= hs;

end;


procedure clr_inpuffer;
var
  ch     : char;
begin
  while (ser.waitingdataex> 0) do
  begin
    ch:= chr(ser.recvbyte(1));
    sleep(dtime_clear);
  end;
end;

procedure wait_inpuffer;
begin
  while (ser.waitingdataex= 0) do
  begin
  end;
end;

procedure rtsdtr_pulse;
begin
  ser.dtr:= true;
  ser.flush;
  sleep(150);
  ser.dtr:= false;
  ser.flush;
  sleep(150);
  ser.dtr:= true;
  ser.flush;

  ser.rts:= true;
  ser.flush;
  sleep(150);
  ser.rts:= false;
  ser.flush;
  sleep(150);
  ser.rts:= true;
  ser.flush;
end;

procedure FlashMCU(datnam : string);
var
  ub          : byte;
  ch          : char;
  tdat        : text;
  s, hs, hs2  : string;
  err         : integer;
  bytelen     : byte;
  fadress     : word;
  highadress  : word;

  banz        : word;
  wcx         : word;
  lz, lz2     : comp;
  ts          : ttimestamp;
  b, b1       : byte;
  dump        : boolean;
  blkanz      : byte;


begin
  dump:= false;
  highadress:= 0;
  assign(tdat,datnam);
  {$i-} reset(tdat); {$i+}
  if (ioresult <> 0) then
  begin
    writeln('Error: file ',datnam,' not found');
    halt;
  end;

  ts:= DateTimeToTimeStamp(Now);
  lz:= TimeStampToMSecs(ts);


  while(not(eof(tdat))) do
  begin
    readln(tdat,s);

    if (copy(s,1,1)= ':') then                      // Einleitung einer Zeile ":"
    begin
      if (copy(s,8,2)= '00') then                   // Position fuer "normale" Datenbytes
      begin
        hs:= '$'+copy(s,2,2);                       // Position fuer die Anzahl der Datenbytes
        val(hs,bytelen,err);
        dec(bytelen);
        hs:= '$'+copy(s,4,2);                       // Position Hi-Byte Speicheradresse
        val(hs,fadress,err);
        fadress:= fadress * 256;
        hs:= '$'+copy(s,6,2);
        val(hs,ub,err);
        fadress:= fadress+ub;

        for banz:= 0 to (( length(s)-11 ) div 2) - 1 do

        begin
          hs:= copy(s,10 +(banz * 2),1);
          hs2:= '$' + hs + copy(s,11 +(banz * 2),1);
          val(hs2,ub,err);
          binarray[fadress]:= ub;
          inc(fadress);
          if (fadress> highadress) then highadress:= fadress;

        end;

      end;
    end;
  end;
  close(tdat);
  dec(highadress);

  if (highadress > maxadress) then
  begin
    ser.free;
    writeln;
    writeln(' ERROR: Programm stored in hex-file is too large');
    writeln;
    halt;
  end;

  // zu Testzwecken, ob die Intel-Hex Datei korrekt eingelesen wird
  if (dump) then
  begin
    writeln('Hexdump'+chr(13)+chr(10));
    writeln(highadress);

    for wcx:= $8000 to highadress do
    begin
      if ((wcx mod 16)= 0) then
      begin
      writeln();
      write(word2hex(wcx),' ');
      end;
      write(byte2hex(binarray[wcx]),' ');
    end;
    writeln();
  end;

  blkanz:= (highadress-$8000) div pagesize;
  if (((highadress-$8000) mod pagesize)<> 0) then inc(blkanz);

  rtsdtr_pulse;

  clr_inpuffer;

  ser.sendbyte(ord('u'));               // Anfrage ob Bootloader ready
  ser.flush;
  sleep(dtime_lo);

  wait_inpuffer;
  ch:= chr(ser.recvbyte(1));            // Programmer sendet ein 'o' zurueck
  ser.flush;
  sleep(dtime_lo);

// den Resetvektor senden
  ser.sendbyte(binarray[$8000]);
  ser.flush;
  sleep(dtime_lo);

  ser.sendbyte(binarray[$8001]);
  ser.flush;
  sleep(dtime_lo);

  ser.sendbyte(binarray[$8002]);
  ser.flush;
  sleep(dtime_lo);

  ser.sendbyte(binarray[$8003]);
  ser.flush;
  sleep(dtime_hi);

// Anzahl zu schreibender Pagebyte Bloecke senden
  ser.sendbyte(blkanz);
  ser.flush;
  sleep(dtime_lo);

// Resetvektor und Blockanzahl vom STM8 zuruecklesen (zur Kontrolle)
  b:= ser.recvbyte(1);
  sleep(dtime_lo);

  b:= ser.recvbyte(1);
  sleep(dtime_lo);

  b:= ser.recvbyte(1);
  sleep(dtime_lo);

  b:= ser.recvbyte(1);
  sleep(dtime_lo);

  b:= ser.recvbyte(1);

  writeln('stm8_bootflashv2 v0.1 by R. Seelig 2017, 2019', chr(10));
  writeln(' Device             : ', devicename);
  writeln(' available flashsize: ', maxadress-$8000, ' Bytes');
  writeln(chr(10),' Bytes to write: ', (highadress-$8000));
  if (showtxbar) then
  begin
    writeln;
    txpbarscala(51);
    write(chr(13));
  end else
    writeln(' please wait...');


  clr_inpuffer;

  wcx:= 0;
  for b:= 1 to blkanz do
  begin
    for b1:= 1 to pagesize do
    begin
      ser.sendbyte(binarray[$8000+wcx]);
      ser.flush;
      inc(wcx);

      if (showtxbar) then
      begin
        txpbar(lz,highadress-$8000, wcx, '#', 50);
        write(chr(13));
      end;

    end;
    wait_inpuffer;
    ub:= ser.recvbyte(1);
  end;

  if (showtxbar) then
  begin
    txpbar100(lz, 50);
    writeln;
  end;

  writeln;

  ts:= DateTimeToTimeStamp(Now);
  lz2:= TimeStampToMSecs(ts);
  writeln(' Flashing time: ',((lz2 - lz) / 1000):3:2,'s  ');
  writeln;
  writeln(' stm8_bootflash done, thank you');

  sleep(dtime_clear);
  rtsdtr_pulse;
end;

{*  -----------------------------------------------------------------------
                                M-A-I-N
    ----------------------------------------------------------------------- *}
begin
  pagesize:= 64;                          // low densitiy
  pagesize:= 128;                         // high density
  if (paramcount < 3) then
  begin
    writeln('stm8_bootflashv2 v0.1');
    writeln('by R. Seelig 2017, 2019');
    writeln('');
    writeln('Syntax:');
    writeln('stm8_bootflashv2 device port filename [notxbar]'); writeln();
    writeln('  device    : MCU - device');
    writeln('  port      : port were the adapter is connected');
    writeln('  filename  : file to be uploaded');
    writeln('  notxbar   : optional, no bar graph is shown');
    writeln('              (use it by call from an IDE)');
    writeln();
    writeln('  Example   : stm8_bootflashv2 stm8s103f3 /dev/ttyUSB0 helloworld.ihx');
    writeln();
    writeln('  Supported devices:');
    writeln('    - STM8S103F3 ');
    writeln('    - STM8S003F3 ');
    writeln('    - STM8S105K4 ');
    writeln('    - LOWDENSITY ');
    writeln('    - HIGHDENSITY ');
    writeln();
    writeln('   Note:');
    writeln('    - The implemented two pseudodevices called lowdensity');
    writeln('       and highdensity set 64 Byte pages (for low density devices)');
    writeln('       or 128 Byte pages (high density devices');
    writeln();

    halt;
  end;

  devicename:= paramstr(1);
  portname:= paramstr(2);
  filename:= paramstr(3);

  devicename:= lowercase(devicename);

  if ((devicename = 'stm8s105k4') or (devicename = 'highdensity'))
  then
  begin
    pagesize:= 128;
    maxadress:= maxadress_hidensity;
  end
  else
  begin
    pagesize:= 64;
    maxadress:= maxadress_lodensity;
  end;

  showtxbar:= true;
  if (paramcount > 3) then
  begin
    if (paramstr(4)= 'notxbar') then showtxbar:= false;
  end;

  ser:= tblockserial.create;
  ser.raiseExcept:= false;
  ser.linuxlock:= false;
  ser.connect(portname);
  ser.config(57600, 8, 'N', 1, false, false);

  clr_inpuffer;

  flashmcu(filename);

  ser.free;

end.
