/* ---------------------------------------------------------
                         uhr_tm1637.c

     6-Digit Uhr mit Anzeigetreiber TM1637 fuer 7-Segment-
     anzeigen und Temperatursensor DS18B20

     MCU      : STM8Sx03 / STM8Sx05
     Compiler : SDCC 4.01 oder neuer

     02.01.2025   R. Seelig
   --------------------------------------------------------- */
#include <stdint.h>
#include <stdlib.h>

#include "stm8s.h"
#include "stm8_init.h"
#include "stm8_gpio.h"

#include "tm16xx.h"
#include "my_time.h"


#undef F_CPU

#define F_CPU     8000000ul

/* ---------------------------------------------------------
                 Anschluss Temperatursensor
       (Anschluesse fuer Anzeigecontroller in tm16xx.h)
   --------------------------------------------------------- */

#define onewire_clr()     { PC3_output_init(); PC3_clr(); }
#define onewire_set()     PC3_input_init();
#define onewire_readb()   is_PC3()


/* Wartezyklen fuer den OneWire-Bus und Programmablauf. Notwendig, da die Delay-Einheiten
   durch Assembler-NOPS in stm8s_init.h realisiert und auf 16 MHz kalibriert sind.
   Wartezyklen fuer einen 8 MHz Coretakt sind daher abweichend
*/

#if (F_CPU == 16000000ul)

  #define owdel1     480
  #define owdel2     420
  #define owdel3      60
  #define owdel4      45
  #define owdel5      14

  #define ndel1       50
  #define ndel2       20
  #define ndel3      150
  #define ndel4       10

#elif (F_CPU == 8000000ul)

  #define owdel1     240
  #define owdel2     210
  #define owdel3      30
  #define owdel4      22
  #define owdel5       6

  #define ndel1       25
  #define ndel2       10
  #define ndel3       75
  #define ndel4        5

#endif


// ########################################################################################
// Kalibriervariable, sollte vom EEProm-Inhalt waehrend der Laufzeit ueberschrieben werden

// Wert hier entspricht der Anzahl der Sekunden, um den die Uhr alle 24h um 00:01.30
// korrigiert wird. Zulaessige Werte sind: +-29 Sekunden

                               int8_t sek_korrektur = 0;

// ########################################################################################

uint8_t dosekcalib = 1;

uint8_t anzmode = 0;           // Anzeigemodus, 0 => permanente Uhranzeige
                               //               1 => Uhr-Datumsanzeige im Wechsel

uint8_t hell = 7;

int16_t temperature = 200;

volatile uint32_t sekcount;          // Sekundenzaehler und somit der Zeitstempel
volatile uint8_t no_intchange= 0;

/*
          1
         ----
    20  | 40 | 2
         ----
    10  |    | 4
         ----
          8

    n = 0x54, S = 0x6d, t = 0x78, M = 0x33 - 0x27,    r= 0x50
    i = 0x10, A = 0x77, b = 0x7c, C = 0x39, d = 0x5e, E = 0x79
    F = 0x71, o = 0x5c, Y = 0x66, J = 0x0e, H = 0x76, T = 0x07
    G = 0x3d, L = 0x38
*/

// Anzeigen Bitmaps
const uint8_t tx_year[4] = { 0x0e, 0x77, 0x76, 0x50 };    // "JAHr"
const uint8_t tx_mon[4]  = { 0x33, 0x27, 0x5c, 0x54 };    // "Mon"
const uint8_t tx_day[4]  = { 0x07, 0x77, 0x3d, 0x00 };    // "TaG"
const uint8_t tx_std[4]  = { 0x6d, 0x78, 0x5e, 0x00 };    // "Std"
const uint8_t tx_min[4]  = { 0x33, 0x27, 0x10, 0x54 };    // "Min"
const uint8_t tx_sec[4]  = { 0x6d, 0x79, 0x39, 0x00 };    // "SEC"

const uint8_t tx_set[4]  = { 0x6d, 0x79, 0x78, 0x00 };    // "SEt"
const uint8_t tx_cal[4]  = { 0x39, 0x77, 0x38, 0x00 };    // "CAL"
const uint8_t tx_mod[4]  = { 0x33, 0x27, 0x5c, 0x5e };    // "Mod"

const uint8_t tx_hell[4] = { 0x76, 0x79, 0x38, 0x38 };    // "HELL"

// Makros um die "Texte" auf die Anzeige zu bringen
#define show_yeartx()  tx2seg7(&tx_year[0])
#define show_montx()   tx2seg7(&tx_mon[0])
#define show_daytx()   tx2seg7(&tx_day[0])
#define show_stdtx()   tx2seg7(&tx_std[0])
#define show_mintx()   tx2seg7(&tx_min[0])
#define show_sectx()   tx2seg7(&tx_sec[0])

#define show_helltx()  tx2seg7(&tx_hell[0])

#define show_settx()   tx2seg7(&tx_set[0])
#define show_caltx()   tx2seg7(&tx_cal[0])
#define show_modtx()   tx2seg7(&tx_mod[0])

#define is_tastsel()   (tm16_readkey() == 1)
#define is_tastp()     (tm16_readkey() == 3)
#define is_tastm()     (tm16_readkey() == 2)


#define eep_maxaddr    127                      // STM8S003 EEProm-Groesse 128 Byte, maxaddr = 127
                                                // STM8S103 EEProm-Groesse 640 Byte, maxaddr = 639

#define eeprom_lock()    (FLASH_IAPSR &= ~(FLASH_IAPSR_DUL))

/* -----------------------------------------------------------
                     eeprom_unlock

     :-) Funktionsname sagt alles
   ----------------------------------------------------------- */
void eeprom_unlock(void)
{
  if (!(FLASH_IAPSR & FLASH_IAPSR_DUL))
  FLASH_DUKR= 0xae;
  FLASH_DUKR= 0x56;
}


/* -----------------------------------------------------------
                           eep_write

     schreibt den Wert value an die Adresse addr im internen
     EEPROM

     Uebergabe:
       addr  : Adresse unter der der Wert zu speichern ist
       value : zu speichernder Wert

     Rueckgabe:
             1 bei zulaessiger Adresse
             0 bei unzulaessiger Adresse
   ----------------------------------------------------------- */
int8_t eep_write(uint16_t addr, int8_t value)
{
  uint8_t *address = (char *) EEPROM_BASE_ADDR + addr;

  if (addr > eep_maxaddr) return 0;            // Pruefung auf zulaessige Speicheradresse

  eeprom_unlock();

  *address = (int8_t) value;

  eeprom_lock();
  return 1;
}

/* -----------------------------------------------------------
                          eep_read

     liest einen gespeicherten Wert aus dem internen EEProm
     uas

     Uebergabe:
       addr  : Adresse unter der der Wert gespeichert ist

     Rueckgabe:
       gelesener Wert, 0xff bei Adressueberschreitung

   ----------------------------------------------------------- */
int8_t eep_read(uint16_t addr)
{
  int8_t value;
  uint8_t *address = (char *) EEPROM_BASE_ADDR + addr;

  if (addr > eep_maxaddr) return 0xff;

  eeprom_unlock();

  value= *address;

  eeprom_lock();
  return value;
}

/* -----------------------------------------------------------------------
                               ds18b20_reset

     initialisiert (und resetet somit) den Sensor
   -----------------------------------------------------------------------*/
uint8_t ds18b20_reset()
{
  uint8_t i;

  onewire_clr();
  delay_us(owdel1);

  onewire_set();
  delay_us(owdel3);

  // Bus lesen und warten
  i = onewire_readb();
  delay_us(owdel2);

  return i;               // 0: Bus ok, 1: Fehler
}

/* -----------------------------------------------------------------------
                               ds18b20_writebit

     schreibt ein einzelnes Bit auf dem Bus
   -----------------------------------------------------------------------*/
void ds18b20_writebit(uint8_t bit)
{
  onewire_clr();
  delay_us(1);

  if(bit) onewire_set();
  delay_us(owdel3);                 // 0-1 Bit fuer 60us stehen lassen

  onewire_set();                    // Funktion mit 1 auf Busleitung beenden
}

/* -----------------------------------------------------------------------
                               ds18b20_readbit

     ein Bit vom Bus lesen
   -----------------------------------------------------------------------*/
uint8_t ds18b20_readbit(void)
{
  uint8_t bit=0;

  onewire_clr();
  delay_us(1);

  onewire_set();
  delay_us(owdel5);

  if(onewire_readb()) bit=1;

  delay_us(owdel4);

  return bit;
}

/* -----------------------------------------------------------------------
                               ds18b20_writebyte

     schreibt ein Byte auf dem Bus
   -----------------------------------------------------------------------*/
void ds18b20_writebyte(uint8_t data)
{
  uint8_t i= 8;

  while(i--)
  {
    ds18b20_writebit(data & 1);
    data = data >> 1;
  }
}

/* -----------------------------------------------------------------------
                               ds18b20_readbyte

     schreibt ein Byte auf dem Bus
   -----------------------------------------------------------------------*/
uint8_t ds18b20_readbyte(void)
{
  uint8_t i= 8, data= 0;

  while(i--)
  {
    data = data >> 1;
    data |= (ds18b20_readbit() << 7);
  }
  return data;
}

/* -----------------------------------------------------------------------
                               ds18b20_gettemp

     startet Messvorgang und gibt die gelesene Temperatur als Pseudo-
     kommazahl zurueck.

     Bsp.:  271 entspricht 27,1 oC
   -----------------------------------------------------------------------*/
int16_t ds18b20_gettemp(void)
{
  uint8_t temp_lo_by;
  uint8_t temp_hi_by;

  ds18b20_reset();
  ds18b20_writebyte(0xcc);                      // cmd: skip ROM
  ds18b20_writebyte(0x44);                      // cmd: Messvorgang starten

  while(!ds18b20_readbit());                    // warten bis Messvorgang abgeschlossen

  ds18b20_reset();                              // reset
  ds18b20_writebyte(0xcc);                      // cmd: skip ROM
  ds18b20_writebyte(0xbe);                      // cmd: read scratchpad

  // 2 Bytes aus Scratchpad lesen (12 relevante Bits)
  temp_lo_by = ds18b20_readbyte();
  temp_hi_by = ds18b20_readbyte();

  // setzt die beiden 8 Bit Werte zusammen und gibt die Temperatur als
  // Pseudokommazahl zurueck
  return  ( ( ( temp_hi_by << 8 ) + temp_lo_by ) * 10) / 16;
}


/* -----------------------------------------------------------
                            tx2seg7

      zeigt Zeichenbitmaps auf den linken 4 Digits des
      Displays an

      Uebergabe:
        *buf: Zeiger auf ein 4 Byte grosses Array, das die
              Bitmaps fuer den Framebuffer enthaelt
   ----------------------------------------------------------- */
void tx2seg7(const uint8_t *buf)
{
  uint8_t i;

  for (i= 0; i <4; i++)
  {
    tm16_setbmp(i, *buf);
    buf++;
  }
  tm16_setbmp(4, 0);
  tm16_setbmp(5, 0);
}

/* -----------------------------------------------------------
                           set_signeddez3

     schreibt einen 3-stelligen vorzeichenbehafteten Wert
     als 2 stelligen Wert mit einer Nachkommastelle auf die
     Anzeige.

     Uebergabe:
        value : zu schreibender Wert
        pos   : Anzeigeposition auf der Anzeige, 1== links
                                                 3== rechts
   ----------------------------------------------------------- */
void set_signeddez3(int16_t value, uint8_t pos)
{
  tm16_setbmp(5-pos+1, led7sbmp[abs(value) % 10]);
  value= value / 10;

  if (abs(value) < 10)
  {
    tm16_setbmp(5-(pos+1),0x00);                         // einstellige Ziffer die fuehrende 0 ausblenden
    tm16_setbmp(5-pos, led7sbmp[abs(value)] | 0x80);     // 0x80 => Dezimalpunkt
  }
  else
  {
    tm16_setdez2(abs(value), pos, 1);
  }
  if (value< 0)
  {
    tm16_setbmp(5-(pos+2),0x40);      // Minuszeichen anzeigen
  }
  else
  {
    tm16_setbmp(5-(pos+2),0x00);      // Anzeige auf "Minusposition" loeschen
  }

}

/* -----------------------------------------------------------
                        temperature_show
   ----------------------------------------------------------- */
void temperature_show(void)
{
  set_signeddez3(temperature,3);
  tm16_setbmp(4,0x63);            // hochgestelltes kleines "o"
  tm16_setbmp(5,0x39);            // Anzeige "C"
}

/* -----------------------------------------------------------
                           time_show
   ----------------------------------------------------------- */
void time_show(void)
{
  tm16_setdez2(mydate.sec,0,0);
  tm16_setdez2(mydate.min,2,1);
  tm16_setdez2(mydate.hour,4,1);
}

/* -----------------------------------------------------------
                           date_show
   ----------------------------------------------------------- */
void date_show(void)
{
  tm16_setdez2(mydate.year % 100,0,0);
  tm16_setdez2(mydate.month,2,1);
  tm16_setdez2(mydate.day,4,1);
}

/* --------------------------------------------------
              Timer1 overflow - interrupt
    -------------------------------------------------- */
void tim1_ovf(void) __interrupt(11)
// 11 ist der Interruptvektor fuer Timer1 overflow
{
  uint8_t tmp;

  sekcount++;
  if (no_intchange== 0)
  {
    mydate_get(sekcount);

    // es erfolgt eine Zeitkorrektur um 00:01.30 Uhr
    if ((mydate.hour== 0) && (mydate.min== 1) && (mydate.sec== 30) && dosekcalib)
    {
      sekcount += sek_korrektur;
      dosekcalib= 0;
    }

    if ((mydate.hour== 0) && (mydate.min== 2) && (mydate.sec== 0) && !dosekcalib)
    {
      dosekcalib= 1;
    }

    if (anzmode== 0)
    {
      time_show();
    }
    else
    {
      tmp= sekcount % 40;
      // 5 Sekunden lang Datum anzeigen
      if ((tmp> 3) && (tmp< 10))
      {
        date_show();
      }
      // 15 Sekunden lang Uhrzeit anzeigen
      else if ((tmp> 9) && (tmp< 16))
      {
        if (tmp== 10) temperature= ds18b20_gettemp();
        temperature_show();
      }
      else
      {
        time_show();
      }
    }
  }

  TIM1_SR1 &= ~TIM1_SR1_UIF;        // Interrupt quittieren

}

/* --------------------------------------------------
                       tim1_init
      intialisiert Timer mit Taktteiler / 16000
      (Prescaler somit = 15999) und startet diesen im
      Autoreloadmodus mit Autoreloadwert 999 (jeder
      1000ste Impuls erzeugt Interrupt)

      Bei F_CPU = 16 MHz gilt:

      16 MHz / 16000 = 1 KHz => 1 ms
      Autoreloadwert 999 => Wert / 1000

      => 1 ms / 1000 = 1 s !!!!!
   -------------------------------------------------- */
void tim1_init(void)
{

//  #define clockdivisor     ((uint16_t) 15999)      // Taktteiler fuer F_CPU == 16 MHz
  #define clockdivisor     ((uint16_t) 7999)      // Taktteiler fuer F_CPU == 8 MHz

  TIM1_PSCRH= (uint8_t) (clockdivisor >> 8);
  TIM1_PSCRL= (uint8_t) (clockdivisor & 0x0000ff);

  // Enable overflow
  TIM1_IER |= TIM1_IER_UIE;

  // Timer1 starten
  TIM1_CR1 |= TIM1_CR1_CEN;     // Timer1 enable ( CEN = ClockENable )
  TIM1_CR1 |= TIM1_CR1_ARPE;    // Timer1 autoreload ( mit Werten in TIM1_ARR )

  // Autoreloadwert = 999 (jeder 1000ste Impuls ein Interrupt)
  TIM1_ARRH= (uint8_t) (999 >> 8);
  TIM1_ARRL= (uint8_t) (999 & 0x00ff);

  int_enable();
}

/* -----------------------------------------------------------
                           getint8fromkey

     liest einen 2-stelligen 8-Bit Zahlenwert ueber die
     Plus- Minus-Tasten ein. Ein laengeres Betaetigen einer
     Taste laesst entsprechend den Zaehlerwert "val"
     automatisch inkrementieren / dekrementieren.

     Uebergabe:
       entry :  vorbelegter Zahlenwert
       min   :  kleinster Zahlenwert der eingegeben werden
                kann
       max   :  groesster Zahlenwert der eingegeben werden
                kann
       pm    :  plusminus
                1 => zeigt bei negativen Werten das Minus-
                zeichen auf Digit 2 an, ansonsten wird
                Digit 2 geloescht
                0 => keine Vorzeichenangabe

     Rueckgabe:
       eingegebener Zahlenwert
   ----------------------------------------------------------- */
int8_t getint8fromkey(int8_t entry, int8_t min, int8_t max, uint8_t pm)
{
  int8_t  val;
  uint8_t cx;

  val= entry;
  tm16_setdez2(abs(val),0,0);
  if (pm)
  {
    if (val< 0)
    {
      tm16_setbmp(3,0x40);      // Minuszeichen anzeigen
    }
    else
    {
      tm16_setbmp(3,0x00);      // Anzeige auf "Minusposition" loeschen
    }
  }
  while (!(is_tastsel() ))
  {

    // inkrementieren
    if (is_tastp())
    {
      val++;
      if (val> max) val= min;
      tm16_setdez2(abs(val),0,0);
      if (pm)
      {
        if (val< 0)
        {
          tm16_setbmp(3,0x40);      // Minuszeichen anzeigen
        }
        else
        {
          tm16_setbmp(3,0x00);      // Anzeige auf "Minusposition" loeschen
        }
      }

      delay(ndel1);

      cx= 0;
      while(is_tastp())
      {
        delay(ndel2);
        cx++;

        // automatisches, schnelles inkrementieren
        if ((is_tastp() && (cx> 35)))
        {
          while(is_tastp())
          {
            val++;
            if (val> max) val= min;
            tm16_setdez2(abs(val),0,0);
            if (pm)
            {
              if (val< 0)
              {
                tm16_setbmp(3,0x40);      // Minuszeichen anzeigen
              }
              else
              {
                tm16_setbmp(3,0x00);      // Anzeige auf "Minusposition" loeschen
              }
            }
            delay(ndel3);
          }
        }
      }
      delay(ndel1);
    }

    // denkrementieren
    if (is_tastm())
    {
      val--;
      if (val< min) val= max;
      tm16_setdez2(abs(val),0,0);
      if (pm)
      {
        if (val< 0)
        {
          tm16_setbmp(3,0x40);      // Minuszeichen anzeigen
        }
        else
        {
          tm16_setbmp(3,0x00);      // Anzeige auf "Minusposition" loeschen
        }
      }
      delay(ndel1);

      cx= 0;
      while(is_tastm())
      {
        delay(ndel2);
        cx++;

        // automatisches, schnelles dekrementieren
        if ((is_tastm() && (cx> 35)))
        {
          while(is_tastm())
          {
            val--;
            if (val< min) val= max;
            tm16_setdez2(abs(val),0,0);
            if (pm)
            {
              if (val< 0)
              {
                tm16_setbmp(3,0x40);      // Minuszeichen anzeigen
              }
              else
              {
                tm16_setbmp(3,0x00);      // Anzeige auf "Minusposition" loeschen
              }
            }
            delay(ndel3);
          }
        }
      }
      delay(ndel1);
    }
  }
  delay(ndel1);
  while (is_tastsel())
  delay(ndel1);
  return val;
}

/* -----------------------------------------------------------
                           uhr_stellen
   ----------------------------------------------------------- */
void uhr_stellen(void)
{
  uint8_t val;

  mydate_get(sekcount);           // Umrechnen Zeitstempel ==> Datum

  // Jahr eingeben
  show_yeartx();
  val= getint8fromkey(mydate.year-2000,0,99, 0);
  mydate.year= 2000 + val;

  // Monat eingeben
  show_montx();
  val= getint8fromkey(mydate.month,1,12, 0);
  mydate.month= val;

  // Tag eingeben
  // Maximalwert fuer Tag wird errechnet aus zuvor eingegebenen Werten
  // aus Jahr und Monat (Schaltjahr wird beruecksichtigt)
  show_daytx();
  val= getint8fromkey(mydate.day,1,month_len(mydate.year, mydate.month), 0);
  mydate.day= val;

  // Stunde eingeben
  show_stdtx();
  val= getint8fromkey(mydate.hour,0,23, 0);
  mydate.hour= val;

  // Minute eingeben
  show_mintx();
  val= getint8fromkey(mydate.min,0,59, 0);
  mydate.min= val;

  // Sekunde eingeben
  show_sectx();
  val= getint8fromkey(mydate.sec,0,59, 0);
  mydate.sec= val;
  sekcount= mydate_getstamp();              // Umrechnen Datum ==> Zeitstempel

  TIM1_CNTRH= 0;                            // Timer auf 0 zurueckstellen
  TIM1_CNTRL= 0;

  // Uhrzeit sofort anzeigen
  tm16_setdez2(mydate.sec,0,0);
  tm16_setdez2(mydate.min,2,1);
  tm16_setdez2(mydate.hour,4,1);

}

/* -----------------------------------------------------------
                         uhr_menutext

     zeigt den "Text" fuer Set, Mod und Cal auf der Anzeige
     an.
   ----------------------------------------------------------- */
void uhr_menutext(uint8_t val)
{
  switch (val)
  {
    case 0: show_settx(); break;
    case 1: show_modtx(); break;
    case 2: show_caltx(); break;
    case 3: show_helltx(); break;
    default: break;
  }
}

/* -----------------------------------------------------------
                         uhr_menu

     stellt die Auswahlmoeglichkeit fuer "Uhr stellen" (set),
     "Anzeigemodus" (mod), "Kalibrieren" (cal) und
     Helligkeit (hell) zur Verfuegung
   ----------------------------------------------------------- */
void uhr_menu(void)
{
  uint8_t wahl;

  wahl= 0;
  show_settx();
  while(1)
  {
    if (is_tastp())
    {
      wahl++;
      wahl %= 4;
      uhr_menutext(wahl);
      delay(ndel1);
      while(is_tastp() );
      delay(ndel1);
    }
    if (is_tastm())
    {
      if (wahl> 0) wahl--; else wahl= 3;
      uhr_menutext(wahl);
      delay(ndel1);
      while(is_tastm() );
      delay(ndel1);
    }
    if (is_tastsel() )
    {
      delay(ndel1);
      while(is_tastsel() );
      delay(ndel1);
      switch (wahl)
      {
        case 0 : uhr_stellen(); break;
        case 1 :
        {
          anzmode= getint8fromkey(anzmode,0,1, 0);
          eep_write(3,anzmode);
          break;
        }
        case 2 :
        {
          sek_korrektur= getint8fromkey(sek_korrektur,-29,29, 1);
          eep_write(2, sek_korrektur);
          break;
        }
        case 3 :
        {
          hell= getint8fromkey(hell, 0, 7, 0);
          tm16_setbright(hell);
          break;
        }
        default: break;
      }
      while (is_tastsel() );
      delay(ndel1);
      return;
    }
  }
}

/* -----------------------------------------------------------
                                 main
   ----------------------------------------------------------- */
int main(void)
{
  int16_t i;
  int16_t cx;
  uint32_t oldsek;

  sysclock_init(0);                  // erst internen Takt
  delay(ndel1);
  sysclock_init(1);                  // und dann auf externen Takt umschalten

  tm16_init();
  tm16_xofs= 0;

  sek_korrektur= eep_read(2);        // Calibrierwert aus EEProm lesen
  anzmode= eep_read(3);

  temperature= ds18b20_gettemp();
  if (anzmode > 1)
  {
    anzmode= 0;
    eep_write(3,anzmode);
  }

  sekcount= 1735734600ul;            // 01.01.2025  12:30
  mydate_get(sekcount);

  delay(ndel1);

  tim1_init();
  sysclock_init(1);

  while(1)
  {
    delay(2);
    if (is_tastp())
    {
      no_intchange= 1;
      tm16_clear();
      tm16_setbmp(4,0x63);            // hochgestelltes kleines "o"
      tm16_setbmp(5,0x39);            // Anzeige "C"
      delay(ndel1);
      while(is_tastp());
      delay(ndel1);
      oldsek= sekcount;
      while(oldsek== sekcount);          // warten bis der Anzeigeinterrupt gerade beendent ist
      temperature= ds18b20_gettemp();
      for (cx= 0; cx< 400; cx++)
      {
        temperature_show();
        delay(ndel4);
      }
      no_intchange= 0;
    }
    delay(2);
    if (is_tastm())
    {
      no_intchange= 1;
      date_show();
      delay(ndel1);
      while(is_tastm());
      delay(ndel1);
      oldsek= sekcount;
      while(oldsek== sekcount);          // warten bis der Anzeigeinterrupt gerade beendent ist
      for (cx= 0; cx< 400; cx++)
      {
        delay(ndel4);
      }
      no_intchange= 0;
    }
    delay(2);
    if (is_tastsel())
    {

      i= 0;
      while((is_tastsel()) && (i < 5))
      {
        delay(10);
        i++;
      }

      if (is_tastsel())
      {
        delay(ndel1);
        while(is_tastsel());
        delay(ndel1);
        no_intchange= 1;
        uhr_menu();
        no_intchange= 0;
      }
    }
  }
}
