/*

  oscillating servo controller  

*/

#define F_CPU 2048000       // Hz

#include "avr/io.h"
#include "avr/interrupt.h"
#include "avr/sleep.h"
#include "util/delay.h"
#include "avr/pgmspace.h"

#define Fmin 5              // /0.1Hz, min frequency
#define Fmax 50             // /0.1Hz, max frequency
#define F_DDS 1000          // /0.1Hz, DDS clock
#define FTW_MUL (0x10000 / F_DDS)

const int8_t sinus[256] PROGMEM = {
	  0,   2,   3,   5,   6,   8,   9,  11, 
	 12,  14,  15,  17,  18,  20,  21,  23, 
	 24,  26,  27,  28,  30,  31,  32,  34, 
	 35,  36,  38,  39,  40,  41,  42,  43, 
	 45,  46,  47,  48,  49,  50,  51,  52, 
	 52,  53,  54,  55,  56,  56,  57,  58, 
	 58,  59,  59,  60,  60,  61,  61,  61, 
	 62,  62,  62,  63,  63,  63,  63,  63, 
	 63,  63,  63,  63,  63,  63,  62,  62, 
	 62,  61,  61,  61,  60,  60,  59,  59, 
	 58,  58,  57,  56,  56,  55,  54,  53, 
	 52,  52,  51,  50,  49,  48,  47,  46, 
	 45,  43,  42,  41,  40,  39,  38,  36, 
	 35,  34,  32,  31,  30,  28,  27,  26, 
	 24,  23,  21,  20,  18,  17,  15,  14, 
	 12,  11,   9,   8,   6,   5,   3,   2, 
	  0,  -2,  -3,  -5,  -6,  -8,  -9, -11, 
	-12, -14, -15, -17, -18, -20, -21, -23, 
	-24, -26, -27, -28, -30, -31, -32, -34, 
	-35, -36, -38, -39, -40, -41, -42, -43, 
	-45, -46, -47, -48, -49, -50, -51, -52, 
	-52, -53, -54, -55, -56, -56, -57, -58, 
	-58, -59, -59, -60, -60, -61, -61, -61, 
	-62, -62, -62, -63, -63, -63, -63, -63, 
	-63, -63, -63, -63, -63, -63, -62, -62, 
	-62, -61, -61, -61, -60, -60, -59, -59, 
	-58, -58, -57, -56, -56, -55, -54, -53, 
	-52, -52, -51, -50, -49, -48, -47, -46, 
	-45, -43, -42, -41, -40, -39, -38, -36, 
	-35, -34, -32, -31, -30, -28, -27, -26, 
	-24, -23, -21, -20, -18, -17, -15, -14, 
	-12, -11,  -9,  -8,  -6,  -5,  -3,  -2
};


volatile uint8_t flag10ms, servo;
uint8_t set_amplitude, set_frequency;
uint16_t ftw; 

uint8_t read_adc(int channel) {
    
    ADMUX   = (ADMUX & ~0x3) | (channel & 0x3);
    ADCSRA |= (1<<ADSC);
    while(ADCSRA & (1<<ADSC));
    return ADCH;
}

void calc_ftw(void) {
    uint16_t f;

    f = Fmin + (((Fmax-Fmin) * set_frequency)>>8);
    ftw = f * FTW_MUL;
}

void calc_servo(void) {
    static uint16_t phase;
    uint8_t tmp;
    int8_t amp;

    phase += ftw;                       // DDS phase accu
    tmp = phase >> 8;
    amp = pgm_read_byte(&sinus[tmp]);
    amp = ((int)amp * set_amplitude) >> 7;
    servo = 192 + amp;
}

int main(void) {

    // system clock prescaler /4
    CLKPR = (1<<CLKPCE);
    CLKPR = 2;

    // IOs init
    DDRB  = (1<<PB1) | (1<<PB0);

    // Timer 0 init
    // mode 1, phase correct PWM, prescaler 8
    TCCR0A = (1<<COM0B1) | (1<<WGM00);
    TCCR0B = (1<<CS01);
    TIMSK0 = (1<<TOIE0);    // Interrupt enable

    // init ADC
	ADMUX  = (1<<ADLAR);    			            // VREF = VCC, left adjust, 8 bit
    ADCSRA = (1<<ADEN) | (1<ADPS2) | (1<<ADPS0);	// Prescaler 32
	read_adc(0);                                    // dummy read

    OSCCAL = 81;      // adjust to 100 +/-1Hz servo pulse frequency
    sei();
    while(1) {
        set_sleep_mode(SLEEP_MODE_IDLE);
        sleep_mode();

        if (flag10ms) {
            flag10ms = 0;
            //PINB = (1<<PB0);    // TEST
            set_frequency = read_adc(2);
            set_amplitude = read_adc(3) / 2;
            calc_ftw();
            calc_servo();
        }
    }
    return 0;
}

// TIMER 0 ISR

ISR(TIM0_OVF_vect) {
    static uint8_t cnt10ms;

    cnt10ms++;
    if (cnt10ms == 5) {
        cnt10ms  = 0;
        flag10ms = 1;
        OCR0B = servo;
    } else {
        OCR0B = 0;
    }
}
