# Datenstruktor mit Klasse und Methoden ohne Initialwerte
# mit _ für theoretischen Zugriffschutz, Zugriff "muss" über Getter/Setter Methoden erfolgen
# Diese Datei ist immer die aktuelle Version

# 13.10.2025

#from typing import Optional
#from datetime import datetime
#from enum import Enum, auto
from dataclasses import dataclass
from dataclasses import dataclass, replace
import copy
    
class KeithleySCPI:
    ### privat ####################################################################       

    # übersetzt von 'wie es im Display steht' zum 'SCPI Syntax' laut Manual
    def _getMode(self, measure: str):
        match measure:
            case "DCV":
                self._unit = "V"
                return "VOLTage:DC"
            case "DCI":
                self._unit = "A"
                return "CURRent:DC"
            case "ACV":
                self._unit = "V"
                return "VOLTage:AC"
            case "ACI":
                self._unit = "A"
                return "CURRent:AC"     
            case "2W":
                self._unit = "Ω"
                return "RESistance"
            case "4W":
                self._unit = "Ω"
                return "FRESistance"
            case "Freq":
                return "FREQuency"
            case "Period":
                return "PERiod"
            case "Diode":
                self._unit = "V"
                return "DIODe"
            case "Temp": 
                self._unit = "°C"
                return "TEMPerature"    
            case "Cap": 
                self._unit = "F"
                return "CAPacitance"                               
            case _:
                return "unknown"  
               
    @dataclass(frozen=True) # täuscht hinweg, Elemente sind dennoch änderbar, nur Objekt ist read only
    class _FunctionItem:
        name: str = ""
        command: str = ""

    # erweiterbare Funktionsliste, mittels Namen wird der hinterlegte SCPI Syntax ausgelesen
    _FUNCTION_LIST = [
        _FunctionItem("getTime",       ":SYSTem:TIME? 1"),              # Reference Manual Seite 12-158
        _FunctionItem("setTime",       ":SYSTem:TIME "),                # Reference Manual Seite 12-158 
        _FunctionItem("setFunctionON", ':SENSe:FUNCtion:ON "MESSMODE"'),
        _FunctionItem("measure",       ":MEASure:MESSMODE?"),           # Reference Manual Seite 12-137  
        _FunctionItem("read",          ":READ?"),                       # Reference Manual Seite 12-10
        _FunctionItem("lastBufferWriteIndex", ":TRACe:ACTual:END?"),    # Reference Manual Seite 12-160        
        _FunctionItem("readBuffer",    ":TRACe:DATA?"),                 # Reference Manual Seite 12-165  
        _FunctionItem("averageCount",  ":SENSE:MESSMODE:AVERage:COUNt "),
        _FunctionItem("averageRepeat", ":SENSE:MESSMODE:AVERage:TCONTrol REPeat"),
        _FunctionItem("averageMoving", ":SENSE:MESSMODE:AVERage:TCONTrol MOVing"),
        _FunctionItem("averageHybrid", ":SENSE:MESSMODE:AVERage:TCONTrol HYBRid"),
        _FunctionItem("averageON",     ":SENSE:MESSMODE:AVERage ON"),
        _FunctionItem("averageOFF",    ":SENSE:MESSMODE:AVERage OFF"),
        _FunctionItem("deleteBuffer",  ":TRACe:CLEar"),                 # Reference Manual Seite 12-164
        _FunctionItem("localMode",     ":TRIGger:CONTinuous RESTart")
    ]
        
    # Bsp.: print(_getSCPIcommandByName("averageON"))  # → ":SENSE:MESSMODE:AVERage ON"
    def _getSCPIcommandByName (self, name: str) -> str:
        item = self._functionMap.get(name)
        if item is None:
            raise ValueError(f"Unbekannter Funktionsname: {name}.")
        return item.command

    ### public ####################################################################    
      
    # Konstruktor
    def __init__(self, name):
        self._deviceName: str = name
        self._functionList = [] # zur Verwendung als Kopie von _FUNCTION_LIST
        self._functionMap = {item.name: item for item in self._FUNCTION_LIST} # init notwendig
        #self._mode          = None  # 'mode' soll ein String sein,     ?
        #self._function: str = None  # 'function' soll ein String sein, aktuelle Meßgröße
        self._unit: str      = None  # 'unit' soll ein String sein,     aktuelle Meßeinheit
        #self._value: int    = 0     # 'value' soll ein Integer sein,   aktueller Meßwert
        self.lastBufferReadIndex: int = 0   # zeigt auf den aktuellen Messwerteintrag im Buffer, 0 == leer
        
    # def formatFloat (self, value: float):
    #     formatValue = "{:.3f}".format(value)
    #     return formatValue
              
    def getUnit(self) -> str:
        return self._unit

    def setUnit(self, u: str):
        if u != "":
            self._unit = u
        else:
            print(f"{u} ist keine gültige Einheit!")    

    def getLastBufferReadIndex (self) -> int:
        return (self.lastBufferReadIndex+1)
    
    def setLastBufferReadIndex (self, count: int):
        self.lastBufferReadIndex = count
          
    def __str__(self) -> str:
        return (f"Ich bin das Keithley DMM6500 SCPI Modul")  
   
    def getDeviceName (self) -> str:
        return (self._deviceName)
    
    def getTime(self) -> str:   # Reference Manual Seite 12-158
        return self._getSCPIcommandByName("getTime")                        
          
    def setTime(self) -> str:   # Reference Manual Seite 12-158
        return self._getSCPIcommandByName("setTime") 

    def changeFunction (self, func: str):
        self._functionList = copy.deepcopy(self._FUNCTION_LIST)  # Kopie anlegen
        
        # zu suchender Eintrag und zu ersetzenden Eintrag festlegen bzw. ermitteln
        # in der kompletten SCPI-Syntax , der ersetzt werden soll
        target: str = "MESSMODE"
        replacement: str = self._getMode(func)
    
        # Ersetze für jeden Eintrag den Funktionsnamen/Messmodus und zwischenspeichern 
        functionListModified = [
            replace(item, command=item.command.replace(target, replacement))
            for item in self._functionList
        ]
        
        # Mapping aus neuer Liste (wenn man darauf per Name zugreift statt Indexnummer)
        self._functionMap = {item.name: item for item in functionListModified}
        #print(self._functionMap)  # Debug
                    
    def getFunction (self) -> str: 
        return self._getSCPIcommandByName("setFunctionON")
        
    def getMeasureOnce (self) -> str:  
        return self._getSCPIcommandByName("measure")

    def getReadOnce(self) -> str:     
        return self._getSCPIcommandByName("read")

    def getReadWithTimestamp(self, name: str) -> str:    
        scpiCode = (f"{self._getSCPIcommandByName("read")}").strip()
        # Komma nach '{scpiCode}' hier, weil erneut ein Zeilenumbruch drin wäre
        scpiCode = (f'{scpiCode} "{name}", TST, READ') 
        print(scpiCode)
        return scpiCode 
        
    def getReadBuffer(self, start, end, name: str) -> str:    
        scpiCode = (f'{self._getSCPIcommandByName("readBuffer")} {start}, {end}').strip()
        # Komma nach '{scpiCode}' hier, weil mit/nach '{end},' erneut ein Zeilenumbruch drin wäre
        scpiCode = (f'{scpiCode}, "{name}"') 
        return scpiCode 

    def getReadBufferWithTimestamp(self, start, end, name: str) -> str:    
        scpiCode = (f'{self._getSCPIcommandByName("readBuffer")} {start}, {end}').strip()
        # Komma nach '{scpiCode}' hier, weil mit/nach '{end},' erneut ein Zeilenumbruch drin wäre
        scpiCode = (f'{scpiCode}, "{name}", TST, READ') 
        #print(scpiCode)
        return scpiCode 
    
    def getLastBufferWriteIndex(self, name: str) -> str:    
        scpiCode = f'{self._getSCPIcommandByName("lastBufferWriteIndex")} "{name}"'
        return scpiCode 
        
    def getAverageCount(self) -> str:   
        return self._getSCPIcommandByName("averageCount")

    def getAverageRepeat(self) -> str:     
        return self._getSCPIcommandByName("averageRepeat")

    def getAverageMoving(self) -> str:     
        return self._getSCPIcommandByName("averageMoving")
    
    def getAverageHybrid(self) -> str:    
        return self._getSCPIcommandByName("averageHybrid")
        
    def getAverageOn(self) -> str:     
        return self._getSCPIcommandByName("averageON")

    def getAverageOff(self) -> str:       
        return self._getSCPIcommandByName("averageOFF")     
    
    def getDeleteBuffer (self, name: str) -> str:
        scpiCode = f'{self._getSCPIcommandByName("deleteBuffer")} "{name}"'
        return scpiCode 
    
    def getLocalMode (self) -> str:
        return self._getSCPIcommandByName("localMode")
   
        
                                  