#include <WiFi.h>
#include <HTTPClient.h>
#include <ArduinoJson.h>
#include <time.h>
#include <nvs_flash.h>
#include <nvs.h>

// ======================================================
// Funktions-Prototypen
// ======================================================
bool ensureWiFi(unsigned long timeoutMs = 15000);
struct ShellyTemps getShellyTemps();
bool fetchBrennwertFromServerWithStatus();
void blinkFlash(int count, int duration = 150);

// ======================================================
// Shelly Temperaturstruktur
// ======================================================
struct ShellyTemps {
  float temp_out;
  float temp_vorlauf;
  float temp_ruecklauf;
};

// ======================================================
// User Config
// ======================================================
const char* WIFI_SSID     = "MeineSSID";                                                    //ändern
const char* WIFI_PASSWORD = "MeingeheimesWLANPasswort";                                     //ändern

const char* SERVER_UPLOAD_URL     = "https://www.PfadzumeinemServer/gasupload.php";         //ändern
const char* SERVER_BRENNWERT_URL  = "https://www.PfadzumeinemServer/brennwertabfrage.php";  //ändern
const char* AUTH_KEY              = "meinGeheimerAuthToken";                                //frei wählbar, muss aber identisch mit dem Token in config.php sein
const char* DEVICE_ID             = "ESP32CAM_GAS_1";

const char* SHELLY_STATUS_URL = "http://IPvonmeinemShelly/rpc/Shelly.GetStatus";            //ändern

// ======================================================
// Hardware Pins (ESP32-CAM)
// ======================================================
const int PIN_REED  = 13;
const int PIN_FLASH = 4;

// ======================================================
// Energetische Konstanten / Intervalle
// ======================================================
const double K_TICK_M3 = 0.01;
double      H_KWH_PER_M3 = 11.54;

const unsigned long INTERVAL_MS          = 30UL * 60UL * 1000UL;
const unsigned long NVS_SAVE_INTERVAL_MS = 12UL * 60UL * 60UL * 1000UL;
const uint32_t      NVS_TICKS_MIN_DELTA  = 50;

// ======================================================
// Entprellung
// ======================================================
const unsigned long DEBOUNCE_US = 200000UL;

// ======================================================
// Zähler
// ======================================================
volatile uint32_t cumulative_ticks = 0;
volatile uint64_t lastPulseUs      = 0;

// ======================================================
// Timing
// ======================================================
unsigned long lastIntervalTs = 0;
unsigned long lastNvsSaveTs  = 0;
uint32_t      lastNvsTicks   = 0;

// ======================================================
// NVS
// ======================================================
nvs_handle_t nvsHandle;
const char*  NVS_NAMESPACE       = "gaslogger";
const char*  NVS_KEY_TICKS       = "ticks";
const char*  NVS_KEY_BRENNWERT   = "brennwert";

// ======================================================
// Datenstruktur für Upload
// ======================================================
struct Frame {
  uint32_t cumulative_ticks;
  double energy_kwh;
  double avg_power_kw;
  float temp_out_c;
  float temp_vorlauf_c;
  float temp_ruecklauf_c;
  String timestamp;
};

// ======================================================
// ISR: Reedkontakt
// ======================================================
void IRAM_ATTR onReedPulse() {
  uint64_t nowUs = (uint64_t)esp_timer_get_time();
  if (nowUs - lastPulseUs >= DEBOUNCE_US) {
    lastPulseUs = nowUs;
    cumulative_ticks++;
  }
}

// ======================================================
// Blinkfunktion
// ======================================================
void blinkFlash(int count, int duration) {
  for (int i = 0; i < count; i++) {
    digitalWrite(PIN_FLASH, HIGH);
    delay(duration);
    digitalWrite(PIN_FLASH, LOW);
    delay(duration);
  }
}

// ======================================================
// NVS Initialisierung
// ======================================================
void initNVS() {
  esp_err_t err = nvs_flash_init();
  if (err == ESP_ERR_NVS_NO_FREE_PAGES || err == ESP_ERR_NVS_NEW_VERSION_FOUND) {
    nvs_flash_erase();
    nvs_flash_init();
  }
  nvs_open(NVS_NAMESPACE, NVS_READWRITE, &nvsHandle);
}

// ======================================================
// NVS Werte laden
// ======================================================
void loadNvsValues() {
  uint32_t ticks = 0;
  if (nvs_get_u32(nvsHandle, NVS_KEY_TICKS, &ticks) == ESP_OK) {
    cumulative_ticks = ticks;
    lastNvsTicks     = ticks;
  }

  uint32_t br_int = 0;
  if (nvs_get_u32(nvsHandle, NVS_KEY_BRENNWERT, &br_int) == ESP_OK) {
    H_KWH_PER_M3 = (double)br_int / 1000.0;
  }
}

// ======================================================
// NVS Ticks speichern
// ======================================================
void saveNvsTicksIfNeeded() {
  unsigned long now = millis();
  if ((now - lastNvsSaveTs) < NVS_SAVE_INTERVAL_MS) return;

  noInterrupts();
  uint32_t currentTicks = cumulative_ticks;
  interrupts();

  if (currentTicks >= lastNvsTicks + NVS_TICKS_MIN_DELTA) {
    nvs_set_u32(nvsHandle, NVS_KEY_TICKS, currentTicks);
    nvs_commit(nvsHandle);
    lastNvsTicks   = currentTicks;
    lastNvsSaveTs  = now;
  }
}

// ======================================================
// NVS Brennwert speichern
// ======================================================
void saveNvsBrennwert() {
  uint32_t br_int = (uint32_t)round(H_KWH_PER_M3 * 1000.0);
  nvs_set_u32(nvsHandle, NVS_KEY_BRENNWERT, br_int);
  nvs_commit(nvsHandle);
}

// ======================================================
// Shelly Temperaturen
// ======================================================
ShellyTemps getShellyTemps() {
  ShellyTemps t = {NAN, NAN, NAN};

  HTTPClient http;
  if (!http.begin(SHELLY_STATUS_URL)) return t;

  int code = http.GET();
  if (code == 200) {
    String payload = http.getString();
    StaticJsonDocument<4096> doc;
    if (!deserializeJson(doc, payload)) {
      t.temp_out       = doc["temperature:100"]["tC"].as<float>(); //hier ggf. ID 100 ändern
      t.temp_ruecklauf = doc["temperature:101"]["tC"].as<float>(); //hier ggf. ID 101 ändern
      t.temp_vorlauf   = doc["temperature:102"]["tC"].as<float>(); //hier ggf. ID 102 ändern
    }
  }
  http.end();
  return t;
}

// ======================================================
// Brennwert vom Server holen
// ======================================================
bool fetchBrennwertFromServerWithStatus() {
  if (!ensureWiFi()) return false;

  HTTPClient http;
  if (!http.begin(SERVER_BRENNWERT_URL)) return false;

  int code = http.GET();
  if (code == 200) {
    String payload = http.getString();
    StaticJsonDocument<1024> doc;
    if (!deserializeJson(doc, payload)) {
      if (doc.containsKey("brennwert")) {
        double bw = doc["brennwert"].as<double>();
        if (bw > 5.0 && bw < 20.0) {
          H_KWH_PER_M3 = bw;
          saveNvsBrennwert();
          http.end();
          return true;
        }
      }
    }
  }
  http.end();
  return false;
}

// ======================================================
// WLAN verbinden
// ======================================================
bool ensureWiFi(unsigned long timeoutMs) {
  if (WiFi.status() == WL_CONNECTED) return true;

  WiFi.mode(WIFI_STA);
  WiFi.begin(WIFI_SSID, WIFI_PASSWORD);

  unsigned long t0 = millis();
  while (WiFi.status() != WL_CONNECTED && millis() - t0 < timeoutMs) {
    delay(200);
  }
  return WiFi.status() == WL_CONNECTED;
}

// ======================================================
// Zeit initialisieren
// ======================================================
const char* ntpServer = "pool.ntp.org";
const long gmtOffset_sec = 3600;
const int daylightOffset_sec = 3600;

void initTime() {
  configTime(gmtOffset_sec, daylightOffset_sec, ntpServer);
}

// ======================================================
// Timestamp holen
// ======================================================
String getTimestamp() {
  struct tm timeinfo;
  if (!getLocalTime(&timeinfo)) return "";
  char buf[25];
  strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S", &timeinfo);
  return String(buf);
}

// ======================================================
// Frame berechnen
// ======================================================
Frame computeFrame(uint32_t ticks_total, uint32_t ticks_interval) {
  Frame f;
  f.cumulative_ticks = ticks_total;

  double volume_m3 = ticks_interval * K_TICK_M3;
  f.energy_kwh     = volume_m3 * H_KWH_PER_M3;
  f.avg_power_kw   = f.energy_kwh * 2.0;

  f.temp_out_c       = NAN;
  f.temp_vorlauf_c   = NAN;
  f.temp_ruecklauf_c = NAN;
  f.timestamp        = "";

  return f;
}

// ======================================================
// Upload
// ======================================================
bool uploadFrame(const Frame& f) {
  if (!ensureWiFi()) return false;

  HTTPClient http;
  if (!http.begin(SERVER_UPLOAD_URL)) return false;

  http.addHeader("Content-Type", "application/x-www-form-urlencoded");

  String postData =
    "device=" + String(DEVICE_ID) +
    "&auth=" + String(AUTH_KEY) +
    "&cumulative_ticks=" + String(f.cumulative_ticks) +
    "&energy_kwh=" + String(f.energy_kwh, 6) +
    "&avg_power_kw=" + String(f.avg_power_kw, 6) +
    "&temp_out_c=" + (isnan(f.temp_out_c) ? "" : String(f.temp_out_c, 2)) +
    "&temp_vorlauf_c=" + (isnan(f.temp_vorlauf_c) ? "" : String(f.temp_vorlauf_c, 2)) +
    "&temp_ruecklauf_c=" + (isnan(f.temp_ruecklauf_c) ? "" : String(f.temp_ruecklauf_c, 2)) +
    "&timestamp=" + f.timestamp;

  int code = http.POST(postData);
  http.end();
  return (code >= 200 && code < 300);
}

// ======================================================
// Setup
// ======================================================
void setup() {
  Serial.begin(115200);
  delay(200);

  pinMode(PIN_REED, INPUT_PULLUP);
  pinMode(PIN_FLASH, OUTPUT);
  digitalWrite(PIN_FLASH, LOW);

  initNVS();
  loadNvsValues();

  attachInterrupt(digitalPinToInterrupt(PIN_REED), onReedPulse, FALLING);

  ensureWiFi();
  initTime();

  bool br_ok = fetchBrennwertFromServerWithStatus();

  if (br_ok) {
    blinkFlash(3);
  } else if (H_KWH_PER_M3 >= 9.0 && H_KWH_PER_M3 <= 12.0) {
    blinkFlash(4);
  } else {
    blinkFlash(6);
  }

  lastIntervalTs = millis();
  lastNvsSaveTs  = millis();
}

// ======================================================
// Loop
// ======================================================
void loop() {
  int reedState = digitalRead(PIN_REED);
  digitalWrite(PIN_FLASH, reedState == LOW ? HIGH : LOW);

  saveNvsTicksIfNeeded();

  unsigned long now = millis();
  if (now - lastIntervalTs >= INTERVAL_MS) {
    static uint32_t last_ticks_snapshot = 0;

    noInterrupts();
    uint32_t ticks_snapshot = cumulative_ticks;
    interrupts();

    uint32_t ticks_interval = ticks_snapshot - last_ticks_snapshot;
    last_ticks_snapshot     = ticks_snapshot;

    Frame f = computeFrame(ticks_snapshot, ticks_interval);

    ShellyTemps temps = getShellyTemps();
    f.temp_out_c       = temps.temp_out;
    f.temp_vorlauf_c   = temps.temp_vorlauf;
    f.temp_ruecklauf_c = temps.temp_ruecklauf;

    f.timestamp = getTimestamp();

    uploadFrame(f);

    lastIntervalTs = now;
  }
}
