# Datenstruktor mit Klasse und Methoden ohne Initialwerte
# mit _ für theoretischen Zugriffschutz, Zugriff "muss" über Getter/Setter Methoden erfolgen
# Diese Datei ist immer die aktuelle Version

# 24.10.2025 - v0.0.5 war die letzte alte Version ###
#            - ab v0.0.6 jetzt Umbau auf mehr Python Slang :-)
# 17.01.2026 - ab v0.1.1 alles geändert mit 'messmode' live Ersetzung

#from datetime import datetime
#from enum import Enum, auto
#from dataclasses import dataclass, replace
#import copy
import sys
from typing import Optional
from string import Template    
# from MeineModule import KeithleyDMM6500SCPI
    
class Keithley_SCPI:    
    
    ### private #####################################################################      
    
    # _fm ... function_mode
    def _fm(unit: Optional[str], function: str) -> dict:
        return {'unit': unit, 'function': function}  
    
    _modes: dict[str, dict[Optional[str], str]] = {
        "DCV":      _fm("V",   "VOLTage:DC"),
        "DCI":      _fm("A",   "CURRent:DC"),
        "ACV":      _fm("V",   "VOLTage:AC"),
        "ACI":      _fm("A",   "CURRent:AC"),
        "2W":       _fm("Ω",   "RESistance"),
        "4W":       _fm("Ω",   "FRESistance"),
        "Freq":     _fm("Hz",  "FREQuency"),
        "Period":   _fm("s",   "PERiod"),
        "Diode":    _fm("V",   "DIODe"),
        "Temp":     _fm("°C",  "TEMPerature"),
        "Cap":      _fm("F",   "CAPacitance"),
        "Digi V":   _fm("V",   "VOLTage"),
        "Digi I":   _fm("A",   "CURRent")
    }
    
    #################################################################################    
                   
    # erweiterbare Funktionsliste, mittels Namen wird der hinterlegte SCPI Syntax ausgelesen
    _syntax_table: dict = {
        "getTime":                  ":SYSTem:TIME? 1",         # Reference Manual Seite 12-158
        "setTime":                  ":SYSTem:TIME ",           # Reference Manual Seite 12-158 
        "setFunctionON":            ':SENSe:FUNCtion:ON "${messmode}"', # Reference Manual Seite 12-137  
        "setDigitizeFunctionON":    ':SENSe:DIGItize:FUNCtion:ON "${messmode}"',
        "measure":                  ":MEASure:${messmode}?",   # Reference Manual Seite 12-5  
        "setMeasureCounts":         ":SENSe:COUNt ",           # Reference Manual Seite 12-134
        "setMeasureDigitizeCounts": ":SENSe:DIGItize:COUNt ",  # Reference Manual Seite 12-135     
        "setSampleRateDigitize":    ":SENSe:${messmode}:SRATe ",  # Reference Manual Seite 12-118    
        "read":                     ":READ?",                  # Reference Manual Seite 12-10
        "lastBufferWriteIndex":     ":TRACe:ACTual:END?",      # Reference Manual Seite 12-160        
        "readBuffer":               ":TRACe:DATA?",            # Reference Manual Seite 12-165  
        "averageCount":             ":SENSE:${messmode}:AVERage:COUNt ",
        "averageRepeat":            ":SENSE:${messmode}:AVERage:TCONTrol REPeat",
        "averageMoving":            ":SENSE:${messmode}:AVERage:TCONTrol MOVing",
        "averageHybrid":            ":SENSE:${messmode}:AVERage:TCONTrol HYBRid",
        "averageON":                ":SENSE:${messmode}:AVERage ON",
        "averageOFF":               ":SENSE:${messmode}:AVERage OFF",
        "deleteBuffer":             ":TRACe:CLEar",            # Reference Manual Seite 12-164
        "localMode":                ":TRIGger:CONTinuous RESTart",
        "None":                     "None"
    }

    # Bsp.: print(_getFunctionByName("averageON"))  # → ":SENSE:${messmode}:AVERage ON"
    def _getFunctionByName (self, name: str) -> str:
        item = self._syntax_table.get(name)
        if item is None:
            raise ValueError(f"Unbekannter Funktionsname: {name}.")
        #return item.command
        return self._function[name]

    
    #################################################################################           
      
    # Konstruktor
    def __init__(self, model):
        #self._deviceName:   str = name
        self._model:        str = model
        self._measureMode:  str = None
        self._unit:         str = None
        self._function:     str = None
        self._stateDigitze: bool = False

    ### public ####################################################################   
              
    def __str__(self) -> str:
        return (f"Ich bin das Keithley DMM6500 SCPI Modul")  
              
    def setFunction(self, func: str) -> str:
        try:
            mode = self._modes[func]
            self._unit = mode["unit"]
            self._function = mode["function"]   # in Syntax übersetzte Funktion zurück
            if func in ("Digi V", "Digi I"):
                self._stateDigitze = True
            else:
                self._stateDigitze = False      
            return f'function found → {func}'
        except KeyError:
            self._unit = None
            self._function = None
            print (f'function unknown → {func}')
            sys.exit("Abbruch")
    
    def getFunction (self) -> str: 
        # um diese Aufrufunterscheidung geht es letztlich
        # ':SENSe:FUNCtion:ON "MESSMODE"'
        # ':SENSe:DIGItize:FUNCtion:ON "MESSMODE"'
        if not self._stateDigitze:    
            return self.build_scpi_syntax("setFunctionON")
        else:
            return self.build_scpi_syntax("setDigitizeFunctionON")
                
    def build_scpi_syntax (self, name: str, debug: bool = False) -> str:  # Bsp. "averageON" → ":SENSE:MESSMODE:AVERage ON"
        try:
            syntax = Template(self._syntax_table[name]).substitute(messmode = self._function)
            if debug:
                modified_list = {
                    key: Template(cmd).substitute(messmode = self._function)
                    for key, cmd in self._syntax_table.items()
                }
                max_key_len = max(len(key) for key in modified_list.keys())
                for key, cmd in modified_list.items():
                    print(f"{key:<{max_key_len}} {cmd}")
        
            return syntax
        except KeyError:
            return f'command unknown → {name}'
        
    @property   # Getter - Aufruf ohne ()
    def function(self) -> str:
        return self._function
    
    @property   # Getter - Aufruf ohne ()
    def unit(self) -> str:
        return self._unit

    ### für Debug ####
    def debug_ausgabe_formatiert (self, obj: dict):
        print()
        max_key_len = max(len(key) for key in obj.keys())
        for key, cmd in obj.items():
            print(f"{key:<{max_key_len}} {cmd}")

    def getLastBufferReadIndex (self) -> int:
        return (self._lastBufferReadIndex+1)    # +1 wegen, dass man ab dem nächsten Index weiterlesen kann
    
    def setLastBufferReadIndex (self, count: int):
        self._lastBufferReadIndex = count
          
    def __str__(self) -> str:
        return (f"Ich bin das Keithley DMM6500 SCPI Modul")  
   
    def getDeviceName (self) -> str:
        return (self._model)
    
    def getTime(self) -> str:   # Reference Manual Seite 12-158
        return self.build_scpi_syntax("getTime")                        
          
    def setTime(self) -> str:   # Reference Manual Seite 12-158
        print(self.build_scpi_syntax("setTime"))
        return self.build_scpi_syntax("setTime") 
        
    def getMeasureCounts (self) -> str: 
        # um diese Aufrufunterscheidung geht es letztlich
        # "SENSe:COUNt "
        # "SENSe:DIGItize:COUNt "
        if not self._stateDigitze:    
            return self.build_scpi_syntax("setMeasureCounts")
        else:
            return self.build_scpi_syntax("setMeasureDigitizeCounts")    
    
    def getSampleRate (self) -> str: 
        # um diese Aufrufunterscheidung geht es letztlich
        # "SENSe:COUNt "
        # "SENSe:DIGItize:COUNt "
        if not self._stateDigitze:    
            return self.build_scpi_syntaxe("setMeasureCounts")
        else:
            return self.build_scpi_syntax("setMeasureDigitizeCounts")   
            
    def getMeasureOnce (self) -> str:  
        return self.build_scpi_syntax("measure")

    def getReadOnce(self) -> str:     
        return self.build_scpi_syntax("read")

    def getReadWithTimestamp(self, name: str) -> str:    
        scpiCode = f'{self.build_scpi_syntax("read")} "{name}", TST, READ'
        #print(scpiCode)
        return scpiCode 
        
    def getReadBuffer(self, start, end, name: str) -> str:    
        scpiCode = f'{self.build_scpi_syntax("readBuffer")} {start}, {end} "{name}"'
       #print(scpiCode)
        return scpiCode 
    
    def getReadBufferWithTimestamp(self, start, end, name: str) -> str:    
        scpiCode = f'{self.build_scpi_syntax("readBuffer")} {start}, {end}, "{name}", TST, READ'
        #print(scpiCode)
        return scpiCode 
    
    def getLastBufferWriteIndex(self, name: str) -> str:    
        scpiCode = f'{self.build_scpi_syntax("lastBufferWriteIndex")} "{name}"'
        return scpiCode 
        
    def getAverageCount(self) -> str:   
        return self.build_scpi_syntax("averageCount")

    def getAverageRepeat(self) -> str:     
        return self.build_scpi_syntax("averageRepeat")

    def getAverageMoving(self) -> str:     
        return self.build_scpi_syntax("averageMoving")
    
    def getAverageHybrid(self) -> str:    
        return self.build_scpi_syntax("averageHybrid")
        
    def getAverageOn(self) -> str:     
        return self.build_scpi_syntax("averageON")

    def getAverageOff(self) -> str:       
        #return self._getFunctionByName("averageOFF")   
        return self.build_scpi_syntax("averageOFF")  
    
    def getDeleteBuffer (self, name: str) -> str:
        scpiCode = f'{self.build_scpi_syntax("deleteBuffer")} "{name}"'
        return scpiCode 
    
    def getLocalMode (self) -> str:
        return self.build_scpi_syntax("localMode")

            
### für schnelle Tests im Modul ###############################

if __name__ == '__main__':
    k = Keithley_SCPI("Keithley DMM6500")
                
    print(k.setFunction("2W"))
    print(k.function)                                   # Debug
    print(k.unit)                                       # Debug
    print(k.build_scpi_syntax("measure"))               # Debug
    
    print(k.setFunction("Temp"))
    print(k.function)                                   # Debug
    print(k.unit)                                       # Debug
    #print(k.build_scpi_syntax("averageMoving", True))   # Debug
    print(k.getAverageOff())   # Debug
    #print(k.getTime())   # Debug


