# --- Anpassen des Skripts: ---
# Passe das Skript an dein Messgerät an.
# Ersetze die Ressource 'USB0::0x1AB1::0x0640::DM3O16290027::INSTR'
# durch die tatsächliche Ressource deines Geräts, die du durch Aufrufen von rm.list_resources() erhälst.
#
# --- Information für Datenimport in Excel. ---
# Daten > Daten abrufen > Aus Datei > Aus Text/CSV importieren,
# '65001:Unicode UTF-8' kontrollieren/auswählen,
# Trennzeichen Tabstopp auswählen,
# Daten transformieren auswählen (nicht sofort laden),
# im sich öffnenden Power Query Editor die Datumsspalte auswählen,
# Datentyp 'Datum' statt 'Ganze Zahl' auswählen und 'Aktuelle ersetzen'
# anschließend 'schließen & laden'
# zu guter Letzt Spalte vom Zeitstempel auswählen
# und das Zellformat Benutzerdefiniert auf hh:mm:ss,000 ändern,
# mehr als 3 Nachkommastellen funktioniert nicht,
# die Messwertspalte je nach Bedarf einheitlich auf 2 oder 3 Kommastellen

# 18.01.2026 - v0.1.2.0
# angepasst auf neue Modul Version v0.1.1 funktioniert mit Main v0.1.2.0

import pyvisa   # (zusätzlich installiert)
from typing import Optional
from datetime import datetime, time, timedelta
import time
import os
from pathlib import Path
import platform, socket, getpass, psutil # (zusätzlich installiert)
import socket

from Modules import KeithleyDMM6500SCPI

### User Parameter #############################################################

OUTPUT_FORMAT: str = "FLOAT"    # Alternativ "EXP" für Exponent
SAVE_DATA: bool = True          # Option 'False/True' für (nicht) speichern in Datei
DEBUG = False   # internes Debug        
    
# Datei Pfad und Namen festlegen
ordner = Path.home() / "Documents"
dateiName = "KeithleyMessDaten.txt"
    
# Initialisieren Sie den Resource Manager
rm = pyvisa.ResourceManager()
keithleySCPI = KeithleyDMM6500SCPI.Keithley_SCPI("Keithley DMM6500")
keithleyUSB  = rm.open_resource('USB0::0x05E6::0x6500::04461469::INSTR')

### Ende User Parameter #############################################################

# Alle verfügbaren Ressourcen auflisten
resources = rm.list_resources()
print("Gefundene Geräte:", resources)
    
# eine Struktur in Python mit einer Klasse
class Device:
    def __init__(self, scpi, line):
        self.scpi = scpi
        self.line = line

# eine Objekt anlegen:
keithley = Device(keithleySCPI, keithleyUSB)

# Setzen Sie den Timeout auf 60 Sekunden
keithley.scpi.timeout = 60000

### Funktionen für Zeit und Datum ##############################################
    
class DateAndTime:
    def __init__(self):
        self.tagName: int = 0   # Wochentag (0=Montag, 6=Sonntag)
        self.stunde:  int = 0
        self.minute:  int = 0
        self.sekunde: int = 0
        self.tag:     int = 0
        self.monat:   int = 0
        self.jahr:    int = 0

systemZeit = DateAndTime()
keithleyZeit = DateAndTime()

def getSystemTime(sysTime, mode: str = "SUFFIX"):
    sysTime.tagName = datetime.now().strftime("%A") # Wochentag
    sysTime.stunde  = datetime.now().strftime("%H") # Stunde (24h)
    sysTime.minute  = datetime.now().strftime("%M") # Minute
    sysTime.sekunde = datetime.now().strftime("%S") # Sekunde
    sysTime.tag     = datetime.now().strftime("%d") # Wochentag Ziffer
    sysTime.monat   = datetime.now().strftime("%m") # Monat
    sysTime.jahr    = datetime.now().strftime("%Y") # Jahreszahl vierstellig
    zeitstempel: str = None
    # aktuelles Datum und Uhrzeit zusammenbauen
    if ("SUFFIX" == mode):
        zeitstempel = datetime.now().strftime("%Y%m%d_%H%M%S")
    if ("DATE" == mode):
        zeitstempel = datetime.now().strftime("%d.%m.%Y %H:%M:%S")
    #print(zeitstempel)    
    return zeitstempel

def getDeviceTime(device, timeObj):
    scpiCode = device.scpi.getTime()
    zeitstempel = device.line.query(scpiCode)
    #print(zeitstempel)
    # Alle mehrfachen Leerzeichen durch ein Einzelnes ersetzen
    cleaned = ' '.join(zeitstempel.split())
    dt = datetime.strptime(cleaned, "%a %b %d %H:%M:%S %Y")
    zeitstempel = dt.strftime("%d.%m.%Y %H:%M:%S")
    #print(zeitstempel)
    # Datumselemente dem DateAndTime Objekt 'tObj' zuordnen    
    timeObj.tagName = dt.strftime("%A") # Wochentag
    timeObj.stunde  = dt.strftime("%H") # Stunde (24h)
    timeObj.minute  = dt.strftime("%M") # Minute
    timeObj.sekunde = dt.strftime("%S") # Sekunde
    timeObj.tag     = dt.strftime("%d") # Wochentag Ziffer
    timeObj.monat   = dt.strftime("%m") # Monat
    timeObj.jahr    = dt.strftime("%Y") # Jahreszahl vierstellig
    return zeitstempel

def buildDateTime (device, t, mode: str = "GET"):
    if ("SET" == mode):
        dateFormat = str(t.jahr) + ", " + str(t.monat) + ", " + str(t.tag) + ", " + str(t.stunde) + ", " + str(t.minute) + ", " + str(t.sekunde)
        scpiCode = device.scpi.setTime() + dateFormat
    if ("GET" == mode):   
        scpiCode = device.scpi.getTime()
    #print(scpiCode)
    return scpiCode

def checkDateAndTime (system, device):
    state: bool = False
    for attr, systemValue in vars(system).items():
        keithleyValue = getattr(device, attr)
        if systemValue != keithleyValue:
            state = True
            #print(f"Abweichung bei '{attr}': Systemzeit = {systemValue}, {keithley.scpi.getDeviceName()} = {keithleyValue}")
    # Bei Abweichung von nur 1s nicht korrigieren, Fehler kann vom Sekundenwechsel während Abfrage stammen   
    diff:int = abs(int(system.sekunde) - int(device.sekunde))
    if 2 > diff:
        state = False            
    if state:
        print(f"Es gab eine Abweichung von {diff} Sekunde(n).")
        print(f"{getSystemTime(systemZeit, "DATE")} {platform.system()} {platform.release()} {socket.gethostname()}")  
        print(f"{getDeviceTime(keithley, keithleyZeit)} {keithley.scpi.getDeviceName()}")
        print(f"{keithley.scpi.getDeviceName()} wird gestellt.")
        keithley.line.write(buildDateTime(keithley, systemZeit, "SET"))
    print(f"{getSystemTime(systemZeit, "DATE")} {platform.system()} {platform.release()} {socket.gethostname()}")  
    print(f"{getDeviceTime(keithley, keithleyZeit)} {keithley.scpi.getDeviceName()}")

def modifiziereDateiName(datei):
    dateiname, dateiendung = os.path.splitext(datei)
    neuerDateiname = f"{dateiname}_{getSystemTime(systemZeit, "SUFFIX")}{dateiendung}"
    if DEBUG:
        print(f"alter Dateiname: {dateiname}{dateiendung}")   
        print("neuer Dateiname:", neuerDateiname)        
    return neuerDateiname  

dateipfad: str = ordner / (modifiziereDateiName(dateiName))
print(dateipfad) 

### Funktionen zum Messen ####################################################### 
           
def setFunction(device, function: str):
    device.scpi.setFunction(function)
    scpiCode: str = device.scpi.getFunction()
    unitSymbol = (f"Unit: {device.scpi.unit}") 
    if SAVE_DATA:
        with open(dateipfad, 'a', encoding="utf-8") as file:
            file.write(f"{scpiCode}\n")
            file.write(f"{unitSymbol}\n")
    time.sleep(1)   # nach einem Funktionswechsel ist eine Zwangspause erforderlich  
    print(f"new function set {scpiCode}")
    print(unitSymbol)
    device.line.write(scpiCode) 

def outputFormatting(valueRaw: str):      
    # valueRaw ist ein String, der enthaltende Zeilenumbruch '\n' wird mit strip() entfernt    
    valueStr: str = valueRaw.strip()
    if "FLOAT" == OUTPUT_FORMAT:
        valueStr = "{:.3f}".format(float(valueStr))
        valueStr = str(valueStr.replace('.', ',')) # Komma statt Punkt
    return valueStr 

def convertDateFormat(dateTimeStr):
    dateStr, timeStr = dateTimeStr.split(' ')
    month, day, year = dateStr.split('/')
    # Spezialbehandlung für Excelimport, Millisekunden auch 3 Stellen kürzen und runden    
    # Zerlegen in Zeit und Nanosekunden
    t_part, ns_part = timeStr.split(".")
    ns_part = (ns_part + "0" * 9)[:9]           # auffüllen auf 9 Stellen
    nanoseconds = int(ns_part)
    # Sekundenanteil + Nanosekunden mit 'timedelta()' berechnen
    base_time = datetime.strptime(t_part, "%H:%M:%S")
    dtFull = base_time + timedelta(microseconds = nanoseconds / 1000)
    # Runden auf Millisekunden
    micro = dtFull.microsecond
    roundedMicro = round(micro / 1000) * 1000   # Millisekunde runden
    # Prüfen auf Sekundenüberlauf
    if roundedMicro == 1000000:
        dtFull = dtFull.replace(microsecond = 0) + timedelta(seconds = 1)
    else:
        dtFull = dtFull.replace(microsecond = roundedMicro)
    # Zeitausgabe formatieren
    timeStr = dtFull.strftime("%H:%M:%S,%f")[:-3]   # auf 3 Nachkommastellen kürzen
    return f"{day}.{month}.{year}\t{timeStr}"

def measureOnce(device, log: str = "yes"):  
    if "yes" == log:
        comment: str = "measure Once"
        print(comment)
        # mit erkennt Excel-Import Datenformat nicht 
        if SAVE_DATA:
                with open(dateipfad, 'a', encoding="utf-8") as file:
                    file.write(f"{comment}\n")  
    scpiCode: str = device.scpi.getMeasureOnce()
    # Messung auslösen, nur Messwert wird hier noch nicht benötigt, liegt im Buffer
    rowData: str = device.line.query(scpiCode) 

def measureSampling(device, count: int = 5, delay: int = 0):  
    comment: str = (f"{count} Sampling(s) ...")  
    print(comment)
    if SAVE_DATA:
            with open(dateipfad, 'a', encoding="utf-8") as file:
                file.write(f"{comment}\n")  
    for i in range(count):
        measureOnce(device, "noComment")
        time.sleep(delay)    
     
def setMeasureCounts(device, count: int = 3):
    scpiCode = device.scpi.getMeasureCounts()  
    scpiCode = (f"{scpiCode}{count}") 
    print(scpiCode) 
    device.line.write(scpiCode)
    if SAVE_DATA:
        with open(dateipfad, 'a', encoding="utf-8") as file:
            file.write(f"{scpiCode}\n")  
                     
def setAverageMode(device, type: str = "Repeat", count: int = 10):
    comment: str = (f"{type} Average von je {count} Messungen")  
    print(comment)
    # senden aus wievielen Messungen soll der Average ermittelt werden
    device.line.write(device.scpi.getAverageCount() + str(count))
    scpiAverageType = ""
    match type:
        case "Repeat":
            scpiAverageType = device.scpi.getAverageRepeat()
        case "Moving":
            scpiAverageType = device.scpi.getAverageMoving()
        case "Hybrid":
            scpiAverageType = device.scpi.getAverageHybrid()
        case _:
            scpiAverageType = "unknown"
    if scpiAverageType != "unknown":   
        device.line.write(scpiAverageType)              # Average Type senden
        device.line.write(device.scpi.getAverageOn())   # Average Mode einschalten
        scpiCode = device.scpi.getAverageOn()
        print(scpiCode)
        if SAVE_DATA:
            with open(dateipfad, 'a', encoding="utf-8") as file:
                file.write(f"{comment}\n")  
                file.write(f"{scpiCode}\n") # mit erkennt Excel-Import Datenformat nicht 
    else:
        print("Unbekannter Average Modus.")

def clearAverageMode(device):
    device.line.write(device.scpi.getAverageOff())  # Average Mode ausschalten
    scpiCode = device.scpi.getAverageOff()
    print(scpiCode) 
    if SAVE_DATA:
        with open(dateipfad, 'a', encoding="utf-8") as file:
            file.write(f"{scpiCode}\n")
                  
def getlastBufferIndex(device, name: str = "defbuffer1"):
    scpiCode: str = device.scpi.getLastBufferWriteIndex(name)
    endIndex: str = device.line.query(scpiCode)
    print(f'last Index {endIndex}', end="")
    return int(endIndex)
       
def saveBufferWithTimestamp(device, name: str = "defbuffer1"):
    print(f'Speichere Buffer "{name}" mit Zeitstempel in Logdatei.') 
    # letzter gelesener Buffer Index (+1) für nächste Auslesung
    startIndex = device.scpi.getLastBufferReadIndex()
    endIndex = getlastBufferIndex(device, name)
    if 1 > endIndex:
        print("Buffer ist leer.")
        return  # Funktion wird hier beendet
    if startIndex > endIndex:
        print("Index verhaspelt.")
        return  # Funktion wird hier beendet
    scpiCode = device.scpi.getReadBufferWithTimestamp(startIndex, endIndex, name)            
    print(scpiCode)
    rawData = device.line.query(scpiCode) 
    device.scpi.setLastBufferReadIndex(endIndex) # aktuellen Index sichern
    # In einzelne Werte aufsplitten
    items = rawData.split(',')
    # In Zweiergruppen (Datum+Zeit, Messwert)
    rows = [items[i:i+2] for i in range(0, len(items), 2)]
    # Datei speichern
    if SAVE_DATA:
        with open(dateipfad, 'a', encoding="utf-8") as file:
            file.write(f"{scpiCode}\n")
            for dt, valueStr in rows:
                dtConverted = convertDateFormat(dt)
                formStr = outputFormatting(valueStr)
                dataFormat = (f'{startIndex}\t{dtConverted}\t{formStr}')
                file.write(f'\t{dataFormat}\n')
                print(f'{dataFormat}')
                startIndex += 1
                    
def clearBuffer(device, name: str = "defbuffer1"):
    device.scpi.setLastBufferReadIndex(0)   # mitgeführten Index zurücksetzen
    scpiCode: str = device.scpi.getDeleteBuffer(name)
    device.line.write(scpiCode) 
    if SAVE_DATA:
        with open(dateipfad, 'a', encoding="utf-8") as file:
            file.write(f"{scpiCode}\n")
    print (scpiCode)
         
def changeToLokalMode (device):
    try:
        print("Script beendet.")
        print("Sende *OPC?")
        scpiCode = keithley.line.query("*OPC?")
        print("OPC Antwort:", scpiCode)
    except Exception as e:
        print("Fehler bei *OPC?:", e)
    print("Das Ende ist unaufhaltsam ...")
    # wechselt in lokalen Modus und misst kontinuierlich weiter
    scpiCode = device.scpi.getLocalMode()
    device.line.write(scpiCode)
    #print(scpiCode)
    # Kurze Pause zur Stabilisierung (z. B. 1 ms)
    time.sleep(0.1)
    print("Local Mode aktiv.")
    keithley.line.close()
    print("Verbindung geschlossen.") 
    
################################################################################  
        
# Logdatei anlegen  
if SAVE_DATA:
    file = open(dateipfad, 'a', encoding="utf-8")
    with open(dateipfad, 'a', encoding="utf-8") as file:
        # Dummyeintrag Spaltenbeschriftung mit Tabstopp für Excel Import
        file.write(f'Funktion\tIndex\tDatum\tZeitstempel\tMesswert\n')
        # notwendige Dummyeinträge damit Excel die Datenformatierug beim Import erkennt
        for i in range (10):
            file.write(f'Dummy\t0\t01.01.2000\t00:00:00,123\t12345,678\n')
 
# Automatische Keithley Zeitkorrekur wenn Differenz größer 1 Sekunde
getSystemTime(systemZeit, "DATE")           # notwendig
getDeviceTime(keithley, keithleyZeit)       # notwendig
checkDateAndTime(systemZeit, keithleyZeit)  # notwendig   
   
clearBuffer(keithley)                   # Messwertspeicher löschen
setFunction(keithley, "2W")             # Einstellung ändern

measureSampling(keithley, 5, 0)         # 5 Einzelmesssungen mit 0sec Pause dazwischen
saveBufferWithTimestamp(keithley)
   
setAverageMode(keithley, "Repeat", 5)   # Average Filter Faktor 5
measureSampling(keithley, 5, 0)         # 5 Einzelmesssungen mit 0sec Pause dazwischen
clearAverageMode(keithley)
saveBufferWithTimestamp(keithley)

setMeasureCounts(keithley)
measureSampling(keithley, 5, 0)         # 5 Einzelmesssungen mit 0sec Pause dazwischen
saveBufferWithTimestamp(keithley)

setFunction(keithley, "Temp")           # Einstellung ändern
measureSampling(keithley, 5, 0)         # 5 Einzelmesssungen mit 0sec Pause dazwischen
saveBufferWithTimestamp(keithley)

setFunction(keithley, "Digi I")         # Einstellung ändern
setMeasureCounts(keithley)
measureSampling(keithley, 5, 0)         # 5 Einzelmesssungen mit 0sec Pause dazwischen
saveBufferWithTimestamp(keithley)

setFunction(keithley, "Digi V")         # Einstellung ändern
setMeasureCounts(keithley)
measureSampling(keithley, 5, 0)         # 5 Einzelmesssungen mit 0sec Pause dazwischen
saveBufferWithTimestamp(keithley)

# stehen geblieben bei SampleRate

### Ende Maßnahmen ###########################################################

clearAverageMode(keithley)  # ggf. falls man es vergisst    
#clearBuffer(keithley)      # ggf. denke das ist hier sinnvoll

# Egal was passiert, auf jeden Fall Datei schließen    
if SAVE_DATA:    
    file.close()
    print("File closed.")

changeToLokalMode(keithley)
          
