'''
Raumklima

Create a webpage access point to show sensor data

Hardware:
ESP32 (sensor chip temperature)
DS18B20 temperature sensor
Sensirion SCD41 CO2 sensor

MicroPython v1.26.1 on 2025-09-11; Generic ESP32 module with ESP32

External libraries:
https://microdot.readthedocs.io/en/latest/intro.html
https://github.com/peter-l5/MicroPython_SCD4X

2026-01-20 mchris

'''

import network
import uasyncio as asyncio
from microdot import Microdot
import esp32
import machine
import onewire
import ds18x20
import scd4x
import gc
import time

gc.collect()

async def ap_mode(ssid, password):
    print("Free heap before init:", gc.mem_free())
    
    # DS18B20 on GPIO 4
    dat = machine.Pin(4)
    ds = ds18x20.DS18X20(onewire.OneWire(dat))
    ds_sensors = ds.scan()
    print('DS18B20 sensors:', ds_sensors)
    gc.collect()
    
    # SCD41 I2C
    i2c = machine.I2C(1, scl=machine.Pin(22), sda=machine.Pin(21), freq=100000)
    scd = scd4x.SCD4X(i2c)
    print('SCD41 init...')
    scd.start_periodic_measurement()
    await asyncio.sleep(5)
    print('SCD41 ready')
    gc.collect()
    
    print("Free heap before WiFi:", gc.mem_free())
    
    # WiFi AP
    ap = network.WLAN(network.AP_IF)
    ap.active(True)
    ap.config(essid=ssid, password=password)
    
    while not ap.active():
        await asyncio.sleep(0.1)
    print('AP IP:', ap.ifconfig()[0])
    print("Free heap after WiFi:", gc.mem_free())
    
    app = Microdot()
    app.ds_sensors = ds_sensors
    app.ds_device = ds
    app.scd_device = scd

    @app.route('/')
    async def index(request):
        html = '''<!DOCTYPE html>
<html><head><title>Raumklima</title>
<meta name="viewport" content="width=device-width,initial-scale=1">
<style>body{font-family:Arial;text-align:center;padding:20px;background:#f0f8ff;}
h1{color:#2c3e50;font-size:28px;}
.reading { font-size: 64px; color: #007bff; margin: 10px 0; }
.time-format { font-size: 38px; color: #7b7bff; margin: 10px 0; }
.co2 { font-size: 64px; color: #28a745; font-weight: bold; margin: 15px 0; }
.temp { font-size: 36px; color: #ff6b35; font-weight: bold; margin: 15px 0; }
<h1>Raumklima</h1>
<div class="reading temp">ESP32: <span id="esp32">-- C</span></div>
<div class="reading temp">DS18B20: <span id="ds18">-- C</span></div>
<div class="reading temp">SCD41 T: <span id="scd_t">-- C</span></div>
<div class="co2">CO2: <span id="co2">-- ppm</span></div>
<div class="reading">RH: <span id="rh">-- %</span></div>
<div class="time-format">Time: <span id="time">-- %</span></div>

<script>
function update(){
fetch('/data').then(r=>r.json()).then(d=>{
document.getElementById('esp32').textContent=d.internal+' C';
document.getElementById('ds18').textContent=d.ds18b20+' C';
document.getElementById('scd_t').textContent=d.scd_temp+' C';
document.getElementById('co2').textContent=d.scd_co2+' ppm';
document.getElementById('rh').textContent=d.scd_rh+' %';
document.getElementById('time').textContent=d.time;
}).catch(e=>console.log('error',e));
}
setInterval(update,5500);update();
</script></body></html>'''
        return html, 200, {'Content-Type': 'text/html'}

    @app.route('/data')
    async def get_data(request):
        # Internal temp
        temp_raw = esp32.raw_temperature()
        internal = round((temp_raw - 32) * 5/9, 1)
        
        # DS18B20
        ds18b20 = -99
        if app.ds_sensors:
            try:
                app.ds_device.convert_temp()
                await asyncio.sleep_ms(750)
                ds18b20 = round(app.ds_device.read_temp(app.ds_sensors[0]), 1)
            except:
                pass
        
        # SCD41
        scd_temp = scd_co2 = scd_rh = -99
        try:
            if app.scd_device.data_ready:
                scd_temp = round(app.scd_device.temperature, 1)
                scd_co2 = int(app.scd_device.CO2)
                scd_rh = round(app.scd_device.relative_humidity, 1)
        except:
            pass
        
        gc.collect()
        t = time.localtime()
        timestamp = "{:02d}:{:02d}:{:02d}".format(t[3], t[4], t[5])
        return {
            'internal': internal,
            'ds18b20': ds18b20,
            'scd_temp': scd_temp,
            'scd_co2': scd_co2,
            'scd_rh': scd_rh,
            'time': timestamp
        }, 200, {'Content-Type': 'application/json'}

    print("Starting server... Free heap:", gc.mem_free())
    await app.start_server(port=80)

gc.collect()
asyncio.run(ap_mode('pipico', '12345678'))
