/**
 * @file lin_sniffer.c
 * @brief Passive LIN Bus Sniffer implementation
 */

#include "lin_sniffer.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "freertos/queue.h"
#include "esp_log.h"
#include "esp_timer.h"
#include "driver/uart.h"
#include <string.h>

static const char *TAG = "LIN_SNIFFER";

#define UART_BUF_SIZE           1024
#define FRAME_QUEUE_SIZE        32
#define SYNC_TIMEOUT_MS         100
#define DATA_TIMEOUT_MS         50

// Global state
static struct {
    bool initialized;
    bool running;
    uart_port_t uart_num;
    TaskHandle_t sniffer_task_handle;
    QueueHandle_t frame_queue;
    lin_frame_callback_t callback;
    void *callback_user_data;
    lin_sniffer_stats_t stats;
    bool verbose;
} sniffer = {0};

// Forward declarations
static void sniffer_task(void *arg);
static bool detect_break(uart_port_t uart_num, uint16_t *duration_us);
static bool read_sync_byte(uart_port_t uart_num);
static bool read_pid(uart_port_t uart_num, uint8_t *pid);
static bool read_data_and_checksum(uart_port_t uart_num, lin_sniffed_frame_t *frame);

esp_err_t lin_sniffer_init(const lin_sniffer_config_t *config)
{
    if (config == NULL) {
        return ESP_ERR_INVALID_ARG;
    }

    if (sniffer.initialized) {
        ESP_LOGW(TAG, "Sniffer already initialized");
        return ESP_ERR_INVALID_STATE;
    }

    // Configure UART for passive listening
    uart_config_t uart_config = {
        .baud_rate = config->baudrate,
        .data_bits = UART_DATA_8_BITS,
        .parity = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_DISABLE,
        .source_clk = UART_SCLK_DEFAULT,
    };

    esp_err_t ret = uart_param_config(config->uart_num, &uart_config);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "UART param config failed: %s", esp_err_to_name(ret));
        return ret;
    }

    // Set RX-only pins (no TX pin for passive sniffing)
    ret = uart_set_pin(config->uart_num, UART_PIN_NO_CHANGE, config->rx_pin, 
                       UART_PIN_NO_CHANGE, UART_PIN_NO_CHANGE);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "UART set pin failed: %s", esp_err_to_name(ret));
        return ret;
    }

    // Install UART driver
    ret = uart_driver_install(config->uart_num, UART_BUF_SIZE * 2, UART_BUF_SIZE * 2, 
                              0, NULL, 0);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "UART driver install failed: %s", esp_err_to_name(ret));
        return ret;
    }

    // Create frame queue
    sniffer.frame_queue = xQueueCreate(FRAME_QUEUE_SIZE, sizeof(lin_sniffed_frame_t));
    if (sniffer.frame_queue == NULL) {
        uart_driver_delete(config->uart_num);
        return ESP_ERR_NO_MEM;
    }

    sniffer.uart_num = config->uart_num;
    sniffer.verbose = config->verbose_logging;
    sniffer.initialized = true;
    memset(&sniffer.stats, 0, sizeof(lin_sniffer_stats_t));

    ESP_LOGI(TAG, "LIN sniffer initialized on UART%d (RX:%d) at %lu baud", 
             config->uart_num, config->rx_pin, config->baudrate);

    return ESP_OK;
}

esp_err_t lin_sniffer_deinit(void)
{
    if (!sniffer.initialized) {
        return ESP_ERR_INVALID_STATE;
    }

    if (sniffer.running) {
        lin_sniffer_stop();
    }

    uart_driver_delete(sniffer.uart_num);
    
    if (sniffer.frame_queue) {
        vQueueDelete(sniffer.frame_queue);
    }

    memset(&sniffer, 0, sizeof(sniffer));
    
    ESP_LOGI(TAG, "LIN sniffer deinitialized");
    return ESP_OK;
}

esp_err_t lin_sniffer_register_callback(lin_frame_callback_t callback, void *user_data)
{
    if (!sniffer.initialized) {
        return ESP_ERR_INVALID_STATE;
    }

    sniffer.callback = callback;
    sniffer.callback_user_data = user_data;
    
    return ESP_OK;
}

esp_err_t lin_sniffer_start(void)
{
    if (!sniffer.initialized) {
        return ESP_ERR_INVALID_STATE;
    }

    if (sniffer.running) {
        ESP_LOGW(TAG, "Sniffer already running");
        return ESP_ERR_INVALID_STATE;
    }

    // Clear UART buffers
    uart_flush(sniffer.uart_num);

    // Create sniffer task
    BaseType_t result = xTaskCreate(sniffer_task, "lin_sniffer", 4096, NULL, 
                                    10, &sniffer.sniffer_task_handle);
    
    if (result != pdPASS) {
        ESP_LOGE(TAG, "Failed to create sniffer task");
        return ESP_ERR_NO_MEM;
    }

    sniffer.running = true;
    ESP_LOGI(TAG, "LIN sniffer started");
    
    return ESP_OK;
}

esp_err_t lin_sniffer_stop(void)
{
    if (!sniffer.running) {
        return ESP_ERR_INVALID_STATE;
    }

    sniffer.running = false;
    
    if (sniffer.sniffer_task_handle) {
        vTaskDelete(sniffer.sniffer_task_handle);
        sniffer.sniffer_task_handle = NULL;
    }

    ESP_LOGI(TAG, "LIN sniffer stopped");
    
    return ESP_OK;
}

esp_err_t lin_sniffer_get_stats(lin_sniffer_stats_t *stats)
{
    if (stats == NULL) {
        return ESP_ERR_INVALID_ARG;
    }

    memcpy(stats, &sniffer.stats, sizeof(lin_sniffer_stats_t));
    return ESP_OK;
}

esp_err_t lin_sniffer_reset_stats(void)
{
    memset(&sniffer.stats, 0, sizeof(lin_sniffer_stats_t));
    return ESP_OK;
}

bool lin_sniffer_validate_pid(uint8_t pid)
{
    uint8_t id = pid & 0x3F;
    
    // Calculate expected parity bits
    uint8_t p0 = ((id >> 0) & 1) ^ ((id >> 1) & 1) ^ ((id >> 2) & 1) ^ ((id >> 4) & 1);
    uint8_t p1 = ~(((id >> 1) & 1) ^ ((id >> 3) & 1) ^ ((id >> 4) & 1) ^ ((id >> 5) & 1)) & 1;
    
    uint8_t expected_pid = id | (p0 << 6) | (p1 << 7);
    
    return (pid == expected_pid);
}

uint8_t lin_sniffer_get_id_from_pid(uint8_t pid)
{
    return pid & 0x3F;
}

bool lin_sniffer_is_ibs_frame(uint8_t id)
{
    // Common IBS frame IDs (based on Hella IBS sensors)
    return (id >= 0x20 && id <= 0x2F);
}

const char* lin_sniffer_get_ibs_frame_description(uint8_t id)
{
    switch (id) {
        case 0x22: return "Current measurement";
        case 0x23: return "Voltage and temperature";
        case 0x24: return "SOC and capacity";
        case 0x25: return "SOH and error status";
        case 0x26: return "Calibration data";
        case 0x27: return "Serial number and version";
        case 0x28: return "Extended data 1";
        case 0x29: return "Extended data 2";
        default:
            if (id >= 0x20 && id <= 0x2F) {
                return "Unknown IBS frame";
            }
            return "Non-IBS frame";
    }
}

void lin_sniffer_print_frame(const lin_sniffed_frame_t *frame)
{
    const char *type_str[] = {"UNKNOWN", "HEADER", "MASTER_TX", "SLAVE_TX", "INVALID"};
    const char *dir_str[] = {"M->S", "S->M", "???"};
    
    ESP_LOGI(TAG, "┌─────────────────────────────────────────────────────────");
    ESP_LOGI(TAG, "│ Time: %llu us (%.3f ms)", 
             frame->timestamp_us, frame->timestamp_us / 1000.0);
    ESP_LOGI(TAG, "│ Type: %s, Dir: %s", 
             type_str[frame->type], dir_str[frame->direction]);
    ESP_LOGI(TAG, "│ ID: 0x%02X, PID: 0x%02X %s", 
             frame->id, frame->pid, frame->pid_valid ? "✓" : "✗");
    
    if (lin_sniffer_is_ibs_frame(frame->id)) {
        ESP_LOGI(TAG, "│ IBS: %s", lin_sniffer_get_ibs_frame_description(frame->id));
    }
    
    if (frame->len > 0) {
        ESP_LOG_BUFFER_HEX_LEVEL(TAG, frame->data, frame->len, ESP_LOG_INFO);
        ESP_LOGI(TAG, "│ Checksum: 0x%02X (calc: 0x%02X) %s", 
                 frame->checksum, frame->calculated_checksum,
                 frame->checksum_valid ? "✓" : "✗");
    }
    
    ESP_LOGI(TAG, "└─────────────────────────────────────────────────────────");
}

void lin_sniffer_print_frame_hex(const lin_sniffed_frame_t *frame)
{
    printf("%llu,0x%02X,0x%02X,", frame->timestamp_us, frame->id, frame->pid);
    
    for (int i = 0; i < frame->len; i++) {
        printf("%02X", frame->data[i]);
        if (i < frame->len - 1) printf(" ");
    }
    
    printf(",0x%02X,%s,%s\n", 
           frame->checksum,
           frame->checksum_valid ? "OK" : "ERR",
           frame->pid_valid ? "OK" : "ERR");
}

int lin_sniffer_export_csv(const lin_sniffed_frame_t *frame, char *buffer, size_t buffer_size)
{
    int pos = snprintf(buffer, buffer_size, 
                      "%llu,0x%02X,0x%02X,", 
                      frame->timestamp_us, frame->id, frame->pid);
    
    for (int i = 0; i < frame->len && pos < buffer_size - 10; i++) {
        pos += snprintf(buffer + pos, buffer_size - pos, "%02X ", frame->data[i]);
    }
    
    pos += snprintf(buffer + pos, buffer_size - pos, 
                   ",0x%02X,%d,%d\n",
                   frame->checksum, 
                   frame->checksum_valid ? 1 : 0,
                   frame->pid_valid ? 1 : 0);
    
    return pos;
}

// Main sniffer task
static void sniffer_task(void *arg)
{
    lin_sniffed_frame_t frame;
    uint8_t pid;
    
    ESP_LOGI(TAG, "Sniffer task started, waiting for LIN traffic...");
    
    while (sniffer.running) {
        // Wait for break field (indicated by frame error or extended low period)
        uint16_t break_duration;
        if (!detect_break(sniffer.uart_num, &break_duration)) {
            vTaskDelay(pdMS_TO_TICKS(10));
            continue;
        }
        
        memset(&frame, 0, sizeof(frame));
        frame.timestamp_us = esp_timer_get_time();
        frame.break_duration_us = break_duration;
        
        // Update stats
        if (sniffer.stats.total_frames == 0) {
            sniffer.stats.first_frame_time_us = frame.timestamp_us;
        }
        sniffer.stats.last_frame_time_us = frame.timestamp_us;
        sniffer.stats.total_frames++;
        
        // Read SYNC byte
        if (!read_sync_byte(sniffer.uart_num)) {
            sniffer.stats.sync_errors++;
            if (sniffer.verbose) {
                ESP_LOGW(TAG, "SYNC error at %llu", frame.timestamp_us);
            }
            continue;
        }
        
        // Read PID
        if (!read_pid(sniffer.uart_num, &pid)) {
            sniffer.stats.timeout_errors++;
            continue;
        }
        
        frame.pid = pid;
        frame.id = lin_sniffer_get_id_from_pid(pid);
        frame.pid_valid = lin_sniffer_validate_pid(pid);
        
        if (!frame.pid_valid) {
            sniffer.stats.pid_errors++;
        }
        
        // Try to read data and checksum
        if (read_data_and_checksum(sniffer.uart_num, &frame)) {
            frame.type = LIN_FRAME_TYPE_SLAVE_TX;
            frame.direction = LIN_DIR_SLAVE_TO_MASTER;
            
            if (frame.checksum_valid) {
                sniffer.stats.valid_frames++;
            } else {
                sniffer.stats.checksum_errors++;
            }
        } else {
            frame.type = LIN_FRAME_TYPE_HEADER;
        }
        
        // Send to callback if registered
        if (sniffer.callback) {
            sniffer.callback(&frame, sniffer.callback_user_data);
        }
        
        // Send to queue for processing
        xQueueSend(sniffer.frame_queue, &frame, 0);
    }
    
    vTaskDelete(NULL);
}

static bool detect_break(uart_port_t uart_num, uint16_t *duration_us)
{
    // Simple approach: wait for any data with timeout
    // A proper implementation would measure the actual break duration
    size_t available;
    uart_get_buffered_data_len(uart_num, &available);
    
    if (available > 0) {
        *duration_us = 0; // Not measured in this simple implementation
        return true;
    }
    
    return false;
}

static bool read_sync_byte(uart_port_t uart_num)
{
    uint8_t sync;
    int len = uart_read_bytes(uart_num, &sync, 1, pdMS_TO_TICKS(SYNC_TIMEOUT_MS));
    
    if (len == 1 && sync == LIN_SYNC_BYTE) {
        return true;
    }
    
    return false;
}

static bool read_pid(uart_port_t uart_num, uint8_t *pid)
{
    int len = uart_read_bytes(uart_num, pid, 1, pdMS_TO_TICKS(DATA_TIMEOUT_MS));
    return (len == 1);
}

static bool read_data_and_checksum(uart_port_t uart_num, lin_sniffed_frame_t *frame)
{
    uint8_t buffer[LIN_MAX_DATA_LEN + 1]; // +1 for checksum
    
    // Try to read maximum possible data (8 bytes + checksum)
    int len = uart_read_bytes(uart_num, buffer, sizeof(buffer), 
                             pdMS_TO_TICKS(DATA_TIMEOUT_MS));
    
    if (len < 2) {
        // No data, probably header-only frame
        return false;
    }
    
    // Last byte is checksum
    frame->len = len - 1;
    memcpy(frame->data, buffer, frame->len);
    frame->checksum = buffer[len - 1];
    
    // Calculate enhanced checksum (includes PID)
    uint16_t sum = frame->pid;
    for (int i = 0; i < frame->len; i++) {
        sum += frame->data[i];
        if (sum > 0xFF) {
            sum -= 0xFF;
        }
    }
    frame->calculated_checksum = (uint8_t)(~sum);
    frame->checksum_valid = (frame->checksum == frame->calculated_checksum);
    
    return true;
}
