/**
 * @file lin_sniffer.h
 * @brief Passive LIN Bus Sniffer for ESP32-S3
 * 
 * This module provides passive monitoring of LIN bus communication
 * between BCM and IBS sensor without interfering with the traffic.
 */

#ifndef LIN_SNIFFER_H
#define LIN_SNIFFER_H

#include <stdint.h>
#include <stdbool.h>
#include "driver/uart.h"
#include "esp_err.h"

#ifdef __cplusplus
extern "C" {
#endif

#define LIN_SYNC_BYTE           0x55
#define LIN_MAX_DATA_LEN        8
#define LIN_FRAME_BUFFER_SIZE   256

/**
 * @brief LIN frame types
 */
typedef enum {
    LIN_FRAME_TYPE_UNKNOWN = 0,
    LIN_FRAME_TYPE_HEADER,      // Break + Sync + PID
    LIN_FRAME_TYPE_MASTER_TX,   // Master sends data
    LIN_FRAME_TYPE_SLAVE_TX,    // Slave sends data (IBS response)
    LIN_FRAME_TYPE_INVALID      // Invalid/corrupted frame
} lin_frame_type_t;

/**
 * @brief LIN frame direction
 */
typedef enum {
    LIN_DIR_MASTER_TO_SLAVE = 0,    // BCM -> IBS
    LIN_DIR_SLAVE_TO_MASTER = 1,    // IBS -> BCM
    LIN_DIR_UNKNOWN = 2
} lin_direction_t;

/**
 * @brief Complete LIN frame structure for sniffing
 */
typedef struct {
    uint64_t timestamp_us;          /*!< Timestamp in microseconds */
    uint8_t id;                     /*!< Frame ID (6 bits) */
    uint8_t pid;                    /*!< Protected ID with parity */
    uint8_t data[LIN_MAX_DATA_LEN]; /*!< Data bytes */
    uint8_t len;                    /*!< Data length */
    uint8_t checksum;               /*!< Received checksum */
    uint8_t calculated_checksum;    /*!< Calculated checksum for verification */
    bool checksum_valid;            /*!< Checksum validation result */
    bool pid_valid;                 /*!< PID parity validation result */
    lin_frame_type_t type;          /*!< Frame type */
    lin_direction_t direction;      /*!< Communication direction */
    uint16_t break_duration_us;     /*!< Measured break duration */
} lin_sniffed_frame_t;

/**
 * @brief LIN sniffer statistics
 */
typedef struct {
    uint32_t total_frames;          /*!< Total frames captured */
    uint32_t valid_frames;          /*!< Valid frames */
    uint32_t checksum_errors;       /*!< Checksum errors */
    uint32_t pid_errors;            /*!< PID parity errors */
    uint32_t timeout_errors;        /*!< Timeout errors */
    uint32_t sync_errors;           /*!< Sync byte errors */
    uint64_t first_frame_time_us;   /*!< Timestamp of first frame */
    uint64_t last_frame_time_us;    /*!< Timestamp of last frame */
} lin_sniffer_stats_t;

/**
 * @brief LIN sniffer configuration
 */
typedef struct {
    uart_port_t uart_num;           /*!< UART port number */
    int rx_pin;                     /*!< RX pin (connect to TJA1020 RXD) */
    uint32_t baudrate;              /*!< Expected LIN baudrate (typically 19200) */
    bool auto_detect_baudrate;      /*!< Try to auto-detect baudrate */
    bool verbose_logging;           /*!< Enable verbose debug output */
} lin_sniffer_config_t;

/**
 * @brief Callback function type for frame reception
 * 
 * @param frame Pointer to captured frame
 * @param user_data User-provided context data
 */
typedef void (*lin_frame_callback_t)(const lin_sniffed_frame_t *frame, void *user_data);

/**
 * @brief Initialize LIN sniffer (passive mode)
 * 
 * @param config Pointer to sniffer configuration
 * @return ESP_OK on success, error code otherwise
 */
esp_err_t lin_sniffer_init(const lin_sniffer_config_t *config);

/**
 * @brief Deinitialize LIN sniffer
 * 
 * @return ESP_OK on success, error code otherwise
 */
esp_err_t lin_sniffer_deinit(void);

/**
 * @brief Register callback for frame reception
 * 
 * @param callback Callback function
 * @param user_data User context data passed to callback
 * @return ESP_OK on success, error code otherwise
 */
esp_err_t lin_sniffer_register_callback(lin_frame_callback_t callback, void *user_data);

/**
 * @brief Start sniffing LIN bus traffic
 * 
 * @return ESP_OK on success, error code otherwise
 */
esp_err_t lin_sniffer_start(void);

/**
 * @brief Stop sniffing LIN bus traffic
 * 
 * @return ESP_OK on success, error code otherwise
 */
esp_err_t lin_sniffer_stop(void);

/**
 * @brief Get current statistics
 * 
 * @param stats Pointer to statistics structure to fill
 * @return ESP_OK on success, error code otherwise
 */
esp_err_t lin_sniffer_get_stats(lin_sniffer_stats_t *stats);

/**
 * @brief Reset statistics
 * 
 * @return ESP_OK on success, error code otherwise
 */
esp_err_t lin_sniffer_reset_stats(void);

/**
 * @brief Decode and print frame in human-readable format
 * 
 * @param frame Pointer to frame to decode
 */
void lin_sniffer_print_frame(const lin_sniffed_frame_t *frame);

/**
 * @brief Print frame in hexadecimal format for analysis
 * 
 * @param frame Pointer to frame to print
 */
void lin_sniffer_print_frame_hex(const lin_sniffed_frame_t *frame);

/**
 * @brief Export frame in CSV format
 * 
 * @param frame Pointer to frame
 * @param buffer Buffer to store CSV string
 * @param buffer_size Size of buffer
 * @return Number of bytes written to buffer
 */
int lin_sniffer_export_csv(const lin_sniffed_frame_t *frame, char *buffer, size_t buffer_size);

/**
 * @brief Validate PID parity bits
 * 
 * @param pid Protected identifier
 * @return true if parity is valid, false otherwise
 */
bool lin_sniffer_validate_pid(uint8_t pid);

/**
 * @brief Get frame ID from PID
 * 
 * @param pid Protected identifier
 * @return Frame ID (6 bits)
 */
uint8_t lin_sniffer_get_id_from_pid(uint8_t pid);

/**
 * @brief Detect if frame is IBS-related (Ford Mondeo MK4)
 * 
 * @param id Frame ID
 * @return true if ID is typically used for IBS communication
 */
bool lin_sniffer_is_ibs_frame(uint8_t id);

/**
 * @brief Get description of IBS frame ID
 * 
 * @param id Frame ID
 * @return String description of frame purpose
 */
const char* lin_sniffer_get_ibs_frame_description(uint8_t id);

#ifdef __cplusplus
}
#endif

#endif // LIN_SNIFFER_H
