#include <stdio.h>
#include "stm32f3xx.h"
#include "usb.h"

// The current clock frequency
uint32_t SystemCoreClock=8000000;

// Counts milliseconds
volatile uint32_t systick_count=0;

// Interrupt handler
void SysTick_Handler(void)
{
    systick_count++;
}

// Delay some milliseconds
void delay_ms(int ms)
{
    uint32_t start=systick_count;
    while (systick_count-start<ms);
}

// Change system clock to 48Mhz using 8Mhz crystal
void init_clock()
{
    // Because the debugger switches PLL on, we may need to switch
    // back to the HSI oscillator before we can configure the PLL

    // Enable HSI oscillator
    SET_BIT(RCC->CR, RCC_CR_HSION);

    // Wait until HSI oscillator is ready
    while(!READ_BIT(RCC->CR, RCC_CR_HSIRDY)) {}

    // Switch to HSI oscillator
    MODIFY_REG(RCC->CFGR, RCC_CFGR_SW, RCC_CFGR_SW_HSI);

    // Wait until the switch is done
    while ((RCC->CFGR & RCC_CFGR_SWS_Msk) != RCC_CFGR_SWS_HSI) {}

    // Disable the PLL
    CLEAR_BIT(RCC->CR, RCC_CR_PLLON);

    // Wait until PLL is fully stopped
    while(READ_BIT(RCC->CR, RCC_CR_PLLRDY)) {}

    // Flash latency 1 wait state
    MODIFY_REG(FLASH->ACR, FLASH_ACR_LATENCY, 1 << FLASH_ACR_LATENCY_Pos);

    // Enable HSE oscillator
    SET_BIT(RCC->CR, RCC_CR_HSEON);

    // Wait until HSE oscillator is ready
    while(!READ_BIT(RCC->CR, RCC_CR_HSERDY)) {}

    // 48Mhz using the 8Mhz HSE oscillator with 6x PLL, lowspeed I/O runs at 24Mhz
    WRITE_REG(RCC->CFGR, RCC_CFGR_SWS_HSI + RCC_CFGR_PLLSRC_HSE_PREDIV + RCC_CFGR_PLLMUL6 + RCC_CFGR_PPRE1_DIV2);

    // Enable PLL
    SET_BIT(RCC->CR, RCC_CR_PLLON);

    // Wait until PLL is ready
    while(!READ_BIT(RCC->CR, RCC_CR_PLLRDY)) {}

    // Select PLL as clock source
    MODIFY_REG(RCC->CFGR, RCC_CFGR_SW, RCC_CFGR_SW_PLL);

    // Update variable
    SystemCoreClock=48000000;

    // Set USB prescaler to 1 for 48 MHz clock
    MODIFY_REG(RCC->CFGR, RCC_CFGR_USBPRE, RCC_CFGR_USBPRE_DIV1);
}

void init_io()
{
    // Enable USB
    SET_BIT(RCC->APB1ENR, RCC_APB1ENR_USBEN);

    // Enable Port A and C
    SET_BIT(RCC->AHBENR, RCC_AHBENR_GPIOAEN + RCC_AHBENR_GPIOCEN);

    // PC13 = Mode 01 Output (for the LED)
    MODIFY_REG(GPIOC->MODER,  GPIO_MODER_MODER13, 0b01 << GPIO_MODER_MODER13_Pos);

    // PA11 = USB D-, mode 10 alternate function 14 push/pull (see data sheet)
    MODIFY_REG(GPIOA->AFR[1], GPIO_AFRH_AFRH3,    14 << GPIO_AFRH_AFRH3_Pos);
    MODIFY_REG(GPIOA->MODER,  GPIO_MODER_MODER11, 0b10 << GPIO_MODER_MODER11_Pos);

    // PA12 = USB D+, mode 10 alternate function 14 push/pull (see data sheet)
    MODIFY_REG(GPIOA->AFR[1], GPIO_AFRH_AFRH4,    14 << GPIO_AFRH_AFRH4_Pos);
    MODIFY_REG(GPIOA->MODER,  GPIO_MODER_MODER12, 0b10 << GPIO_MODER_MODER12_Pos);
}

// Redirect standard output to the USB port, e.g. from puts() or printf()
int _write(int file, char *ptr, int len)
{
    return UsbSendBytes(ptr, len, 10);
}

int main()
{
    init_clock();
    init_io();
    UsbSetup();

    // Initialize system timer
    SysTick_Config(SystemCoreClock/1000);

    while (1)
    {
        // LED On (Low)
        WRITE_REG(GPIOC->BSRR, GPIO_BSRR_BR_13);
        delay_ms(100);

        UsbSendStr("Hello World!\n",10);
        // Alternative: puts("Hello World!");

        // If a line has been received, then send an echo back
        if (UsbRxBufferContains('\n'))
        {
            char buf[256];
            UsbGetStr(buf,sizeof(buf),'\n',0);
            UsbSendStr(buf,10);
        }

        // LED Off (High)
        WRITE_REG(GPIOC->BSRR, GPIO_BSRR_BS_13);
        delay_ms(900);
    }
}
