/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file remotecontrol.c
 *
 * Copyright (c) 2011 Robert Meyer, Frank Meyer - frank(at)fli4l.de
 *
 * Fuses (without bootloader):
 *                                         L-Fuse  H-Fuse  E-Fuse
 *  Internal RC Osc at 8MHz,      5.0 V:    0xE2    0xDC    0xF9
 *  Internal RC Osc at 8MHz,      4.5 V:    0xE2    0xDD    0xF9
 *  External Crystal Osc at 8MHz, 5.0 V:    0xFF    0xDC    0xF9
 *  External Crystal Osc at 8MHz, 4.5 V:    0xFF    0xDD    0xF9
 *
 * Fuses (with bootloader):
 *                                         L-Fuse  H-Fuse  E-Fuse
 *  Internal RC Osc at 8MHz,      5.0 V:    0xE2    0xDC    0xF8
 *  Internal RC Osc at 8MHz,      4.5 V:    0xE2    0xDD    0xF8
 *  External Crystal Osc at 8MHz, 5.0 V:    0xFF    0xDC    0xF8
 *  External Crystal Osc at 8MHz, 4.5 V:    0xFF    0xDD    0xF8
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <inttypes.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <avr/sleep.h>
#include <avr/wdt.h>

#include "irmpconfig.h"
#include "irmp.h"
#include "irsndconfig.h"
#include "irsnd.h"

#define BOOTLOADER_SUPPORT      1                       // 1: support bootload, 0: no bootloader
#define UART_DEBUG              0                       // 1: debug, 0: no debug

static volatile uint16_t        idle_cnt;
static volatile uint8_t         ms_cnt;

/*-----------------------------------------------------------------------------------------------------------------------
 * LEDs
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define LED_1_PORT          PORTD
#define LED_1_DDR           DDRD
#define LED_1               6

#define LED_2_PORT          PORTD
#define LED_2_DDR           DDRD
#define LED_2               7

#define LED_3_PORT          PORTB
#define LED_3_DDR           DDRB
#define LED_3               0

#define LED_1_ON            LED_1_PORT |= 1<<LED_1
#define LED_1_OFF           LED_1_PORT &= ~(1<<LED_1)
#define LED_2_ON            LED_2_PORT |= 1<<LED_2
#define LED_2_OFF           LED_2_PORT &= ~(1<<LED_2)
#define LED_3_ON            LED_3_PORT |= 1<<LED_3
#define LED_3_OFF           LED_3_PORT &= ~(1<<LED_3)

/*-----------------------------------------------------------------------------------------------------------------------
 * Initialize LEDs
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
led_init (void)
{
    LED_1_OFF;
    LED_1_DDR |= (1<<LED_1);

    LED_2_OFF;
    LED_2_DDR |= (1<<LED_2);

    LED_3_OFF;
    LED_3_DDR |= (1<<LED_3);
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Switch all LEDs on
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
leds_on (void)
{
    LED_1_ON;
    LED_2_ON;
    LED_3_ON;
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Switch all LEDs off
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
leds_off (void)
{
    LED_1_OFF;
    LED_2_OFF;
    LED_3_OFF;
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Show active device
 *-----------------------------------------------------------------------------------------------------------------------
 */
static uint8_t     device_number = 0;

static void
show_device_number (uint8_t device_number)
{
    leds_off();

    switch (device_number)
    {
        case 0: LED_1_ON; break;
        case 1: LED_2_ON; break;
        case 2: LED_3_ON; break;
    }
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Called (back) from IRSND module
 *-----------------------------------------------------------------------------------------------------------------------
 */
void
led_callback (uint8_t on)
{
    if (on)
    {
        switch (device_number)
        {
            case 0: LED_1_ON; break;
            case 1: LED_2_ON; break;
            case 2: LED_3_ON; break;
        }
    }
    else
    {
        switch (device_number)
        {
            case 0: LED_1_OFF; break;
            case 1: LED_2_OFF; break;
            case 2: LED_3_OFF; break;
        }
    }
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Keys
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define KEY_POWER_PORT      PORTC
#define KEY_POWER_DDR       DDRC
#define KEY_POWER_PIN       PINC
#define KEY_POWER           4
#define KEY_POWER_VALUE     0

#define KEY_UP_PORT         PORTC
#define KEY_UP_DDR          DDRC
#define KEY_UP_PIN          PINC
#define KEY_UP              0
#define KEY_UP_VALUE        1

#define KEY_DOWN_PORT       PORTC
#define KEY_DOWN_DDR        DDRC
#define KEY_DOWN_PIN        PINC
#define KEY_DOWN            1
#define KEY_DOWN_VALUE      2

#define KEY_LEFT_PORT       PORTC
#define KEY_LEFT_DDR        DDRC
#define KEY_LEFT_PIN        PINC
#define KEY_LEFT            2
#define KEY_LEFT_VALUE      3

#define KEY_RIGHT_PORT      PORTC
#define KEY_RIGHT_DDR       DDRC
#define KEY_RIGHT_PIN       PINC
#define KEY_RIGHT           3
#define KEY_RIGHT_VALUE     4

#define MAX_KEYS            5                                                       // max. number of keys to store
#define MAX_DEVICES         3                                                       // max. number of different devices
#define MAX_CODES           5                                                       // max. number of codes per key

#define KEY_SELECT_PORT     PORTC
#define KEY_SELECT_DDR      DDRC
#define KEY_SELECT_PIN      PINC
#define KEY_SELECT          5
#define KEY_SELECT_VALUE    5

#define KEY_PROG_VALUE      6

#define KEY_FLASH_VALUE     7


static volatile uint8_t     global_key = 0xFF;

/*-----------------------------------------------------------------------------------------------------------------------
 * Initialize keys
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
key_init (void)
{
    KEY_SELECT_DDR &= ~(1<<KEY_SELECT);
    KEY_SELECT_PORT |= (1<<KEY_SELECT);

    KEY_POWER_DDR &= ~(1<<KEY_POWER);
    KEY_POWER_PORT |= (1<<KEY_POWER);

    KEY_UP_DDR &= ~(1<<KEY_UP);
    KEY_UP_PORT |= (1<<KEY_UP);

    KEY_DOWN_DDR &= ~(1<<KEY_DOWN);
    KEY_DOWN_PORT |= (1<<KEY_DOWN);

    KEY_LEFT_DDR &= ~(1<<KEY_LEFT);
    KEY_LEFT_PORT |= (1<<KEY_LEFT);

    KEY_RIGHT_DDR &= ~(1<<KEY_RIGHT);
    KEY_RIGHT_PORT |= (1<<KEY_RIGHT);

    PCMSK1 |= (1<<PCINT8) | (1<<PCINT9) | (1<<PCINT10) | (1<<PCINT11) | (1<<PCINT12) | (1<<PCINT13);
    PCICR |= (1<<PCIE1);
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Poll keys
 *-----------------------------------------------------------------------------------------------------------------------
 */
static uint8_t
key_poll (void)
{
    uint8_t value;

    if (!(KEY_SELECT_PIN & (1<<KEY_SELECT)))
    {
 #if BOOTLOADER_SUPPORT == 1
       if (!(KEY_POWER_PIN & (1<<KEY_POWER)))
        {
            value = KEY_FLASH_VALUE;
        }
        else
#endif
        {
            value = KEY_SELECT_VALUE;
        }
    }
    else if (!(KEY_POWER_PIN & (1<<KEY_POWER)))
    {
        value = KEY_POWER_VALUE;
    }
    else if (!(KEY_UP_PIN & (1<<KEY_UP)))
    {
        value = KEY_UP_VALUE;
    }
    else if (!(KEY_DOWN_PIN & (1<<KEY_DOWN)))
    {
        value = KEY_DOWN_VALUE;
    }
    else if (!(KEY_LEFT_PIN & (1<<KEY_LEFT)))
    {
        value = KEY_LEFT_VALUE;
    }
    else if (!(KEY_RIGHT_PIN & (1<<KEY_RIGHT)))
    {
        value = KEY_RIGHT_VALUE;
    }
    else
    {
        value = 0xFF;
    }

    return value;
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Get a key
 *-----------------------------------------------------------------------------------------------------------------------
 */
static uint8_t
getkey (void)
{
    uint8_t k;

    k = global_key;

    if (k != 0xFF)
    {
        cli();
        global_key = 0xFF;
        sei();
    }

    return k;
}

/*-----------------------------------------------------------------------------------------------------------------------
 * IR RECEIVER switch
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define REC_PORT            PORTB
#define REC_DDR             DDRB
#define REC                 1
#define REC_OFF             REC_PORT |= (1<<REC)                                // enable receiver
#define REC_ON              REC_PORT &= ~(1<<REC)                               // disable receiver

/*-----------------------------------------------------------------------------------------------------------------------
 * Initialize IR receiver switch
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
rec_init (void)
{
    REC_OFF;                                                                    // set output to 1 (disable receiver)
    REC_DDR |= (1<<REC);                                                        // set pin to output
}

#if UART_DEBUG == 1
/*-----------------------------------------------------------------------------------------------------------------------
 * UART routines
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define BAUD                                    9600L
#include <util/setbaud.h>

#ifdef UBRR0H

#define UART0_UBRRH                             UBRR0H
#define UART0_UBRRL                             UBRR0L
#define UART0_UCSRA                             UCSR0A
#define UART0_UCSRB                             UCSR0B
#define UART0_UCSRC                             UCSR0C
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE0)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ01)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ00)
#ifdef URSEL0
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL0)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN0)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN0)
#define UART0_UDR                               UDR0
#define UART0_U2X                               U2X0
#define UART0_RXC                               RXC0

#else

#define UART0_UBRRH                             UBRRH
#define UART0_UBRRL                             UBRRL
#define UART0_UCSRA                             UCSRA
#define UART0_UCSRB                             UCSRB
#define UART0_UCSRC                             UCSRC
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ1)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ0)
#ifdef URSEL
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN)
#define UART0_UDR                               UDR
#define UART0_U2X                               U2X
#define UART0_RXC                               RXC

#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Initialize  UART
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
uart_init (void)
{
    UART0_UBRRH = UBRRH_VALUE;                                                                      // set baud rate
    UART0_UBRRL = UBRRL_VALUE;

#if USE_2X
    UART0_UCSRA |= (1<<UART0_U2X);
#else
    UART0_UCSRA &= ~(1<<UART0_U2X);
#endif

    UART0_UCSRC = UART0_UCSZ1_BIT_VALUE | UART0_UCSZ0_BIT_VALUE | UART0_URSEL_BIT_VALUE;
    UART0_UCSRB |= UART0_TXEN_BIT_VALUE;                                                            // enable UART TX
    UART0_UCSRB |= UART0_RXEN_BIT_VALUE;                                                            // activate UART RX
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Send character
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
uart_putc (unsigned char ch)
{
    while (!(UART0_UCSRA & UART0_UDRE_BIT_VALUE))
    {
        ;
    }

    UART0_UDR = ch;
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Send string
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
uart_puts (char * s)
{
    while (*s)
    {
        uart_putc (*s++);
    }
}

#endif // #if UART_DEBUG == 1

/*-----------------------------------------------------------------------------------------------------------------------
 * Initialize timer
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
timer_init (void)
{
    OCR1A   =  (F_CPU / F_INTERRUPTS) - 1;                                      // compare value: 1/15000 of CPU frequency
    TCCR1B  = (1 << WGM12) | (1 << CS10);                                       // switch CTC Mode on, set prescaler to 1
    TIMSK1  = 1 << OCIE1A;                                                      // OCIE1A: Interrupt by timer compare
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Timer interrupt: called 15000 times per second
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define KEY_SHORT_HOLD_TIME     (F_INTERRUPTS / 15)                             // 1/15 sec
#define KEY_LONG_HOLD_TIME      (F_INTERRUPTS)                                  // 1 sec
#define IDLE_TIME               (F_INTERRUPTS / 2)                              // 1/2 sec

ISR(TIMER1_COMPA_vect)
{
    uint8_t         k;
    static uint8_t  timecnt;
    static uint16_t keycnt;
    static uint8_t  last_k = 0xFF;

    if (! irsnd_ISR())                                                          // call irsnd ISR
    {                                                                           // nothing to send...
        (void) irmp_ISR();                                                      // ... then call irmp ISR
    }

    timecnt++;

    if (timecnt >= (uint8_t) (F_INTERRUPTS / 1000))
    {
        timecnt = 0;

        if (ms_cnt < 255)
        {
            ms_cnt++;
        }
    }

    k = key_poll ();                                                            // key pressed?

    if (k == 0xFF && last_k != KEY_SELECT_VALUE)                                // no....
    {
        keycnt = 0;                                                             // reset key counter

        if (idle_cnt < 0xFFFF)                                                  // increment idle counter
        {
            idle_cnt++;
        }
    }
    else
    {                                                                           // key pressed!
        idle_cnt = 0;                                                           // reset idle counter

        if (k == last_k)                                                        // key is same as last key?
        {
            if (keycnt < 0xFFFF)
            {
                keycnt++;                                                       // yes, increment key counter
            }

            if ((k < MAX_KEYS || k == KEY_FLASH_VALUE) && keycnt >= KEY_SHORT_HOLD_TIME)
            {                                                                   // key 1/15 sec pressed?
                keycnt = 0;                                                     // yes, reset key counter
                last_k = 0xFF;                                                  // reset last key value
                global_key = k;                                                 // store actual key
            }
            else if (k == KEY_SELECT_VALUE && keycnt >= KEY_LONG_HOLD_TIME)     // SELECT key 1 sec pressed?
            {
                keycnt = 0;                                                     // yes, reset key counter
                last_k = 0xFF;                                                  // reset last key value
                global_key = KEY_PROG_VALUE;                                    // store KEY_PROG as pressed key
            }
        }
        else
        {                                                                       // key is NOT the same as last key...
            if (last_k == KEY_SELECT_VALUE)                                     // last key was SELECT...
            {
                if (keycnt >= KEY_SHORT_HOLD_TIME)                              // SELECT only 1/15 sec pressed?
                {
                    global_key = KEY_SELECT_VALUE;                              // yes, store SELECT as pressed key
                    k = 0xFF;                                                   // reset k (which will be stored as last key, see below)
                }
            }

            keycnt = 0;
            last_k = k;
        }
    }
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Pin Change Interrupt: used to wakeup cpu from sleep mode
 *-----------------------------------------------------------------------------------------------------------------------
 */
ISR(PCINT1_vect)                                                                // key pressed (PCINT service routine)
{
    ;
}

/*-----------------------------------------------------------------------------------------------------------------------
 * EEPROM routines
 *-----------------------------------------------------------------------------------------------------------------------
 */
static IRMP_DATA    ee_irmp_data[MAX_DEVICES][MAX_KEYS][MAX_CODES]          EEMEM;
static uint8_t      ee_pause[MAX_DEVICES][MAX_KEYS][(MAX_CODES + 1) / 2]    EEMEM;

/*-----------------------------------------------------------------------------------------------------------------------
 * Read a block from EEPROM
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
my_eeprom_read_block (void * addr, void * ee_addr, size_t size)
{
#if 0
    uint8_t tmp_sreg;

    tmp_sreg = SREG;                                                            // save interrupt status
    cli();                                                                      // disable interrupts
#endif

    eeprom_busy_wait ();                                                        // wait until eeprom is ready
    eeprom_read_block (addr, ee_addr, size);                                    // read block

#if 0
    SREG = tmp_sreg;                                                            // restore interrupt status
#endif
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Write a block into EEPROM
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
my_eeprom_write_block (void * addr, void * ee_addr, size_t size)
{
#if 0
    uint8_t tmp_sreg;

    tmp_sreg = SREG;                                                            // save interrupt status
    cli();                                                                      // disable interrupts
#endif

    eeprom_busy_wait ();                                                        // wait until eeprom is ready
    eeprom_write_block (addr, ee_addr, size);                                   // write block

#if 0
    SREG = tmp_sreg;                                                            // restore interrupt status
#endif
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Delay: wait x seconds
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
my_delay_s (uint8_t sec)
{
    while (sec--)
    {
        _delay_ms (1000);                                                       // wait 1000 msec
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * HEX: convert hex digit into decimal value
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint8_t
xtoi (unsigned char * buf)
{
    uint8_t x = buf[0];

    if (x >= '0' && x <= '9')
    {
        x -= '0';
    }
    else if (x >= 'A' && x <= 'F')
    {
        x -= 'A' - 10;
    }
    else if (x >= 'a' && x <= 'f')
    {
        x -= 'a' - 10;
    }
    else
    {
        x = 0;
    }

    return (x);
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * HEX: convert two hex digits into decimal value
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint8_t
xxtoi (unsigned char * buf)
{
    uint8_t v;

    v = (xtoi (buf) << 4) | xtoi (buf + 1);
    return (v);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * HEX: convert four hex digits into decimal value
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint16_t
xxxxtoi (unsigned char * buf)
{
    uint16_t v;

    v = (xxtoi (buf) << 8) | xxtoi (buf + 2);
    return (v);
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * IR-Out routines
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static IRMP_DATA    irmp_data[MAX_CODES];
static uint8_t      pause[MAX_CODES];

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * send IR data bound by a key
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
send_key (uint8_t devno, uint8_t key)
{
    uint8_t     cnt;
    uint8_t     p;
    uint8_t     k;

    my_eeprom_read_block (&(irmp_data[0]), &(ee_irmp_data[devno][key][0]), MAX_CODES * sizeof (IRMP_DATA));
    my_eeprom_read_block (&(pause[0]), &(ee_pause[devno][key][0]), (MAX_CODES + 1) / 2 * sizeof (uint8_t));

    for (k = 0; k < MAX_CODES; k++)
    {
        if (irmp_data[k].protocol == 0xFF)                          // valid protocol?
        {                                                           // no, break
            break;
        }

        cnt = irmp_data[k].flags + 1;                               // get number of frames to send
        irmp_data[k].flags = 0;                                     // reset number of frames!

        if (k & 0x01)                                               // odd?
        {
            p = pause[k/2] & 0x0F;                                  // yes, use lower nibble as pause value
        }
        else
        {
            p = pause[k/2] >> 4;                                    // even, use upper nibble as pause value
        }

        my_delay_s (p);                                             // wait x seconds

        while (cnt > 0)                                             // send cnt frames
        {
            irsnd_send_data (&(irmp_data[k]), TRUE);                // send IR code now

            while (irsnd_is_busy ())                                // HACK: wait until IRSND is ready
            {
                ;
            }

            _delay_ms (50);                                         // wait 50 msec to force a pause between frames
            cnt--;
        };
    }
}

/*-----------------------------------------------------------------------------------------------------------------------
 * MAIN
 *-----------------------------------------------------------------------------------------------------------------------
 */
int
main (void)
{
    uint8_t             p;
    uint8_t             cancelled;
    uint8_t             key_learned;
    uint8_t             leds_are_on = FALSE;
    uint8_t             framecnt;
    uint8_t             key;
    uint8_t             k;

    wdt_disable ();
    led_init ();                                                                // initialize LED, KEY, timer
    key_init ();
    timer_init ();
    rec_init ();                                                                // initialize REC, IRMP, IRSND
    irmp_init ();
    irsnd_init();
    irsnd_set_callback_ptr (led_callback);

#if UART_DEBUG == 1
    uart_init ();
#endif

    sei ();                                                                     // enable interrupts

    set_sleep_mode (SLEEP_MODE_PWR_DOWN);                                       // set CPU into powerdown mode
    sleep_enable ();                                                            // enable sleep mode..
    sleep_cpu ();

    while (1)
    {
        if ((key = getkey()) != 0xFF)                                           // read key, pressed?
        {                                                                       // yes
#if BOOTLOADER_SUPPORT == 1
            if (key == KEY_FLASH_VALUE)
            {
                void (*funcptr) (void) = (void *) 0x3800;
                cli ();
                funcptr ();
                break;
            }
            else
#endif
            if (key == KEY_SELECT_VALUE)                                   // select key pressed?
            {
                if (leds_are_on)                                                // LEDs already on, switch to next device
                {
                    device_number++;

                    if (device_number >= MAX_DEVICES)
                    {
                        device_number = 0;
                    }
                }

                show_device_number (device_number);                             // show current device
                leds_are_on = TRUE;
            }
            else if (key == KEY_PROG_VALUE)                                     // switch into learning mode
            {
                leds_on();                                                      // switch all LEDs on: we are ready to receive IR signal
                REC_ON;                                                         // enable receiver

                _delay_ms (500);

                while ((key = getkey()) == KEY_PROG_VALUE || key == KEY_SELECT_VALUE)   // empty key data
                {
                    ;
                }

                cancelled = FALSE;
                key_learned = FALSE;

                for (k = 0; k < MAX_CODES; k++)
                {
                    while (irmp_get_data (&(irmp_data[k])))                     // empty irmp data
                    {
                        ;
                    }

                    while (getkey() != 0xFF)                                    // empty key data
                    {
                        ;
                    }

                    p = 0;

                    while (! irmp_get_data (&irmp_data[k]))                     // receive IR signal
                    {
                        key = getkey();

                        if (key < MAX_KEYS)                                     // pressed a key?
                        {                                                       // yes
                            irmp_data[k].protocol = 0xFF;
                            key_learned = TRUE;
                            break;
                        }
                        else if (key == KEY_PROG_VALUE)
                        {
                            cancelled = TRUE;                                   // cancel learning mode
                            break;
                        }
                        else if (key == KEY_SELECT_VALUE)
                        {
                            if (p < 15)
                            {
                                p++;
                            }
                            leds_off(); 
                            _delay_ms (250);
                            leds_on();
                        }
                    }

                    leds_off(); 

                    framecnt = 0;
                    cli ();
                    ms_cnt = 0;
                    sei();

                    while (ms_cnt < 250)
                    {
                        // some devices need a repeated frame to accept the command, e.g. POWER command on Toshiba TV to switch off
                        if (! cancelled && irmp_data[k].protocol != 0xFF)       // after 250 ms....
                        {
                            IRMP_DATA id;

                            if (irmp_get_data (&id))                            // ... receive IR signal again
                            {
                                if (id.flags == 1)                              // it's a repeated frame, store it!
                                {
                                    framecnt++;
                                }
                                cli ();
                                ms_cnt = 0;
                                sei ();
                            }
                        }
                    }

                    irmp_data[k].flags = framecnt;

                    leds_on();
                    _delay_ms (250);

                    if (k & 0x01)
                    {
                        pause[k/2] |= p;
                    }
                    else
                    {
                        pause[k/2] = (p << 4);
                    }

                    if (cancelled || key_learned)
                    {
                        break;
                    }
                } // for ...

                if (! cancelled)
                {
                    if (key == 0xFF)
                    {
                        while ((key = getkey ()) >= MAX_KEYS)                   // pressed a key?
                        {                                                       // yes
                            if (key == KEY_PROG_VALUE)
                            {
                                break;
                            }
                        }
                    }

                    if (key < MAX_KEYS)                                         // valid key pressed?
                    {
                        my_eeprom_write_block (&(irmp_data[0]), &(ee_irmp_data[device_number][key][0]), MAX_CODES * sizeof (IRMP_DATA));
                        my_eeprom_write_block (&(pause[0]), &(ee_pause[device_number][key][0]), (MAX_CODES + 1) / 2 * sizeof (uint8_t));
                                                   // let LEDs blink 2 times: code stored

#if UART_DEBUG == 1
                        {
                            char buf[5];

                            uart_puts ("p=");
                            utoa (irmp_data[0].protocol, buf, 16);
                            uart_puts (buf);
                            uart_puts (" a=0x");
                            utoa (irmp_data[0].address, buf, 16);
                            uart_puts (buf);
                            uart_puts (" c=0x");
                            utoa (irmp_data[0].command, buf, 16);
                            uart_puts (buf);
                            uart_puts ("\r\n");
                        }
#endif
                        _delay_ms (250);
                        leds_off();
                        _delay_ms (250);
                        leds_on();
                        _delay_ms (250);
                    }
                }

                leds_off();
                REC_OFF;                                                        // disable receiver to save energy (current)

                while (getkey() != 0xFF)                                        // empty key data again (avoid sending yet)
                {
                    ;
                }
            }
            else if (key < MAX_KEYS)
            {                                                                   // send IR code (stored in RAM)
                send_key (device_number, key);
            }
        }
        else
        {
            if (idle_cnt >= IDLE_TIME)                                          // 1/2 second idle...
            {
                cli ();                                                         // reset idle counter
                idle_cnt = 0;
                sei ();

                leds_are_on = FALSE;                                            // reset LED flag
                leds_off ();                                                    // switch LEDs off
                sleep_cpu ();                                                   // get into powerdown mode
            }
        }
    }
}
