//***************************************************************************************
//***
//***	Projectname:	Control Interface	
//***	Filename:		ASCInterface.c
//***	Shortcut:		ASC
//***	Author:			Ralf Hochhausen
//***	Date:			20.10.2003
//***	Acual Version:	V1.0 (Initial Version)
//***	
//***	Description:	Hardware dirver for the asynchronous interface of the mega 128	
//***
//***	File-History:	20.10.2003
//***					V1.0	Initial Version
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Includes
//***
//***************************************************************************************
#include <avr\io.h>
#include <avr\signal.h>
#include "Types.h"
#include "ASCInterface.h"
#include "C_Interf.h"
#include "Debug.h"

//***************************************************************************************
//***
//***	Local Definitions
//***
//***************************************************************************************
#define ASC_CPU_FREQUENCY		8000000L	//CPU runs at 8MHz
//#define ASC_BAUDRATE			9600L		//Baudrate of the asynchronous interface
//#define ASC_BAUDRATE			19200L		//Baudrate of the asynchronous interface
#define ASC_BAUDRATE			38400L		//Baudrate of the asynchronous interface
#define	ASC_SPEEDMODE			16			//Interface operates in normal mode
//Calculation of the Baudrate Register
#define ASC_UBRR				ASC_CPU_FREQUENCY/(ASC_SPEEDMODE*ASC_BAUDRATE)-1

#define ASC_RECEIVEBUFFERSIZE	30			//Buffersize of the receive buffer
#define ASC_TRANSMITBUFFERSIZE	30			//Buffersize of the transmit buffer

//***************************************************************************************
//***
//***	Local Enumerations
//***
//***************************************************************************************
enum
{
	ASC_NORMAL_MODE			= 0x00,			//asynchronous interface in normal mode
	ASC_RXBUFFER_FULL		= 0x01,			//receive buffer is full
	ASC_TXBUFFER_FULL		= 0x02,			//transmit buffer is full
	ASC_RXBUFFER_OVERRUN	= 0x04,			//receive buffer overrun
	ASC_TXBUFFER_OVERRUN	= 0x08			//transmit buffer overrun
};

//***************************************************************************************
//***
//***	Local Variables
//***
//***************************************************************************************
static uchar ASC_ucReceiveBuffer[ASC_RECEIVEBUFFERSIZE];	//Array for receive data
static uchar ASC_ucTransmitBuffer[ASC_TRANSMITBUFFERSIZE];	//Array for transmit data
static uchar ASC_ucWriteRxBufferIdx;	//Index for writing data into receive buffer
static uchar ASC_ucReadRxBufferIdx;		//Index for reading receive buffer
static uchar ASC_ucWriteTxBufferIdx;	//Index for writing data into transmit buffer
static uchar ASC_ucReadTxBufferIdx;		//Index for reading transmit buffer
static uchar ASC_ucStatus;				//Status of the asynchronous interface

//***************************************************************************************
//***
//***	Local functions and interrupt service routines
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Function:		SIGNAL(SIG_UART1_RECV)
//***	
//***	Return Type:	void
//***
//***	Parameters:		none
//***	
//***	Receive interrupt service routine 
//***	Checks if receive buffer overrun occured. If not received data is written into
//***	ring buffer. If receive buffer is full, flag is set
//***	
//***************************************************************************************
SIGNAL(SIG_UART1_RECV)
{
	if(ASC_ucStatus && ASC_RXBUFFER_FULL)			//buffer overrun
	{
		ASC_ucStatus |= ASC_RXBUFFER_OVERRUN;		//set overrun flag
		return;
	}

	ASC_ucReceiveBuffer[ASC_ucWriteRxBufferIdx] = UDR1;

	ASC_ucWriteRxBufferIdx++;

	if(ASC_ucWriteRxBufferIdx == ASC_RECEIVEBUFFERSIZE)
	{
		ASC_ucWriteRxBufferIdx = 0;
	}

	if(ASC_ucWriteRxBufferIdx == ASC_ucReadRxBufferIdx)
	{
		ASC_ucStatus |= ASC_RXBUFFER_FULL;			//buffer is full
	}
}


//***************************************************************************************
//***
//***	Global Functions
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Function:		void ASC_vHandleTask(void)
//***	
//***	Return Type:	void
//***
//***	Parameters:		none
//***	
//***	Checks if date has to be send. If yes, the hardware buffer is checked. If it
//***	is empty, the next data byte is written into the hardware buffer.
//***	After this, the transmit buffer read index is set to the next read position
//***
//***************************************************************************************
void ASC_vHandleTask(void)
{
	if(ASC_ucReadTxBufferIdx != ASC_ucWriteTxBufferIdx)		//any data to send?
	{
		if(UCSR1A & (1 << UDRE1))									//hw transmit buffer empty?
		{
			UDR1 = ASC_ucTransmitBuffer[ASC_ucReadTxBufferIdx];		//write data into buffer
			
			ASC_ucReadTxBufferIdx++;		//Set index to the next position
			if(ASC_ucReadTxBufferIdx == ASC_TRANSMITBUFFERSIZE)
			{
				ASC_ucReadTxBufferIdx = 0;
			}
			if(ASC_ucStatus & ASC_TXBUFFER_FULL)	//transmit buffer has been full?
			{
				//clear transmit buffer full and overrun flag
				ASC_ucStatus &= (~(ASC_TXBUFFER_FULL | ASC_TXBUFFER_OVERRUN));
			}
		}
	}
}

//***************************************************************************************
//***
//***	Function:		uchar ASC_vGetReceivedByte(void)
//***	
//***	Return Type:	Received data byte out of the ring buffer
//***
//***	Parameters:		none
//***	
//***	Returns the first received data byte out of the ring buffer. If the buffer
//***	has been full, the flags are cleared
//***
//***************************************************************************************
uchar ASC_ucGetReceivedByte(void)
{
	uchar ucData_;		//temporary data variable

	ucData_ = ASC_ucReceiveBuffer[ASC_ucReadRxBufferIdx];

	ASC_ucReadRxBufferIdx++;		//set read index to next position

	if(ASC_ucReadRxBufferIdx == ASC_RECEIVEBUFFERSIZE)
	{
		ASC_ucReadRxBufferIdx = 0;
	}

	if(ASC_ucStatus & ASC_RXBUFFER_FULL)		//buffer has been full?
	{
		//reset status flags
		ASC_ucStatus &= (~(ASC_RXBUFFER_FULL | ASC_RXBUFFER_OVERRUN));
	}
	
	return ucData_;
}

//***************************************************************************************
//***
//***	Function:		uchar ASC_vTransmitByte(uchar *ucData_)
//***	
//***	Return Type:	TRUE, if data can be transmitted (data has been written
//***					into buffer), else FALSE
//***
//***	Parameters:		uchar *ucData_:	Pointer on the data byte that has to be transmitted
//***	
//***	Writes one data byte into the transmit buffer. If buffer is full or overrun
//***	has occured, status flags are set
//***
//***************************************************************************************
uchar ASC_vTransmitByte(uchar ucData_)
{
	if(ASC_ucStatus & ASC_TXBUFFER_FULL)			//transmit buffer full
	{
		ASC_ucStatus |= ASC_TXBUFFER_OVERRUN;		//transmit buffer overrun
		return FALSE;
	}

	//write data into transmit buffer
	ASC_ucTransmitBuffer[ASC_ucWriteTxBufferIdx] = ucData_;

	ASC_ucWriteTxBufferIdx++;			//set index to the next position
	if(ASC_ucWriteTxBufferIdx == ASC_TRANSMITBUFFERSIZE)
	{
		ASC_ucWriteTxBufferIdx = 0;
	}

	if(ASC_ucWriteTxBufferIdx == ASC_ucReadTxBufferIdx)	//transmit buffer full?
	{
		ASC_ucStatus |= ASC_TXBUFFER_FULL;
	}

	return TRUE;
}

//***************************************************************************************
//***
//***	Function:		uchar ASC_ucIsTransmitBufferFree(void)
//***	
//***	Return Type:	TRUE: Buffer is free; FALSE: buffer is full
//***
//***	Parameters:		none
//***	
//***	returns TRUE if at least one byte is free in transmit buffer, else false
//***
//***************************************************************************************
uchar ASC_ucIsTransmitBufferFree(void)
{
	return ~(ASC_ucStatus & ASC_TXBUFFER_FULL);
}

//***************************************************************************************
//***
//***	Function:		uchar ASC_ucIsByteReceived(void)
//***	
//***	Return Type:	TRUE: Data is in buffer; FALSE: buffer is empty
//***
//***	Parameters:		none
//***	
//***	returns TRUE if at least one byte in receive buffer, else false
//***
//***************************************************************************************
uchar ASC_ucIsByteReceived(void)
{
	if((ASC_ucReadRxBufferIdx != ASC_ucWriteRxBufferIdx) ||
	   (ASC_ucStatus & ASC_RXBUFFER_FULL))
	{
		return TRUE;
	}

	return FALSE;
}

//***************************************************************************************
//***
//***	Function:		void ASC_vInit(void)
//***	
//***	Return Type:	void
//***
//***	Parameters:		none
//***	
//***	Initialization of the asynchronous interface (hardware driver for the atmega128)
//***	Initialization of the transmit and receive buffer
//***	
//***	Parameters for the asynchronous interface:
//***	Baudrate:		9600
//***	Databits:		8
//***	Stopbits:		1
//***	Parity:			no
//***
//***************************************************************************************
void ASC_vInit(void)
{
	uchar ucTemp_;

	UCSR1A = 0x00;			//USART operates at normal speed
							//Multi-Processor mode is disabled

	UCSR1B = 0x98;			//Enable transmitter and receiver
							//Enable receive complete interrupt

	UCSR1C = 0x06;			//Configure USART for asynchronous operation
							//1 stopbit, no patity

	UBRR1H = 0x00;			//Configure Baudrate Register
	UBRR1L = ASC_UBRR;

	//Initialize local Variables
	for(ucTemp_ = 0; ucTemp_ < ASC_RECEIVEBUFFERSIZE; ucTemp_++)
	{
		ASC_ucReceiveBuffer[ucTemp_] = 0x00;
	}
	for(ucTemp_ = 0; ucTemp_ < ASC_TRANSMITBUFFERSIZE; ucTemp_++)
	{
		ASC_ucTransmitBuffer[ucTemp_] = 0x00;
	}
	ASC_ucWriteRxBufferIdx = 0;	
	ASC_ucReadRxBufferIdx = 0;
	ASC_ucWriteTxBufferIdx = 0;
	ASC_ucReadTxBufferIdx = 0;

	ASC_ucStatus = ASC_NORMAL_MODE;
}

