//***************************************************************************************
//***
//***	Projectname:	Compact Flash Interface
//***	Filename:		CF_Card_IF.c
//***	Shortcut:		CFC
//***	Author:			Ralf Hochhausen
//***	Date:			06.12.2003
//***	Acual Version:	V1.0 (Initial Version)
//***	
//***	Description:	Memory Interface to a compact flash card.
//***					Compact flash card is used in memory mapped mode
//***					Registers are located at 0xE000 in external RAM space of Mega128
//***
//***	File-History:	06.12.2003
//***					V1.0	Initial Version
//***					Identify drive and read sector functions are implemented.
//***					Writing data to the card isn't possible yet.
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Includes
//***
//***************************************************************************************
#include "Types.h"
#include "CF_Card_IF.h"
#include "ASCInterface.h"

//***************************************************************************************
//***
//***	Local Definitions
//***
//***************************************************************************************

//***************************************************************************************
//***	Definition of the register addresses
//***************************************************************************************
#define		CFC_DATA_REG			(*(volatile uchar*)(0xE000l))	//Data register
#define		CFC_ERROR_REG			(*(volatile uchar*)(0xE001l))	//Error register (read)
#define		CFC_FEATURE_REG			(*(volatile uchar*)(0xE001l))	//Feature register (write)
#define		CFC_SECTOR_COUNT_REG	(*(volatile uchar*)(0xE002l))	//Sector count register
#define		CFC_SECTOR_NUMBER_REG	(*(volatile uchar*)(0xE003l))	//Sector number register	LBA07..LBA00	
#define		CFC_CYLINDER_LOW_REG	(*(volatile uchar*)(0xE004l))	//Cylinder low register		LBA15..LBA08
#define		CFC_CYLINDER_HIGH_REG	(*(volatile uchar*)(0xE005l))	//Cylinder high register	LBA23..LBA16
#define		CFC_DRIVE_HEAD_REG		(*(volatile uchar*)(0xE006l))	//Drive head register		LBA27..LBA24
#define		CFC_STATUS_REG			(*(volatile uchar*)(0xE007l))	//Status register (read)
#define		CFC_COMMAND_REG			(*(volatile uchar*)(0xE007l))	//Command register (write)

//***************************************************************************************
//***	Definition of the compact flash (ATA) commands
//***************************************************************************************
#define		CFC_CMD_READ_SECTOR		0x20		//Read sector command
#define		CFC_CMD_IDENTIFY_DRIVE	0xEC		//Identify Drive Command

//***************************************************************************************
//***	Definition of the status register bits
//***
//***	Bit7															Bit0
//***	BSY		DRDY		DWF		DSC		DRQ		CORR		IDX		ERR
//***************************************************************************************
#define	CFC_STATUS_BSY				0x80		//Busy flag
#define	CFC_STATUS_DRDY				0x40		//Drive ready
#define	CFC_STATUS_DWF				0x20		//Drive write fault
#define	CFC_STATUS_DSC				0x10		//Drive seek complete
#define	CFC_STATUS_DRQ				0x08		//Data request
#define	CFC_STATUS_CORR				0x04		//Corrected data
#define	CFC_STATUS_IDX				0x02		//Index
#define	CFC_STATUS_ERR				0x01		//Error

//***************************************************************************************
//***	Definition of the drive head register bits
//***
//***	Bit7															Bit0
//***	Obsolete	LBA		Obsolete	DRV		HeadNo3	HeadNo2	HeadNo1	HeadNo0		
//***************************************************************************************
#define	CFC_DRV_HEAD_LBA			0x40


//***************************************************************************************
//***
//***	Local functions
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Function:		uchar ucReadStatusRegister(void)
//***	
//***	Return Type:	uchar
//***
//***	Parameters:		none
//***	
//***	Returns the content of the status register
//***
//***************************************************************************************
uchar ucReadStatusRegister(void)
{
	return (CFC_STATUS_REG);
}

//***************************************************************************************
//***
//***	Function:		uchar ucReadErrorRegister(void)
//***	
//***	Return Type:	uchar
//***
//***	Parameters:		none
//***	
//***	Returns the content of the error register
//***
//***************************************************************************************
uchar ucReadErrorRegister(void)
{
	return (CFC_ERROR_REG);
}

//***************************************************************************************
//***
//***	Function:		void vSetLBAAddress(lba_address ulAddress_)
//***	
//***	Return Type:	void
//***
//***	Parameters:		lba_address pucSectorAddress:		LBA Address
//***	
//***	Write the LBA address into the CF card registers
//***
//***************************************************************************************
void vSetLBAAddress(lba_address lbaAddress_)
{
	while(!(CFC_STATUS_REG & CFC_STATUS_DRDY)){}
	CFC_SECTOR_NUMBER_REG  = lbaAddress_.byte[0];
	CFC_CYLINDER_LOW_REG   = lbaAddress_.byte[1];
	CFC_CYLINDER_HIGH_REG  = lbaAddress_.byte[2];
	CFC_DRIVE_HEAD_REG     = ((lbaAddress_.byte[3] & 0x0F) | 0xA0 | CFC_DRV_HEAD_LBA);
	return;
}


//***************************************************************************************
//***
//***	Global Functions
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Function:		void CFC_vIdentifyDrive(uchar* pucBuffer_)
//***	
//***	Return Type:	void
//***
//***	Parameters:		uchar* pucBuffer_:	Memory location, where the drive information
//***										has to be stored
//***	
//***	Executes the identify drive command and stores the retrieved data
//***	on the given memory location
//***
//***************************************************************************************
void CFC_vIdentifyDrive(uchar* pucBuffer_)
{
	uint uiByteCounter_ = 0;

	CFC_COMMAND_REG = CFC_CMD_IDENTIFY_DRIVE;		//Write identify drive command

	for(uiByteCounter_ = 0; uiByteCounter_ < 512; uiByteCounter_++)
	{
		while(CFC_STATUS_REG != 0x58){}				//Wait for CF card ready
		*pucBuffer_ = CFC_DATA_REG;
		pucBuffer_++;	
	}

}

//***************************************************************************************
//***
//***	Function:		void CFC_vReadSector(lba_address lbaSectorAddress, uchar* pucBuffer)
//***	
//***	Return Type:	void
//***
//***	Parameters:		lba_address lbaSectorAddress:		Address of the Sector that has
//***														to be read
//***					uchar* pucBuffer:					Pointer to the memory location,
//***														where the data has to be stored			
//***	
//***	Function to read one sector of the compact flash card
//***
//***************************************************************************************
void CFC_vReadSector(lba_address lbaSectorAddress_, uchar* pucBuffer_)
{
	uint uiByteCounter_ = 0;

	vSetLBAAddress(lbaSectorAddress_);				//Write LBA address into registers

	CFC_SECTOR_COUNT_REG = 0x01;					//Read one sector
	
	CFC_COMMAND_REG = CFC_CMD_READ_SECTOR;			//Write command for reading sector

	for(uiByteCounter_ = 0; uiByteCounter_ < 512; uiByteCounter_++)
	{
		while(CFC_STATUS_REG != 0x58){}				//Wait for CF card ready
		*pucBuffer_ = CFC_DATA_REG;
		pucBuffer_++;	
	}
	return;	
}

//***************************************************************************************
//***
//***	Function:		void CFC_vInit(void)
//***	
//***	Return Type:	void
//***
//***	Parameters:		none
//***	
//***	Function is used to initialize the comapct flash card. It waites until the
//***	busy flag of the card is cleared
//***
//***************************************************************************************
void CFC_vInit(void)
{
	while(CFC_STATUS_REG & CFC_STATUS_BSY){}		//Wait for CF card ready
}
			
