//***************************************************************************************
//***
//***	Projectname:	Control Interface	
//***	Filename:		System.c
//***	Shortcut:		SYS
//***	Author:			Ralf Hochhausen
//***	Date:			22.10.2003
//***	Acual Version:	V1.0 (Initial Version)
//***	
//***	Description:	Operating system with ms time tick
//***
//***	File-History:	22.10.2003
//***					V1.0	Initial Version
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Includes
//***
//***************************************************************************************
#include <avr\io.h>
#include <avr\signal.h>
#include "TYPES.h"
#include "GlobalDef.h"
#include "ASCInterface.h"
#include "CF_Card_IF.h"

//***************************************************************************************
//***
//***	Local Definitions
//***
//***************************************************************************************
#define SYS_TCNT2CMPVAL		124	//Compare Value for timer/counter 2 (1ms time tick)
#define SYS_EXTRAM_STARTADR	0x2000l

//***************************************************************************************
//***
//***	Local Variables
//***
//***************************************************************************************
static uchar		SYS_bSystemTimeTick;
static uchar*		SYS_pucDataPointer;
static uchar		SYS_ucState;
static lba_address	SYS_ulSectorAddress;
static uchar		SYS_ucCounter;

//***************************************************************************************
//***
//***	Local functions and interrupt service routines
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Function:		SIGNAL(SIG_OUTPUT_COMPARE2)
//***	
//***	Return Type:	void
//***
//***	Parameters:		none
//***	
//***	Interrupt handler of the 1.04ms time tick. Sets trigger Flag
//***
//***************************************************************************************
SIGNAL(SIG_OUTPUT_COMPARE2)
{
	SET(SYS_bSystemTimeTick);
}

//***************************************************************************************
//***
//***	Global Functions
//***
//***************************************************************************************

//***************************************************************************************
//***
//***	Function:		void SYS_vHandleTask(void)
//***	
//***	Return Type:	void
//***
//***	Parameters:		none
//***	
//***	Task handler of the operating system. 1.04ms time tick and time control
//***
//***************************************************************************************
void SYS_vHandleTask(void)
{
	if(SYS_bSystemTimeTick)
	{
		CLR(SYS_bSystemTimeTick);

		if(!(--SYS_ucCounter))
		{
			SYS_ucCounter = 10;
			switch(SYS_ucState)
			{
				case 0:
				{
					SYS_pucDataPointer = (uchar*)SYS_EXTRAM_STARTADR;
					CFC_vReadSector(SYS_ulSectorAddress, SYS_pucDataPointer);
					SYS_ucState = 1;
					SYS_pucDataPointer = (uchar*)SYS_EXTRAM_STARTADR;
					break;
				}	
				case 1:
				{	
					if(ASC_ucIsTransmitBufferFree())
					{
						ASC_vTransmitByte(*SYS_pucDataPointer);
						SYS_pucDataPointer++;
						if(SYS_pucDataPointer == (uchar*)(SYS_EXTRAM_STARTADR + 0x200))
						{
							SYS_ucState = 2;
						}
					}
					break;
				}
				case 2:
				{
					
					if(ASC_ucIsByteReceived())
					{
						if(ASC_ucGetReceivedByte() == 'n')
						{	
							SYS_ulSectorAddress.longbyte++;
							SYS_pucDataPointer = (uchar*)SYS_EXTRAM_STARTADR;
							SYS_ucState = 0;
							SYS_ucCounter = 10;
						}
					}
					break;
				}
			}			
		}
	}
}


//***************************************************************************************
//***
//***	Function:		void SYS_vInit(void)
//***	
//***	Return Type:	void
//***
//***	Parameters:		none
//***	
//***	Intialization of the 1ms time tick (exact 1.04ms)
//***
//***************************************************************************************
void SYS_vInit(void)
{	
	//Initialization of the module variables
	CLR(SYS_bSystemTimeTick);
	SYS_pucDataPointer = (uchar*)SYS_EXTRAM_STARTADR;
	SYS_ucState = 0;
	SYS_ulSectorAddress.longbyte = 0x000007D8l;
	SYS_ucCounter = 10;

	//Initialization of the external RAM area
	XMCRA = 0x00;
	XMCRB = 0x00;
	MCUCR = 0x80;

	//Initialization of the 1.04ms time tick with timer 3
	TCCR2 = 0x0B;				//Configure timer/counter2 for CTC operation
								//Prescaler is set to 64
								//Output is disabled

	TIFR  &= ~(1 << TOV2);		//Clear overflow flag
	TIFR  &= ~(1 << OCF2);		//Clear compare match flag
	
	TIMSK |=  (1 << OCIE2);		//Enable compare interrupt
	TIMSK &= ~(1 << TOIE2);		//Disable overflow interrupt
	
	OCR2 = SYS_TCNT2CMPVAL;		//Set compare value
	TCNT2 = 0x00;
}
